/* 
 * zdbm_io.c --
 * Chris Hyde
 * created: 8/92
 *
 * Modified: Dean Collins Sat Feb 12 18:53:50 1994
 * Fixed several typos.
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Sat Apr 09 23:50:06 1994
 * Added an #ifdef for SCO.
 *
 * Modified: Chris Hyde 1/10/95
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Added predicate functions for use by next_problem() and others,
 *      see note towards end of file.
 * Changed date format in new_prid(), was %j%Y, now %Y%j, thanks to
 *      Dan Bechard.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 *
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 * 
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */


#ifndef SYSV

#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <limits.h>
#include <stdarg.h>
#if defined(linux) || defined(SCO) || defined(sco)
#include <unistd.h>
#endif /* linux */
/* end not system 5 */

#else /* it is system 5 */

#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#if defined(AIXV3) || defined(OSF1)
#include <sys/time.h>
#else
#include <time.h>
#endif /* AIXV3 */
#include <errno.h>
#include <stdarg.h>

#endif /* system 5 */

#include "zdbm.h"

#ifdef DEBUG
 int open_closed=0,
     locked_unlocked=0;
#endif

/* The top of the database tree. (Formerly PTSROOT.)  Added DC 2/23/94 */
char PtsRoot[COMPLETEPATHLEN]=PTSROOT ;

 
int exists(char *root, char *path, char *fname, char *extension)
{
 static char tpath[COMPLETEPATHLEN];

 zdebug("\nIn exists()\n");
 zdebug2("\tPath - %s%s\n", root, path);
 zdebug1("\tFile name to check - %s\n", fname);
 zdebug1("\tFile extension - %s\n", extension);

 if ((fname[0]=='\0') && (extension[0]=='\0'))
    /* no file name or extension so don't add a '/' */
    sprintf(tpath, "%s%s", root, path);
 else
    /* we're going to need the '/' */
    sprintf(tpath, "%s%s/%s%s", root, path, fname, extension);
 if (access(tpath, F_OK)==ERROR) 
 {
    zdebug1("exists() - the file %s does NOT exist\n", tpath);

    db_errorno=errno;
    return(FALSE);
 }
 else
 {
    zdebug1("exists() - the file %s does exist\n", tpath);

    return(TRUE);
 }
} /* end exists */

 
int append(char *path, char *from1, char *from2, char *to1, char *to2)
{
 ZFILE *temp1, *temp2;
 char file1[COMPLETEPATHLEN],
      file2[COMPLETEPATHLEN],
      buffer[STD_BLK];
 long bottom;

 sprintf(file1, "%s/%s", path, from1);
 sprintf(file2, "%s/%s", path, to1);

 zdebug("\nIn append()\n");
 zdebug4("\tAppending %s%s\nTo %s%s\n", file1, from2, file2, to2);

 if ((temp1=zopen(file1, from2, "r", REP))==NULL)
    return(FAIL); /* could not open "from" file */
 if ((temp2=zopen(file2, to2, "a", REP))==NULL)
 {      /* could not open "to" file */
    zclose(temp1);
    return(FAIL);
 }

bottom=ftell(temp2);

 if (zlock(temp2)==FAIL)
 {      /* the lock failed */
    zclose(temp1);
    zclose(temp2);
    return(FAIL);
 }

 zdebug("append() - performing append - ");

 while (zread(buffer, STD_BLK, 1, temp1)==1)
 {      /* read from the "from" file until the end */

    zdebug("?");

    if (zwrite(buffer, STD_BLK, 1, temp2)<1)
       break; /* there was an error writing to the "to" file */
 }

 zdebug("\n");

 zunlock(temp2);

 if (zerror(temp1) || zerror(temp2))
 {      /* there was an error either reading or writing */
    db_errorno=errno;
    zclose(temp1);
    zclose(temp2);
    trunc_file(file2, to2, bottom);
    return(FAIL);
 }
 else
 {
    zclose(temp1);
    zclose(temp2);
    return(SUCESS);
 }
} /* end append() */

 
ZFILE *zopen(char *leaf, char *extension, char *perm, int index)
{
   char path[COMPLETEPATHLEN],
        *temp;
   int fd, mode;

   if (index==INDEX)  /* want to open an index file */
   {
      if ((temp=(char*)rindex(leaf, '/'))==NULL) /* find the name of the leaf */
         temp=leaf; /* name of the index is the name fo the leaf */
      else
         temp++; /* want the char after the '/' */
      sprintf(path, "%s%s/%s%s", PtsRoot, leaf, temp, extension);
   }
   else /* not opening an index file */
      sprintf(path, "%s%s%s", PtsRoot, leaf, extension);

   zdebug("\nIn zopen()\n");
   zdebug1("\tFile To Open %s\n", path);
   zdebug1("\tAcess method = %s\n", perm);

   if (perm[1]=='+') /* open for both reading and writing */
      mode=O_RDWR;
   else
   {
      if (perm[0]=='a') /* open for appending */
         mode=O_APPEND | O_WRONLY;
      else
         if (perm[0]=='r') /* open for read only */
            mode=O_RDONLY;
         else
            if (perm[0]=='w') /* open for writing only */
               mode=O_WRONLY;
   } /* end else */
   if ((fd=open(path, mode, FILE_PERMS))<0)
   {    /* error opening file */

      zdebug1("zopen() - could not open %s\n", path);
      zdebug1("\terror number %d\n", errno);

      db_errorno=errno;
      return(NULL);
   }
   else
   {
      zdebug1("zopen() - opened %s\n", path);
      zdebug1("\tOpens - Closes = %d\n", ++open_closed);

      return(fdopen(fd, perm));
   } /* end else */
} /* end zopen */

 
int zclose(ZFILE *db_pointer)
{
int status;

 zdebug1("In zclose()\n\tOpens - Closes = %d\n", --open_closed);

 status=(fclose(db_pointer));

 zdebug1("\tstatus=%d\n",status) ;

 return(status) ;
} /* end zclose */

 
long file_size(ZFILE *fp, long record_size)
{
  long size;
  struct stat statbuf; /* space to hold info about the file */

  zdebug("\nIn file_size()\n");
  zdebug1("\trecord_size = %d\n", record_size);

  if (fstat(fileno(fp), &statbuf)==ERROR) /* get the file status */
  { /* error getting file status */
     db_errorno=errno;
     return(ERROR);
  }
  size=(long)statbuf.st_size; /* size of file in bytes */
  return(size/record_size);
} /* end file_size */

 
int zlock(ZFILE *fp)
{
  int fd; /* file descriptor */

  zdebug("\nIn zlock()\n");

  fd=fileno(fp);

  zdebug3("\tfd = %d, fp = %d, zerror(fp) = %d\n",fd, fp, zerror(fp)) ;

#ifndef SYSV /* bsd specific file locking */

  if (flock(fd, LOCK_EX | LOCK_NB)<0)
  {  /* lock failed */
     zdebug1("\terrno = %d\n",errno) ;

     if (errno==EWOULDBLOCK)
        db_errorno=ETXTBSY; /* used for consistancy */
     else
        db_errorno=errno;
     return(FAIL);
  }
  else /* lock succeeded */
  {
     zdebug1("\tLocked - Unlocked = %d\n", ++locked_unlocked);

     return(SUCESS);
  } /* end else */
 /* end of BSD specific file locking */

#else /* it is system 5 specific file locking */

  if (lockf(fd, F_TLOCK, 0)==ERROR)
  {  /* lock failed */
     if (errno==EACCES) /* file is already locked */
        db_errorno=ETXTBSY; /* used for consistancy */
     else
        db_errorno=errno;
     return(FAIL);
  }
  else /* lock succeeded */
  {
     zdebug1("\tLocked - Unlocked = %d\n", ++locked_unlocked);

     return(SUCESS);
  } /* end else */
#endif /* end of system 5 specific file locking */
} /* end lock_file */

 
int zunlock(ZFILE *fp)
{
  int fd; /* file descriptor */

  zdebug("\nIn zunlock()\n");

  fd=fileno(fp);

#ifndef SYSV /* begin of BSD specific file unlocking */
  if (flock(fd, LOCK_UN)<0)
  {  /* unlocking failed */
     db_errorno=errno;
     return(FAIL);
  }
  else /* unlocking suceeded */
  {
     zdebug1("\tLocks - Unlocks = %d\n", --locked_unlocked);

     return(SUCESS);
  } /* end else */
 /* end of system 5 specfic file unlocking */

#else /* begin of system 5 specific file unlocking */
  if (lockf(fd,F_ULOCK, 0)==ERROR)
  {  /* unlocking failed */
     db_errorno=errno;
     return(FAIL);
  }
  else /* unlocking suceeded */
  {
     zdebug1("\tLocks - Unlocks = %d\n", --locked_unlocked);

     return(SUCESS);
  } /* end else */
#endif /* end of system 5 specific file locking */
} /* end zunlock */

 
int make_branch(char *db_path)
{
  char path[COMPLETEPATHLEN];
  char *cp, 
       tpath[DBPATHLEN]; /* since strtok plays with the string passed to it
              * we need to place the db_path into another string
              * so the original won't be affected */
  ZFILE *dummy;

  strcpy(path, PtsRoot); /* copy the database root into the path */
  nl_strncpy(tpath, db_path, DBPATHLEN);

  zdebug("\nIn make_branch()\n");
  zdebug1("\tdb_path = %s\n", db_path);

  cp=strtok(tpath, "/"); /* get the first part of the path */
  strcat(path, cp); /* place the first part of the new branch in path */

  while(cp!=NULL) /* while there are more partial paths */
  {
     zdebug1("make_branch() - partial path = %s\n", path);

     if (exists("", path, "", "")==FAIL)
 /* if the partial path does not exist, then create it with the permissions
  * as defined by DIR_PERMS (in zdbm.h)
  */
        if (mkdir(path, DIR_PERMS)==ERROR)
        { /* error creating directory */
           db_errorno=errno;
           return(FAIL);
        }
     cp=strtok(NULL, "/"); /* get next portion of pathlist */
     if (cp!=NULL)
     {
        strcat(path, "/");
        strcat(path, cp);
     }
  } /* end while */

/* we made the directory path, now make the new index file */

  cp=(char*)rindex(path, '/'); /* get the name of the last directory in the list */
  if (cp==NULL) /* the path has only one directory */
     cp=db_path;
  else
     cp++; /* we want the char after the '/' */
/* create the index file, since create_file() leaves the created file open
 * we have to close it before exiting from the module.
 */
  if ((dummy=create_file(db_path, cp, ".index", FILE_PERMS))==NULL)
     /* error occured durring the creation */
     return(FAIL);
  else
  {
     zclose(dummy);
     return(SUCESS);
  }
} /* end make_branch */

 
ZFILE *create_file(char *path, char *fname, char *extension, int perm)
{
 char tpath[COMPLETEPATHLEN];
 int fd;

 zdebug2("\nIn create_file()\n\tpath = %s%s\n", PtsRoot, path);
 zdebug3("\tfname = %s\n\textension = %s\n\tperm = %d\n", fname,
         extension, perm);

 sprintf(tpath, "%s%s/%s%s", PtsRoot, path, fname, extension);
 if ((fd=open(tpath, O_WRONLY | O_CREAT | O_EXCL, perm))<0)
 {  /* error creating file */
    db_errorno=errno;
    return(NULL);
 }
 zdebug1("create_file() - created %s\n", tpath);
 zdebug1("\tOpened - Closed = %d\n", ++open_closed);

 return(fdopen(fd, "w"));
} /* end create_file */

 
int remove_file(char *path, char *fname, char *extension)
{
 char tpath[COMPLETEPATHLEN];

 zdebug("\nIn remove_file()\n");
 zdebug4("\tpath = %s%s\n\tfname = %s\n\textension = %s\n", PtsRoot, 
        path, fname, extension);

 sprintf(tpath, "%s%s/%s%s", PtsRoot, path, fname, extension);
 if (unlink(tpath)==ERROR)
 {
    db_errorno=errno;
    return(FAIL);
 }
 else
    return(SUCESS);
} /* end remove_file */

 
char *new_prid(long size)
{
 struct timeval timeclock;
 struct timezone tz;
 static char prid[MAXPRID];

 zdebug("\nIn new_prid()\n");
 zdebug1("\tsize = %d.\n", size);

 if (gettimeofday(&timeclock, &tz)==ERROR)
 {
    db_errorno=errno;
    return(NULL);
 }
 if (strftime(prid, MAXPRID, "%Y%j", localtime(&(timeclock.tv_sec)))==FAIL)
 {
    db_errorno=STRINGERROR;
    return(NULL);
 }
 sprintf(prid, "%s%05d", prid, size);
 return(prid);
} /* end new_prid */

 
int trunc_file(char *path, char *extension, long flen)
{
 char tpath[COMPLETEPATHLEN];

 zdebug("\nIn trunc_file()\n");
 zdebug3("\tPath = %s\n\tExtension = %s\n\tAmount = %d\n", path,
                extension, flen);

 sprintf(tpath, "%s%s", path, extension);
 if (truncate(tpath, flen)==ERROR)
 {  /* error truncating file */
    db_errorno=errno;
    return(FAIL);
 }
 else
    return(SUCESS);
} /* end trunc_file */

 
int rename_file(char *from_path, char *from1, char *from2, 
        char *to_path, char *to1, char *to2)
{
 char tpath1[COMPLETEPATHLEN],
      tpath2[COMPLETEPATHLEN];

 zdebug("\nIn rename_file().\n");
 zdebug6("\tRenaming %s%s%s\n to \n\t%s%s%s\n", from_path, from1, from2, 
 to_path, to1, to2);

 sprintf(tpath1, "%s%s/%s%s", PtsRoot, from_path, from1, from2);
 sprintf(tpath2, "%s%s/%s%s", PtsRoot, to_path, to1, to2);

 if (rename(tpath1, tpath2)==ERROR)
 {  /* error renameing file */
    db_errorno=errno;
    return(FAIL);
 }
 return(SUCESS);
} /* end rename_file */

 
int copy_file(char *from1, char *from2, char *from3,
              char *to1, char *to2, char *to3)
{
 char from[FNAMELEN],
      buffer[STD_BLK];
 ZFILE *infile,
      *outfile;
 int in;

 zdebug("\nIn copy_file()\n");
 zdebug3("\tFrom1 - %s\n\tFrom2 - %s\n\tFrom3 - %s\n", from1, from2, from3);
 zdebug3("\tTo1 - %s\n\tTo2 - %s\n\tTo3 - %s\n", to1, to2, to3);

 sprintf(from, "/%s%s", from2, from3);
 if ((infile=zopen(from1, from, "r", REP))==NULL)
    return(FAIL);
 if ((outfile=create_file(to1, to2, to3, FILE_PERMS))==NULL)
 {
    zclose(infile);
    return(FAIL);
 }

 zdebug("Copying File");

 while ((in=zread(buffer, sizeof(char), STD_BLK, infile)) > 0)
 {
    zdebug("!") ;

    if (zwrite(buffer, sizeof(char), in, outfile)<in)
       break;
 }

 zdebug("\n") ;

 if (zerror(infile) || zerror(outfile))
 {      /* error reading or writing */
    db_errorno=errno;
    zclose(infile);
    zclose(outfile);
    remove_file(to1, to2, to3);
    return(FAIL);
 }
 zclose(infile);
 zclose(outfile);
 return(SUCESS);
} /* end copy_file */

 
int write_log(char *path, char *prid, char *new_log)
{
 char tname[FNAMELEN];
 ZFILE *temp;

 zdebug("\nIn write_log()\n");
 zdebug3("\tLog path - %s%s/%s.log\n", PtsRoot, path, prid);

 nl_strncpy(tname, TEMPORARY, FNAMELEN);
 if ((temp=create_file(path, (char *)mktemp(tname), "", FILE_PERMS))
            ==NULL)
    return(FAIL); /* the creation failed */

 zdebug("write_log() - created new log, now writing it out\n");

/* we don't want to write the terminating null out to the file.  We will
 * add it back on we we read the log in
 */
 if (zwrite(new_log, sizeof(char), strlen(new_log), temp)<strlen(new_log))
 {      /* the write failed */
    db_errorno=errno;
    zclose(temp);
    return(FAIL);
 }
 else   /* the write suceeded */
 {
    zdebug("write_log() - wrote file for new log\n");

    zclose(temp);

    zdebug("write_log() - renaming temp file to log file\n");

    if (rename_file(path, tname, "", path, prid, ".log")==FAIL)
    {   /* couldn't rename the new problem log */
       remove_file(path, tname, "");
       return(FAIL);
    }
    else /* rename suceeded */
       return(SUCESS);
 } /* end else write suceeded */
} /* end write_log() */

 
char *zdbm_strerror(int zerror)
{
 int index;
 static char *error_messages[]=
    {
     "BADTYPE - a bad parameter was passed to a database function.\n",
     "NOTFOUND - the problem record asked for was not found in the leaf given.\n",
     "DOUBLEREP - a reporter has tried to report a problem twice.\n",
     "DELETED - the problem record asked for has been deleted.\n",
     "ISSOLVED - the problem record given is solved.\n",
     "ISUNSOLVED - the problem record given is unsloved.\n",
     "NOTOWNER - the person requesting the end of an edit session is not the one who started it.\n",
     "NOREP - the problem record is missing either the old reporters list, or the reporters list.\n",
     "EXISTS - the problem requested has already been requested for reopening.\n",
     "LINKERR - permission to delete a problem record that is an original link has not been given.\n",
     "REMOVEERROR - an error occured while a record was being removed.\n",
     "STRINGERROR - an error occured during a string operation.\n",
     "ROLLEDOVER - there are no problem records of the desired type in the database leaf given.\n",
     "SAMELEAF - a request was made to link or move a problem to a leaf in which it already resides.\n",
     "NOTINEDIT - a request was made to unlock a problem from edit that was not being edited\n"
    }; /* end of error_messages */

 zdebug("In zdbm_strerror()\n");
 zdebug1("\tzdbm error number = %d\n", zerror);

 if (zerror<BEGIN_OF_ZDBM_ERROR_CODES) 
    return((char*)strerror(zerror)); /* the error code is for a lib function */
 else
    if (zerror>END_ZDBM_ERROR_CODES) /* zerror is above the range for zdbm */
    {
       db_errorno=BADTYPE;
       return("Unknown ZDBM error code.\n");
    } /* end else zerror above range */
    else /* zerror is in the range of zdbm codes */
    {
       index=zerror-BEGIN_OF_ZDBM_ERROR_CODES;
       return(error_messages[index]);
    } /* end else  zerror is in range */
} /* end of zdbm_strerror() */

/* the following functions are predicates for use by other functions, although
 * they are designed for use by next_problem() which takes a predicate
 * function as a parameter to use while searching for the desired problem
 * type.
 */

int is_solved(index_record *index, char *dummy)
{
 return(index->sf==SOLVED);
} /* end of is_solved() */

int is_unsolved(index_record *index, char *dummy)
{
 return(index->sf==UNSOLVED);
} /* end of is_unsolved() */

int eq_priority(index_record *index, char *priority)
{
 return(index->priority==atoi(priority));
} /* end of eq_priority() */

int ge_priority(index_record *index, char *priority)
{
 return(index->priority==atoi(priority));
} /* end of ge_priority() */

int le_priority(index_record *index, char *priority)
{
 return(index->priority==atoi(priority));
} /* end of le_priority() */

char *get_date(index_record *index)
{
 static char date[8];
 
 strncpy(date, get_prid(index), 7); /* first 7 chars are date */
 date[7]='\0'; /* append NULL just to be sure */
 return(date);
} /* end of get_date() */

int eq_date(index_record *index, char *date)
{
 return(strcmp(get_date(index), date)==0);
} /* end of eq_date() */

int ge_date(index_record *index, char *date)
{
 return(strcmp(get_date(index), date)>=0); /* return problems after date
                                            * given */
} /* end of ge_date() */

int le_date(index_record *index, char *date)
{
 return(strcmp(get_date(index), date)<=0); /* return problems before date
                                            * given */
} /* end of le_date() */

/* end of zdbm_io.c */
