/*
 * reopen_problem.c -- 
 * Chris Hyde
 * created: 
 * 
 * Modified: Dean Collins Sat Feb 12 18:50:53 1994
 * Fixed several typos.
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Sat Jan 14 12:34:10 PST 1995
 * Changed DEBUG output to go to stderr.
 */

/*
 * Copyright (c) 1992,1993,1994 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "zdbm.h"


int reopen_request(char *db_path, char *prid, char *reasons,
		   reporter_record *rep_rec)
{
  index_record index;
  long num_rec;
  FILE *temp;
  static char path[DBPATHLEN];
  static char tprid[MAXPRID];

#ifdef DEBUG
 fprintf(stderr,"\nIn reopen_request().\n");
 fprintf(stderr,"\tPath of problem to reopen - %s/%s\n", db_path, prid);
#endif

 if (!exists(PtsRoot, REOPENDIR, "", "")) /* does the reopen directory */
 {		/* exist ? */
    make_branch(REOPENDIR); /* no so create it */
    if ((temp=open_file(REOPENDIR, ".index", "w", INDEX))==NULL)
       return(FAIL);
/* after creating the reopen directory, we need to write out the number of
 * records that are in the directory, 0 in this case. */
    num_rec=0;
    if (fwrite(&num_rec, sizeof(long), 1, temp)<1)
    {	/* the write failed */
       db_errorno=errno;
       close_file(temp);
       return(FAIL);
    }
    close_file(temp);
#ifdef DEBUG
 fprintf(stderr,"reopen_request() - created reopen directory.\n");
#endif
 }

#ifdef DEBUG
 fprintf(stderr,"reopen_request() - found reopen directroy.\n");
#endif

 if ((temp=open_file(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL); /* the open failed */

 if (find_problem(temp, prid, &index)==NULL)
 {  /* error searching for problem */
    close_file(temp);
    return(FAIL);
 }

#ifdef DEBUG
 fprintf(stderr,"reopen_request() - found problem to reopen.\n");
#endif

 if (index.sf!=SOLVED) /* can only reopen SOLVED problems */
 {
    db_errorno=ISUNSOLVED;
    close_file(temp);
    return(FAIL);
 }
 if (index.mrd==GONE) /* can't reopen a deleted problem */
 {
    db_errorno=DELETED;
    close_file(temp);
    return(FAIL);
 }
 if (index.mrd==LINKED) /* the problem was linked to the leaf given */
 {
#ifdef DEBUG
 fprintf(stderr,"reopen_request() - problem to reopen is a link.\n");
#endif
    strcpy(path, index.sd); /* path is used to access the records */
    strcpy(tprid, index.ln_prid); /* tprid is used to access records */
 }
 else /* the problem resides in the leaf given */
 {
    strcpy(path, db_path);
    strcpy(tprid, prid);
 }
 close_file(temp);

 if ((temp=open_file(REOPENDIR, ".index", "r+", INDEX))==NULL)
    return(FAIL); /* the open failed */

 if (lock_file(temp)==0) /* lock the reopen index */
 {  /* the lock failed */
    close_file(temp);
    return(FAIL);
 }
 if (fread(&num_rec, sizeof(long), 1, temp)==NULL) /* get the number of */
 {   /* records in the index, report failure if read fails */
    db_errorno=errno;
    unlock_file(temp);
    close_file(temp);
    return(FAIL);
 }
#ifdef DEBUG
fprintf(stderr,"reopen_request() - there have been %d reopened requests made.\n", num_rec);
#endif

 if (double_problem(temp, path, tprid)) /* check for duplicate reports */
 { /* the problem has already been reported */
    db_errorno=EXISTS;
    unlock_file(temp);
    close_file(temp);
    return(FAIL);
 }
#ifdef DEBUG
 fprintf(stderr,"reopen_request() - creating reopened index record.\n");
#endif

 index.mrd=REOPENED;
 index.sf=UNSOLVED;
 index.ef=FALSE;
 index.account_name[0]=NULL;
 index.node_name[0]=NULL;
 strcpy(index.ln_prid, tprid);
 strcpy(index.sd, path);
 strcpy(index.prid, new_prid(num_rec));
 if (index.prid[0]==NULL)
 {  /* error creating id */
    unlock_file(temp);
    close_file(temp);
    return(FAIL);
 }
#ifdef DEBUG
 fprintf(stderr,"reopen_request() - created record, now appending it.\n");
#endif

 fseek(temp, 0L, SEEK_END); /* move to the end of the index for appending */

 if (fwrite(&index, sizeof(index_record), 1, temp)<1)
 { /* error writing to the index */
    db_errorno=errno;
    unlock_file(temp);
    close_file(temp);
    return(FAIL);
 }
#ifdef DEBUG
fprintf(stderr,"reopen_request() - wrote new record, now updating record count\n");
#endif

 /* now go back to the beginning and write out a new record count */
 rewind(temp);
 if (++num_rec>MAXREC) /* we've maxed out the # of possible records */
    num_rec=0; /* hopefully the low numbers have been removed */

 fwrite(&num_rec, sizeof(long), 1, temp);
 unlock_file(temp);
 close_file(temp);

#ifdef DEBUG
 fprintf(stderr,"reopen_request() - updated record count.\n");
 fprintf(stderr,"reopen_request() - now creating reasons log.\n");
#endif

 if (write_log(REOPENDIR, index.prid, reasons)==FAIL)
 { /* error creating reasons log */
    delete_record(index.prid);
    return(FAIL);
 }

#ifdef DEBUG
 fprintf(stderr,"reopen_request() - wrote reasons log.\n");
 fprintf(stderr,"reopen_request() - creating reporters file.\n");
#endif
 
 if ((temp=create_file(REOPENDIR, index.prid, ".rep", FILE_PERMS))==NULL)
 { /* error creating reporters file */
    delete_record(index.prid);
    return(FAIL);
 }
 if (fwrite(rep_rec, sizeof(reporter_record), 1, temp)<1)
 {	/* error writing reporter record to file */
    db_errorno=errno;
    delete_record(index.prid);
    close_file(temp);
    return(FAIL);
 }
#ifdef DEBUG
 fprintf(stderr,"reopen_request() - created reporter file.\n");
#endif

 close_file(temp);
 return(SUCESS);
} /* end reopen_request */


int double_problem(FILE *db_pointer, char *db_path, char *prid)
{
 index_record index;
 int no_doubles=TRUE;

#ifdef DEBUG
 fprintf(stderr,"\nIn double_problem().\n");
 fprintf(stderr,"\tProblem to search for - %s/%s.\n", db_path, prid);
 fprintf(stderr,"\tSearching");
#endif

 while ((no_doubles) &&
		(fread(&index, sizeof(index_record), 1, db_pointer)!=NULL))
 {
#ifdef DEBUG
 fprintf(stderr,"?");
#endif

    if ((strcmp(index.sd, db_path)==0) &&
		(strcmp(index.ln_prid, prid)==0))
       no_doubles=FALSE;
 }
#ifdef DEBUG
 if (no_doubles)
    fprintf(stderr,"\ndouble_problem() - no doubles found.\n");
 else
    fprintf(stderr,"\ndouble_problem() - found a double.\n");
#endif
 if (ferror(db_pointer))
 {	/* there was an i/o error */
    db_errorno=errno;
    return(ERROR);
 }
 else
    return(!(no_doubles));
} /* end double_problem */


reopened_record *view_reopen(char *reopen_prid)
{
 static reopened_record reop;
 static long location=sizeof(long);
 static char db_path[DBPATHLEN];
 index_record index;
 int size;
 char *log;
 FILE *temp;

#ifdef DEBUG
 fprintf(stderr,"\nIn view_reopen().\n");
#endif

/* Open the reopend problem index, then seek to the last location looked
 * at.  If that happens to be the end of file then rewind the pointer and
 * seek past the record counter at the begining of the file.
 */

 if ((temp=open_file(REOPENDIR,".index", "r", INDEX))==NULL)
    return(NULL); /* the open failed */

 if (reopen_prid!=NULL) /* does the caller wants a particular problem */
 {
#ifdef DEBUG
 fprintf(stderr,"view_reopen() - looking for a particular problem - %s\n", reopen_prid);
#endif

    if (find_reopen(temp, &index, reopen_prid)==NULL)
    {
       close_file(temp);
       return(NULL);
    } /* end if */
 }
 else
 {
#ifdef DEBUG
 fprintf(stderr,"view_reopen() - seeking past last record viewed.\n");
#endif

    fseek(temp, location, SEEK_SET);

#ifdef DEBUG
 fprintf(stderr,"view_reopen() - reading the next record.\n");
#endif

/* Now read the index record */

/* the reopen directory is a virtual circular list, when the end is reached
 * you start at the beginning.  The following goto makes the code more
 * "elegant".
 */

    kluge:
    if (fread(&index, sizeof(index_record), 1, temp)==NULL)
    {  /* error reading from file */
       if (feof(temp))
       {
          rewind(temp);
          location=sizeof(long); /* seek past the record counter */
          fseek(temp, location, SEEK_SET);
          goto kluge;
       }
       else /* it's an error */
       {
          db_errorno=errno;
          close_file(temp);
          return(NULL);
       } /* end error */
    } /* end read */
#ifdef DEBUG
 fprintf(stderr,"view_reopen() - index record read.\n");
 print_index(&index);
#endif

/* increment the location pointer */
    location+=sizeof(index_record);
 } /* end else the user didn't want a particular problem */

 close_file(temp);

/* now create the reopened record */
 strcpy(reop.prid, index.prid); /* the problem id of the reopened problem */
 strcpy(reop.db_path, index.sd); /* the path to the leaf containing the problem
				  * to be reopened */

#ifdef DEBUG
 fprintf(stderr,"view_reopen() - creating reopened record.\n");
 fprintf(stderr,"\tgetting reasons log.\n");
#endif

 sprintf(db_path, "%s/%s", REOPENDIR, reop.prid);
 if ((temp=open_file(db_path, ".log", "r", REP))==NULL)
 {	/* error opening reasons log */
    return(NULL);
 }
 size=file_size(temp, sizeof(char));
 if (reop.reasons_log!=NULL)
    free(reop.reasons_log);
 if ((log=(char *)calloc(size, sizeof(char)))==NULL)
 {	/* error allocating space for reasons log */
    db_errorno=errno;
    close_file(temp);
    return(NULL);
 }
#ifdef DEBUG
 fprintf(stderr,"view_reopen() - allocated space for reasons log.\n");
#endif

 if (fread(log, sizeof(char), size, temp)==NULL)
 {	/* error reading reasons log */
    db_errorno=errno;
    close_file(temp);
    return(NULL);
 }
 reop.reasons_log=log;
 close_file(temp);
#ifdef DEBUG
 fprintf(stderr,"view_reopen() - read reasons log.\n");
 fprintf(stderr,"\tgetting reporter record.\n");
#endif

 sprintf(db_path, "%s/%s", REOPENDIR, reop.prid);
 if ((temp=open_file(db_path, ".rep", "r", REP))==NULL)
    return(NULL);  /* the open failed */
 if (fread(&reop.rep_rec, sizeof(reporter_record), 1, temp)==NULL)
 {  /* error reading file */
    db_errorno=errno;
    close_file(temp);
    return(NULL);
 }
 close_file(temp);

#ifdef DEBUG
 fprintf(stderr,"view_reopen() - got reporter record.\n");
 fprintf(stderr,"\tgetting problem record to reopen.\n");
#endif

/* Now get the original problem */
 if ((reop.pro_rec=get_problem(index.sd, index.ln_prid, PARTICULAR, 
				NULL, LINKS))==NULL)
    return(NULL); /* there was an error getting the problem record */
 else 
 {
    reop.pro_rec->status=REOPENED; /* the problem is reopened */
    reop.pro_rec->reop_rec=&reop; /* points back to the reopened record */

#ifdef DEBUG
 fprintf(stderr,"view_reopen() - changed problem status to REOPENED\n");
 fprintf(stderr,"\tstatus - %d\n", reop.pro_rec->status);
#endif

    return(&reop);
 }
} /* end view_reopen */


char *confirm_reopen(char *reop_prid, reporter_record *reop_rec,
		     char *sys_string, char *reasons)
{
 index_record index;
 static index_record org_index;
 FILE *temp,
      *org_temp,
      *temp_two;
 char not_found=TRUE;
 static char path[COMPLETEPATHLEN];
 long count,
      bottom;

#ifdef DEBUG
 fprintf(stderr,"\nIn confirm_reopen().\n");
 fprintf(stderr,"\tPRID to confirm - %s\n\tSysop - %s\n", reop_prid,
		sys_string);
#endif

 if ((temp=open_file(REOPENDIR, ".index", "r+", INDEX))==NULL)
    return(NULL); /* the open failed */
 else
    if (lock_file(temp)==FAIL) /* lock the reopen index file */
    {  /* the lock failed */
       close_file(temp);
       return(NULL);
    }

 if (find_reopen(temp, &index, reop_prid)==NULL)
 {
    unlock_file(temp);
    close_file(temp);
    return(NULL);
 }

#ifdef DEBUG
 fprintf(stderr,"\nconfirm_reopen() - found desired problem.\n");
 fprintf(stderr,"\tnow find original problem record.\n");
#endif

/* Open the index for the leaf that holds the problem to be ropened */
 if ((org_temp=open_file(index.sd, ".index", "r+", INDEX))==NULL)
 {  /* could not open the index file */
    unlock_file(temp);
    close_file(temp);
    return(NULL);
 }

/* Now find the problem to be reopend */
 if (find_problem(org_temp, index.ln_prid, &org_index)==NULL)
 {  /* error looking for the problem */
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    return(NULL);
 }
#ifdef DEBUG
 fprintf(stderr,"confirm_reopen() - found original problem.\n");
 fprintf(stderr,"\tnow creating the new reporter file.\n");
#endif

/* create the new reporters list and add the new reporters record */
 if ((temp_two=create_file(index.sd, org_index.prid, ".rep",
			   FILE_PERMS))==NULL)
 {  /* error creating new file */
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    return(NULL);
 }
 if (fwrite(reop_rec, sizeof(reporter_record), 1, temp_two)<1)
 {  /* error writing to the file */
    db_errorno=errno;
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    close_file(temp_two);
    remove_file(index.sd, org_index.prid, ".rep");
    return(NULL);
 }
 else
    close_file(temp_two);

#ifdef DEBUG
 fprintf(stderr,"confirm_reopen() - created new reporter file.\n");
 fprintf(stderr,"\tappending sysop string to log.\n");
#endif

/* Now add the name of the person confirming the reopen (sys_string) and
 * the reasons for the reopening to the problem log, but first get the position
 * of the current end of file.  This way if an error occurs before completing
 * the reopening the additions can be removed (at least in theory).
 */
 sprintf(path, "%s/%s", index.sd, org_index.prid);
 if ((temp_two=open_file(path, ".log", "a", REP))==NULL)
 {  /* error opening log file */
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    remove_file(index.sd, org_index.prid, ".rep");
    return(NULL);
 }
 bottom=ftell(temp_two); /* get the position of the end of the file */

 fwrite(sys_string, sizeof(char), strlen(sys_string), temp_two);

#ifdef DEBUG
 fprintf(stderr,"confirm_reopen() - wrote sysop string.\n");
 fprintf(stderr,"\tappending reasons.\n");
#endif

 if (fwrite(reasons, sizeof(char), strlen(reasons)+1, temp_two)
		<=strlen(reasons))
 {  /* error writing to the log file */
    db_errorno=errno;
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    close_file(temp_two);
    trunc_file(path, ".log", bottom); /* cut off the extra stuff */
    remove_file(index.sd, org_index.prid, ".rep");
    return(NULL);
 }
 else
    close_file(temp_two);

#ifdef DEBUG
 fprintf(stderr,"confirm_reopen() - appended reasons.\n");
 fprintf(stderr,"\tnow update the index entry.\n");
#endif

/* now lock the index file and write out a new index entry for the reopened
 * problem.
 */

 if (lock_file(org_temp)==FAIL)
 {  /* the locking failed */
    unlock_file(temp);
    close_file(temp);
    close_file(org_temp);
    trunc_file(path, ".log", bottom);
    remove_file(index.sd, org_index.prid, ".rep");
    return(NULL);
 }

 org_index.sf=UNSOLVED;
 fseek(org_temp, -(long)sizeof(index_record), SEEK_CUR);
 if (fwrite(&org_index, sizeof(index_record), 1, org_temp)<1)
 {  /* error updateing index */
    db_errorno=errno;
    unlock_file(temp);
    unlock_file(org_temp);
    close_file(temp);
    close_file(org_temp);
    trunc_file(path, ".log", bottom);
    remove_file(index.sd, org_index.prid, ".rep");
    return(NULL);
 }
#ifdef DEBUG
 fprintf(stderr,"confirm_reopen() - updated index entry.\n");
 fprintf(stderr,"\tnow remove the reopen record.\n");
#endif
 /* now unlock and close everything, then remove the entry from the reopend
  * directory.  We are not overly concerned if the record removal fails, since
  * the update has occured.
  */

  unlock_file(org_temp);
  close_file(org_temp);
  unlock_file(temp);
  close_file(temp);
  if (delete_record(reop_prid)==FAIL)
  	/* deletion failed */
     return(NULL);
  else
     return(org_index.prid);
} /* end confirm_reopen */


int deny_reopen(char *reop_prid)
{
#ifdef DEBUG
 fprintf(stderr,"\nIn deny_reopen().\n");
 fprintf(stderr,"\tPRID - %s.\n", reop_prid);
#endif

 return(delete_record(reop_prid));
} /* end deny_reopen */


int delete_record(char *reop_prid)
{
 index_record index;
 char *iname;
 static char tfile[FNAMELEN+1];
 FILE *temp,
      *temp_two;
 long num_rec;

#ifdef DEBUG
 fprintf(stderr,"\nIn delete_record().\n\tRecord to remove - %s\n", reop_prid);
#endif
 strcpy(tfile, TEMPFILE);

 if ((temp=open_file(REOPENDIR, ".index", "r", INDEX))==NULL)
    return(FAIL);
 if (lock_file(temp)==FAIL)
 {  /* the lock of the index failed */
    close_file(temp);
    return(FAIL);
 }

#ifdef DEBUG
 fprintf(stderr,"delete_record() - creating temporary file.\n");
#endif
 if ((temp_two=create_file(REOPENDIR, (char *)mktemp(tfile), NULL,
			   FILE_PERMS))==NULL)
 {  /* the creation failed */
    unlock_file(temp);
    close_file(temp);
    return(FAIL);
 }
 if (lock_file(temp_two)==FAIL)
 {  /* the lock the temp file failed */
    unlock_file(temp);
    close_file(temp);
    close_file(temp_two);
    remove_file(REOPENDIR, tfile, NULL);
    return(FAIL);
 }

/* we need to read the record count from the index file and write it to the
 * new index file before searching for the records. */

 fread(&num_rec, sizeof(long), 1, temp);
 fwrite(&num_rec, sizeof(long), 1, temp_two);

#ifdef DEBUG
 fprintf(stderr,"delete_record() - searching for record to remove");
#endif

 while (fread(&index, sizeof(index_record), 1, temp)!=NULL)
 {
#ifdef DEBUG
 fprintf(stderr,"?");
#endif

    if (strcmp(index.prid, reop_prid)!=0)
       if (fwrite(&index, sizeof(index_record), 1, temp_two)<1)
       {  /* the write failed */
	  db_errorno=errno;
	  unlock_file(temp);
	  close_file(temp);
	  unlock_file(temp_two);
	  close_file(temp_two);
	  remove_file(REOPENDIR, tfile, NULL);
          return(FAIL);
       } /* end writing to temp_two */
 } /* end while */

 if (ferror(temp)) /* did we leave the loop because of an error reading? */
 {
    db_errorno=errno;
    unlock_file(temp);
    close_file(temp);
    unlock_file(temp_two);
    close_file(temp_two);
    remove_file(REOPENDIR, tfile, NULL);
    return(FAIL);
 }
#ifdef DEBUG
 fprintf(stderr,"\ndelete_record() - removed record from index.\n");
 fprintf(stderr,"\trenaming temp file.\n");
#endif

/* now that we've removed the offending index record, delete the old index
 * and rename the temp file, so it will become the new index.  Since in theory
 * someone could try to access the reopen directory between the time the old
 * index is removed and the new one is renamed, the calls are made while the
 * file is still open and locked.  In this way the system will not perform the
 * deletion and renaming, until the files are closed and unlocked.
 */

 iname=rindex(REOPENDIR, '/');
 if (iname==NULL)
    iname=REOPENDIR; /* the name of the index file is the name of the dir */
 else
    iname++; /* want char after '/' */
 unlock_file(temp);
 unlock_file(temp_two);
 close_file(temp);
 close_file(temp_two);
 rename_file(REOPENDIR, tfile, "", REOPENDIR, iname, ".index");
 remove_file(REOPENDIR, reop_prid, ".rep");
 remove_file(REOPENDIR, reop_prid, ".log");
 return(SUCESS);
} /* end delete_record */


index_record *find_reopen(FILE *db_pointer, index_record *index, char *prid)
{

/* call search_index() to look for the particular problem */

 return(search_index(db_pointer, prid, index, sizeof(long)));
} /* end find_reopen() */
 
/* end of reopen_problem.c */
