/*
 * signal.c -- Signal handler routines for the Problem Tracking System (PTS)
 *             database software.
 * Dean Collins
 * created: Sun Nov 15 02:59:13 PST 1992
 */

/*
 * Copyright (c) 1995,1994,1993,1992 Dean Collins.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that neither the name of the University of
 * Idaho nor the name of Dean Collins be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission from both parties.  Neither The University of Idaho
 * nor Dean Collins make any representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO AND DEAN COLLINS DISCLAIM ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY OF IDAHO
 * OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

/* Relavent compile-time defines:
 *
 * NOSIGNALS	-- Disable the signal handler entirely.
 * CRUDESIGNALS	-- Don't popup any warning windows before shutting down
 *			xpts.  Do it immediatly.  This option is not
 *			officially recommended, since it violates some of
 *			the rules for signal handlers.  It may or may not
 *			work as expected.  But, it may be better than NOSIGNALS.
 */


#ifndef NOSIGNALS

#include <stdlib.h>             /* General C utilities header file */
#include <stdio.h>		/* Standard input/output utilities hdr. file */
#include <signal.h>		/* Signal handler routines header file */
#include <errno.h>              /* Error utilities header file */

#include <X11/Intrinsic.h>      /* Include standard Toolkit Header file. */
#include <X11/StringDefs.h>     /* Include StringDefs header file */
#include <X11/Shell.h>          /* Include the Shell header file for popups */

#include <X11/Xaw/AsciiText.h>  /* Include the ascii Text widget hdr. file */
#include <X11/Xaw/Box.h>        /* Include the Box widget header file */
#include <X11/Xaw/Command.h>    /* Include the Command widget header file */
#include <X11/Xaw/Dialog.h>     /* Include the Dialog widget header file */
#include <X11/Xaw/Form.h>       /* Include the Form widget header file */
#include <X11/Xaw/List.h>       /* Include the List widget header file */
#include <X11/Xaw/Paned.h>      /* Include the Paned widget header file */
#include <X11/Xaw/Viewport.h>   /* Include the Viewport widget header file */

#include <X11/Xaw/Cardinals.h>  /* Definitions of ZERO, ONE, etc... */

#include "zdbm.h"               /* Zombie Database Manager header file */
#include "cloud.h"		/* Nebulous Cloud header file */
#include "clouderror.h"         /* Nebulous Cloud error rtn. header file */
#include "config.h"             /* Config. file parser header file */
#include "xpts.h"               /* Xpts header file */

extern problem_record * ProblemRecord ;
extern char Path[] ;
extern int WarningFlag ;

/* Things that make you go hmmm....*/
/*
#if DEBUG
#undef stdout
#define stdout stderr
#endif
*/
	/* NICE means "ask before shutting down." */
	/* MEAN means "shut down NOW! */
#define NICE 1
#define MEAN (!(NICE))

extern Widget GlobalMainMenuWidget ;

int	TheEnd(int Type, int Signal) ;
void	SuperErrorPopup  (String name, Widget parent, String message) ;


SigType SigRec ;


/* Some signals aren't defined on all systems.  Let me know if
 * there are more to add to this list.   Calling signal() with -1
 * will simply return with an error which can be ignored.
 * 
 * Linux: SIGEMT and SIGSYS aren't defined, apparently.
 */
#define NeverHappen (-1)

#ifndef SIGEMT
#define SIGEMT NeverHappen
#endif /*SIGEMT*/

#ifndef SIGSYS
#define SIGSYS NeverHappen
#endif /*SIGSYS*/



/******************** MODULES *************************************/

/*
______________________________________________________________________
InitSignalHandler()

FUNCTION DESCRIPTION:

	This function initializes the signal handlers.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins            Date:  11/15/92
______________________________________________________________________
*/


void
InitSignalHandler(void)

     /* Interface description:
         INPUT:
          None.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
	  /* NONE  */
          /* Minor */
	  /* NONE  */

   /*-------------- Start of InitSignalHandler() routine ---------------*/

#ifdef DEBUG
   printf("XPTS(signal.c): Initializing signal handlers...\n") ;
#endif

     /* Init. signal handler for common, critical, system-caused signals. */
#  ifndef linux
   signal(SIGBUS,  XSignalHandler) ;  /* bus error */
#  endif /*!linux*/
   signal(SIGSEGV, XSignalHandler) ;  /* segmentation violation */
   signal(SIGSYS,  XSignalHandler) ;  /* bad argument to system call */

     /* Init. signal handler for critical system-caused signals. */
   signal(SIGILL,  XSignalHandler) ;  /* illegal instruction */
   signal(SIGTRAP, XSignalHandler) ;  /* trace trap */
   signal(SIGIOT,  XSignalHandler) ;  /* IOT instruction */
   signal(SIGABRT, XSignalHandler) ;  /* used by abort */
#  ifndef linux
   signal(SIGEMT,  XSignalHandler) ;  /* EMT instruction */
#  endif /*!linux*/
   signal(SIGFPE,  XSignalHandler) ;  /* floating point exception */

     /* Init. signal handler for user-caused signals. */
   signal(SIGHUP,  XSignalHandler) ;  /* hangup */
   signal(SIGINT,  XSignalHandler) ;  /* interrupt (rubout) */
   signal(SIGQUIT, XSignalHandler) ;  /* quit (ASCII FS) */
   signal(SIGTERM, XSignalHandler) ;  /* software termination signal from kill */

#ifdef DEBUG
   printf("XPTS(signal.c): Done initializing signal handlers.\n") ;
#endif

}  /*------------------ End of InitSignalHandler() ---------------------*/


/*
______________________________________________________________________
XSignalHandler()

FUNCTION DESCRIPTION:

        This function is the signal handler.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  11/15/92
______________________________________________________________________
*/

SIGHANDLER XSignalHandler(int Signal)

     /* Interface description:
         INPUT:
	  Signal	- The signal that caused this interrupt.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   static int flag=FALSE ;
          /* Minor */
          /* NONE  */

   /*----------------- Start of XSignalHandler() routine ------------------*/

#ifdef DEBUG
   /* I know you're not supposed to call library functions in a signal handler.
    * I've got to get SOME debugging output, though.
    */
   printf("XPTS(signal.c): Entering XSignalHandler() with Signal=%d...\n",
	  Signal) ;

#endif

     /* Reinitialize the signal handler for this signal so it calls
      * this function.  This may not be necessary on all systems.
      */
   signal(Signal, XSignalHandler) ;

     /* Update the SigRec values */
   ++(SigRec.count) ;
   if (SigRec.signal == 0) SigRec.signal = Signal ; 


   switch (Signal)
   {
      case SIGINT:
      case SIGQUIT:
      case SIGTERM:	SigRec.type = USERSIG ;
			break ;

      default:		SigRec.type = SYSSIG ;
   }


   /* It's up to the main program to detect that a signal has occured, now. */
#ifdef DEBUG

   printf("XPTS(signal.c): Leaving XSignalHandler() with Signal=%d...\n",
	  Signal) ;
#endif

   flag=FALSE ;

#  ifdef CRUDESIGNALS
     /* The HP 730 doesn't behave like the other systems.  Call
      * ShutDownDB right now since it will never get called otherwise.
      * I probably just don't understand something about signals on this
      * platform, but this should work better than the current situation.
      * However, it makes library calls, so it may not work.
      */
   ShutDownDB(FALSE) ;
   exit(1) ;
#  endif /* CRUDESIGNALS */

   if (SigRec.count > 1)  /* Caught a second signal, so get out now */
   {
#ifdef DEBUG
      printf("XPTS(signal.c): caught a second signal.  Quitting xpts now...\n") ;
#endif
      ShutDownDB(FALSE) ;
      exit(1) ;
   }

   return ;
}  /*--------------------- End of XSignalHandler() ------------------------*/



/*
______________________________________________________________________
DoSignal()

FUNCTION DESCRIPTION:

        This function will cause a popup to appear for a signal,
then it cleans up the database.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  11/15/92
______________________________________________________________________
*/


void
DoSignal(int Signal)

     /* Interface description:
         INPUT:
          Signal        - The signal that caused this interrupt.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*----------------- Start of DoSignal() routine ------------------*/

#ifdef DEBUG
   fprintf(stderr,"XPTS(signal.c): Entering DoSignal with signal %d.\n", Signal) ;
#endif

   switch(Signal)
   {
      case SIGINT:
      case SIGQUIT:
      case SIGTERM:
	 fprintf(stderr, "%s\n", signalstr(Signal)) ;
         if(TheEnd(NICE, Signal)) exit(Signal) ;
         break ;

		/* Parent process died, so get out without popping up
		 * any windows.
                 */
      case SIGHUP:
	 fprintf(stderr, "%s\n", signalstr(Signal)) ;
	 ShutDownDB(FALSE) ;
         exit (Signal) ;
         break ;

      default:
	 fprintf(stderr, "%s\n", signalstr(Signal)) ;
         if(TheEnd(MEAN, Signal)) exit(Signal) ;
   }

#ifdef DEBUG
   fprintf(stderr,"XPTS(signal.c): Leaving DoSignal()\n") ;
#endif

   return ;
}  /*--------------------- End of DoSignal() ------------------------*/


/*
______________________________________________________________________
TheEnd()

FUNCTION DESCRIPTION:

        This function will cause xpts to shut down nicely.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  10/15/92
______________________________________________________________________
*/


int 
TheEnd(int Type,
       int Signal)

     /* Interface description:
         INPUT:
	  Type		- Type of shutdown: MEAN --> Don't ask user, tell user.
				            NICE --> Ask user if should shutdown
	  Signal	- The signal that caused this interrupt.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   char msg[1024] ;
          /* Minor */
          /* NONE  */


   /*----------------- Start of TheEnd() routine ------------------*/

#ifdef DEBUG
   printf("XPTS(signal.c): Entering TheEnd()\n") ;
#endif

   fprintf(stderr, "\nShutting down XPTS") ;
   if (Type == NICE)
       fprintf(stderr, " (nicely)") ;
   fprintf(stderr, "...\n") ;

#ifdef DEBUG
   fprintf(stderr, "\n\t(I'm too young to die!)\n\n") ;
#endif


   if (Type == NICE)
   {   /* Call SuperWarningPopup()
        * if (!WarningFlag)
        *   Update the SigRec values and return.
        */
       sprintf(msg, "%s\nXPTS has received signal %d  !\n"
                    "\nShut down XPTS now? \n", signalstr(Signal), Signal) ;
       if (XtIsWidget(GlobalMainMenuWidget))
       {  SuperWarningPopup("signalWarning", GlobalMainMenuWidget, msg) ;
          if (!WarningFlag)
          {
#ifdef DEBUG
             printf("XPTS(signal.c): "
                    "Returned from SuperWarningPopup with 'false'!\n") ;
#endif
             --(SigRec.count) ;
             SigRec.signal=0 ;
             return(FALSE) ;  /* OK, you said so.... */
          }
#ifdef DEBUG
          else 
	     printf("XPTS(signal.c): "
                    "Returned from SuperWarningPopup with 'true'!\n") ;
#endif
       }

   } else  /* Type != NICE */
   {   /* Call SuperErrorPopup() */
       sprintf(msg, "%s\nXPTS has received signal %d  !\n"
                    "\nShutting down XPTS... \n", signalstr(Signal), Signal) ;
       if (XtIsWidget(GlobalMainMenuWidget))
       {  SuperErrorPopup("signalError", GlobalMainMenuWidget, msg) ;
#ifdef DEBUG
          printf("Returned from SuperErrorPopup!\n") ;
#endif
       }

   } /* End if (Type == NICE) */

     /* Unlock any locked problems... */

   ShutDownDB(TRUE) ;


#ifdef DEBUG
   printf("XPTS(signal.c): Possibly useful values:\n") ;
   printf("XPTS(signal.c): errno = %d -- (%s)\n", errno, strerror(errno)) ;
/*CLD*/   printf("XPTS(signal.c): %s\n", CloudStrerror()) ;
   printf("XPTS(signal.c): db_errorno = %d\n\t\t%s\n", db_errorno,
/*ZDBM*/  zdbm_strerror(db_errorno)) ;

   printf("XPTS(signal.c): Leaving TheEnd()\n") ;
#endif

   return(TRUE) ;

}  /*--------------------- End of TheEnd() ------------------------*/
   /*                      ^^^^^^^^^^^^^^^ (what a concept... :-)  */


/*
______________________________________________________________________
signalstr()

FUNCTION DESCRIPTION:

        This function will convert a signal number into a string
describing that signal.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  11/15/92
______________________________________________________________________
*/


char *
signalstr(int Signal)

     /* Interface description:
         INPUT:
	  Signal	- The signal that caused this interrupt.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   static char sigstr[1000] ; /*For the lack of a better constant */
          /* Minor */
          /* NONE  */


   /*----------------- Start of signalstr() routine ------------------*/

   switch(Signal)
   {
#     ifndef linux
      case SIGBUS:
         sprintf(sigstr, "\n***** Signal SIGBUS caught!  (Bus error) *****\n") ;
         break ;
#     endif /*!linux*/

      case SIGSEGV:
         sprintf(sigstr, "\n***** Signal SIGSEGV caught!  "
                         "(Segmentation violation) *****\n") ;
         break ;

      case SIGSYS:
         sprintf(sigstr, "\n***** Signal SIGSYS caught!  "
                         "(bad argument to system call) *****\n") ;
         break ;

      case SIGHUP:
         sprintf(sigstr, "\n***** Signal SIGHUP caught!  (Hangup) *****\n") ;
         break ;

      case SIGINT:
         sprintf(sigstr, "\n***** Signal SIGINT caught!  (Interrupt) *****\n") ;
         break ;

      case SIGQUIT:
         sprintf(sigstr, "\n***** Signal SIGQUIT caught!  (Quit) *****\n") ;
         break ;

      case SIGTERM:
         sprintf(sigstr, "\n***** Signal SIGTERM caught!  "
                        "(Software termination signal from kill) *****\n") ;
         break ;

      case SIGILL:
         sprintf(sigstr, "\n***** Signal SIGILL caught!  "
                         "(Illegal Instruction) *****\n") ;
         break ;

      case SIGTRAP:
         sprintf(sigstr, "\n***** Signal SIGTRAP caught!  "
                         "(Trace Trap) *****\n") ;
         break ;

      case SIGABRT:
         sprintf(sigstr, "\n***** Signal SIGABRT caught!  (Abort) *****\n") ;
         break ;

#ifndef linux
      case SIGEMT:
         sprintf(sigstr, "\n***** Signal SIGEMT caught!  "
                         "(EMT Instruction) *****\n") ;
         break ;
#endif /*!linux*/

      case SIGFPE:
         sprintf(sigstr, "\n***** Signal SIGFPE caught!  "
                         "(Floating point execption) *****\n") ;
         break ;

      default:
         sprintf(sigstr, "\n***** Signal caught!  Signal value is %d. *****\n",
                 Signal) ;
   }

   return (sigstr) ;

}  /*--------------------- End of signalstr() ------------------------*/


/*
______________________________________________________________________
SuperErrorPopup()

FUNCTION DESCRIPTION:

        This function will inform the user of an error and waits
for them to acknowledge.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  11/15/92
______________________________________________________________________
*/


void
SuperErrorPopup(String     name,
                Widget     parent,
                String     message)

     /* Interface description:
         INPUT:
          name           - The resource name of the popup.
          parent         - The parent of the popup widget.
          message        - The specific error message to display.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   char btnname[80] ;
   XEvent event ;
   int done=0 ;
   XtAppContext app_con ;

          /* Minor */
          /* NONE  */
   
   /*----------------- Start of SuperErrorPopup() routine ------------------*/

#ifdef DEBUG
   printf("XPTS(signal.c): Entering SuperErrorPopup.\n") ;
#endif
     /* Start by calling ErrorPopup() to define the widgets. */
   ErrorPopup(name, parent, message) ;

     /* Add a callback for the quitButton to call SuperErrorSelect() */
   sprintf(btnname, "*%s*quitButton", name) ;
#ifdef DEBUG
   printf("XPTS(signal.c): SuperErrorPopup() converting name '%s' to widget...\n",
		btnname) ;
#endif
   XtAddCallback(XtNameToWidget(parent, btnname), XtNcallback, 
		 SuperErrorSelect, parent) ;

     /* Loop, watching for a DestroyNotify event */
   app_con = XtWidgetToApplicationContext(parent) ;
   while (!done)
   {  XtAppNextEvent(app_con, &event) ;
      if (event.type == DestroyNotify)
      {  
#ifdef DEBUG
         printf("XPTS(signal.c): Found a 'DestroyNotify' event.\n") ;
#endif
         done=1 ;   /* Popup has been destroyed, so we're done looping. */
      }
      XtDispatchEvent(&event) ;
   }

#ifdef DEBUG
   printf("XPTS(signal.c): Exiting SuperErrorPopup.\n") ;
#endif

}  /*--------------------- End of SuperErrorPopup() ------------------------*/


/*
______________________________________________________________________
SuperErrorSelect()

FUNCTION DESCRIPTION:

        This function is the callback for the SuperErrorPopup quit
button.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  11/15/92
______________________________________________________________________
*/


void
SuperErrorSelect(   Widget        w,
                 XtPointer        client_data,
                 XtPointer        call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback,
			   this is unused.
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*-------------- Start of SuperErrorSelect() routine ---------------*/

/* This does nothing, but it does it well. */

#ifdef DEBUG
   fprintf(stderr, "XPTS(signal.c): Called SuperErrorSelect!!!!!\n") ;
#endif

}  /*------------------ End of SuperErrorSelect() ---------------------*/


#endif /* NOSIGNALS */

/* end of signal.c */
