/* Author:  Luke Sheneman. */

/* Modified: Dean Collins Sat Feb 12 15:35:02 PST 1994
 * Modification recommended by magnush@isy.liu.se (Magnus Hammerin).
 * Switched from using getegid() to using getgroups() to determine
 * whether the user is in the sysop group.  This will eliminate the
 * need to make sysop the current group (ala "newgrp sysop")
 * before using this software.  Thanks Mangus.
 *
 * Modified: Dean Collins Wed Feb 23 17:53:02 1994
 * Added call to config_misc(), which reads things like dbtop, printer,
 * etc., from the config file.
 *
 * Modified: Dean Collins Fri Mar 18 23:27:41 1994
 * Ensured that NGROUPS_MAX was defined per John Neil's request.
 *
 * Modified: Dean Collins Mon Jan  2 16:39:12 PST 1995
 * Changed all printfs in DEBUG sections to fprintf(stderr...).
 *
 * Modified: Dean Collins Sun May 21 13:42:45 PDT 1995
 * Changed some incorrect return(NULL)s to return(False)s.  NULL is
 * often a pointer, and the function in question (Setup) returns an int.
 */

/*
 * Copyright (c) 1995,1994,1993 Dean Collins.
 * Copyright (c) 1992 Luke Sheneman.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the names of the University of Idaho
 * Dean Collins and Luke Sheneman not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written 
 * prior permission.  The University of Idaho, Luke Sheneman and Dean Collins 
 * make no representations about the suitability of this software for 
 * any purpose.  It is provided "as is" without express or implied warranty.
 *  
 * THE UNIVERSITY OF IDAHO, LUKE SHENEMAN, AND DEAN COLLINS  DISCLAIM ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY OF IDAHO
 * LUKE SHENEMAN OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT OR 
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */


 /*
 *
 *      setup.c - this source file contains the following functions:
 *
 *	Setup() - Gets information about the user and reads the problem 
 * 		hierarchy.  It also fills in the UserInfo and ProblemTree 
 *		structures.
 *	BuildTree() - Actually builds the problem tree.  Called by Setup().
 *	BreakUpPath() - Breaks up the given path into a dynamically allocated
 *		array of strings.  (dynamic for memory problems)
 *	BreakUpPath2() - Breaks up the given path into a statically
 *		declared array of strings. (static for speed)
 *      BuildTreePath() - Builds a portion of the problem tree, given a path.
 *	FindBranch() - This returns a branch of the problem tree, given a 
 *		branch in the problem tree and a path segment.
 *	AddNode() - Adds a Full_path node to the end of a linked list of 
 *		Full_path nodes.	
 *	TraverseLeaves() - Given a tree, this function returns all of the
 *		leaves in that tree.
 *	TreverseTree() - Given a tree, this function returns all branches and
 * 		leaves in the tree.
 *
 */ 
 
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <pwd.h>
#include <zdbm.h>
#include "cloud.h"
#include "clouderror.h"

/* Added 3/18/94 by Dean Collins */
#ifndef NGROUPS_MAX
#define NGROUPS_MAX 32
#endif /*NGROUPS_MAX*/


/* The command to print files will be stored here. */
char PrinterCmd[COMPLETEPATHLEN]=PRINTER ;
/* The options for PrinterCmd used to print files will be stored here. */
char PrinterOpts[COMPLETEPATHLEN]=PRINTER_OPTS ;

/*
  Setup() fills in the UserInfo structure that is passed to it from the ui.  
  The UserInfo structure contains a flag which holds whether or not the 
  user is a sysop.  It also contains the user's real name and their user name
  Setup() also builds a tree from the linked-list data structure which is 
  passed to it from the section which reads the config file.  This tree will 
  hold all the problem tree (a tree of problem types), which the ui can then 
  utilize further.  Also, call umask() to set up default permissions.
	

	Author:  Luke Sheneman			Date:  08/04/92
	Revision:  Luke Sheneman		Date:  08/06/92

		Made the pproblemtree variable that gets passed into this 
	function a pointer to the pointer of a problem tree structure.  This
	was necessary so that the user interface have the problem tree 
	returned properly.

		This function was black-box tested with a driver that simulated
	P_code() and the user interface. 
		This function has been thouroughly tested.
*/



int Setup(userinfo,pproblemtree)
UserInfo *userinfo;
ProblemTree **pproblemtree;
{
	/* First, lets fill in the UserInfo structure.  This requires that
	   a pointer to an already malloc'd UserInfo structure be passed into
	   this function.
	*/

	
	int error;			   /* Hold our error condition */
	int loop;			   /* A simple counter 	       */
	struct passwd *passwd;		   /* Will hold passwd structure */
	ProblemTree *problemtree;
	struct Full_path *pathlist;	   /* Will contain head of problem 
					           type tree linked list */
	char *realname,*temp_real_name;
	struct Full_path list;
		/* Group membership info, added 2/14/94 DC*/
	int gidset[NGROUPS_MAX];	/* Array of groups we're a member of*/
	int gidsetlen=NGROUPS_MAX;	/* Length of gidset[] array */
	int gid_found;			/* Flag: True==>sysop group member*/
	StringListRec *sysops=NULL ;	/* List of sysop names, 2/25/94, DC*/

	umask(PTS_UMASK);		   /* Set default permissions. */

	list.next=NULL;
	strcpy(list.path,"/");
					/* Let's fill in the sysopflag */
					/* Modified 2/14/94 DC.  Use   */
					/* getgroups instead of getegid*/
	gidsetlen = getgroups(gidsetlen, gidset) ;
	gid_found = False ;
        loop=0 ;
	while (!gid_found && loop<gidsetlen)
		gid_found = (gidset[loop++] == SYSOPGID) ;

		/* Added since getgroups was returning zero groups 
		 * on some systems (well... on _one_ system).
		 * Can't hurt to do it as a backup.  DC 3/7/94
		 */
	if (!gid_found)	
		gid_found = (getegid () == SYSOPGID) ;

	userinfo->sysopflag = gid_found ;


	passwd=getpwuid( getuid() );	   /* Get the passwd structure from
					      the user's effective UID */

	if(!passwd)
	{	
		cld_errno=CLD_ERR1;
		return(False);
	}

	userinfo->username=(char *)malloc(strlen(passwd->pw_name)+1);
	realname=(char *)malloc(strlen(passwd->pw_gecos)+1);
	temp_real_name=(char *)malloc(strlen(passwd->pw_gecos)+1);
	strcpy(realname,passwd->pw_gecos);
	loop=0;
	while( (realname[loop]!='\0') && (realname[loop]!=',') )
	{
		temp_real_name[loop]=realname[loop];
		loop++;
	}
	temp_real_name[loop]='\0';
	userinfo->realname=(char *)malloc(strlen(temp_real_name)+1);
	strcpy(userinfo->username,passwd->pw_name);	/* copy the username */
	strcpy(userinfo->realname,temp_real_name);	/* copy the realname */

	/*  The UserInfo structure is now completely filled.  It is time to 
	    fill in the ProblemTree structure based on the configuration file.
	    For this, merely call BuildTree(), passing the head of the linked
	    list as a parameter.
	*/

	if(config_tree(&pathlist))	/* Returns 1 on error. */
	{
		cld_errno=CLD_ERR2;
		return(False);		/* Return failure to user interface */
	}

	BuildTree(&problemtree,pathlist);	/* Lets build the tree! */
	*pproblemtree=problemtree;


	/*  Now read the miscellanious information from the config file.
	    This is a second pass, which, if we were overly ambitious,
	    we could eliminate.  However, the config file is small.
	    Added by Dean Collins, Wed Feb 23 17:52:27 1994
	*/
	if(config_misc())		/* Returns 1 on error. */
	{
		cld_errno=CLD_ERR2;
		return(False);		/* Return failure to user interface */
	}


        /*  Now run through the sysop list, filled in by config_misc() above.
	    This list is only needed if the user cannot use a sysop group.
	    Added by Dean Collins, Fri Feb 25 09:44:37 1994
            Fixed 3/4/95 DC.  AddToStringList works different now.
        */
        sysops = SysopList;
        if (sysops)
           while (sysops)	/*The last record is null */
           {  if (strcmp(sysops->string, userinfo->username) == 0)
                 userinfo->sysopflag = TRUE ;
		 sysops = sysops->next ;
	   }

	
	return(True);			       /* Back to the user interface */ 
}


/*  BuildTree() - This function actually builds the problem tree given an
	pointer to the uninitialized tree and a linked list of problem types.

	Author:  Luke Sheneman			Date:  08/04/92
	Revision:  Luke Sheneman		Date:  08/06/92

		Many revisions were made to this function.  Most notibly, the
	ptree parameter must be passed into BuildTree() as a pointer-pointer 
	instead of a simple pointer to a ProblemTree structure.
		This function was black-box tested with drivers.
*/

void BuildTree(ptree,list)
ProblemTree **ptree;
struct Full_path *list;
{
	struct Full_path *pathtemp;	 /*  Placemarker in list	*/
	ProblemTree *tree;		 /*  the tree that we will use	*/ 
	ProblemTree *current; 		 /*  marks current head of tree	*/
	char parts[MAXBRANCHES][DBPATHLEN]; /*  array of path segments 	
						this is statically declared
						in this function for speed */
	ProblemTree *subtree;		 /*  Keep track of sub-trees 	*/
	int loop=0;			 /*  Just a counter 		*/

	pathtemp=list;		   /*  Set the temp pointer to head of list */

	/* Here, we malloc and fill in the root of the tree structure.  We
	   set everything to NULL inside of the structure, but set the 
 	   fullpath and path entries to the ROOTSTR 
	*/

	tree=(ProblemTree *)malloc(sizeof(ProblemTree)); /* allocate root */
	tree->subhead=(ProblemTree *)NULL;
	tree->next=(ProblemTree *)NULL;
	strcpy(tree->fullpath,ROOTSTR);
	strcpy(tree->path,ROOTSTR);

	current=tree;		/*  Set the current sub-tree to be the root */



	/* Let's build the tree now, based on the path linked list  */
	while(pathtemp)
	{
		int l=0;

		BreakUpPath2(pathtemp->path,parts);	/* Go static.  Need speed here */
		
	  	while(strcmp(parts[loop],""))
		{
			subtree=FindBranch(current,parts[loop]); /* See if branch already exists */
			if(subtree)	/* if it does... */
			{
				current=subtree;	/* set current equal to the subtree that
						 		it returns.  */
				loop++;			/* increment position in path parts 
								array */
			}
			else
			{
				BuildTreePath(&current,parts,loop);  /* Build tree from here */
				strcpy(parts[loop],"");	
			}
		}	
		pathtemp=pathtemp->next;	/* next path, please */
		current=tree;			/* set current equal to head */
		loop=0;				/* initialize our counter */
	}
	*ptree=tree; /*set ptree equal to tree to return to calling function */
}





/*BreakUpPath() - This function breaks up a path where components of that path
  seperated by a '/'.  It places the components in an array of NULL-terminated
  strings for use in BuildTree()

  Author:  Luke Sheneman		Date:  08/05/92
  TESTING HISTORY:

	This module has been tested.
*/

void BreakUpPath(path,parts)
char *path;
char *parts[MAXBRANCHES];
{
	int loop,x,y,l;			/* Some counters and index variables */
        char c,temp[DBPATHLEN];		/* Some temporary storage	     */

/* 	zdebug1("In beginning of BreakUpPath().  path=%s\n",path); 	*/



	/* malloc the bastard - this was the solution to the 3500 barf - must be freed elsewhere*/
	for(loop=0;loop<MAXBRANCHES;loop++)
		parts[loop]=(char *)malloc(sizeof(char)*DBPATHLEN);



	for(loop=0;loop<MAXBRANCHES;loop++)	/* more efficient.  Thanks, dean.  */
		memset(parts[loop],(char *)NULL,DBPATHLEN);	

        l=strlen(path);			
	if(path[l-1]!='/')
        	strcat(path,"/");	/* Make sure path ends with a '/'    */
	if(path[0]=='/')
	{
		strcpy(temp,path);	/* Remove the first character, a '/' */
		for(loop=1;loop<l+1;loop++)
                	path[loop-1]=temp[loop];
	}

        y=x=0;				/* Initialize some variables to 0    */
        for(loop=0;loop<l;loop++)	/* Fill in the array of strings.     */
        {
                c=path[loop];	
                if(c!='/')
        		parts[y][x++]=c;
                else
                {
                        parts[y][x+1]='\0';
                        y++;
                        x=0;
                }
        }
	strcpy(parts[y+1],"");		/* Make marker for end of array */
}





/*BreakUpPath2() - This function breaks up a path where components of that path
  seperated by a '/'.  It places the components in an array of NULL-terminated
  strings for use in BuildTree().  This function differs from BreakUpPath in
  that it is more static.  That is, it uses statically declared variables 
  rather than mallocing them.  Big performance increase, but wasted space.
  This function should only be called from routines in setup.c  

  Author:  Luke Sheneman		Date:  12/05/92
  TESTING HISTORY:

	This module has been tested.
*/


void BreakUpPath2(path,parts)	/* faster...static */
char *path;
char parts[MAXBRANCHES][DBPATHLEN];
{
	int loop,x,y,l;			/* Some counters and index variables */
        char c,temp[DBPATHLEN];		/* Some temporary storage	     */

/* 	zdebug1("In beginning of BreakUpPath().  path=%s\n",path); 	*/



	/* malloc the bastard - this was the solution to the 3500 barf - must be freed elsewhere*/

	for(loop=0;loop<MAXBRANCHES;loop++)	/* more efficient.  Thanks, dean.  */
		memset(parts[loop],(char *)NULL,DBPATHLEN);	

        l=strlen(path);			
	if(path[l-1]!='/')
        	strcat(path,"/");	/* Make sure path ends with a '/'    */
	if(path[0]=='/')
	{
		strcpy(temp,path);	/* Remove the first character, a '/' */
		for(loop=1;loop<l+1;loop++)
                	path[loop-1]=temp[loop];
	}

        y=x=0;				/* Initialize some variables to 0    */
        for(loop=0;loop<l;loop++)	/* Fill in the array of strings.     */
        {
                c=path[loop];	
                if(c!='/')
        		parts[y][x++]=c;
                else
                {
                        parts[y][x+1]='\0';
                        y++;
                        x=0;
                }
        }
	strcpy(parts[y+1],"");		/* Make marker for end of array */
}





/*  BuildTreePath() - This function traverses a path, adding that entire path
	to the problem tree.

	Author:  Luke Sheneman			Date:  08/04/92
	Revision:  Luke Sheneman		Date:  08/06/92

		Again, many changes were made to this function.  pcurrent
	must be passed to this function as a pointer-pointer to a ProblemTree
	structure.
		This function was black-box tested using drivers.
*/

void BuildTreePath( ProblemTree **pcurrent,
			   char parts[MAXBRANCHES][DBPATHLEN],
			   int index )
{
	ProblemTree *current;
	ProblemTree *temptree;		/* temp for linked list of heads  */
	ProblemTree *t;
	int loop,i=0;

	current = *pcurrent;
	loop=index;
	temptree=current;
	while(strcmp(parts[loop],""))
	{
		if(!temptree->subhead)
		{
			temptree->subhead=(ProblemTree *)malloc(sizeof(ProblemTree));
			temptree->subhead->subhead=(ProblemTree *)NULL;
			temptree->subhead->next=(ProblemTree *)NULL;
/*#ifdef AIXV3*/
			strcpy(temptree->subhead->fullpath,"") ;
/*#endif*/
			strcpy(temptree->subhead->path,parts[loop]);
			i=0;
			while( (strcmp(parts[i],"")) && (i<=loop) )
			{
				char dummy[DBPATHLEN];	

				sprintf(dummy,"/%s",parts[i]);
				strcat(temptree->subhead->fullpath,dummy);
				i++;
			}
			temptree=temptree->subhead;

			zdebug1("BuildTreePath() path A ='%s'\n",temptree->fullpath) ;
		}
		else
		{
			t=(ProblemTree *)malloc(sizeof(ProblemTree));
/*#ifdef AIXV3*/
			strcpy(t->fullpath,"") ;
/*#endif*/
			t->subhead=(ProblemTree *)NULL;
			i=0;
			while( (strcmp(parts[i],"")) && (i<=loop) )
			{
				char dummy[DBPATHLEN];

				sprintf(dummy,"/%s",parts[i]);
				strcat(t->fullpath,dummy);
				i++;
			}
			strcpy(t->path,parts[loop]);
			t->next=temptree->subhead;
			temptree->subhead=t;
			temptree=t;

			zdebug1("BuildTreePath() path B ='%s'\n",temptree->fullpath) ;

		}
		loop++;	
	}
	*pcurrent=current;
}






/* FindBranch() - This function finds a particular branch in the problem tree
   by matching the path segment string at that branch with a given string.

	Author:  Luke Sheneman			Date:  08/04/92
	Revision:  Luke Sheneman		Date:  08/06/92

		This function seems to work.  It was used with BuildTree(),
	and BuildTree() seems to work.

*/

	

ProblemTree *FindBranch(current,part)
ProblemTree *current;
char *part;
{
	ProblemTree *t;    

	t=current->subhead;
	if(!t)
		return( (ProblemTree *)NULL );
	else
	{
		while(t)
		{
			if( (!strcmp(t->path,part)) )
				return( (ProblemTree *)t );
			t=t->next;
		}
	}
	return( (ProblemTree *)NULL );
}







/* AddNode() - This function adds a Full_path structure to the end of an 
 	existing linked list of Full_path structures.  

	Author:  Luke Sheneman			Date:  08/04/92
	Revision:  Luke Sheneman		Date:  08/06/92

		This function is trivial and has been tested.

*/	

void AddNode(list,path)
struct Full_path *list;
char *path;
{
        struct Full_path *t;
        struct Full_path *temp;

        temp=list;

        while(temp->next)
                temp=temp->next;

        t=(struct Full_path *)malloc(sizeof(struct Full_path));
        t->next=NULL;
        strcpy(t->path,path);
        temp->next=t;
}











/* TraverseLeaves() - This function traverses a problem tree, and returns 
	a linked list of all leaves.  This is useful in functions like
	ReadUnsolvedSummaries() which attempt to read database information
	at all leaves in the problem tree.  This function uses a brute 
	force recursive algorithm to visit all leaves of the problem tree
	in memory.  It calls TraverseTree() to do this.

	Author:  Luke Sheneman			Date:  08/04/92	
	

		This function is trivial and self-explanatory, and has
		been tested thouroughly.

*/


void TraverseLeaves(tree,list)
ProblemTree *tree;
struct Full_path *list;
{
        if(tree)
        {
                if(!tree->subhead)
                        AddNode(list,tree->fullpath);
                else
                        TraverseTree(tree->subhead,list);
                if(tree->next)
                        TraverseTree(tree->next,list);
        }
}








/* TraverseTree() - This function traverses a given tree by a 
   	brute force recursive algorithm.  

	Author:  Luke Sheneman			Date:  08/04/92

		This function has been tested to a great degree.

*/
 

void TraverseTree(tree,list)
ProblemTree *tree;
struct Full_path *list;
{
        if(tree)
        {
                if(!tree->subhead)
                        AddNode(list,tree->fullpath);
                else
                        TraverseTree(tree->subhead,list);
                if(tree->next)
                        TraverseTree(tree->next,list);
        }
}

