/* convertindex.c
 *
 * These series of functions convert the prid's of an xpts data base to
 * the new format.  It also removes deleted entries and updates all links.
 *
 * Syntax:
 *          convertindex leaffile
 *
 *  where:
 *          leaffile is a file that contains the path lists of the DB leaves.
 *
 * Sugested invocation:
 *
 *          find /usr/local/PTS/DB -type d -print > tmp; convertindex tmp
 *
 * This process is performed in three passes.
 *
 * Pass 1:
 *      Convert the prid's in the indexes and rename the associated files.
 *
 * Pass 2:
 *      Update all the links in the data base.
 *
 * Pass 3:
 *      For each leaf
 *          create a temp file (will contain the new index)
 *          readin each index entry
 *          if it's been deleted or moved then
 *              remove any files in the current leaf
 *              continue
 *          else
 *              write out to new index
 *          sort new index
 *          rename temp file to leaf.index (this also deletes old index)
 *      next leaf
 *
 *  Copyright (C) 1995, Christopher Hyde
 *
 */

#include <strings.h>
#include <errno.h>
#include <sys/stat.h>
#include <stdio.h>

#include "zdbm.h"

char *fixpath(char *string) ;

struct mini_index
{
    char id[MAXPRID];
    long loc;
};

FILE *error_log; /* error messages will be printed here */
int db_errorno;

/* main
 *
 * Controls the conversion process.
 *
 */

int main(int argc, char **argv)
{
 time_t clock;
 FILE *lf;
 char db_path[DBPATHLEN];

/* first we open the error log and print out the date and time we started */
 if ((error_log=fopen("error_log", "w"))==NULL)
    exit(errno);

/* now get the name of the leaf file */
 if (argc!=2)
 {
    fprintf(error_log, "No leaf file specified\n");
/* may need to check the time functions for differences */
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(1);
 }

 if ((lf=fopen(argv[1], "r"))==NULL)
 {
    fprintf(error_log, "Could not open leaf file - %s\n", argv[1]);
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(errno);
 }

/* for each leaf given in the leaf file do */
/*fprintf(error_log, "Began Pass 1 at %s\n", ctime(time(&clock)));*/
 while (fgets(db_path, DBPATHLEN, lf)!=NULL) 
 {
    fixpath(db_path); /* replaces '\n' with '\0' */
    if (pass_1(db_path)==FAIL)
    {
       fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
       exit(db_errorno);
    }
 } /* end pass 1 while */
 if (ferror(lf)) /* did we exit because of an error */
 {
    fprintf(error_log, "Error reading leaf file %s\n", argv[1]);
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(errno);
 }

/* for each leaf given in the leaf file do */
 rewind(lf); 
 fprintf(error_log, "Began Pass 2 at %s\n", ctime(time(&clock)));
 while (fgets(db_path, DBPATHLEN, lf)!=NULL)
 {
    fixpath(db_path);
    if (pass_2(db_path)==FAIL)
    {
       fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
       exit(db_errorno);
    }
 } /* end while for pass 2 */
 if (ferror(lf)) /* did we exit because of an error */
 {
    fprintf(error_log, "Error reading leaf file %s\n", argv[1]);
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(errno);
 }

/* for each leaf given in the leaf file do */
 fprintf(error_log, "Began Pass 3 at %s\n", ctime(time(&clock)));
 rewind(lf);
 while (fgets(db_path, DBPATHLEN, lf)!=NULL)
 {
    fixpath(db_path);
    if (pass_3(db_path)==FAIL)
    {
       fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
       exit(db_errorno);
    }
 } /* end while for pass 3 */
 if (ferror(lf)) /* did we exit because of an error */
 {
    fprintf(error_log, "Error reading leaf file %s\n", argv[1]);
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(errno);
 }

/* now lets fix the reopen leaf */
 fprintf(error_log, "Converting the reopen leaf at %s\n", ctime(time(&clock)));
 if (convert_reopen(REOPENDIR)==FAIL)
 {
    fprintf(error_log, "Execution halted at %s\n", ctime(time(&clock)));
    exit(db_errorno);
 }

 fprintf(error_log, "Completed execution at %s\n", ctime(time(&clock)));
 fclose(error_log);
} /* end main */


/* pass_1
 *
 *  This function changes the prid of each index entry in the current index.
 *  This includes deleted and moved entries, since they may have non-updated
 *  links associated with them.  It also renames the files associated with each
 *  index entry.
 *
 */

int pass_1(char *db_path)
{
 FILE *fp;
 char new_id[MAXPRID];
 index_record inrec;

 db_errorno=0;

 if (exists(PtsRoot, db_path, "", ""))
 {
    fprintf(error_log, "Opening db_path - %s\n", db_path);
    if ((fp=open_file(db_path, ".index", "r+", INDEX))==NULL)
       return(FAIL);
    fprintf(error_log, "Locking index\n");
    lock_file(fp);
    fprintf(error_log, "Reading index\n");
    while (fread(&inrec, sizeof(index_record), 1, fp)==1)
    {
       new_id[0]=inrec.prid[3];
       new_id[1]=inrec.prid[4];
       new_id[2]=inrec.prid[5];
       new_id[3]=inrec.prid[6];
       new_id[4]=inrec.prid[0];
       new_id[5]=inrec.prid[1];
       new_id[6]=inrec.prid[2];
       new_id[7]=inrec.prid[7];
       new_id[8]=inrec.prid[8];
       new_id[9]=inrec.prid[9];
       new_id[10]=inrec.prid[10];
       new_id[11]=inrec.prid[11];
       new_id[12]=inrec.prid[12];
       new_id[13]='\0';
 /* now rename the log and reporter files */
       rename_file(db_path, inrec.prid, ".log", db_path, new_id, ".log");
       rename_file(db_path, inrec.prid, ".rep", db_path, new_id, ".rep");
       rename_file(db_path, inrec.prid, ".rep_old",
                    db_path, new_id, ".rep_old");
 /* change the index entry */
       strcpy(inrec.prid, new_id);
 /* if the current entry is a link, update the original prid */
       if (inrec.mrd==LINKED)
       {
          new_id[0]=inrec.ln_prid[3];
          new_id[1]=inrec.ln_prid[4];
          new_id[2]=inrec.ln_prid[5];
          new_id[3]=inrec.ln_prid[6];
          new_id[4]=inrec.ln_prid[0];
          new_id[5]=inrec.ln_prid[1];
          new_id[6]=inrec.ln_prid[2];
          new_id[7]=inrec.ln_prid[7];
          new_id[8]=inrec.ln_prid[8];
          new_id[9]=inrec.ln_prid[9];
          new_id[10]=inrec.ln_prid[10];
          new_id[11]=inrec.ln_prid[11];
          new_id[12]=inrec.ln_prid[12];
          new_id[13]='\0';
          strcpy(inrec.ln_prid, new_id);
       } /* end if linked */

/* now back up the file 1 record and write out the new index record */
       fseek(fp, -(long)sizeof(index_record), SEEK_CUR); 
       if (fwrite(&inrec, sizeof(index_record), 1, fp) == 0)
          break; /* out of loop */
    } /* end while */
    unlock_file(fp);
    if (ferror(fp))
    {
       db_errorno=errno;
       return(FAIL);
    }
 } /* end if index exists */
 else
    fprintf(error_log, "Leaf - %s - does not exist\n", db_path);
 return(SUCESS);
} /* end pass_1() */


/* pass_2
 *
 * This function goes through each leaf given and updates any linked
 * problems that are found.
 *
 */

int pass_2(char *db_path)
{
 FILE *fp;
 index_record inrec;

 db_errorno=0;

 if (exists(PtsRoot, db_path, "", ""))
 {
    fprintf(error_log, "Opening db_path - %s\n", db_path);
    if ((fp=open_file(db_path, ".index", "r+", INDEX))==NULL)
       return(FAIL);
    fprintf(error_log, "Reading index\n");
    while (fread(&inrec, sizeof(index_record), 1, fp)==1)
    {
       if (inrec.mrd==LINKED)
          if (check_index(fp, &inrec)==NULL) /* problem updating */
          {
             fprintf(error_log, "Could not update current entry, deleting\n");
             lock_file(fp);
             inrec.mrd=GONE;
             fseek(fp, -(long)sizeof(index_record), SEEK_CUR);
             fwrite(&inrec, sizeof(index_record), 1, fp);
             unlock_file(fp);
          }
    } /* end while */
    if (ferror(fp))
    {
       db_errorno=errno;
       return(FAIL);
    }
 } /* end index exists */
 else
    fprintf(error_log, "Leaf - %s - does not exist\n", db_path);
} /* end pass_2() */


/* pass_3
 *
 * This function goes through a leaf removing un-needed index entries and
 * then resorts the index.  The sorting is done by using an insertion sort on a
 * mini_index in memory (prid, file location).  Next a new index file is
 * created and renamed.
 *
 */

int pass_3(char *db_path)
{
 index_record inrec;
 struct mini_index *mini;
 FILE *fp,
      *tmp;
 char *iname,
      tfile[FNAMELEN];
 long numrec,
      i,
      j;

 db_errorno=0;

 if (exists(PtsRoot, db_path, "", ""))
 {
    fprintf(error_log, "Opening db_path - %s\n", db_path);
    if ((fp=open_file(db_path, ".index", "r+", INDEX))==NULL)
       return(FAIL);
    numrec=file_size(fp, (long)sizeof(index_record));
    fprintf(error_log, "Creating mini index\n");
    if ((mini = (struct mini_index *) calloc(numrec,
          sizeof(struct mini_index))) == NULL)
    {
       db_errorno=errno;
       return(FAIL);
    }
    if (lock_file(fp)==FAIL)
    {
       free(mini);
       return(FAIL);
    }
    fprintf(error_log, "Reading index\n");
    i=0;
    while (fread(&inrec, sizeof(index_record), 1, fp)==1)
    {
       if ((inrec.mrd==GONE) || (inrec.mrd==MOVED))
       { /* the current entry is either deleted or moved so make sure the
          * files are gone and ignore the entry */
          remove_file(db_path, inrec.prid, ".log");
          remove_file(db_path, inrec.prid, ".rep");
          remove_file(db_path, inrec.prid, ".rep_old");
       }
       else /* the current entry is ok so add it to the new index */
       {
          strcpy(mini[i].id, inrec.prid);
          mini[i].loc=ftell(fp)-sizeof(index_record);
          i++;
       }
    } /* end while */
    if (ferror(fp))
    {
       db_errorno=errno;
       free(mini);
       unlock_file(fp);
       return(FAIL);
    }
    fprintf(error_log, "Sorting index\n");
    isort(mini, i);
    fprintf(error_log, "Creating new index file\n");
    strcpy(tfile, TEMPFILE);
    if ((tmp=create_file(db_path, (char *)mktemp(tfile), ".index",
             FILE_PERMS))==NULL)
    {
       free(mini);
       unlock_file(fp);
       return(FAIL);
    }
    fprintf(error_log, "Loading new index\n");
 /* i is 1 more than the number of elements in mini index */
    for (j=0; j<i; j++)
    {
       fseek(fp, mini[j].loc, SEEK_SET);
       if (fread(&inrec, sizeof(index_record), 1, fp)<1)
       {
          db_errorno=errno;
          break; /* out of loop */
       }
       if (fwrite(&inrec, sizeof(index_record), 1, tmp)<1)
       {
          db_errorno=errno;
          break; /* out of loop */
       }
    } /* end for */
    free(mini);
    unlock_file(fp);
    close_file(fp);
    close_file(tmp);
    if (ferror(fp) || ferror(tmp))
    {
       remove_file(db_path, tfile, ".index");
       return(FAIL);
    }
    iname=rindex(db_path, '/');
    if (iname=NULL)
       iname=db_path;
    else
       iname++;
    fprintf(error_log, "Renaming new index file\n");
    if (rename_file(db_path, tfile, ".index", db_path, iname, ".index")==FAIL)
    {
       remove_file(db_path, tfile, ".index");
       return(FAIL);
    }
 } /* end index exists */
 else
    fprintf(error_log, "Leaf - %s - does not exist\n");
 return(SUCESS);
} /* end pass_3() */


/* isort
 *
 * This is a simple insertion sort.  Since the indexes should be in almost
 * sorted order (probably in sorted order), this sort will work well.
 *
 */

int isort(struct mini_index *mini, long max)
{
 long i, j, v;
 char cid[MAXPRID];

 for (i=1; i<max; i++)
 {
    strcpy(cid, mini[i].id);
    v=mini[i].loc;
    for (j=i; strcmp(mini[j-1].id, cid)>0; j--)
    {
       strcpy(mini[j].id, mini[j-1].id);
       mini[j].loc=mini[j-1].loc;
    }
    strcpy(mini[j].id, cid);
    mini[j].loc=v;
 }
} /* end isort() */


/* covnvert_reopen
 *
 * This function converts the reopen leaf, if it exists.  The reopen leaf is
 * a special case, since the index file contains different information.
 *
 */

int convert_reopen(char *db_path)
{
 long numrec;
 char new_id[MAXPRID];
 index_record inrec;
 FILE *fp;

 if (exists(PtsRoot, db_path, "", ""))
 {
    fprintf(error_log, "Opening reopen index\n");
    if ((fp=open_file(db_path, ".index", "r+", INDEX))==NULL)
       return(FAIL);
    fprintf(error_log, "Locking index\n");
    if (lock_file(fp)==FAIL)
       return(FAIL);
/* now read the record counter from the index file */
    fread(&numrec, sizeof(long), 1, fp);
    while (fread(&inrec, sizeof(index_record), 1, fp)==1)
    {
 /* first we change the prid and the file names associated with it */
       new_id[0]=inrec.prid[3];
       new_id[1]=inrec.prid[4];
       new_id[2]=inrec.prid[5];
       new_id[3]=inrec.prid[6];
       new_id[4]=inrec.prid[0];
       new_id[5]=inrec.prid[1];
       new_id[6]=inrec.prid[2];
       new_id[7]=inrec.prid[7];
       new_id[8]=inrec.prid[8];
       new_id[9]=inrec.prid[9];
       new_id[10]=inrec.prid[10];
       new_id[11]=inrec.prid[11];
       new_id[12]=inrec.prid[12];
       new_id[13]='\0';
       rename_file(db_path, inrec.prid, ".log",
                    db_path, new_id, ".log");
       rename_file(db_path, inrec.prid, ".rep",
                    db_path, new_id, ".rep");
       strcpy(inrec.prid, new_id);
 /* then we change the link to the orignal problem */
       new_id[0]=inrec.ln_prid[3];
       new_id[1]=inrec.ln_prid[4];
       new_id[2]=inrec.ln_prid[5];
       new_id[3]=inrec.ln_prid[6];
       new_id[4]=inrec.ln_prid[0];
       new_id[5]=inrec.ln_prid[1];
       new_id[6]=inrec.ln_prid[2];
       new_id[7]=inrec.ln_prid[7];
       new_id[8]=inrec.ln_prid[8];
       new_id[9]=inrec.ln_prid[9];
       new_id[10]=inrec.ln_prid[10];
       new_id[11]=inrec.ln_prid[11];
       new_id[12]=inrec.ln_prid[12];
       new_id[13]='\0';
       strcpy(inrec.ln_prid, new_id);

 /* now back up the file pointer and write out the new entry */
       fseek(fp, (long)sizeof(index_record), SEEK_CUR);
       fwrite(&inrec, sizeof(index_record), 1, fp);
    } /* end while */
    if (ferror(fp)) /*did we leave the loop due to an error? */
    {
       unlock_file(fp);
       return(FAIL);
    }
 } /* end reopen index exists */
 else
    fprintf(error_log, "The reopen leaf, does not exist\n");
} /* end convert_reopen() */

/* fixpath
 *
 * This function gets rid of the '\n' after a call to fgets().
 *
 */

char *fixpath(char *string)
{
 int len;

 len=strlen(string);
 string[len-1]='\0';
 return(string);
} /* end fixpath() */

/* end convertindex.c */
