////////////////////////////////////////////////////////////////////////////////
//
//	Class : mxMailAddressBook
//
//	Author : Andy Jefferson
//
//	Description :
//		mxMailAddressBook is a representation of an address book.
//		The address book consists of an array (vector) of mail addresses
//		(mxMailAddress), as well as a filename, and a method of sorting
//		the array of addresses.
//
//	Public Methods :
//		(Constructor) requires :
//			filename	Filename where the addressbook is stored
//
//		sortByName()	Sort the addresses by the full-name.
//		sortByAlias()	Sort the addresses by the alias.
//		sortByAddress()	Sort the addresses by the mail address
//		insert()	Insert an address entry into the array.
//		remove()	Remove an address entry from the array.
//		modify()	Modify an address entry in the array.
//		read()		Read the addressbook from file.
//		write()		Write the addressbook back to file, overwriting
//				the previous contents.
//		addressForAlias()	Returns an address given the alias.
//		nameForAlias()		Returns the name given the alias.
//		nameForAddress()	Returns the name given the address.
//		filename()	Returns the filename of the address book.
//		sortMethod()	Returns the sort method of the address book.
//		noOfAddresses()	Returns the no of addresses in the store.
//		address()	Returns the requested mxMailAddress object.
//		addressTitle()	Returns a title for the requested address.
//
//	Limitations :
//		Assumes that the addressbook is made up of entries on
//		individual lines, and fields within entries are separated
//		by a SINGLE tab in the order
//
//		alias	name	address	other
//
////////////////////////////////////////////////////////////////////////////////
#ifndef mxMailAddressBook_h
#define mxMailAddressBook_h
#ifdef __cplusplus

#include <vector>
#include <algorithm>
#include <string>
#include "mxMailAddress.h"

typedef enum
{
	SORT_BY_ALIAS,SORT_BY_NAME,SORT_BY_ADDRESS
} mxMailAddressSortMethod;

class mxMailAddressBook
{
	public:

//		Constructor, Destructor, assignment operator etc

		mxMailAddressBook();
		mxMailAddressBook(string the_filename);
		~mxMailAddressBook();
		mxMailAddressBook& operator=(const mxMailAddressBook&);

//		Access functions

		const string&		filename() const;
		int			noOfAddresses() const;
		mxMailAddressSortMethod	sortMethod() const;
		const mxMailAddress&	address(int addr_no) const;

		string		addressForAlias(string alias) const;
		string		nameForAlias(string alias) const;
		string		nameForAddress(string addr) const;
		string		addressTitle(int addr_no) const;

//		Update functions

		void		read();
		void		write();

		void		sortByAlias();
		void		sortByName();
		void		sortByAddress();

		void		insert(const mxMailAddress& addr);
		void		remove(int addr_no);
		void		modify(int addr_no,const mxMailAddress& addr2);

		friend ostream&	operator<< (ostream& s,mxMailAddressBook& book);

	private:

		void		reSort();

		// Data members

		string			_filename;
		mxMailAddressSortMethod	_sortMethod;
		vector<mxMailAddress>	_addresses;
};

inline
const string&	mxMailAddressBook::filename() const
	{return _filename;}
inline
int		mxMailAddressBook::noOfAddresses() const
	{return _addresses.size();}
inline
mxMailAddressSortMethod	mxMailAddressBook::sortMethod() const
	{return _sortMethod;}
inline
const mxMailAddress&	mxMailAddressBook::address(int addr_no) const
{
	if (addr_no < 0 || addr_no >= _addresses.size())
	  return _addresses[0];
	else
	  return _addresses[addr_no];
}
inline
void	mxMailAddressBook::remove(int addr_no)
{
	if (addr_no < 0 || addr_no >= _addresses.size())
	  return;
	else
	  _addresses.erase(_addresses.begin()+addr_no);
	reSort();
}
inline
void	mxMailAddressBook::modify(int addr_no,const mxMailAddress& addr)
{
	if (addr_no < 0 || addr_no >= _addresses.size())
	  return;
	else
	  *(_addresses.begin()+addr_no) = addr;
	reSort();
}
inline
void	mxMailAddressBook::insert(const mxMailAddress& addr)
{
	_addresses.push_back(addr);
	reSort();
}

#endif
#endif
