////////////////////////////////////////////////////////////////////////////////
//
//	Class : akApp
//
//	Author : Andy Jefferson
//
//	Description :
//		The akApp class is the top-level definition of any ak-based
//		program. It provides for initialisation of X, setting up of the
//		event queue, as well as management of all top-level shells.
//		The pointer at the foot of the header points to an instance
//		of the akApp class. When the user creates an instance of the
//		akApp class (at startup, as a global), theAkApp is set to
//		point to it. From this, the rest of the application has access
//		to the akApp object (by including this file), and can
//		subsequently manage/unmanage/iconise the application.
//		The akApp class maintains a list of all top-level windows
//		- allowing addition/deletion via the functions
//		"registerWindow"/"unregisterWindow" respectively.
//
//	Public Methods :
//		(Constructor) requires :
//			name		Application name - for resources
//			title		Application title
//			version		Application version
//		manage()		Manages the application
//		unmanage()		Unmanages the application
//		iconify()		Iconises the application
//		busy()			Turns the cursor to a stopwatch
//		ready()			Turns the cursor to a stopwatch
//
//
//	Limitations :
//
////////////////////////////////////////////////////////////////////////////////
// This code is ADAPTED from the book:
//
// Object-Oriented Programming with C++ and OSF/Motif, 2nd Edition
//    by
// Douglas Young
// Prentice Hall, 1995
// ISBN 0-13-20925507
//
// Copyright 1995 by Prentice Hall
// All Rights Reserved
//
// Permission to use, copy, modify, and distribute this software for 
// any purpose except publication and without fee is hereby granted, provided 
// that the above copyright notice appear in all copies of the software.
//
////////////////////////////////////////////////////////////////////////////////
#ifndef _akApp_h
#define _akApp_h
#ifdef __cplusplus

#include "akComponent.h"
#include <vector>
class akWindow;

class akApp : public akComponent
{

		// Allow main & akWindow to access protected member funcs

		friend int main(int argc,char **argv);
		friend akWindow;	// akWindow needs to call 
					// private functions for registration
	public:

		akApp(const string app_name,
		      const string app_version,
		      const string app_title);

		// Manage/Unmanage

		void manage(void);
		void unmanage(void);

		// Iconify

		void iconify(void);

		// Busy/Ready

		void busy(void);
		void ready(void);

		// Colormap control

		void		setColormap(Colormap cmap);

		// Convenient access functions

		Display 	*display(void)
			{return (_display);}
		Colormap 	colormap(void)
			{return (_colormap);}
		XtAppContext	appContext(void)
			{return (_appContext);}
		const string&	applicationName(void)
			{return (_applicationName);}
		const string&	applicationVersion(void)
			{return (_applicationVersion);}
		const string&	applicationTitle(void)
			{return (_applicationTitle);}
		virtual const char *const className(void)
			{return ("akApp");}

	protected:

		// Support commonly needed data structures as a convenience

		Display		*_display;
		Colormap	_colormap;
		XtAppContext 	_appContext;

		// Functions to handle Xt interface

		virtual void initialize(int *argc,char **argv);  
		virtual void handleEvents(void);

		string			_applicationName;
		string 			_applicationVersion;
		string			_applicationTitle;
		vector<akWindow*>  	_windows;   

	private:

		// Functions for registering and unregistering top level windows

		void registerWindow(akWindow *window);
		void unregisterWindow(akWindow *window);
};

// Pointer to single global instance
extern akApp *theAkApp; 

inline
void	akApp::setColormap(Colormap cmap)
{
	_colormap = cmap;
	XtVaSetValues(_w,XmNcolormap,cmap,0);
}

#endif
#endif
