//////////////////////////////////////////////////////////////////////////////
// This example code is from the book:
//
// Object-Oriented Programming with C++ and OSF/Motif, 2nd Edition
//   by
// Douglas Young
// Prentice Hall, 1995
// ISBN 0-13-20925507
//
// Copyright 1995 by Prentice Hall
// All Rights Reserved
//
// Permission to use, copy, modify, and distribute this software for 
// any purpose except publication and without fee is hereby granted, provided 
// that the above copyright notice appear in all copies of the software.
///////////////////////////////////////////////////////////////////////////////
#include "akApp.h"
#include "akWindow.h"
#include <Xm/MainW.h>
#include <X11/cursorfont.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/IntrinsicP.h>
#include <X11/CompositeP.h>
#include <X11/CoreP.h>
#include <Xm/Protocols.h>
#include <assert.h>
#include <iostream.h>
////////////////////////////////////////////////////////////////////////////////
//
//	Constructor
//
////////////////////////////////////////////////////////////////////////////////
akWindow::akWindow(const char *name)
	: akComponent(name),
	  _icon((akPixmap *)0),
	  _workArea((Widget)0),
	  _iconic(FALSE)
{
//	akApp object must exist before any akWindow object

	assert(theAkApp);

	theAkApp->registerWindow(this);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destructor
//
////////////////////////////////////////////////////////////////////////////////
akWindow::~akWindow(void)
{
	// Unregister this window with the akApp object

	theAkApp->unregisterWindow(this);
	if (_icon)
	  delete _icon;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Initialize function - create the shell etc
//
////////////////////////////////////////////////////////////////////////////////
void akWindow::initialize(void)
{
	Arg	args[10];

	// Only initialize if not yet initialized

	if (!_w)
	  {

	   // All top-level windows in the MotifApp framework are 
	   // implemented as a popup shell off the akApp's
	   // base widget

	   XtSetArg(args[0],XmNtitle,theAkApp->applicationName().c_str());
	   _w = XtCreatePopupShell(_name, 
				   applicationShellWidgetClass,
				   theAkApp->baseWidget(),
				   args,1);
	   installDestroyHandler();

	   // Install handler to keep track of state changes

	   XtAddEventHandler(_w,StructureNotifyMask,FALSE,
			     &akWindow::stateChangedCB,(XtPointer)this);

	   // Use a Motif XmMainWindow widget to handle window layout
  
	   _main = XtCreateManagedWidget("mainWindow", 
				         xmMainWindowWidgetClass,
				         _w,NULL,0);

	   // Called derived class to create the work area

	   _workArea = createWorkArea(_main);  
	   assert(_workArea);

	   // Designate the _workArea widget as the XmMainWindow
	   // widget's XmNworkArea widget
  
	   XtVaSetValues(_main,XmNworkWindow,_workArea,NULL);
  
	   // Manage the work area if the derived class hasn't already
  
	   if (!XtIsManaged(_workArea))
             XtManageChild(_workArea); 
	  }
}
////////////////////////////////////////////////////////////////////////////////
//
//	Manage the window
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::manage()
{
	assert(_w);
	XtPopup(_w,XtGrabNone);

	// Map the window, in case the window is iconified

	if (XtIsRealized(_w))
	XMapRaised(theAkApp->display(),XtWindow(_w));
}
////////////////////////////////////////////////////////////////////////////////
//
//	Unmanage the window
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::unmanage()
{
	assert(_w);
	XtPopdown(_w);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Install the Window Manager close mechanism
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::installWMClose()
{
	assert(_w);

	// Add destruction mechanism for Window Manager

	Atom	WM_DELETE_WINDOW;

	XtVaSetValues(_w,XmNdeleteResponse,XmDO_NOTHING,NULL);
	WM_DELETE_WINDOW = XmInternAtom(XtDisplay(_w),"WM_DELETE_WINDOW",False);
	XmAddWMProtocolCallback(_w,WM_DELETE_WINDOW,
				&akWindow::wmCloseCB,(XtPointer)this);
}
////////////////////////////////////////////////////////////////////////////////
//
//	wmCloseCB - close function for window manager
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::wmCloseCB(Widget,XtPointer clientData,XtPointer)
{
	akWindow	*obj = (akWindow *)clientData;

	obj->unmanage();
}
////////////////////////////////////////////////////////////////////////////////
//
//	stateChangedCB - Window state has changed
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::stateChangedCB(Widget,XtPointer clientData,XEvent *event,Boolean *flag)
{
	akWindow	*obj = (akWindow *)clientData;

//	Window has been iconised (unmapped)

	if (event->type == UnmapNotify)
	  obj->_iconic = true;

//	Window has been de-iconised (mapped)

	else if (event->type == MapNotify)
	  obj->_iconic = false;

//	Something else has changed ....

	else
	  {}
}
////////////////////////////////////////////////////////////////////////////////
//
//	Iconise the window
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::iconify(void)
{
	assert(_w);

	// Set the widget to have an initial iconic state
	// in case the base widget has not yet been realized

	XtVaSetValues(_w,XmNiconic,TRUE,NULL);

	// If the widget has already been realized, iconify it

	if (XtIsRealized(_w))
	  XIconifyWindow(theAkApp->display(),XtWindow(_w),0);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the icon - pixmap
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::setIcon(string name,char **pix)
{
	Pixmap	pixmap;

	if (_icon)
	  delete _icon;
	_icon = new akPixmap(pix);
	pixmap = _icon->pixmap();
	if (pixmap)
	  XtVaSetValues(_w,
			XmNiconPixmap,pixmap,
			XmNiconName,(char *)name.c_str(),
			0);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the icon - bitmap
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::setIcon(string name,int pix_width,int pix_height,char pix_bits[])
{
	assert(_w);

	Pixmap	pixmap;

	if (_icon)
	  delete _icon;
	_icon = new akPixmap(pix_width,pix_height,pix_bits);
	pixmap = _icon->pixmap();
	if (pixmap != (Pixmap)0)
	  XtVaSetValues(_w,
			XmNiconPixmap,pixmap,
			XmNiconName,(char *)name.c_str(),
			0);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the window title
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::setTitle(string title)
{
	assert(_w);
	XtVaSetValues(_w,XmNtitle,title.c_str(),0);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the window (and popups) to busy
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::busy(void)
{
	assert(_w);

//	Set cursor for shell

	akComponent::busy();

//	Set cursor for all popups

	for (int i=0;i<_w->core.num_popups;i++)
	  if (XtIsRealized(_w->core.popup_list[i]))
	    busyWidget(_w->core.popup_list[i]);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the window (and popups) to ready
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::ready(void)
{
	assert(_w);

//	Set cursor for shell

	akComponent::ready();

//	Set cursor for all popups

	for (int i=0;i<_w->core.num_popups;i++)
	  if (XtIsRealized(_w->core.popup_list[i]))
	    readyWidget(_w->core.popup_list[i]);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set the widget to busy
//
////////////////////////////////////////////////////////////////////////////////
void 	akWindow::busyWidget(Widget w)
{
	Display		*display=theAkApp->display();

	assert(w);

	if (display != (Display *)0 && XtWindow(w) != (Window)0)
	  XDefineCursor(display,XtWindow(w),_busyCursor);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set a widget to ready
//
////////////////////////////////////////////////////////////////////////////////
void	akWindow::readyWidget(Widget w)
{
	Display	*display=theAkApp->display();

	assert(w);

	if (display != (Display *)0 && XtWindow(w) != (Window)0)
	  XUndefineCursor(display,XtWindow(w));
}
