#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <iostream.h>
#include <fstream.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "mxMailBox.h"

////////////////////////////////////////////////////////////////////////////////
//	Sorting algorithms for "array" of mxMsgSummary objects
////////////////////////////////////////////////////////////////////////////////
class SortBySize
{
	public:
		bool operator() (const mxMsgSummary& p,const mxMsgSummary& q) const
			{return p.length() < q.length();}
};
class SortByDate
{
	public:
		bool operator() (const mxMsgSummary& p,const mxMsgSummary& q) const
			{return p.position() < q.position();}
};
class SortByAuthor
{
	public:
		bool operator() (const mxMsgSummary& p,const mxMsgSummary& q) const
			{return p.author() < q.author();}
};
class SortBySubject
{
	public:
		bool operator() (const mxMsgSummary& p,const mxMsgSummary& q) const
			{return p.subject() < q.subject();}
};
////////////////////////////////////////////////////////////////////////////////
//
//	Default Constructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailBox::mxMailBox()
	:_filename(""),
	 _type(FOLDER),
	 _sortMethod(SORT_BY_DATE),
	 _readTime(0)
{
//	Empty
}
////////////////////////////////////////////////////////////////////////////////
//
//	Constructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailBox::mxMailBox(string the_filename,mxMailBoxType the_type)
	:_filename(the_filename),
	 _type(the_type),
	 _sortMethod(SORT_BY_DATE),
	 _readTime(0)
{
	read();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailBox::~mxMailBox()
{
	write();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Assignment operator
//
////////////////////////////////////////////////////////////////////////////////
mxMailBox&	mxMailBox::operator=(const mxMailBox& box)
{
	if (this != &box)
	  {
	   _filename   = box._filename;
	   _type       = box._type;
	   _sortMethod = box._sortMethod;
	   _messages   = box._messages;
	   _readTime   = box._readTime;
	  }
	return *this;
}
////////////////////////////////////////////////////////////////////////////////
//
//	ostream operator
//
////////////////////////////////////////////////////////////////////////////////
ostream&	operator<<(ostream& s,mxMailBox& box)
{
	string		type_name;
	string		sort_name;

	if (box._type == INBOX)
	  type_name = "INBOX";
	else if (box._type == OUTBOX)
	  type_name = "OUTBOX";
	else
	  type_name = "FOLDER";
	if (box._sortMethod == SORT_BY_AUTHOR)
	  sort_name = "SORT_BY_AUTHOR";
	else if (box._sortMethod == SORT_BY_SUBJECT)
	  sort_name = "SORT_BY_SUBJECT";
	else if (box._sortMethod == SORT_BY_SIZE)
	  sort_name = "SORT_BY_SIZE";
	else
	  sort_name = "SORT_BY_DATE";
	return s <<
		"mxMailBox" << endl <<
		"  Filename:       " << box._filename << endl <<
		"  No Of Messages: " << box._messages.size() << endl <<
		"  Box Type:       " << type_name << endl <<
		"  Sort Method:    " << sort_name << endl;
}
////////////////////////////////////////////////////////////////////////////////
//
//	sortBySize - sort the messages by the size
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailBox::sortBySize()
{
	if (_sortMethod == SORT_BY_SIZE)
	  return;
	_sortMethod = SORT_BY_SIZE;
	stable_sort(_messages.begin(),_messages.end(),SortBySize());
}
////////////////////////////////////////////////////////////////////////////////
//
//	sortByDate - sort the messages by the date
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailBox::sortByDate()
{
	if (_sortMethod == SORT_BY_DATE)
	  return;
	_sortMethod = SORT_BY_DATE;
	stable_sort(_messages.begin(),_messages.end(),SortByDate());
}
////////////////////////////////////////////////////////////////////////////////
//
//	sortByAuthor - sort the messages by the author
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailBox::sortByAuthor()
{
	if (_sortMethod == SORT_BY_AUTHOR)
	  return;
	_sortMethod = SORT_BY_AUTHOR;
	stable_sort(_messages.begin(),_messages.end(),SortByAuthor());
}
////////////////////////////////////////////////////////////////////////////////
//
//	sortBySubject - sort the messages by the subject
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailBox::sortBySubject()
{
	if (_sortMethod == SORT_BY_SUBJECT)
	  return;
	_sortMethod = SORT_BY_SUBJECT;
	stable_sort(_messages.begin(),_messages.end(),SortBySubject());
}
////////////////////////////////////////////////////////////////////////////////
//
//	messageText - returns the message text for reqd message no
//
////////////////////////////////////////////////////////////////////////////////
string	mxMailBox::messageText(int msg_no) const
{
	string	message_text;
	FILE	*fp;
	char	*s;

//	Check for erroneous input

	if (msg_no < 0 || msg_no >= _messages.size())
	  message_text = "";
	else
	   {
	    // Read message from mail-box - using start position and length
	    fp = fopen(_filename.c_str(),"r");
	    if (fp)
	       {	
		if (fseek(fp,_messages[msg_no].position(),0) == 0)
		   {
		    s = new char[_messages[msg_no].length()+1];
		    memset(s,'\0',_messages[msg_no].length()+1);
		    fread(s,_messages[msg_no].length(),1,fp);
		    fclose(fp);
		    message_text = s;
		    delete s;
		   }
	       }
	   }
	return message_text;
}
////////////////////////////////////////////////////////////////////////////////
//
//	write - Update the mailbox file
//	Update 'Status' of messages whether they are 'read' or not
//
////////////////////////////////////////////////////////////////////////////////
void 	mxMailBox::write()
{
//	Read in any additional messages that have arrived since (if any)

	read();

//	Read all messages 'text' in before writing mail box back

	for (int msg_no=0;msg_no<_messages.size();msg_no++)
	  _messages[msg_no].setActualText(messageText(msg_no));

//	Add 'Status: ' tag if not present, and check it is correct

	vector<mxMsgSummary>::iterator	iter;
	for (iter=_messages.begin();iter<_messages.end();iter++)
	   {
	    int		pos_status;
	    string	msg_text=(*iter).actualText();

	    pos_status = msg_text.find("\nStatus: ");

	    // Already got a 'Status:' tag - remove it
	    if (pos_status >= 0)
	       {
		int	pos_status_end=msg_text.find("\n",pos_status+1);
		msg_text.remove(pos_status+1,pos_status_end-pos_status);
	       }

	    // Add 'Status: ' tag reflecting 'Read' status.
	    // Position it as the last Header tag (just before "\n\n")
	    int	pos_status_insert;
	    int	pos_header_end;

	    // Find location of End-of-Header - put Status before it
	    pos_header_end = msg_text.find("\n\n");
	    if (pos_header_end >= 0 && pos_header_end < msg_text.size())
	      pos_status_insert = pos_header_end+1;
	    else
	      pos_status_insert = -1;

	    // Found valid insert position so add the tag
	    if (pos_status_insert >= 0)
	       {
		if ((*iter).status())
		  msg_text.insert(pos_status_insert,"Status: RO\n");
		else
		  msg_text.insert(pos_status_insert,"Status: O\n");
		(*iter).setActualText(msg_text);
	       }
	   }

//	Write updated mail-box - in date order

	ofstream	outfile;

	sort(_messages.begin(),_messages.end(),SortByDate());
	outfile.open(_filename.c_str());
	if (outfile)
	  for (iter=_messages.begin();iter<_messages.end();iter++)
	    outfile << (*iter).actualText();
	else
	  cerr << "mxMailBox : Couldn't open mailbox for writing" << endl;
	outfile.close();

//	Delete all messages and re-read them since positions have changed

	while (_messages.size() > 0)
	  _messages.pop_back();
	_readTime = 0;
	read();	// This will automatically put messages back in reqd order
}
////////////////////////////////////////////////////////////////////////////////
//
//	read - Read the mailbox file and update the internal message store
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailBox::read()
{
	char 		line[1024];
	int		linelen=1024;
	long 		start;
	long 		length;
	long		lineStartPos;
	int		msg_no=0;
	int		start_msg_no=0;
	int		end_msg_no=0;
	string		msg_text;
	mxMsgSummary	*new_msg=(mxMsgSummary *)0;
	struct stat	stbuf;

//	Check last modified time for mailbox - return if not updated recently

	if (stat(_filename.c_str(),&stbuf) != -1)
	  if (stbuf.st_mtime <= _readTime)
	    return;

//	Check no of messages in box - set start and end message no's

	start_msg_no = _messages.size();
	end_msg_no   = noOfMsgsLatest() - 1;

	// No of messages is same or reduced, but somehow changed, so delete
	// all existing messages, and re-read the lot
	if (end_msg_no+1 <= start_msg_no)
	  {
	   while (_messages.size() > 0)
	     _messages.pop_back();
	   start_msg_no = 0;
	  }
	// More messages in mailbox, so read the new ones
	else
	  {
	  }

//	Sort existing messages (if any) by date before reading

	sort(_messages.begin(),_messages.end(),SortByDate());

//	Open mail file for reading - reading messages from
//	'start_msg_no' to 'end_msg_no'

	FILE	*fp = fopen(_filename.c_str(),"r");
	time(&_readTime);

//	Jump to end of last message (by date) in mailbox

	if (end_msg_no > _messages.size() && _messages.size() > 0)
	  {
	   start  = _messages[_messages.size()-1].position();
	   length = _messages[_messages.size()-1].length();
	   fseek(fp,start+length,0);
	   msg_no = start_msg_no;
	  }

//	Read the file from this point

	while ((fp != 0) && fgets(line,linelen,fp))
	  {
	   lineStartPos = ftell(fp) - strlen(line);

//	   At end of message (start of next message) - add message to mailbox

	   if (AtStartOfMailMessage(line))
	     {

//	      If not first, start pt and message is required

	      if (msg_no>0 && (msg_no-1)>=start_msg_no && (msg_no-1)<=end_msg_no)
		{
		 // Add message to mxMailBox
		 new_msg = new mxMsgSummary;
		 new_msg->setPosition(start);
		 new_msg->setMessageHeaders(msg_text);
		 _messages.push_back(*new_msg);
	         delete new_msg;
		}
	      msg_text.remove();

//	      Set start of next message

	      msg_no++;
	      start = lineStartPos;
	     }
	   msg_text += line;
	  }

	if ((msg_no-1)>=start_msg_no && (msg_no-1)<=end_msg_no)
	  {
	   // Add message to mxMailBox
	   new_msg = new mxMsgSummary;
	   new_msg->setPosition(start);
	   new_msg->setMessageHeaders(msg_text);
	   _messages.push_back(*new_msg);
	   delete new_msg;
	  }
  	fclose(fp);

//	Reorder back to required order

	if (_sortMethod == SORT_BY_AUTHOR)
	  sortByAuthor();
	else if (_sortMethod == SORT_BY_SUBJECT)
	  sortBySubject();
	else if (_sortMethod == SORT_BY_SIZE)
	  sortBySize();
	else if (_sortMethod == SORT_BY_DATE)
	  sortByDate();
}
////////////////////////////////////////////////////////////////////////////////
//
//	noOfMsgsLatest - Reads mailbox and finds no of messages present.
//
////////////////////////////////////////////////////////////////////////////////
int	mxMailBox::noOfMsgsLatest() const
{
	char		line[1024];
	int		linelen=1024;
	int		nmessages;
	struct stat	stbuf;
	ifstream	infile;

//	Check last modified time for mailbox - return current no if not updated

	if (stat(_filename.c_str(),&stbuf) != -1)
	  if (stbuf.st_mtime <= _readTime)
	    return noOfMsgs();

//	Read file and check no of messages

	nmessages = 0;
	infile.open(_filename.c_str());
	if (infile)
	  {
	   while (infile.getline(line,linelen,'\n'))
	     if (AtStartOfMailMessage(line))
	       nmessages++;
	   infile.close();
	  }
	else
	  return _messages.size();

	return nmessages;
}
////////////////////////////////////////////////////////////////////////////////
//
//	noOfMsgsLatestRead - Reads mailbox and finds no of messages 'read'.
//
////////////////////////////////////////////////////////////////////////////////
int	mxMailBox::noOfMsgsLatestRead() const
{
	char		line[1024];
	int		linelen=1024;
	int		nmessages=0;
	struct stat	stbuf;
	ifstream	infile;

//	Check last modified time for mailbox - return current no if not updated

	if (stat(_filename.c_str(),&stbuf) != -1)
	  if (stbuf.st_mtime <= _readTime)
	    return noOfMsgsRead();

//	Read file and check no of messages

	infile.open(_filename.c_str());
	if (infile)
	   {
	    bool	checking_for_status=false;
	    while (infile.getline(line,linelen,'\n'))
	       {
	        if (AtStartOfMailMessage(line))
		   {
		    checking_for_status = true;
		   }
		// Check for status tag
		if (checking_for_status)
		   {
		    string	temp=line;
		    int		pos=temp.find("Status: ");

		    // Tag found
		    if (pos == 0)
		       {
			pos = temp.find("Status: RO");
			if (pos == 0)
			  nmessages++;
			checking_for_status = false;
		       }
		   }
	       }
	    infile.close();
	   }

	return nmessages;
}
////////////////////////////////////////////////////////////////////////////////
//
//	AtStartOfMailMessage
//	Takes as input the line, and returns an indicator for whether the line
//	is the start of a mail message.
//
//	A valid line will be in the form :-
//	From <user> <weekday> <month> <day> <hr:min:sec>  ....
//		.... [TZ1 [TZ2]] <year> [remote from sitelist]
//
////////////////////////////////////////////////////////////////////////////////
bool 	mxMailBox::AtStartOfMailMessage(char *line)
{
	char	field[255]; // Buffer for current field of line
	int	len;        // Length of current field
	int	pos;        // Position within line string

//	Find "From"

	if ((len = GetWordFromString(line,0,field,sizeof(field),&pos)) < 0 ||
	    (strcmp(field,"From") != 0) || (pos != 0))
	  return false;
	line += len;

//	Find <user>

	if ((len = GetWordFromString(line,0,field,sizeof(field),&pos)) < 0)
	  return false;
	line += len;

//	Find <weekday>

	if ((len = GetWordFromString(line,0,field,sizeof(field),&pos)) < 0)
	  return false;
	line += len;

//	Find <month>

	if ((len = GetWordFromString(line,0,field,sizeof(field),&pos)) < 0)
	   return false;
	line += len;

	return true;
}
////////////////////////////////////////////////////////////////////////////////
//	GetWordFromString
//	Extracts the next white-space delimited word from the "line" starting
//	"start" characters into the line and skipping any leading white-space
//	there.  Handles backslash-quoted characters and double-quote bracketed
//	strings as an atomic unit.  The resulting word, up to "wordlen" bytes
//	long, is saved in "word".  Returns the line index where extraction is
//	terminated, e.g. the next word can be extracted by starting at
//	start+<return-val>. Returns -1 if no words are found.
//
////////////////////////////////////////////////////////////////////////////////
int	mxMailBox::GetWordFromString(char *line,int start,char *word,
			  	   int wordlen,int *pos)
{
    	register int len;
    	register char *p;
	register char *c,quot;

    	for (p = line+start;isspace(*p);++p);

    	if (*p == '\0')
	  return (-1);	 // nothing IN line!
	if (pos != 0)	 // save the position ("From" valid only as first char)
	  *pos = (int)(p - (line+start));

//	This finds the length of the next part of the address/data field
//	It calculates the length of the next part of the
//	string field containing address/data.  It takes into account
//	quoting via " as well as \ escapes.
//	Quoting via ' is not taken into account, as RFC-822 does not
//	consider a ' character a valid 'quoting character'
//	len = 1 for a single character unless:
//	len = 0 at end of string.
//	len = 2 for strings that start \
//	The length of quoted sections is returned for quoted fields

    	while (*p != '\0')
	  {
	   len = LengthOfNextPart(p);
	   if (len == 1 && isspace(*p))
	     break;

	   while (--len >= 0)
	     {
	      if (--wordlen > 0)
		*word++ = *p;
	      ++p;
	     }
          }

    	*word = '\0';
    	return (p - line);
}
////////////////////////////////////////////////////////////////////////////////
int	mxMailBox::LengthOfNextPart(register char *s)
{
	register char	*c,quot;

	quot = *s;
	if (quot == '\0')
	  return(0);
	if (quot == '\\')
	  return(*++s != '\0' ? 2 : 1);
	if (quot != '"')
	  return(1);

   	for (c = s+1; *c; c++)
	  {
	   if (*c == quot)
	     return(1+c-s);
           if (*c == '\\')
	     if (*c++)
	       c++;
	  }

	return(c-s);
}
