#include <stdio.h>
#include <ctype.h>
#include <iostream.h>
#include <fstream.h>
#include <pwd.h>
#include <stdlib.h>
#include <sys/types.h>
#include "mxMailAddressBook.h"

////////////////////////////////////////////////////////////////////////////////
class SortByAlias
{
	public:
		bool	operator() (const mxMailAddress& p,const mxMailAddress& q) const
			{return p.alias() < q.alias();}
};
class SortByName
{
	public:
		bool	operator() (const mxMailAddress& p,const mxMailAddress& q) const
			{return p.name() < q.name();}
};
class SortByAddress
{
	public:
		bool	operator() (const mxMailAddress& p,const mxMailAddress& q) const
			{return p.address() < q.address();}
};
////////////////////////////////////////////////////////////////////////////////
//
//	Default Constructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailAddressBook::mxMailAddressBook()
	:_filename(""),
	 _sortMethod(SORT_BY_ALIAS)
{
//	Empty
}
////////////////////////////////////////////////////////////////////////////////
//
//	Constructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailAddressBook::mxMailAddressBook(string the_filename)
	:_filename(the_filename),
	 _sortMethod(SORT_BY_ALIAS)
{
	read();
	sortByAlias();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailAddressBook::~mxMailAddressBook()
{
	write();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Assignment operator
//
////////////////////////////////////////////////////////////////////////////////
mxMailAddressBook&	mxMailAddressBook::operator=(const mxMailAddressBook& book)
{
	if (this != &book)
	  {
	   _filename  = book._filename;
	   _addresses = book._addresses;
	  }
	return *this;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Re-Sort function
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailAddressBook::reSort()
{
	if (_sortMethod == SORT_BY_ALIAS)
	  stable_sort(_addresses.begin(),_addresses.end(),SortByAlias());
	else if (_sortMethod == SORT_BY_NAME)
	  stable_sort(_addresses.begin(),_addresses.end(),SortByName());
	else if (_sortMethod == SORT_BY_ADDRESS)
	  stable_sort(_addresses.begin(),_addresses.end(),SortByAddress());
}
////////////////////////////////////////////////////////////////////////////////
//
//	Sort by alias function
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailAddressBook::sortByAlias()
{
	if (_sortMethod == SORT_BY_ALIAS)
	  return;
	_sortMethod = SORT_BY_ALIAS;
	stable_sort(_addresses.begin(),_addresses.end(),SortByAlias());
}
////////////////////////////////////////////////////////////////////////////////
//
//	Sort by name function
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailAddressBook::sortByName()
{
	if (_sortMethod == SORT_BY_NAME)
	  return;
	_sortMethod = SORT_BY_NAME;
	stable_sort(_addresses.begin(),_addresses.end(),SortByName());
}
////////////////////////////////////////////////////////////////////////////////
//
//	Sort by name function
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailAddressBook::sortByAddress()
{
	if (_sortMethod == SORT_BY_ADDRESS)
	  return;
	_sortMethod = SORT_BY_ADDRESS;
	stable_sort(_addresses.begin(),_addresses.end(),SortByAddress());
}
////////////////////////////////////////////////////////////////////////////////
//
//	addressTitle - Return a title for the specified address
//
////////////////////////////////////////////////////////////////////////////////
string 	mxMailAddressBook::addressTitle(int addr_no) const
{
	string		address_title;

	if (addr_no < 0 || addr_no >= _addresses.size())
	  address_title = "";
	else
	  {
	   address_title = _addresses[addr_no].alias();
	   while (address_title.size() < 10)
	     address_title += " ";
	   address_title+= " - " + _addresses[addr_no].name() + " (" + _addresses[addr_no].address() + ")";
	  }

	return address_title;
}
////////////////////////////////////////////////////////////////////////////////
//
//	write - Write addressbook back to file
//
////////////////////////////////////////////////////////////////////////////////
void 	mxMailAddressBook::write()
{
	ofstream	outfile;
	vector<mxMailAddress>::iterator	iter;

//	Write the addressbook

	outfile.open(_filename.c_str());
	if (outfile)
	  {
	   for (iter=_addresses.begin();iter<_addresses.end();iter++)
	     outfile << (*iter).alias() << "\t" <<
			(*iter).name() << "\t" <<
			(*iter).address() << "\t" <<
			(*iter).other() << endl;
	   outfile.close();
	  }
}
////////////////////////////////////////////////////////////////////////////////
//
//	read - Read addressbook from file
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailAddressBook::read()
{
	ifstream	infile;
	char		line[BUFSIZ];
	mxMailAddress	*new_address;

//	Delete existing addresses

	while (_addresses.size() > 0)
	  _addresses.pop_back();

//	Open the addressbook file for reading

	infile.open(_filename.c_str());
	if (infile)
	  {
	   while (infile.getline(line,BUFSIZ,'\n'))
	     {
	      string	tempstr;
	      string	tempalias;
	      string	tempname;
	      string	tempaddress;
	      string	tempother;
	      bool	alias_done=false;
	      bool	name_done=false;
	      bool	address_done=false;

	      new_address = new mxMailAddress;
	      for (int i=0;i<strlen(line);i++)
		{
		 if (line[i] == '\t')
		   {
		    if (!alias_done)
		      {
		       alias_done = true;
		       tempalias = tempstr;
		      }
		    else if (!name_done)
		      {
		       name_done = true;
		       tempname = tempstr;
		      }
		    else if (!address_done)
		      {
		       address_done = true;
		       tempaddress = tempstr;
		      }
		    else
		      tempother = tempstr;
		    tempstr.remove();
		   }
		 else
		   tempstr += line[i];
		}
	      if (tempstr.size() > 0)
		{
		 if (!alias_done)
		   tempalias = tempstr;
		 else if (!name_done)
		   tempname = tempstr;
		 else if (!address_done)
		   tempaddress = tempstr;
		 else
		   tempother = tempstr;
		}

	      new_address->setDetails(tempalias,tempname,tempaddress,tempother);
	      insert(*new_address);
	      delete new_address;
	     }
	   infile.close();
	  }
}
////////////////////////////////////////////////////////////////////////////////
//
//	iostream function
//
////////////////////////////////////////////////////////////////////////////////
ostream&	operator<<(ostream& s,mxMailAddressBook& book)
{
	return s <<
		"mxMailAddressBook" << endl <<
		"  Filename:        " << book._filename << endl <<
		"  No of addresses: " << book._addresses.size() << endl;
}
////////////////////////////////////////////////////////////////////////////////
//
//	addressForAlias - address lookup function
//
////////////////////////////////////////////////////////////////////////////////
string	mxMailAddressBook::addressForAlias(string the_alias) const
{
	string	the_address;

//	Search through all addresses for requested alias
//	Return the requested address if available, else return the alias

	the_address = the_alias;
	vector<mxMailAddress>::const_iterator	iter;
	for (iter=_addresses.begin();iter<_addresses.end();iter++)
	  {
	   if ((*iter).alias() == the_alias)
	     {
	      the_address = (*iter).address();
	      break;
	     }
	  }

	return the_address;
}
////////////////////////////////////////////////////////////////////////////////
//
//	nameForAlias - name lookup function
//
////////////////////////////////////////////////////////////////////////////////
string	mxMailAddressBook::nameForAlias(string the_alias) const
{
	string	the_name;

//	Search through all addresses for requested alias
//	Return the requested name if available, else return the alias

	the_name = the_alias;
	vector<mxMailAddress>::const_iterator	iter;
	for (iter=_addresses.begin();iter<_addresses.end();iter++)
	  {
	   if ((*iter).alias() == the_alias)
	     {
	      the_name = (*iter).name();
	      break;
	     }
	  }

	return the_name;
}
////////////////////////////////////////////////////////////////////////////////
//
//	nameForAddress - name lookup function using address
//
////////////////////////////////////////////////////////////////////////////////
string	mxMailAddressBook::nameForAddress(string the_address) const
{
	string		the_name;
	struct	passwd	*password;

//	Search through all addresses for requested address
//	Return the requested name if available, else return the address

	password = getpwnam(the_address.c_str());
	if (password != (struct passwd *)0)
	  the_name = password->pw_gecos;
	else
	  the_name = the_address;

	vector<mxMailAddress>::const_iterator	iter;
	for (iter=_addresses.begin();iter<_addresses.end();iter++)
	  {
	   if ((*iter).address() == the_address)
	     {
	      the_name = (*iter).name();
	      break;
	     }
	  }

	return the_name;
}
