/*
 * compressed.c
 *	Utility functions to deal with compressed/gzipped files.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@echo-on.net)
 */
#include	<wlib/rcs.h>
MU_ID("$Mu: mgv/unmangle.c,v 1.17 $")

#include	<assert.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<dirent.h>
#include	<unistd.h>
#include	<string.h>
#include	<errno.h>

#include	<wlib/wlib.h>
#include	<mine/mgv.h>
#include	<mine/unmangle.h>
#include	<mine/strings.h>

/*
 * the mode bits that we want on the temp directories (just a little
 * paranoia)
 */
#define	WANTMODE	(S_IRUSR | S_IWUSR | S_IXUSR)

/*
 * Where the temp files go (allocated with XtMalloc).
 */
static char *tempdir = NULL;

/*
 * atexit() function for cleaning up the temp directory
 */
static void
cleanup(void)
{
	DIR		*dir;
	struct dirent	*d;
	char		pwd[1024];

	if(tempdir == NULL
	|| getcwd(&pwd[0], sizeof(pwd) - 1) == NULL
	|| chdir(tempdir) != 0
	|| (dir = opendir(tempdir)) == NULL)
		return;

	/*
	 * I would prefer to use pointer arithmetic instead of
	 * strcmp() to check for "." and ".." but some systems
	 * (MP-RAS so far) define d_name to be an array of one char
	 * (so that struct dirent has a variable size, i.e. only one
	 * malloc is needed (false laziness!)) and that
	 * causes many compiler warnings about array bounds that
	 * I'd rather not see.
	 */
	while((d = readdir(dir)) != NULL) {
		if(strcmp(d->d_name, ".")  == 0
		|| strcmp(d->d_name, "..") == 0)
			continue;
		remove(d->d_name);
	}

	closedir(dir);
	chdir(pwd);
	rmdir(tempdir);
	XtFree(tempdir);
	tempdir = NULL;
}

/*
 * create the temp directory and ensure some sanity
 */
static int
mk_tmp(char *dir)
{
	struct stat	st;

	assert(dir     != NULL);
	assert(tempdir == NULL);

	atexit(cleanup);

	tempdir = XtMalloc(strlen(dir) + sizeof("/mgv.") + 64);
	sprintf(tempdir, "%s/mgv.%ld", dir, (long)getpid());

	if(mkdir(tempdir, 0) != 0) {
		XtFree(tempdir);
		tempdir = NULL;
		return FALSE;
	}
	memset((void *)&st, '\0', sizeof(st));
	if(stat(tempdir, &st) != 0) {
		remove(dir);
		free(tempdir);
		tempdir = NULL;
		return FALSE;
	}
	if((st.st_mode & WANTMODE) != WANTMODE)
		chmod(tempdir, st.st_mode | WANTMODE);

	return TRUE;
}

/*
 * magic numbers for various compressed files (from gzip.h in gzip 1.2.4)
 */
static	char	magic[][2] = {
	{'\037', '\036'},	/* pack		*/
	{'\037', '\213'},	/* gzip		*/
	{'\037', '\236'},	/* old gzip	*/
	{'\037', '\235'},	/* compress	*/
};

/*
 * we can deal with PDF files here or in dsc.c, since some people might
 * try and compress PDF files (not much point since PDF files are already
 * compressed) dsc.c is where it will end up
 */
#define	FT_NORMAL	0
#define	FT_COMPRESSED	1

static int
filetype(char *file)
{
	int	fd, i, n;
	char	buf[5];

	if((fd = open(file, O_RDONLY)) == -1)
		return FT_NORMAL;
	n = read(fd, (void *)&buf[0], sizeof(buf));
	close(fd);
	if(n != 2 && n != sizeof(buf))
		return FT_NORMAL;

	/*
	 * gzip.h says one byte order but the files sometimes say
	 * another so we'll just check both
	 */
	for(i = 0; i < (int)(sizeof(magic)/sizeof(magic[0])); ++i)
		if((buf[0] == magic[i][0] && buf[1] == magic[i][1])
		|| (buf[0] == magic[i][1] && buf[1] == magic[i][0]))
			return FT_COMPRESSED;
	return FT_NORMAL;
}

static void
showmsg(MGV *m, char *file, char *tmp)
{
	FILE	*fp;
	char	buf[1024];

	fp = fopen(tmp, "r");
	remove(tmp);
	if(fp == NULL)
		return;
	memset((void *)buf, '\0', sizeof(buf));
	fread((void *)&buf[0], 1, sizeof(buf) - 1, fp);
	fclose(fp);
	if(buf[0] == '\0')
		return;
	wl_warn(m->main, "%s %s:\n%s", mgv_s(m->main, MgvScantDecompress),
								file, buf);
}

/*
 * Uncompress a file and return the new file name, if the file doesn't
 * appear to be compressed or the uncompression fails, the original
 * file name will be returned.  In all cases, the return value is
 * allocated with XtNewString().
 */
char *
mgv_unmangle(MGV *m, char *file)
{
	char	out[1024], cmd[2048], tmp[1024];
	char	*s;
	int	t;

	if((t = filetype(file)) == FT_NORMAL
	|| (tempdir == NULL && !mk_tmp(m->temp_dir)))
		return XtNewString(file);

	if((s = strrchr(file, '/')) == NULL)
		s = file;
	else
		++s;
	sprintf(out, "%s/%s.%ld.%d", tempdir, s,
				(long)XtWindow(m->main), m->inst_num);
	sprintf(tmp, "%s/%ld.%d", tempdir, (long)getpid(), m->inst_num);

	switch(t) {
	case FT_COMPRESSED:
		sprintf(cmd, "%s <%s >%s 2>%s",
					m->decompress, file, out, tmp);
		break;
	default:
		assert("mgv_unmangle(): confused by file type!" != NULL);
		return XtNewString(file);
		break;
	}

	errno = 0;
	if(system(cmd) != 0 && errno != 0)
		strcpy(out, file);
	showmsg(m, file, &tmp[0]);

	return XtNewString(out);
}
