/*
 * Ghostview.c -- Ghostview widget.
 * Copyright (C) 1992  Timothy O. Theisen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Tim Theisen           Systems Programmer
 * Internet: tim@cs.wisc.edu       Department of Computer Sciences
 *     UUCP: uwvax!tim             University of Wisconsin-Madison
 *    Phone: (608)262-0438         1210 West Dayton Street
 *      FAX: (608)262-9777         Madison, WI   53706
 *
 * The GhostviewPSBBox() was added by mu@echo-on.net who also did some
 * code reformatting and wrapped an sprintf() in setlocale()'s to
 * make sure floating point values were written with decimal points.
 * The gsLocale resource was added to control this.  The widget was
 * also modified to make its own copy of the GvNfilename and
 * GvNarguments resources to avoid free memory reads (thank you purify!).
 * The anti-aliasing support was also added.  I also reformatted the code
 * to make it easier on my eyes.
 */
#include	<wlib/rcs.h>
MU_ID("$Mu: mgv/Ghostview.c,v 1.55 $")

#include	<assert.h>
#include	<stdio.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<errno.h>
#include	<unistd.h>
#include	<string.h>
#include	<sys/types.h>
#include	<sys/wait.h>
#include	<signal.h>
#include	<fcntl.h>
#include	<locale.h>
#include	<X11/IntrinsicP.h>
#include	<X11/StringDefs.h>
#include	<X11/Xatom.h>
#include	<X11/Xproto.h>

#include	<mine/GhostviewP.h>

/*
 * GV_BUFSIZ is set to the minimum POSIX PIPE_BUF to ensure that
 * nonblocking writes to ghostscript will work properly.
 */
#define	GV_BUFSIZ	512

static void	ComputeXdpi(Widget, int, XrmValue *);
static void	ComputeYdpi(Widget, int, XrmValue *);

#define	OFFSET(field)	XtOffsetOf(GhostviewRec, ghostview.field)
static XtResource resources[] =
{
	{
		GvNarguments, GvCArguments,
		XtRString, sizeof(String), OFFSET(arguments),
		XtRString, NULL
	}, {
		GvNbottomMargin, XtCMargin,
		XtRInt, sizeof(int), OFFSET(bottom_margin),
		XtRImmediate, (XtPointer)0
	}, {
		GvNbusyCursor, XtCCursor,
		XtRCursor, sizeof(XtPointer), OFFSET(busy_cursor),
		XtRString, "watch"
	}, {
		GvNnotifyCallback, XtCCallback,
		XtRCallback, sizeof(XtPointer), OFFSET(notify_callback),
		XtRCallback, NULL
	}, {
		GvNcursor, XtCCursor,
		XtRCursor, sizeof(XtPointer), OFFSET(cursor),
		XtRString, "crosshair"
	}, {
		GvNfilename, GvCFilename,
		XtRString, sizeof(String), OFFSET(filename),
		XtRString, NULL
	}, {
		XtNforeground, XtCForeground,
		XtRPixel, sizeof(Pixel), OFFSET(foreground),
		XtRString, XtDefaultForeground
	}, {
		GvNinterpreter, GvCInterpreter,
		XtRString, sizeof(String), OFFSET(interpreter),
		XtRString, "gs"
	}, {
		GvNleftMargin, XtCMargin,
		XtRInt, sizeof(int), OFFSET(left_margin),
		XtRImmediate, (XtPointer)0
	}, {
		GvNllx, GvCBoundingBox,
		XtRInt, sizeof(int), OFFSET(llx),
		XtRImmediate, (XtPointer)0
	}, {
		GvNlly, GvCBoundingBox,
		XtRInt, sizeof(int), OFFSET(lly),
		XtRImmediate, (XtPointer)0
	}, {
		GvNmessageCallback, XtCCallback,
		XtRCallback, sizeof(XtPointer), OFFSET(message_callback),
		XtRCallback, NULL
	}, {
		GvNorientation, GvCOrientation,
		GvRPageOrientation, sizeof(GvPageOrientation), OFFSET(orientation),
		XtRImmediate, (XtPointer)GvPageOrientationPortrait
	}, {
		GvNoutputCallback, XtCCallback,
		XtRCallback, sizeof(XtPointer), OFFSET(output_callback),
		XtRCallback, NULL
	}, {
		GvNpalette, GvCPalette,
		GvRPalette, sizeof(GvPalette), OFFSET(palette),
		XtRImmediate, (XtPointer)GvPaletteColor
	}, {
		GvNquiet, GvCQuiet,
		XtRBoolean, sizeof(Boolean), OFFSET(quiet),
		XtRImmediate, (XtPointer)True
	}, {
		GvNrightMargin, XtCMargin,
		XtRInt, sizeof(int), OFFSET(right_margin),
		XtRImmediate, (XtPointer)0
	}, {
		GvNsafer, GvCSafer,
		XtRBoolean, sizeof(Boolean), OFFSET(safer),
		XtRImmediate, (XtPointer)True
	}, {
		GvNtopMargin, XtCMargin,
		XtRInt, sizeof(int), OFFSET(top_margin),
		XtRImmediate, (XtPointer)0
	}, {
		GvNuseBackingPixmap, GvCUseBackingPixmap,
		XtRBoolean, sizeof(Boolean), OFFSET(use_bpixmap),
		XtRImmediate, (XtPointer)True
	}, {
		GvNurx, GvCBoundingBox,
		XtRInt, sizeof(int), OFFSET(urx),
		XtRImmediate, (XtPointer)612
	}, {
		GvNury, GvCBoundingBox,
		XtRInt, sizeof(int), OFFSET(ury),
		XtRImmediate, (XtPointer)792
	}, {
		GvNxdpi, GvCResolution,
		XtRInt, sizeof(int), OFFSET(xdpi),
		XtRCallProc, (XtPointer)ComputeXdpi
	}, {
		GvNydpi, GvCResolution,
		XtRInt, sizeof(int), OFFSET(ydpi),
		XtRCallProc, (XtPointer)ComputeYdpi
	}, {
		GvNgsLocale, GvCGsLocale,
		XtRString, sizeof(String), OFFSET(gs_locale),
		XtRString, (XtPointer)"POSIX"
	}, {
		GvNantialias, GvCAntialias,
		XtRBoolean, sizeof(Boolean), OFFSET(antialias),
		XtRImmediate, (XtPointer)True
	}
};
#undef	offset

static void	Message(Widget, XEvent *, String *, Cardinal *);
static void	Notify(Widget, XEvent *, String *, Cardinal *);
static void	Input(XtPointer, int *, XtInputId *);
static void	Output(XtPointer, int *, XtInputId *);

static void	ClassInitialize(void);
static void	ClassPartInitialize(WidgetClass);
static void	Initialize(Widget, Widget, ArgList, Cardinal *);
static void	Realize(Widget, Mask *, XSetWindowAttributes *);
static void	Destroy(Widget);
static void	Resize(Widget);
static Boolean	SetValues(Widget, Widget, Widget, ArgList, Cardinal *);
static XtGeometryResult	QueryGeometry(Widget,
				XtWidgetGeometry *, XtWidgetGeometry *);

static void	Layout(Widget, Boolean, Boolean);
static Boolean	ComputeSize(Widget, Boolean, Boolean, Dimension *, Dimension *);
static void	ChangeSize(Widget, Dimension, Dimension);
static Boolean	Setup(GhostviewWidget);
static void	StartInterpreter(Widget);
static void	StopInterpreter(Widget);
static void	InterpreterFailed(Widget);

static XtActionsRec actions[] = {
	{"gvMessage",	Message},
	{"gvNotify",	Notify},
};

/*
 * notify takes zero to four parameters.  The first two give the width and
 * height of the zoom requested in the default user coordinate system.
 * If they are omitted, a default value of 72 is provided.  If the second
 * parameter is omitted, the zoom area is assumed to be a square.
 * The next two parameters give the desired resolution of the zoom window.
 * If they are omitted, a default value of 300 is provided. If the four
 * parameter is omitted, the y resolution is assumed to be equal to the
 * x resolution.
 */
static char translations[] =
"<Message>:		gvMessage()	\n\
<Btn1Down>:		gvNotify(72)	\n\
<Btn2Down>:		gvNotify(108)	\n\
<Btn3Down>:		gvNotify(144)	\n\
";

GhostviewClassRec ghostviewClassRec = {
	{	/* core fields */
	/* superclass		*/	(WidgetClass) &coreClassRec,
	/* class_name		*/	"Ghostview",
	/* widget_size		*/	sizeof(GhostviewRec),
	/* class_initialize	*/	ClassInitialize,
	/* class_part_initialize*/	ClassPartInitialize,
	/* class_inited		*/	FALSE,
	/* initialize		*/	Initialize,
	/* initialize_hook	*/	NULL,
	/* realize		*/	Realize,
	/* actions		*/	actions,
	/* num_actions		*/	XtNumber(actions),
	/* resources		*/	resources,
	/* num_resources	*/	XtNumber(resources),
	/* xrm_class		*/	NULLQUARK,
	/* compress_motion	*/	TRUE,
	/* compress_exposure	*/	TRUE,
	/* compress_enterleave	*/	TRUE,
	/* visible_interest	*/	FALSE,
	/* destroy		*/	Destroy,
	/* resize		*/	Resize,
	/* expose		*/	NULL,
	/* set_values		*/	SetValues,
	/* set_values_hook	*/	NULL,
	/* set_values_almost	*/	XtInheritSetValuesAlmost,
	/* get_values_hook	*/	NULL,
	/* accept_focus		*/	NULL,
	/* version		*/	XtVersion,
	/* callback_private	*/	NULL,
	/* tm_table		*/	translations,
	/* query_geometry	*/	QueryGeometry,
	/* display_accelerator	*/	XtInheritDisplayAccelerator,
	/* extension		*/	NULL
	},
	{	/* ghostview fields */
	/* ghostview		*/	NULL,
	/* gv_colors		*/	NULL,
	/* next			*/	NULL,
	/* page			*/	NULL,
	/* done			*/	NULL
	}
};

WidgetClass ghostviewWidgetClass = (WidgetClass)&ghostviewClassRec;

/*
 * not everyone has putenv() and if they do have it, sometimes we
 * can't get at its prototype so here is a bare-bones implementation
 * that should be good enough for our purposes
 */
#if !defined(HAS_PUTENV)
#	define	HAS_PUTENV	0
#endif

#if !HAS_PUTENV
#define putenv gv_putenv

extern	char	**environ;
static int
gv_putenv(const char *e)
{
static	char	**my_environ;
	char	*value;
	char	**p;
	int	len, i;
	char	**tmp;

	if((value = strrchr(e, '=')) == NULL)
		return -1;
	++value;
	len = strlen(e) - strlen(value) - 1;

	/*
	 * see if it is already there
	 */
	for(p = environ, i = 0; p != NULL && *p != NULL; ++p, ++i)
		if(strncmp(e, *p, len) == 0)
			break;
	if(p == NULL || *p == NULL) {
		/*
		 * not there, build a new environment array
		 */
		if(environ == my_environ) {
			tmp = realloc((void *)my_environ, (i+2)*sizeof(char *));
			if(tmp == NULL)
				return -1;
		}
		else {
			tmp = malloc((i+2)*sizeof(char *));
			if(tmp == NULL)
				return -1;
			memcpy((void *)tmp, (void *)environ, i*sizeof(char *));
		}
		my_environ   =
		environ      = tmp;
		environ[i+1] = NULL;
	}

	environ[i] = (char *)e;

	return 0;
}

#endif	/* HAS_PUTENV */

/*
 * Procedures that compute the default xdpi and ydpi from display parameters
 */
static void
ComputeXdpi(Widget w, int offset, XrmValue *value)
{
static	int	xdpi;
	USEUP(offset);
	xdpi  = 25.4 * WidthOfScreen(XtScreen(w));
	xdpi /= WidthMMOfScreen(XtScreen(w));
	value->addr = (XtPointer)&xdpi;
}

static void
ComputeYdpi(Widget w, int offset, XrmValue *value)
{
static	int	ydpi;
	USEUP(offset);
	ydpi  = 25.4 * HeightOfScreen(XtScreen(w));
	ydpi /= HeightMMOfScreen(XtScreen(w));
	value->addr = (XtPointer)&ydpi;
}

/*
 * Message action routine.
 * Passes ghostscript message events back to application via
 * the message callback.  It also marks the interpreter as
 * being not busy at the end of page, and stops the interpreter
 * when it send a "done" message.
 */
static void
Message(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
	GhostviewWidget		gvw = (GhostviewWidget)w;
	GhostviewWidgetClass	gvc = (GhostviewWidgetClass)XtClass(w);
	Atom			page, done;
	String			callback;

	USEUP(params); USEUP(num_params);

	page     = XmuInternAtom(XtDisplay(w), gvc->ghostview_class.page);
	done     = XmuInternAtom(XtDisplay(w), gvc->ghostview_class.done);
	callback = NULL;
	gvw->ghostview.mwin = event->xclient.data.l[0];
	if(event->xclient.message_type == page) {
		gvw->ghostview.busy = False;
		XDefineCursor(XtDisplay(gvw), XtWindow(gvw),
			gvw->ghostview.cursor);
		callback = "Page";
	}
	else if(event->xclient.message_type == done) {
		StopInterpreter(w);
		callback = "Done";
	}
	else {
		return;
	}
	assert(callback != NULL);
	XtCallCallbackList(w, gvw->ghostview.message_callback, callback);
}

/*
 * Notify action routine.
 * Calculates where the user clicked in the default user coordinate system.
 * Call the callbacks with the point of click and size of zoom window
 * requested.
 */
static void
Notify(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
	GhostviewWidget		gvw = (GhostviewWidget)w;
	GhostviewReturnStruct	ret_val;

	/*
	 * Notify takes zero to four parameters.  The first two give the
	 * width and height of the zoom requested in the default user
	 * coordinate system.  If they are omitted, a default value of 72
	 * is provided.  If the second parameter is omitted, the zoom area
	 * is assumed to be a square.  The next two parameters give the
	 * desired resolution of the zoom window.  If they are omitted, a
	 * default value of 300 is provided. If the fourth parameter is
	 * omitted, the y resolution is assumed to be equal to the
	 * x resolution.
	 */
	switch(*num_params) {
	case 0:
		ret_val.width = ret_val.height =  72;
		ret_val.xdpi  = ret_val.ydpi   = 300;
		break;
	case 1:
		ret_val.width = ret_val.height = atoi(params[0]);
		ret_val.xdpi  = ret_val.ydpi   = 300;
		break;
	case 2:
		ret_val.width  = atoi(params[0]);
		ret_val.height = atoi(params[1]);
		ret_val.xdpi   = ret_val.ydpi    = 300;
		break;
	case 3:
		ret_val.width  = atoi(params[0]);
		ret_val.height = atoi(params[1]);
		ret_val.xdpi   = ret_val.ydpi    = atoi(params[2]);
		break;
	default:
		ret_val.width  = atoi(params[0]);
		ret_val.height = atoi(params[1]);
		ret_val.xdpi   = atoi(params[2]);
		ret_val.ydpi   = atoi(params[3]);
		break;
	}

	switch(gvw->ghostview.orientation) {
	case GvPageOrientationPortrait:
		ret_val.psx = gvw->ghostview.llx
				+ event->xbutton.x * 72.0 / gvw->ghostview.xdpi;
		ret_val.psy = gvw->ghostview.ury
				- event->xbutton.y * 72.0 / gvw->ghostview.ydpi;
		break;
	case GvPageOrientationLandscape:
		ret_val.psx = gvw->ghostview.llx
				+ event->xbutton.y * 72.0 / gvw->ghostview.ydpi;
		ret_val.psy = gvw->ghostview.lly
				+ event->xbutton.x * 72.0 / gvw->ghostview.xdpi;
		break;
	case GvPageOrientationUpsideDown:
		ret_val.psx = gvw->ghostview.urx
				- event->xbutton.x * 72.0 / gvw->ghostview.xdpi;
		ret_val.psy = gvw->ghostview.lly
				+ event->xbutton.y * 72.0 / gvw->ghostview.ydpi;
		break;
	case GvPageOrientationSeascape:
		ret_val.psx = gvw->ghostview.urx
				- event->xbutton.y * 72.0 / gvw->ghostview.ydpi;
		ret_val.psy = gvw->ghostview.ury
				- event->xbutton.x * 72.0 / gvw->ghostview.xdpi;
		break;
	}
	ret_val.event = event;
	XtCallCallbackList(w, gvw->ghostview.notify_callback,
							(XtPointer)&ret_val);
}

static Boolean broken_pipe = False;

static void
CatchPipe(int i)
{
	USEUP(i);
	broken_pipe = True;
}

/*
 * Input - Feed data to ghostscript's stdin.
 * Write bytes to ghostscript using non-blocking I/O.
 * Also, pipe signals are caught during writing.  The return
 * values are checked and the appropriate action is taken.  I do
 * this at this low level, because it may not be appropriate for
 * SIGPIPE to be caught for the overall application.
 */
static void
Input(XtPointer client_data, int *source, XtInputId *id)
{
	Widget		w   = (Widget)client_data;
	GhostviewWidget	gvw = (GhostviewWidget)w;
	int		bytes_written;
	struct sigaction	sa, osa;

	USEUP(id); USEUP(source);

	memset((void *)&sa,  '\0', sizeof(sa));
	memset((void *)&osa, '\0', sizeof(osa));
	sa.sa_handler = CatchPipe;
	sigemptyset(&sa.sa_mask);
	sigaction(SIGPIPE, &sa, &osa);
	do {
		if(gvw->ghostview.buffer_bytes_left == 0) {
			/*
			 * Get a new section if required
			 */
			if(gvw->ghostview.ps_input
			&& gvw->ghostview.bytes_left == 0) {
				GVFile	*ps_old = gvw->ghostview.ps_input;
				gvw->ghostview.ps_input = ps_old->next;
				if(ps_old->close)
					fclose(ps_old->fp);
				XtFree((XtPointer)ps_old);
			}

			/*
			 * Have to seek at the beginning of each section
			 */
			if(gvw->ghostview.ps_input
			&& gvw->ghostview.ps_input->seek_needed) {
				if(gvw->ghostview.ps_input->len > 0)
					fseek(gvw->ghostview.ps_input->fp,
						gvw->ghostview.ps_input->begin,
						SEEK_SET);
				gvw->ghostview.ps_input->seek_needed = False;
				gvw->ghostview.bytes_left =
					gvw->ghostview.ps_input->len;
			}

			if(gvw->ghostview.bytes_left > GV_BUFSIZ) {
				gvw->ghostview.buffer_bytes_left =
					fread(gvw->ghostview.input_buffer,
					sizeof (char), GV_BUFSIZ,
					gvw->ghostview.ps_input->fp);
			}
			else if(gvw->ghostview.bytes_left > 0) {
				gvw->ghostview.buffer_bytes_left =
					fread(gvw->ghostview.input_buffer,
					sizeof(char), gvw->ghostview.bytes_left,
					gvw->ghostview.ps_input->fp);
			}
			else {
				gvw->ghostview.buffer_bytes_left = 0;
			}

			if(gvw->ghostview.bytes_left > 0
			&& gvw->ghostview.buffer_bytes_left == 0) {
				/*
				 * Error occured
				 */
				InterpreterFailed(w);
			}
			gvw->ghostview.input_buffer_ptr =
				gvw->ghostview.input_buffer;
			gvw->ghostview.bytes_left -=
				gvw->ghostview.buffer_bytes_left;
		}

		if(gvw->ghostview.buffer_bytes_left > 0) {
			bytes_written = write(gvw->ghostview.interpreter_input,
				gvw->ghostview.input_buffer_ptr,
				gvw->ghostview.buffer_bytes_left);

			if(broken_pipe) {
				/*
				 * Something bad happened
				 */
				broken_pipe = False;
				InterpreterFailed(w);
			}
			else if (bytes_written == -1) {
				if((errno != EWOULDBLOCK)
				&& (errno != EAGAIN)) {
					/*
					 * Something bad happened
					 */
					InterpreterFailed(w);
				}
			}
			else {
				gvw->ghostview.buffer_bytes_left -=
					bytes_written;
				gvw->ghostview.input_buffer_ptr +=
					bytes_written;
			}
		}
	} while(gvw->ghostview.ps_input
	     && gvw->ghostview.buffer_bytes_left == 0);

	sigaction(SIGPIPE, &osa, &sa);
	if(gvw->ghostview.ps_input             == NULL
	&& gvw->ghostview.buffer_bytes_left    == 0
	&& gvw->ghostview.interpreter_input_id != None) {
		XtRemoveInput(gvw->ghostview.interpreter_input_id);
		gvw->ghostview.interpreter_input_id = None;
	}
}

/*
 * Output - receive I/O from ghostscript's stdout and stderr.
 * Pass this to the application via the output_callback.
 */
static void
Output(XtPointer client_data, int *source, XtInputId *id)
{
	Widget		w     = (Widget)client_data;
	GhostviewWidget	gvw   = (GhostviewWidget)w;
	int		bytes = 0;
	char		buf[GV_BUFSIZ+1];
	int		*gvw_src;
	XtInputId	gvw_id;

	USEUP(id);

	gvw_src = NULL;
	gvw_id  = (XtInputId)0;
	if(*source == gvw->ghostview.interpreter_output) {
		gvw_src = &gvw->ghostview.interpreter_output;
		gvw_id  = gvw->ghostview.interpreter_output_id;
	}
	else if(*source == gvw->ghostview.interpreter_error) {
		gvw_src = &gvw->ghostview.interpreter_error;
		gvw_id  = gvw->ghostview.interpreter_error_id;
	}
	if(gvw_src != NULL) {
		bytes = read(*gvw_src, buf, GV_BUFSIZ);
		if(bytes == 0) {
			/*
			 * EOF
			 */
			close(*gvw_src);
			*gvw_src = -1;
			XtRemoveInput(gvw_id);
			return;
		}
		else if(bytes == -1) {
			/*
			 * badness
			 */
			InterpreterFailed(w);
			return;
		}
	}

	if(bytes > 0) {
		buf[bytes] = '\0';
		XtCallCallbackList(w, gvw->ghostview.output_callback,
			(XtPointer)buf);
	}
}

#define	DONE(type, value)					\
	{							\
		if(toVal->addr != NULL) {			\
			if(toVal->size < sizeof(type)) {	\
				toVal->size = sizeof(type);	\
				return False;			\
			}					\
			*(type*)(toVal->addr) = (value);	\
		}						\
		else {						\
			static type static_val;			\
			static_val  = (value);			\
			toVal->addr = (XPointer)&static_val;	\
		}						\
		toVal->size = sizeof(type);			\
		return True;					\
	}

/*
 * PageOrienation Conversion Routine.
 * Returns True if Conversion is successful.
 */
static Boolean
GvCvtStringToPageOrientation(Display *dpy, XrmValue *args, Cardinal *num_args,
			XrmValue *fromVal, XrmValue *toVal, XtPointer *data)
{
static	XrmQuark	GvQEportrait;
static	XrmQuark	GvQElandscape;
static	XrmQuark	GvQEupsideDown;
static	XrmQuark	GvQEseascape;
static	Boolean		haveQuarks = False;
	XrmQuark	q;
	char		*str = (char *)fromVal->addr;
	char		lowerName[1000];

	USEUP(args); USEUP(num_args); USEUP(data);

	if(str == NULL)
		return False;

	if(!haveQuarks) {
		GvQEportrait   = XrmStringToQuark(GvEportrait);
		GvQElandscape  = XrmStringToQuark(GvElandscape);
		GvQEupsideDown = XrmStringToQuark(GvEupsideDown);
		GvQEseascape   = XrmStringToQuark(GvEseascape);
		haveQuarks     = True;
	}

	XmuCopyISOLatin1Lowered(lowerName, str);
	q = XrmStringToQuark(lowerName);

	if(q == GvQEportrait)
		DONE(GvPageOrientation, GvPageOrientationPortrait);
	if(q == GvQElandscape)
		DONE(GvPageOrientation, GvPageOrientationLandscape);
	if(q == GvQEupsideDown)
		DONE(GvPageOrientation, GvPageOrientationUpsideDown);
	if(q == GvQEseascape)
		DONE(GvPageOrientation, GvPageOrientationSeascape);

	XtDisplayStringConversionWarning(dpy, str, GvRPageOrientation);
	return False;
}

/*
 * Palette Conversion Routine.
 * Returns True if Conversion is successful.
 */
static Boolean
GvCvtStringToPalette(Display *dpy, XrmValue *args, Cardinal *num_args,
			XrmValue *fromVal, XrmValue *toVal, XtPointer *data)
{
static	XrmQuark	GvQEmonochrome;
static	XrmQuark	GvQEgrayscale;
static	XrmQuark	GvQEcolor;
static	Boolean		haveQuarks = False;
	XrmQuark    	q;
	char		*str = (XPointer) fromVal->addr;
	char		lowerName[1000];

	USEUP(args); USEUP(num_args); USEUP(data);

	if(str == NULL)
		return False;

	if(!haveQuarks) {
		GvQEmonochrome = XrmStringToQuark(GvEmonochrome);
		GvQEgrayscale  = XrmStringToQuark(GvEgrayscale);
		GvQEcolor      = XrmStringToQuark(GvEcolor);
		haveQuarks      = True;
	}

	XmuCopyISOLatin1Lowered(lowerName, str);
	q = XrmStringToQuark(lowerName);

	if(q == GvQEmonochrome)
		DONE(GvPalette, GvPaletteMonochrome);
	if(q == GvQEgrayscale)
		DONE(GvPalette, GvPaletteGrayscale);
	if(q == GvQEcolor)
		DONE(GvPalette, GvPaletteColor);

	XtDisplayStringConversionWarning(dpy, str, GvRPalette);
	return False;
}

/*
 * Register the type converter required for the PageOrientation.
 * Register the type converter required for the Palette.
 * This routine is called exactly once.
 */
static void
ClassInitialize(void)
{
	XtSetTypeConverter(XtRString, GvRPageOrientation,
		GvCvtStringToPageOrientation, NULL, 0, XtCacheAll, NULL);
	XtSetTypeConverter(XtRString, GvRPalette, GvCvtStringToPalette,
		NULL, 0, XtCacheAll, NULL);
}

/*
 * Get atoms needed to communicate with ghostscript.
 * This routine is called once per display.
 */
static void
ClassPartInitialize(WidgetClass class)
{
	GhostviewWidgetClass	gvc = (GhostviewWidgetClass)class;

	gvc->ghostview_class.ghostview = XmuMakeAtom("GHOSTVIEW");
	gvc->ghostview_class.gv_colors = XmuMakeAtom("GHOSTVIEW_COLORS");
	gvc->ghostview_class.next      = XmuMakeAtom("NEXT");
	gvc->ghostview_class.page      = XmuMakeAtom("PAGE");
	gvc->ghostview_class.done      = XmuMakeAtom("DONE");
}

/*
 * Initialize private state.
 */
static void
Initialize(Widget request, Widget new, ArgList args, Cardinal *num_args)
{
	XGCValues	values;
	XtGCMask	mask;
	GhostviewWidget	ngvw = (GhostviewWidget)new;
	GhostviewWidget	rgvw = (GhostviewWidget)request;

	USEUP(args); USEUP(num_args);

	values.foreground = new->core.background_pixel;
	mask              = GCForeground;

	ngvw->ghostview.gc                    = XtGetGC(new, mask, &values);
	ngvw->ghostview.mwin                  = None;
	ngvw->ghostview.disable_start         = False;
	ngvw->ghostview.interpreter_pid       = -1;
	ngvw->ghostview.input_buffer          = NULL;
	ngvw->ghostview.bytes_left            = 0;
	ngvw->ghostview.input_buffer_ptr      = NULL;
	ngvw->ghostview.buffer_bytes_left     = 0;
	ngvw->ghostview.ps_input              = NULL;
	ngvw->ghostview.interpreter_input     = -1;
	ngvw->ghostview.interpreter_output    = -1;
	ngvw->ghostview.interpreter_error     = -1;
	ngvw->ghostview.interpreter_input_id  = None;
	ngvw->ghostview.interpreter_output_id = None;
	ngvw->ghostview.interpreter_error_id  = None;
	ngvw->ghostview.gs_width              = 0;
	ngvw->ghostview.gs_height             = 0;
	ngvw->ghostview.changed               = False;
	ngvw->ghostview.busy                  = False;

	if(rgvw->ghostview.interpreter != NULL)
		ngvw->ghostview.interpreter = XtNewString(rgvw->ghostview.interpreter);
	if(rgvw->ghostview.arguments != NULL)
		ngvw->ghostview.arguments = XtNewString(rgvw->ghostview.arguments);
	if(rgvw->ghostview.filename != NULL)
		ngvw->ghostview.filename = XtNewString(rgvw->ghostview.filename);

	/*
	 * Compute window size
	 */
	Layout(new, (Boolean)(rgvw->core.width == 0),
					(Boolean)(rgvw->core.height == 0));
}

/*
 * Create Window and start interpreter if needed
 */
static void
Realize(Widget w, Mask *valueMask, XSetWindowAttributes *attributes)
{
	GhostviewWidget	gvw = (GhostviewWidget) w;

	if(gvw->ghostview.cursor != None) {
		attributes->cursor = gvw->ghostview.cursor;
		*valueMask |= CWCursor;
	}

	XtCreateWindow(w, (unsigned)InputOutput, (Visual *)CopyFromParent,
							*valueMask, attributes);

	Setup(gvw);
}

/*
 * Destroy routine: kill the interpreter and release the GC
 */
static void
Destroy(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	StopInterpreter(w);
	XtReleaseGC(w, gvw->ghostview.gc);
	XtFree((XtPointer)gvw->ghostview.input_buffer);
	XtFree((XtPointer)gvw->ghostview.filename);
	XtFree((XtPointer)gvw->ghostview.arguments);
	XtFree((XtPointer)gvw->ghostview.interpreter);
	if(gvw->core.background_pixmap != XtUnspecifiedPixmap)
		XFreePixmap(XtDisplay(w), gvw->core.background_pixmap);
}

/*
 * Process resize request.  Requested size cannot be changed.
 * NOTE: This routine may be called before the widget is realized.
 * (It was a surprise to me.)
 * If the widget is realized, start a new interpreter by calling Setup().
 * If Setup() actually started a new interpreter and it is taking input
 * from stdin, send a refresh message to the application.  This is the
 * only way that the application can be notified that it needs to resend
 * the input because someone forced a new window size on the widget.
 */
static void
Resize(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	Layout(w, False, False);
	if(!XtIsRealized(w))
		return;
	if(Setup(gvw) && gvw->ghostview.filename == NULL)
		XtCallCallbackList(w, gvw->ghostview.message_callback,
								"Refresh");
}

/*
 * SetValues routine.  Set new private state, based on changed values
 * in the widget.  Always returns False, because redisplay is never needed.
 */
static Boolean
SetValues(Widget current, Widget request, Widget new, ArgList args,
						Cardinal *num_args)
{
	GhostviewWidget	cgvw = (GhostviewWidget)current;
	GhostviewWidget	rgvw = (GhostviewWidget)request;
	GhostviewWidget	ngvw = (GhostviewWidget)new;
	String		cfilename, carguments, cinterpreter;
	String		rfilename, rarguments, rinterpreter;

	USEUP(args); USEUP(num_args);

	if(XtIsRealized(new)
	&& !ngvw->ghostview.busy
	&& cgvw->ghostview.cursor != ngvw->ghostview.cursor)
		XDefineCursor(XtDisplay(new), XtWindow(new),
						ngvw->ghostview.cursor);
	if(XtIsRealized(new)
	&& ngvw->ghostview.busy
	&& cgvw->ghostview.busy_cursor != ngvw->ghostview.busy_cursor)
		XDefineCursor(XtDisplay(new), XtWindow(new),
						ngvw->ghostview.busy_cursor);

	if(cgvw->core.background_pixel != rgvw->core.background_pixel) {
		XGCValues	values;
		XtGCMask	mask;

		XtReleaseGC(current, cgvw->ghostview.gc);
		values.foreground = new->core.background_pixel;
		mask = GCForeground;
		ngvw->ghostview.gc = XtGetGC(new, mask, &values);
	}
	if(cgvw->ghostview.filename != rgvw->ghostview.filename) {
		if(rgvw->ghostview.filename != NULL)
			ngvw->ghostview.filename =
					XtNewString(rgvw->ghostview.filename);
		else
			ngvw->ghostview.filename = NULL;
		XtFree((XtPointer)cgvw->ghostview.filename);
		cgvw->ghostview.filename = NULL;
	}
	if(cgvw->ghostview.arguments != rgvw->ghostview.arguments) {
		if(rgvw->ghostview.arguments != NULL)
			ngvw->ghostview.arguments =
					XtNewString(rgvw->ghostview.arguments);
		else
			ngvw->ghostview.arguments = NULL;
		XtFree((XtPointer)cgvw->ghostview.arguments);
		cgvw->ghostview.arguments = NULL;
	}
	if(cgvw->ghostview.interpreter != rgvw->ghostview.interpreter) {
		if(rgvw->ghostview.interpreter != NULL)
			ngvw->ghostview.interpreter =
				XtNewString(rgvw->ghostview.interpreter);
		else
			ngvw->ghostview.interpreter = NULL;
		XtFree((XtPointer)cgvw->ghostview.interpreter);
		cgvw->ghostview.interpreter = NULL;
	}

	if(cgvw->ghostview.antialias != rgvw->ghostview.antialias)
		cgvw->ghostview.antialias = rgvw->ghostview.antialias;

	cfilename    = (cgvw->ghostview.filename == NULL)
			? "(null)"
			: cgvw->ghostview.filename;
	carguments   = (cgvw->ghostview.arguments == NULL)
			? "(null)"
			: cgvw->ghostview.arguments;
	cinterpreter = (cgvw->ghostview.interpreter == NULL)
			? "(null)"
			: cgvw->ghostview.interpreter;
	rfilename    = (rgvw->ghostview.filename == NULL)
			? "(null)"
			: rgvw->ghostview.filename;
	rarguments   = (rgvw->ghostview.arguments == NULL)
			? "(null)"
			: rgvw->ghostview.arguments;
	rinterpreter = (rgvw->ghostview.interpreter == NULL)
			? "(null)"
			: rgvw->ghostview.interpreter;

	if(cgvw->core.width != rgvw->core.width
	|| cgvw->core.height != rgvw->core.height
	|| cgvw->core.background_pixel != rgvw->core.background_pixel
	|| cgvw->ghostview.foreground != rgvw->ghostview.foreground
	|| cgvw->ghostview.palette != rgvw->ghostview.palette
	|| strcmp(cinterpreter, rinterpreter) != 0
	|| strcmp(carguments, rarguments) != 0
	|| cgvw->ghostview.quiet != rgvw->ghostview.quiet
	|| cgvw->ghostview.safer != rgvw->ghostview.safer
	|| strcmp(cfilename, rfilename) != 0
	|| cgvw->ghostview.orientation != rgvw->ghostview.orientation
	|| cgvw->ghostview.use_bpixmap != rgvw->ghostview.use_bpixmap
	|| cgvw->ghostview.xdpi != rgvw->ghostview.xdpi
	|| cgvw->ghostview.ydpi != rgvw->ghostview.ydpi
	|| cgvw->ghostview.bottom_margin != rgvw->ghostview.bottom_margin
	|| cgvw->ghostview.left_margin != rgvw->ghostview.left_margin
	|| cgvw->ghostview.right_margin != rgvw->ghostview.right_margin
	|| cgvw->ghostview.top_margin != rgvw->ghostview.top_margin
	|| cgvw->ghostview.llx != rgvw->ghostview.llx
	|| cgvw->ghostview.lly != rgvw->ghostview.lly
	|| cgvw->ghostview.urx != rgvw->ghostview.urx
	|| cgvw->ghostview.ury != rgvw->ghostview.ury
	|| cgvw->ghostview.antialias != rgvw->ghostview.antialias) {
		ngvw->ghostview.changed = True;
		Layout(new, True, True);
	}

	if(ngvw->ghostview.changed && XtIsRealized(current))
		Setup(ngvw);
	return False;
}

/*
 * This tells the parent what size we would like to be given certain
 * constraints.
 */
static XtGeometryResult 
QueryGeometry(Widget w, XtWidgetGeometry *intended, XtWidgetGeometry *requested)
{
	Dimension	new_width, new_height;
	Boolean		change;
	XtGeometryMask	width_req, height_req;
 
	width_req  = intended->request_mode & CWWidth;
	height_req = intended->request_mode & CWHeight;

	if(width_req != 0)
		new_width = intended->width;
	else
		new_width = w->core.width;

	if(height_req != 0)
		new_height = intended->height;
	else
		new_height = w->core.height;

	requested->request_mode = 0;

	/*
	 * We only care about our height and width.
	 */
	if(width_req == 0 && height_req == 0)
		return XtGeometryYes;

	change = ComputeSize(w, width_req == 0, height_req == 0, &new_width,
								&new_height);

	requested->request_mode |= CWWidth;
	requested->width         = new_width;
	requested->request_mode |= CWHeight;
	requested->height        = new_height;

	return (change) ? XtGeometryAlmost : XtGeometryYes;
}

/*
 * Layout the widget.
 */
static void
Layout(Widget w, Boolean xfree, Boolean yfree)
{
	Dimension	width = w->core.width;
	Dimension	height = w->core.height;

	if(ComputeSize(w, xfree, yfree, &width, &height))
		ChangeSize(w, width, height);
}

/*
 * Compute new size of window, sets xdpi and ydpi if necessary.
 * returns True if new window size is different
 */
static Boolean
ComputeSize(Widget w, Boolean xfree, Boolean yfree, Dimension *width,
							Dimension *height)
{
	GhostviewWidget	gvw        = (GhostviewWidget) w;
	Dimension	new_width  = *width;
	Dimension	new_height = *height;
	int		newxdpi, newydpi;
	Boolean		change;

	if(xfree && yfree) {
		/*
		 * width and height can be changed, calculate window size
		 * according to xpdi and ydpi
		 */
		switch(gvw->ghostview.orientation) {
		case GvPageOrientationPortrait:
		case GvPageOrientationUpsideDown:
			new_width = (gvw->ghostview.urx - gvw->ghostview.llx)
					/ 72.0 * gvw->ghostview.xdpi + 0.5;
			new_height = (gvw->ghostview.ury - gvw->ghostview.lly)
					/ 72.0 * gvw->ghostview.ydpi + 0.5;
			break;
		case GvPageOrientationLandscape:
		case GvPageOrientationSeascape:
			new_width = (gvw->ghostview.ury - gvw->ghostview.lly)
					/ 72.0 * gvw->ghostview.xdpi + 0.5;
			new_height = (gvw->ghostview.urx - gvw->ghostview.llx)
					/ 72.0 * gvw->ghostview.ydpi + 0.5;
			break;
		}
	}
	else if(xfree) {
		/*
		 * height is fixed, preserve aspect ratio by
		 * recomputing ydpi and xdpi
		 */
		switch (gvw->ghostview.orientation) {
		case GvPageOrientationPortrait:
		case GvPageOrientationUpsideDown:
			newydpi = gvw->core.height * 72.0
				/ (gvw->ghostview.ury - gvw->ghostview.lly);
			newxdpi = newydpi * gvw->ghostview.xdpi
				/ gvw->ghostview.ydpi;
			gvw->ghostview.xdpi = newxdpi;
			gvw->ghostview.ydpi = newydpi;
			new_width = (gvw->ghostview.urx - gvw->ghostview.llx)
				/ 72.0 * gvw->ghostview.xdpi + 0.5;
			break;
		case GvPageOrientationLandscape:
		case GvPageOrientationSeascape:
			newydpi = gvw->core.height * 72.0
				/ (gvw->ghostview.urx - gvw->ghostview.llx);
			newxdpi = newydpi * gvw->ghostview.xdpi
				/ gvw->ghostview.ydpi;
			gvw->ghostview.xdpi = newxdpi;
			gvw->ghostview.ydpi = newydpi;
			new_width = (gvw->ghostview.ury - gvw->ghostview.lly)
				/ 72.0 * gvw->ghostview.xdpi + 0.5;
			break;
		}
	}
	else if(yfree) {
		/*
		 * width is fixed, preserve aspect ratio by recomputing
		 * xdpi and ydpi
		 */
		switch(gvw->ghostview.orientation) {
		case GvPageOrientationPortrait:
		case GvPageOrientationUpsideDown:
			newxdpi = gvw->core.width * 72.0 
				/(gvw->ghostview.urx - gvw->ghostview.llx);
			newydpi = newxdpi * gvw->ghostview.ydpi
				/ gvw->ghostview.xdpi;
			gvw->ghostview.xdpi = newxdpi;
			gvw->ghostview.ydpi = newydpi;
			new_height = (gvw->ghostview.ury - gvw->ghostview.lly)
				/ 72.0 * gvw->ghostview.ydpi + 0.5;
			break;
		case GvPageOrientationLandscape:
		case GvPageOrientationSeascape:
			newxdpi = gvw->core.width * 72.0
				/ (gvw->ghostview.ury - gvw->ghostview.lly);
			newydpi = newxdpi * gvw->ghostview.ydpi
				/ gvw->ghostview.xdpi;
			gvw->ghostview.xdpi = newxdpi;
			gvw->ghostview.ydpi = newydpi;
			new_height = (gvw->ghostview.urx - gvw->ghostview.llx)
				/ 72.0 * gvw->ghostview.ydpi + 0.5;
			break;
		}
	}
	else {
		/*
		 * height and width are fixed.  Just have to live with it.
		 */
		switch (gvw->ghostview.orientation) {
		case GvPageOrientationPortrait:
		case GvPageOrientationUpsideDown:
			gvw->ghostview.xdpi = gvw->core.width * 72.0
				/ (gvw->ghostview.urx - gvw->ghostview.llx);
			gvw->ghostview.ydpi = gvw->core.height * 72.0
				/ (gvw->ghostview.ury - gvw->ghostview.lly);
			break;
		case GvPageOrientationLandscape:
		case GvPageOrientationSeascape:
			gvw->ghostview.xdpi = gvw->core.width * 72.0
				/ (gvw->ghostview.ury - gvw->ghostview.lly);
			gvw->ghostview.ydpi = gvw->core.height * 72.0
				/ (gvw->ghostview.urx - gvw->ghostview.llx);
			break;
		}
	}

	change  = (new_width != *width) || (new_height != *height);
	*width  = new_width;
	*height = new_height;
	return change;
}

/*
 * Request a size change.
 */
static void
ChangeSize(Widget w, Dimension width, Dimension height)
{
	XtWidgetGeometry	request, reply;
	Boolean			changed = False;
	Boolean			canx, cany;

	request.request_mode = CWWidth | CWHeight;
	request.width        = width;
	request.height       = height;

	switch(XtMakeGeometryRequest(w, &request, &reply)) {
	case XtGeometryYes:
		changed = True;
		break;
	case XtGeometryNo:
	case XtGeometryDone:
		break;
	case XtGeometryAlmost:
		canx = (Boolean)(request.height != reply.height);
		cany = (Boolean)(request.width  != reply.width);
		ComputeSize(w, canx, cany, &reply.width, &reply.height);
		request = reply;
		switch(XtMakeGeometryRequest(w, &request, &reply)) {
		case XtGeometryYes:
			changed = True;
			break;
		case XtGeometryNo:
		case XtGeometryDone:
			break;
		case XtGeometryAlmost:
			request = reply;
			ComputeSize(w, False, False, &request.width,
							&request.height);
			request.request_mode = CWWidth | CWHeight;
			XtMakeGeometryRequest(w, &request, &reply);
			changed = True;
			break;
		}
		break;
	}

	/*
	 * If success, setup the widet for the new size.
	 */
	if(changed && XtIsRealized(w))
		Setup((GhostviewWidget)w);
}

/*
 * Catch the alloc error when there is not enough resources for the
 * backing pixmap.  Automatically shut off backing pixmap and let the
 * user know when this happens.
 */
static Boolean alloc_error;
static XErrorHandler oldhandler;

static int
catch_alloc(Display *dpy, XErrorEvent *err)
{
	if(err->error_code == BadAlloc)
		alloc_error = True;
	if(alloc_error)
		return 0;
	return oldhandler(dpy, err);
}

/*
 * Setup - sets up the backing pixmap, and GHOSTVIEW property and
 * starts interpreter if needed.
 * NOTE: the widget must be realized before calling Setup().
 * Returns True if a new interpreter was started, False otherwise.
 */
static Boolean
Setup(GhostviewWidget gvw)
{
	Widget			w = (Widget)gvw;
	GhostviewWidgetClass	gvc = (GhostviewWidgetClass)XtClass(w);
	char			buf[GV_BUFSIZ];
	Pixmap			bpixmap;
	XSetWindowAttributes	xswa;
	char			*olocale;

	if(!gvw->ghostview.changed
	&& gvw->core.width  == gvw->ghostview.gs_width
	&& gvw->core.height == gvw->ghostview.gs_height)
		return False;

	StopInterpreter(w);

	if((gvw->core.width  != gvw->ghostview.gs_width
	|| gvw->core.height  != gvw->ghostview.gs_height
	|| !gvw->ghostview.use_bpixmap)
	&& gvw->core.background_pixmap != XtUnspecifiedPixmap) {
		XFreePixmap(XtDisplay(w), gvw->core.background_pixmap);
		gvw->core.background_pixmap = XtUnspecifiedPixmap;
		XSetWindowBackgroundPixmap(XtDisplay(w), XtWindow(w), None);
	}

	if(gvw->ghostview.use_bpixmap) {
		if(gvw->core.background_pixmap == XtUnspecifiedPixmap) {
			/*
			 * Get a Backing Pixmap, but be ready for the
			 * BadAlloc.
			 */
			XSync(XtDisplay(w), False);
			oldhandler  = XSetErrorHandler(catch_alloc);
			alloc_error = False;
			bpixmap     = XCreatePixmap(XtDisplay(w), XtWindow(w),
					gvw->core.width, gvw->core.height,
					gvw->core.depth);
			/*
			 * force the error
			 */
			XSync(XtDisplay(w), False);
			if(alloc_error) {
				XtCallCallbackList(w,
						gvw->ghostview.message_callback,
						"BadAlloc");
				if(bpixmap != None) {
					XFreePixmap(XtDisplay(w), bpixmap);
					/*
					 * force the error
					 */
					XSync(XtDisplay(w), False);
					bpixmap = None;
				}
			}
			oldhandler = XSetErrorHandler(oldhandler);
			if(bpixmap != None) {
				gvw->core.background_pixmap = bpixmap;
				XSetWindowBackgroundPixmap(XtDisplay(w),
						XtWindow(w),
						gvw->core.background_pixmap);
			}
		}
		else {
			bpixmap = gvw->core.background_pixmap;
		}
	}
	else {
		if(gvw->core.background_pixmap != XtUnspecifiedPixmap) {
			XFreePixmap(XtDisplay(w), gvw->core.background_pixmap);
			gvw->core.background_pixmap = XtUnspecifiedPixmap;
			XSetWindowBackgroundPixmap(XtDisplay(w), XtWindow(w),
									None);
		}
		bpixmap = None;
	}

	if(bpixmap != None)
		xswa.backing_store = NotUseful;
	else
		xswa.backing_store = Always;
	XChangeWindowAttributes(XtDisplay(w), XtWindow(w),CWBackingStore,&xswa);

	gvw->ghostview.gs_width  = gvw->core.width;
	gvw->ghostview.gs_height = gvw->core.height;

	/*
	 * This presents a minor localization problem, the %g
	 * format will use commas in some locales but ghostscript
	 * will not always understand this.  Hence, we need to do
	 * the setlocale() dance.
	 *		--mu@echo-on.net.
	 */
	olocale = setlocale(LC_ALL, NULL);
	setlocale(LC_NUMERIC, gvw->ghostview.gs_locale);
	sprintf(buf, "%ld %d %d %d %d %d %g %g %d %d %d %d",
		bpixmap, gvw->ghostview.orientation,
		gvw->ghostview.llx, gvw->ghostview.lly,
		gvw->ghostview.urx, gvw->ghostview.ury,
		(double)gvw->ghostview.xdpi, (double)gvw->ghostview.ydpi,
		gvw->ghostview.left_margin, gvw->ghostview.bottom_margin,
		gvw->ghostview.right_margin, gvw->ghostview.top_margin);
	setlocale(LC_NUMERIC, olocale);
	XChangeProperty(XtDisplay(w), XtWindow(w),
		XmuInternAtom(XtDisplay(w), gvc->ghostview_class.ghostview),
		XA_STRING, 8, PropModeReplace,
		(unsigned char *)buf, (int)strlen(buf));

	sprintf(buf, "%s %ld %ld",
		gvw->ghostview.palette == GvPaletteMonochrome
			? "Monochrome"
			: gvw->ghostview.palette == GvPaletteGrayscale
			? "Grayscale"
			: gvw->ghostview.palette == GvPaletteColor
			? "Color" : "?",
		gvw->ghostview.foreground, gvw->core.background_pixel);
	XChangeProperty(XtDisplay(w), XtWindow(w),
		XmuInternAtom(XtDisplay(w), gvc->ghostview_class.gv_colors),
		XA_STRING, 8, PropModeReplace,
		(unsigned char *)buf, (int)strlen(buf));

	/*
	 * make sure the properties are updated
	 */
	XSync(XtDisplay(w), False);
	StartInterpreter(w);

	return True;
}

/*
 * This routine starts the interpreter.  It sets the DISPLAY and 
 * GHOSTVIEW environment variables.  The GHOSTVIEW environment variable
 * contains the Window that ghostscript should write on.
 *
 * This routine also opens pipes for stdout and stderr and initializes
 * application input events for them.  If input to ghostscript is not
 * from a file, a pipe for stdin is created.  This pipe is setup for
 * non-blocking I/O so that the user interface never "hangs" because of
 * a write to ghostscript.
 */
static void
StartInterpreter(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;
	int	std_in[2];
	int	std_out[2];
	int	std_err[2];
	char	*argv[100];
	char	*cptr;
	int	argc;
	char	*arguments;

	argc      = 0;
	arguments = NULL;

	StopInterpreter(w);

	/*
	 * Clear the window before starting a new interpreter.
	 */
	if(gvw->core.background_pixmap != XtUnspecifiedPixmap) {
		XFillRectangle(XtDisplay(w), gvw->core.background_pixmap,
				gvw->ghostview.gc,
				0, 0, gvw->core.width, gvw->core.height);
	}
	XClearArea(XtDisplay(w), XtWindow(w),
			0, 0, gvw->core.width, gvw->core.height, False);

	if(gvw->ghostview.disable_start)
		return;

	argv[argc++] = gvw->ghostview.interpreter;
	if(gvw->ghostview.antialias) {
		argv[argc++] = "-dNOPLATFONTS";
		argv[argc++] = "-sDEVICE=x11alpha";
	}
	else {
		argv[argc++] = "-sDEVICE=x11";
	}
	argv[argc++] = "-dNOPAUSE";
	if(gvw->ghostview.quiet)
		argv[argc++] = "-dQUIET";
	if(gvw->ghostview.safer)
		argv[argc++] = "-dSAFER";
	if(gvw->ghostview.arguments != NULL) {
		cptr = arguments = XtNewString(gvw->ghostview.arguments);
		while(isspace((int)*cptr & 0xff))
			++cptr;
		while(*cptr != '\0') {
			argv[argc++] = cptr;
			while(*cptr != '\0' && !isspace((int)*cptr & 0xff))
				++cptr;
			if(*cptr != '\0')
				*cptr++ = '\0';
			if(argc + 2 >= (int)(sizeof(argv)/sizeof(argv[0]))) {
				fprintf(stderr,
					"Too many arguments to interpreter.\n");
				exit(EXIT_FAILURE);/*,,,*/
			}
			while(isspace((int)*cptr & 0xff))
				++cptr;
		}
	}

	if(gvw->ghostview.filename != NULL
	&& strcmp(gvw->ghostview.filename, "-") != 0) {
		argv[argc++] = gvw->ghostview.filename;
		argv[argc++] = "-c";
		argv[argc++] = "quit";
	}
	else {
		argv[argc++] = "-";
	}
	argv[argc++] = NULL;

	if(gvw->ghostview.filename == NULL
	|| strcmp(gvw->ghostview.filename, "-") != 0) {
		if(pipe(std_in) == -1) {
			perror("Could not create pipe");
			exit(EXIT_FAILURE);/*,,,*/
		}
	}
	else {
		std_in[0] = open(gvw->ghostview.filename, O_RDONLY, 0);
	}

	if(pipe(std_out) == -1) {
		perror("Could not create pipe");
		exit(EXIT_FAILURE);/*,,,*/
	}
	if(pipe(std_err) == -1) {
		perror("Could not create pipe");
		exit(EXIT_FAILURE); /*,,,*/
	}

	gvw->ghostview.changed = False;
	gvw->ghostview.busy = True;
	XDefineCursor(XtDisplay(gvw), XtWindow(gvw),gvw->ghostview.busy_cursor);

	gvw->ghostview.interpreter_pid = fork();
	if(gvw->ghostview.interpreter_pid == 0) {
		/*
		 * child
		 *
		 * We are safe leaving buf0 and buf1 on the stack
		 * (as opposed to static or dynamically allocated)
		 * since we're doing an execvp() before they go out
		 * of scope and that exec will do the necessary copying.
		 * Normally putenv() just does a plain pointer assignment
		 * so you can't use automatic arrays for setting environment
		 * variables.
		 */
		char buf0[GV_BUFSIZ];
		char buf1[GV_BUFSIZ];

		close(std_out[0]);
		close(std_err[0]);
		dup2(std_out[1], 1);
		close(std_out[1]);
		dup2(std_err[1], 2);
		close(std_err[1]);
		sprintf(buf0, "GHOSTVIEW=%ld", (long)XtWindow(w));
		sprintf(buf1, "DISPLAY=%s", XDisplayString(XtDisplay(w)));
		putenv(&buf0[0]);
		putenv(&buf1[0]);
		if(gvw->ghostview.filename == NULL
		|| strcmp(gvw->ghostview.filename, "-") != 0) {
			close(std_in[1]);
			dup2(std_in[0], 0);
			close(std_in[0]);
		}
		execvp(argv[0], argv);
		sprintf(buf0, "Exec of %s failed", argv[0]);
		perror(buf0);
		_exit(EXIT_FAILURE);
	}
	else {
		if(gvw->ghostview.filename == NULL) {
			int result;
			close(std_in[0]);

			result = fcntl(std_in[1], F_GETFL, 0);
			result = result | O_NONBLOCK;
			result = fcntl(std_in[1], F_SETFL, result);
			gvw->ghostview.interpreter_input    = std_in[1];
			gvw->ghostview.interpreter_input_id = None;
		}
		else if (strcmp(gvw->ghostview.filename, "-")) {
			close(std_in[0]);
		}
		close(std_out[1]);
		gvw->ghostview.interpreter_output    = std_out[0];
		gvw->ghostview.interpreter_output_id = 
			XtAppAddInput(XtWidgetToApplicationContext(w),
				std_out[0], (XtPointer)XtInputReadMask,
				Output, (XtPointer)w);
		close(std_err[1]);
		gvw->ghostview.interpreter_error    = std_err[0];
		gvw->ghostview.interpreter_error_id = 
			XtAppAddInput(XtWidgetToApplicationContext(w),
				std_err[0], (XtPointer)XtInputReadMask,
				Output, (XtPointer)w);
	}
	XtFree((XtPointer)arguments);
}

/*
 * Stop the interperter, if present, and remove any Input sources.
 * Also reset the busy state.
 */
static void
StopInterpreter(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget) w;

	if(gvw->ghostview.interpreter_pid >= 0) {
		struct sigaction sa, osa;
		int	status;

		/*
		 * Make sure SIGCHLD isn't being ignored, if it is, the
		 * waitpid() might not return at all.
		 * This section of code used to use wait(0) but that really
		 * isn't what we want since it assumes that the process
		 * that dies is the one we tried to kill.  The chance of
		 * confusion is pretty minimal but if we have waitpid(),
		 * which does what we really want, we should use it.
		 *	--mu@echo-on.net.
		 */
		memset((void *)&sa,  '\0', sizeof(sa));
		memset((void *)&osa, '\0', sizeof(osa));
		sa.sa_handler = SIG_DFL;
		sigemptyset(&sa.sa_mask);
		sigaction(SIGCHLD, &sa, &osa);
		kill(gvw->ghostview.interpreter_pid, SIGTERM);
		waitpid(gvw->ghostview.interpreter_pid, &status, 0);
		gvw->ghostview.interpreter_pid = -1;
		sigaction(SIGCHLD, &osa, NULL);
	}

	if(gvw->ghostview.interpreter_input >= 0) {
		GVFile	*ps_old;

		close(gvw->ghostview.interpreter_input);
		gvw->ghostview.interpreter_input = -1;
		if(gvw->ghostview.interpreter_input_id != None) {
			XtRemoveInput(gvw->ghostview.interpreter_input_id);
			gvw->ghostview.interpreter_input_id = None;
		}
		while(gvw->ghostview.ps_input) {
			ps_old = gvw->ghostview.ps_input;
			gvw->ghostview.ps_input = ps_old->next;
			if(ps_old->close)
				fclose(ps_old->fp);
			XtFree((XtPointer)ps_old);
		}
	}

	if(gvw->ghostview.interpreter_output >= 0) {
		close(gvw->ghostview.interpreter_output);
		gvw->ghostview.interpreter_output = -1;
		XtRemoveInput(gvw->ghostview.interpreter_output_id);
	}
	if(gvw->ghostview.interpreter_error >= 0) {
		close(gvw->ghostview.interpreter_error);
		gvw->ghostview.interpreter_error = -1;
		XtRemoveInput(gvw->ghostview.interpreter_error_id);
	}
	gvw->ghostview.busy = False;
	XDefineCursor(XtDisplay(gvw), XtWindow(gvw), gvw->ghostview.cursor);
}

/*
 * The interpeter failed, Stop what's left and notify application
 */
static void
InterpreterFailed(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	StopInterpreter(w);
	XtCallCallbackList(w, gvw->ghostview.message_callback, "Failed");
}


	/**
	 ** Public Routines
	 **/


/*
 * Stop any interpreter and disable new ones from starting.
 */
void
GhostviewDisableInterpreter(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	gvw->ghostview.disable_start = True;
	if(XtIsRealized(w))
		StopInterpreter(w);
}

/*
 * Allow an interpreter to start and start one if the widget is
 * currently realized.
 */
void
GhostviewEnableInterpreter(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	gvw->ghostview.disable_start = False;
	if(XtIsRealized(w))
		StartInterpreter(w);
}

/*
 * Returns True if the interpreter is ready for new input.
 */
Boolean
GhostviewIsInterpreterReady(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	return gvw->ghostview.interpreter_pid != -1
		&& !gvw->ghostview.busy
		&& gvw->ghostview.ps_input == NULL;
}

/*
 * Returns True if the interpreter is running.
 */
Boolean
GhostviewIsInterpreterRunning(Widget w)
{
	return ((GhostviewWidget)w)->ghostview.interpreter_pid != -1;
}

/*
 * Returns the current backing pixmap.
 */
Pixmap
GhostviewGetBackingPixmap(Widget w)
{
	GhostviewWidget	gvw = (GhostviewWidget)w;

	return (gvw->core.background_pixmap != XtUnspecifiedPixmap)
		? gvw->core.background_pixmap
		: None;
}

/*
 * Queue a portion of a PostScript file for output to ghostscript.
 * fp: FILE * of the file in question.  NOTE: if you have several
 * Ghostview widgets reading from the same file.  You must open
 * a unique FILE * for each widget.
 * SendPS does not actually send the PostScript, it merely queues it
 * for output.
 * begin: position in file (returned from ftell()) to start.
 * len:   number of bytes to write.
 *
 * If an interpreter is not running, nothing is queued and
 * False is returned.
 */
Boolean
GhostviewSendPS(Widget w, FILE *fp, off_t begin, unsigned len, Bool close)
{
	GhostviewWidget	gvw = (GhostviewWidget) w;
	GVFile		*ps_new;

	if(gvw->ghostview.interpreter_input < 0
	|| (ps_new = (GVFile *)XtMalloc(sizeof(GVFile))) == NULL)
		return False;
	ps_new->fp          = fp;
	ps_new->begin       = begin;
	ps_new->len         = len;
	ps_new->seek_needed = True;
	ps_new->close       = close;
	ps_new->next        = NULL;

	if(gvw->ghostview.input_buffer == NULL
	&& (gvw->ghostview.input_buffer = (char *)XtMalloc(GV_BUFSIZ)) == NULL){
		XtFree((XtPointer)ps_new);
		return False;
	}

	if(gvw->ghostview.ps_input == NULL) {
		gvw->ghostview.input_buffer_ptr  = gvw->ghostview.input_buffer;
		gvw->ghostview.bytes_left        = len;
		gvw->ghostview.buffer_bytes_left = 0;
		gvw->ghostview.ps_input          = ps_new;
		gvw->ghostview.interpreter_input_id =
			XtAppAddInput(XtWidgetToApplicationContext(w),
				gvw->ghostview.interpreter_input,
				(XtPointer)XtInputWriteMask, Input,
				(XtPointer)w);
	}
	else {
		GVFile	*p;
		for(p = gvw->ghostview.ps_input; p->next != NULL; p = p->next)
			;
		p->next = ps_new;
	}

	return True;
}

/*
 * Tell ghostscript to start the next page.
 * Returns False if ghostscript is not running, or not ready to start
 * another page.
 * If another page is started.  Sets the busy flag and cursor.
 */
Boolean
GhostviewNextPage(Widget w)
{
	GhostviewWidget		gvw = (GhostviewWidget)w;
	GhostviewWidgetClass	gvc = (GhostviewWidgetClass)XtClass(w);
	XEvent			event;

	if(gvw->ghostview.interpreter_pid < 0
	|| gvw->ghostview.mwin == None
	|| gvw->ghostview.busy)
		return False;

	gvw->ghostview.busy = True;
	XDefineCursor(XtDisplay(gvw), XtWindow(gvw),gvw->ghostview.busy_cursor);

	event.xclient.type         = ClientMessage;
	event.xclient.display      = XtDisplay(w);
	event.xclient.window       = gvw->ghostview.mwin;
	event.xclient.format       = 32;
	event.xclient.message_type = XmuInternAtom(XtDisplay(w),
						gvc->ghostview_class.next);
	XSendEvent(XtDisplay(w), gvw->ghostview.mwin, False, 0, &event);

	/*
	 * force delivery
	 */
	XFlush(XtDisplay(w));

	return True;
}

/*
 * convert a screen bounding-box to a PostScript bounding-box
 */
void
GhostviewPSBBox(Widget w, int x1, int y1, int x2, int y2,
						GhostviewReturnStruct *gs)
{
#	define	GVABS(x)	(((x) >= 0) ? (x) : -(x))
	GhostviewWidget	gv = (GhostviewWidget)w;
	Position	x, y;

	x        = (x1 + x2) / 2;
	y        = (y1 + y2) / 2;
	gs->xdpi = gv->ghostview.xdpi;
	gs->ydpi = gv->ghostview.ydpi;

	switch(gv->ghostview.orientation) {
	default:
	case GvPageOrientationPortrait:
		gs->psx    = gv->ghostview.llx + (x * 72.0) / gs->xdpi;
		gs->psy    = gv->ghostview.ury - (y * 72.0) / gs->ydpi;
		gs->width  = (GVABS(x2 - x1) * 72.0) / gv->ghostview.xdpi;
		gs->height = (GVABS(y2 - y1) * 72.0) / gv->ghostview.ydpi;
		break;
	case GvPageOrientationLandscape:
		gs->psx    = gv->ghostview.llx + (y * 72.0) / gs->ydpi;
		gs->psy    = gv->ghostview.lly + (x * 72.0) / gs->xdpi;
		gs->width  = (GVABS(y2 - y1) * 72.0) / gv->ghostview.ydpi;
		gs->height = (GVABS(x2 - x1) * 72.0) / gv->ghostview.xdpi;
		break;
	case GvPageOrientationUpsideDown:
		gs->psx    = gv->ghostview.urx - (x * 72.0) / gs->xdpi;
		gs->psy    = gv->ghostview.lly + (y * 72.0) / gs->ydpi;
		gs->width  = (GVABS(x2 - x1) * 72.0) / gv->ghostview.xdpi;
		gs->height = (GVABS(y2 - y1) * 72.0) / gv->ghostview.ydpi;
		break;
	case GvPageOrientationSeascape:
		gs->psx    = gv->ghostview.urx - (y * 72.0) / gs->ydpi;
		gs->psy    = gv->ghostview.ury - (x * 72.0) / gs->xdpi;
		gs->width  = (GVABS(y2 - y1) * 72.0) / gv->ghostview.ydpi;
		gs->height = (GVABS(x2 - x1) * 72.0) / gv->ghostview.xdpi;
		break;
	}
#	undef	GVABS
}

/*
 * initialize a ghostview widget
 */
void
GhostviewInitialize(Widget w)
{
	((GhostviewWidget)w)->ghostview.disable_start = True;
	if(XtIsRealized(w))
		Setup((GhostviewWidget)w);
}
