/*****************************************************************************

	db.c

	Environment:    Unix R40V3/Solaris2/Linux.

	Revision history:	@(#)db.c	1.9	97/06/23


	DESCRIPTION: Part of the Mdb Application.
			Main database routines. Keep
			X-stuff out of here if possible.

        COPYRIGHT NOTICE:
        Permission to use,  copy,  modify,  and  distribute  this
        software  and  its    documentation   is  hereby  granted
        without fee, provided that  the  above  copyright  notice
        appear  in all copies and that both that copyright notice
        and  this  permission   notice   appear   in   supporting
        documentation.  The   author  makes  no   representations
        about   the   suitability   of   this  software  for  any
        purpose.  It  is  provided  "as  is"  without  express or
        implied warranty.

        THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD  TO  THIS
        SOFTWARE,    INCLUDING    ALL   IMPLIED   WARRANTIES   OF
        MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE AUTHOR
        BE  LIABLE  FOR  ANY  SPECIAL,  INDIRECT OR CONSEQUENTIAL
        DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS  OF
        USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
        NEGLIGENCE  OR  OTHER  TORTIOUS   ACTION,   ARISING   OUT
        OF   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
        SOFTWARE.

******************************************************************************/
/******************************************************************************/
#pragma ident "@(#)db.c      1.9		97/06/23"

#define DB
#include "mdb.h"

/*
 * Externals.
 */
extern Boolean	Case;
extern items	Item[];

/*
 * Forward declarations.
 */
static void	db_close(FILE *fd);
static void	db_header(FILE *fd, int cnt);
static FILE	*db_open(char *path, Boolean save);
static int	db_truncate(FILE *fd);

/*
 * Local variables.
 */
static char 	Class[256];
static char 	Folder[256];
static char 	Revision[256];
static char	Folderpath[PATH_MAX];
static Boolean	Writable;
static FILE	*LCKfd = NULL;
static FILE	*Fdup = NULL;


/*
 * Check folder existent. Create if necessary.
 * Synopsis:
 * int db_init( char *folder, char *folderpath )
 * Where:
 *	folder:		name of folder.
 *	folderpath:	path to folder.
 *
 *	If "folder" is defined as the string "PRIVATE",
 *	then "folderpath" will be forced relative the
 *	user's home directory.
 *
 * Return:
 *	0	= OK.
 *	-1	= Error, and errno will be set accordingly.
 */
/*ARGSUSED*/
int db_init( char *folder, char *folderpath )
{

	struct stat sbuff;
	char path[PATH_MAX];
	char buff[PATH_MAX];
	char dbuff[1024];
	char *fpath, *home, *ptr;
	int  i;
	mode_t	mode = 0644;
	FILE *fd;

	Folder[0] = Folderpath[0] = '\0';
	Writable = False;

	if (Fdup) {
		(void)fclose(Fdup);
		Fdup = NULL;
	}

	/*
	 * Private folders are installed in
	 * the user's home directory.
	 */
	if ( ! strcmp( folder, PRIVATE ) ) {
		if (( home = getenv( "HOME" )) == NULL ) {
			errno = EACCES;
			return(-1);
		}
		mode = 0600;
		(void)sprintf( buff, "%s/%s", home, GetPrivateFolderPath() );
		fpath = buff;
		(void)strcpy( folderpath, fpath );
	} else
		fpath = folderpath;

	(void)sprintf( path, "%s/%s%s", fpath, folder, MDBEXT );

	/*
	 * Create or open folder.
	 */
	if ( stat( path, &sbuff ) ) {

		if ( errno == ENOENT ) {
			(void)sprintf( buff, "/bin/mkdir -p %s", path );
			*strrchr( buff, '/' ) = '\0';
			(void)strcat( buff, " 2>/dev/null" );

			(void)system( buff );

			if ( (i = creat( path, mode )) == -1 )
				return(-1);

			(void)close( i );

			if ( (fd = db_open( path, True )) == NULL )
				return(-1);

			(void)strcpy( Folder, folder );
			(void)strcpy( Class, GetFolderClass() );

			(void)db_header( fd, 0 );
			(void)db_close(fd);
			Writable = True;


		} else 
			return(-1);
	} else  {
		if ( (fd = db_open( path, False )) == NULL )
			return(-1);

		/*
		 * Register class and revision.
		 */
		(void)fgets( dbuff, sizeof(dbuff), fd );
		(void)fgets( dbuff, sizeof(dbuff), fd );

		if ( (ptr = strchr( dbuff, ')' )) == NULL ) {
			(void)strcpy( Class, "UNKNOWN" );
			(void)db_close(fd);
			return(0);
		}

		*ptr = '\0';
		if ( (ptr = strchr( dbuff, '(' )) == NULL ) {
			(void)strcpy( Class, "UNKNOWN" );
			db_close(fd);
			return(0);
		}
		(void)strcpy( Class, ++ptr );

		(void)fgets( dbuff, sizeof(dbuff), fd );
		if ( (ptr = strchr( dbuff, ')' )) == NULL ) {
			(void)strcpy( Class, "UNKNOWN" );
			db_close(fd);
			return(0);
		}
		*ptr = '\0';

		if ( (ptr = strchr( dbuff, '(' )) == NULL ) {
			(void)strcpy( Class, "UNKNOWN" );
			db_close(fd);
			return(0);
		}
		(void)strcpy( Revision, ++ptr );

		db_close(fd);

		if ((sbuff.st_mode & S_IWRITE) && (geteuid() == sbuff.st_uid)) 
			Writable = True; else
		if ((sbuff.st_mode & S_IWGRP) && (getgid() == sbuff.st_gid)) 
			Writable = True;
	}

	(void)strcpy( Folder, folder );
	(void)strcpy( Folderpath, path );

	return( 0 );
}


/*
 * Uncase a text string.
 */
/*ARGSUSED*/
static void uncase( unsigned char buff[], Boolean stat )
{
	size_t i, cnt;

	if ( Case == True || stat == True ) return;

	cnt = strlen( (char *)buff );

	for( i = 0; i < cnt; i++ ) {
#if defined(SYSV) && !defined(SVR4)
	/*
	 * Does not support 8 bit international char. set.
	 */
		if ( buff[i] >= 0x7f )
			continue;
#endif
		buff[i] = tolower( buff[i] );
	}
}


/*
 * Compare two strings.
 */
/*ARGSUSED*/
static int strcompare( char *s1, char *s2, Boolean stat )
{
	int len;
	int i = 0;
	int tot = 1;

	/*
	 * Scan for a match within the string
	 * or a left aligned match.
	 */
	if ( ! stat )
		tot = strlen(s2);

	len = strlen(s1);

	if ( !tot || !len ) return(1);

	if (*s1 == '*') return(stat == True ? 1 : 0);

	while( i < tot )
		if ( ! strncmp( s1, &s2[i++], len ) )
			return(0);

	return(1);
}


/*
 * how = 0;	Find all items.
 * how = 1;	Find all by type.
 * how = 2;	Find all by type and items.
 * Everyting has to pass the filter.
 */
/*ARGSUSED*/
static int
checkitem( char *pitems[], char *filter, char *name, int how, FILE *fd )
{

	int rval = 0;
	int item, found;
	char dbuff[1024];

	if ( (filter == NULL) && (pitems) == NULL )
		return(1);

	if ( filter != NULL ) {

		uncase( (unsigned char *)filter, 0 );
		(void)strcpy( dbuff, name );
		uncase( (unsigned char *)dbuff, 0 );
		if ( strcompare( filter, dbuff, 0 ) )
			return(0);

	}

	if ( how == 0 ) found = 1; else found = 0;

	if ( pitems != NULL ) {
		while( fgets( dbuff, sizeof(dbuff), fd ) != NULL ) {

			dbuff[strlen(dbuff)-1] = '\0';

			if (! strncmp( dbuff, DBTAIL, sizeof(DBTAIL)-1 ) )
				break;

			if ( dbuff[0] != '\t' )
				continue;

			if ( ! found ) {
				for ( item = 0; item < GetMaxItems(); item++ ) {
					if ( pitems[item] == NULL )
						break;
					if ( ! strcmp( dbuff, pitems[item] ) ) {
						found = 1;
						if ( how == 1 )
							return(1);
						break;
					}
				}
				if ( found ) continue;
			}

			if ( (!found) && (how) ) continue;

			for ( item = 0; item < GetMaxItems(); item++ ) {
				if ( pitems[item] == NULL )
					break;
				if ( ! strcmp( dbuff, pitems[item] ) ) {
					rval = 1;
					break;
				}
			}

			if ( (rval) && (found) && (how == 2) )
				return(1);

			if ( (rval) && (!how) )
				break;
		}

	} else if ( filter != NULL ) rval = 1;

	return(rval);
}


/*
 * Return a list of selected items.
 * Synopsis:
 * FILE *db_itemlist(char *pitems[], char *filter, int how, int *tot)
 * Where:
 *	pitems:		A NULL terminated list of string pointers
 *			of items we are interested in or NULL.
 *	filter:		A string pattern to filter items or NULL.
 *	how:		= 0 Find all items.
 *			= 1 Find all by type.
 *			= 2 Find all by type and items
 *	tot:		Returned No. of items found.
 *
 * Returns:
 *	An opened FILE descriptor with selected items.
 */
/*ARGSUSED*/
FILE * db_itemlist( char *pitems[], char *filter, int how, int *tot )
{
	char dbuff[1024];
	char *ptr;
	static FILE *fd, *fdi;
	long next;

	*tot = 0;

	if ( (fdi = db_open( Folderpath, False )) == NULL )
		return(NULL);

	if ( (fd = tmpfile()) == NULL ) {
		db_close(fdi);
		return(NULL);
	}

	while( fgets( dbuff, sizeof(dbuff), fdi ) != NULL ) {
		if (! strncmp( dbuff, DBHEAD, sizeof(DBHEAD)-1 ) ) {

			next = strtol( &dbuff[sizeof(DBHEAD)], &ptr, 10 ) +
				( ftell(fdi) - (long)strlen(dbuff) );

			*strrchr( dbuff, ')' ) = '\0';
			ptr = strrchr( dbuff, '(' );
			ptr++;
			if (checkitem( pitems, filter, ptr, how, fdi )) {
				(void)fprintf( fd, "%s\n", ptr );
				(*tot)++;
			}

			(void)fseek( fdi, next, SEEK_SET );
		}
	}

	db_close(fdi);

	if ( fseek ( fd, 0L, SEEK_SET ) ) {
		(void)fclose(fd);
		return(NULL);
	}

	return(fd);
}


/*ARGSUSED*/
static int view_item( FILE *fd )
{
	int item;
	size_t size;
	char *ptr, dbuff[1024];

	/*
	 * An item is found, and we continue to read
	 * the file to update the screen with the
	 * item values found in the file.
	 */

	/*
	 * First blank all fields.
	 */
	for( item = 0; item < GetMaxItems(); item++ )
		(void)xwrite( "", item );

	while ( fgets( dbuff, sizeof(dbuff), fd ) != NULL ) {

		if ( ! strncmp( dbuff, DBTAIL, sizeof(DBTAIL)-1) )
			break;

		if ( (item = atoi(&dbuff[1])) == DELETED )
			continue;

		if ( Item[item].type == STEXTW ) {
			ptr = (char *)malloc( (size = atol(&dbuff[5])) );
			(void)fread( ptr, sizeof(char), size, fd );
			ptr[size-1] = '\0';
			xwrite( ptr, item );
			free(ptr);
		} else {
			dbuff[strlen(dbuff)-1] = '\0';
			xwrite( &dbuff[5], item );
		}
	}

	if ( ferror( fd ) ) 
		return(-1);

	return(0);
}

/*ARGSUSED*/
static int rm_item( FILE *fd,  size_t beg, long end )
{

#if defined(F_FREESP)
	struct flock fl;
#else
	long curpos;
#endif
	size_t cnt;
	char dbuff[8192];

	/*
	 * An item is about to be updated, or deleted.
	 * In either case the existing one has to be removed
	 * from the database, and then eventually appended last.
	 */

	/*
	 * We need an extra duplicate of our db file.
	 */
	if ( Fdup == NULL ) {
		if ( (Fdup = fopen( Folderpath, "r" )) == NULL )
			return(-1);
	}

	if ( fseek ( Fdup, end, SEEK_SET ) )
		return(-1);

	if ( fseek ( fd, (long)beg, SEEK_SET ) )
		return(-1);

	/*
	 * Overwrite the old item.
	 */
	while( (cnt = fread( dbuff, sizeof(char), sizeof(dbuff), Fdup )) ) {
		if ( fwrite( dbuff, sizeof(char), cnt, fd ) != cnt )
			return(-1);
	}

	(void)fflush(fd);

	/*
	 * Truncate from here to EOF.
	 */
#if defined(F_FREESP)
	bzero( &fl, sizeof(struct flock) );
	fl.l_whence = 1;
	fl.l_start  = 0;
	fl.l_len    = 0;
	if ( fcntl( fileno(fd), F_FREESP, &fl ) )
		return(-1);
#else
	curpos = ftell( fd );

	if ( fseek( fd, 0L, SEEK_END ) )
		return(-1);

	if ( ftruncate( fileno(fd), (size_t)(ftell(fd) - (end-beg)) ) < 0 )
		return(-1);

	if ( fseek ( fd, curpos, SEEK_SET ) )
		return(-1);
#endif

	/*
	 * Check errors.
	 */
	if ( ferror( Fdup ) )
		return(-1);

	return(0);
}


/*
 * Synopsis:
 *   long getitem( char *key, FILE *fd, Boolean save,
 *				long nextpos, long *curpos, Boolean exact )
 *
 * Where:	key	is the pattern to search for.
 *		*fd	the file to search.
 *		save	flag to indicate that we are about to updata/save.
 *		nextpos	where to begin search.
 *		curpos	is a return value to be used as nextpos for 'next'.
 *              exact   indicates exact search.
 *
 * If an secondary key item exists, it vill be evaluated too.
 *
 * Returns:	The file position to an existing item.
 *		If not found, return 0.
 */
/*ARGSUSED*/
static long
getitem( char *key, FILE *fd, Boolean save,
			long nextpos, long *curpos, Boolean exact )
{

	char *skey;
	char keyCopy[1024];
	char skeyCopy[1024];
	char keyValueCopy[1024];
	char dbuff[1024];
	char *itemptr, *keyptr, *skeyptr;
	long found, pos;
	long end = 0;
	size_t len;
	size_t beg = 0;
	Boolean stat = save || exact ? True : False;

	pos = found = *curpos = 0;

	if ( nextpos == DB_DELETE )
		nextpos = 0;

	if ( fseek( fd, nextpos, SEEK_SET ) )
		return(-1);

	/*
	 * Save the original key
	 * before we uncase the key string.
	 */
	(void)strcpy( keyCopy, key );
	uncase((unsigned char *)keyCopy, stat );

	while( fgets( dbuff, sizeof(dbuff), fd ) != NULL ) {

		if (! strncmp( dbuff, DBHEAD, sizeof(DBHEAD)-1 ) ) {

			/*
			 * Get the registrated lenght of this item.
			 */
			pos = atol( &dbuff[sizeof(DBHEAD)] );

			len = strlen(dbuff);

			dbuff[len-2] = '\0';

			keyptr = strrchr( dbuff, '(' ); keyptr++;

			(void)strcpy( keyValueCopy, keyptr );

			/*
			 * Check for a secondary key.
			 */
			if ( (skeyptr = strrchr( keyptr, '[' )) != NULL ) {
				*--skeyptr = '\0'; skeyptr += 2;
				dbuff[len-3] = '\0';

				(void)strcpy( skeyCopy, skeyptr );
				uncase((unsigned char *)skeyptr, stat);

				itemptr = strrchr( keyValueCopy, '[' );
				*--itemptr = '\0';
				uncase((unsigned char *)keyptr, stat);

				if ( GetSKeyItem() ) {
					/*
					 * Read the skey from the screen item.
					 */
					skey = xread( GetSKeyItem() );
					uncase( (unsigned char*)skey,stat );

					if ( strcompare(skey, skeyptr, stat) ) {
						(void)fseek( fd,
						    pos-len, SEEK_CUR );
						continue;
					}
				} else {
					(void)fseek( fd, pos-len, SEEK_CUR );
					continue;
				}

			} else
				uncase( (unsigned char *)keyptr, stat );

			/*
			 * Break out if the key(s) match.
			 */
			if ( ! strcompare( keyCopy, keyptr, stat ) ) {
				found = beg = ftell(fd) - len;
				end = beg + pos;
				*curpos = end;
				/*
				 * Return the complete found string.
				 */
				(void)strcpy( key, keyValueCopy );
				break;
			}

			/*
			 * .. if not, go for next.
			 */
			(void)fseek( fd, pos-len, SEEK_CUR );
		}
	}


	if ( found ) {

		if ( save ) {
			if ( rm_item( fd, beg, (long)end ) )
				return(-1);
		} else {
			if ( view_item( fd ) )
				return(-1);
		}
	}

	if ( fseek( fd, 0L, SEEK_END ) )
		return(-1);

	return(found);
}


/*
 * Open the database and perform the requested action - save or just search.
 *
 * Synopsis:
 * int db_access( Boolean save, long nextpos, long *curpos, Boolean exact )
 *
 *
 * Where:
 *  nextpos:
 *	If `nextpos' is positive then db_access() will start the search
 *	at that position.
 *
 * 	If `nextpos' is set to DB_APPEND, then db_access() will not check for
 *	existing items and a write `append' is performed.
 *
 *	If `nextpos' is set to DB_TRUNCATE, then the folder will be truncated
 *      and all existing data will be lost.
 *
 *	If `nextpos' is set to DB_DELETE, then a delete item will be performed.
 *	The three latter cases requires `save' to be true.
 *
 *  exact:
 *	Will force exact search, and thus ignore the state of the Case flag.
 *      This parameter will be forced true if save or delete is in progress.
 *
 * Return values:
 * db_access() returns the file position to a found item, or 0 if not found.
 * A second return parameter; `curpos', will be the starting point for the
 * next consecutive search, i.e, it can be used as the `nextpos' input
 * parameter for consecutive db_access() calls.
 * If an error occurs, db_access() will return -1, and errno will be set to
 * indicate the error.
 *
 * Visual Actions.
 * The actual user input parameters are obtained by reading the keyitem
 * field (and/or the secondary keyitem if it is used). If the keyitems
 * match a registrated item, then the screen will be updated accordingly.
 */
/*ARGSUSED*/
int db_access( Boolean save, long nextpos, long *curpos, Boolean exact )
{

	char dbuff[1024];
	char *ptr;
	long found = 0;
	long end, pos;
	int i;
	FILE *fd;

	if ( (save) && (!Writable) ) {
		errno = EACCES;
		return(-1);
	}

	if ( (nextpos < 0) && (!save) ) {
		errno = EACCES;
		return(-1);
	}


	if ( (fd = db_open( Folderpath, save )) == NULL )
		return(-1);

	if ( nextpos == DB_TRUNCATE ) {

		if ( db_truncate( fd ) )
			return(-1);

		db_header( fd, 0 );
		db_close(fd);
		return(0);
	}

	if ( fseek( fd, 0L, SEEK_END ) ) {
		db_close(fd);
		return(-1);
	}

	errno = 0;

	if ( ! save ) {

		/*
		 * Read.
		 */
		if ( (found = getitem( xread( GetKeyItem() ), fd, save,
				nextpos, curpos, exact )) == -1 ) {
			db_close(fd);
			return(-1);
		}

		if ( ! found )
			errno = 0;
		db_close(fd);
		return((int)found);
	}

	/*
	 * Write.
	 */
	if ( nextpos != DB_APPEND ) {

		if ( (found = getitem( xread( GetKeyItem() ), fd, save,
				nextpos, curpos, exact )) == -1 ) {
			db_close(fd);
			return(-1);
		}
	}


	if ( nextpos == DB_DELETE ) {
		/*
		 * Delete item.
		 */
		if ( fseek( fd, 0L, SEEK_SET ) ) {
			db_close(fd);
			return(-1);
		}

		if ( ! fread( dbuff, 1, sizeof(dbuff), fd ) ) {
			db_close(fd);
			return(-1);
		}

		(void)fseek( fd, 0L, SEEK_SET );

		/*
		 * Update tot items in folder.
		 */
		db_header( fd, atoi( &dbuff[sizeof(DBTOT)] )-1 );
		db_close(fd);
		return((int)found);
	}

	/*
	 * Append item.
	 */
	if ( ( GetSKeyItem() ) ) {
		if ( *xread(  GetSKeyItem() ) )
			(void)sprintf( dbuff, " [%s]", xread( GetSKeyItem() ) );
		else
			dbuff[0] = '\0';
	} else
		dbuff[0] = '\0';

	pos = ftell(fd);

	/*
	 * Plug in a NULL sized template.
	 */
	if ( fprintf( fd, "%s<%06d>(%s%s)\n", DBHEAD, 0,
				xread( GetKeyItem() ), dbuff) < 0 ) {
		db_close(fd);
		return(-1);
	}

	/*
	 * Write all item fields.
	 */
	for( i = 0; i < GetMaxItems(); i++ ) {

		if ( *(ptr = xread(i)) ) {

			if ( Item[i].type == STEXTW ) {
				if ( fprintf( fd, "\t%03d\t%05d\n%s\n",
						i, strlen(ptr)+1, ptr ) < 0 ) {
					db_close(fd);
					return(-1);
				}
			} else {
				if ( fprintf( fd, "\t%03d\t%s\n",
						i, ptr ) < 0 ) {
					db_close(fd);
					return(-1);
				}
			}

		}

	}

	if ( fprintf( fd, "%s(%s)\n", DBTAIL, xread( GetKeyItem() ) ) < 0 ) {
		db_close(fd);
		return(-1);
	}

	end = ftell(fd);

	if ( fseek( fd, pos, SEEK_SET ) ) {
		db_close(fd);
		return(-1);
	}

	/*
	 * Plug in actual sizes.
	 */
	if ( fprintf( fd, "%s<%06ld>", DBHEAD, end-pos ) < 0 ) {
		db_close(fd);
		return(-1);
	}

	if ( ! found ) {
		if ( fseek( fd, 0L, SEEK_SET ) ) {
			db_close(fd);
			return(-1);
		}

		if ( ! fread( dbuff, 1, sizeof(dbuff), fd ) ) {
			db_close(fd);
			return(-1);
		}

		(void)fseek( fd, 0L, SEEK_SET );

		/*
		 * Update tot items in folder.
		 */
		db_header( fd, atoi( &dbuff[sizeof(DBTOT)] )+1 );
	}

	db_close(fd);

	return((int)found);
}


/*
 * Open the database and search it for a `pattern' in any field.
 *
 * Synopsis:
 * Boolean db_gsearch( long nextpos, long *curpos, int *field, char *pattern )
 *
 * Where:
 *  nextpos:
 *      If `nextpos' is positive then db_gsearch() will start the search
 *      at that position.
 *  pattern:
 *	The string we are interested in.
 *
 * Return values:
 * db_gsearch() returns `True' if an item is found. A second return parameter:
 * `curpos', will be the starting point for the next consecutive search, i.e,
 * it can be used as the `nextpos' input parameter for consecutive db_gsearch()
 * calls. A third return parameter: `field' is set to the item field number of
 * the found item.
 * If an error occurs, db_gsearch() will return `False'.
 *
 * Visual Actions.
 * If a match occurs, then the screen will expose the item found.
 */
/*ARGSUSED*/
Boolean db_gsearch( long nextpos, long *curpos, int *field, char *pattern )
{

	char dbuff[1024];
	char *ptr, *ptr1, *ptr2;
	size_t len, cnt;
	size_t len1 = 0;
	size_t pos = 0;
	Boolean found = False;
	FILE *fd;

	*curpos = 0;
	*field = 0;

	if ( (fd = db_open( Folderpath, False )) == NULL )
		return(False);

	if ( fseek( fd, nextpos, SEEK_SET ) ) {
		db_close(fd);
		return(False);
	}

	uncase((unsigned char *)pattern, 0 );

	while( fgets( dbuff, sizeof(dbuff), fd ) != NULL ) {

		if (! strncmp( dbuff, DBHEAD, sizeof(DBHEAD)-1 ) ) {

			len = atol( &dbuff[sizeof(DBHEAD)] );
			len1 = strlen(dbuff);

			pos = ftell(fd) - len1;

			dbuff[len1-2] = '\0';

			if ( (ptr = strrchr( dbuff, '[' )) != NULL )
				*--ptr = '\0';

			/*
			 * No. of bytes to process.
			 */
			cnt = (len-len1) -
				(strlen( strrchr(dbuff, '(' ) ) +
				sizeof(DBTAIL)+1);

			/*
			 * Get memory and read in the entire item.
			 */
			if ( (ptr = (char *)malloc( len-len1 )) == NULL ) {
				db_close(fd);
				return(False);
			}

			if ( fread( ptr, sizeof(char), len-len1, fd ) < cnt ) {
				free(ptr);
				db_close(fd);
				return(False);
			}

			ptr[len-len1] = '\0';

			/*
			 * Scan this item.
			 */
			ptr1 = ptr2 = ptr;
			while( (int)cnt > 0 ) {

				*( ptr1 = strchr( ptr1, '\n' ) ) = '\0';

				cnt -= strlen(ptr2)+1;

				if ( *ptr2 == '\t' ) {
					*field = atoi( ++ptr2 );
					ptr2 += 4;
				}

				uncase( (unsigned char *)ptr2, 0 );

				if ( *field != DELETED ) {
					if ( !strcompare( pattern, ptr2, 0 ) ) {
						found = True;
						*curpos = pos + len;
						break;
					}
				}

				ptr2 = ++ptr1;
			}

			free(ptr);
		}

		if ( found )
			break;
	}

	if ( found ) {

		(void)fseek ( fd, pos+len1, SEEK_SET );

		if ( view_item( fd ) ) {
			db_close(fd);
			*curpos = 0; *field = 0;
			return(False);
		}
		db_close(fd);
		return(True);
	}

	db_close(fd);
	return(found);
}


/*ARGSUSED*/
static FILE *db_open( char *path, Boolean save )
{
	struct flock fl;
	int tm = 5;
	FILE *fd;

	/*
	 * Open and lock the database.
	 */
	if ( LCKfd != NULL ) {
		(void)fseek ( LCKfd, 0L, SEEK_SET );
		return(LCKfd);
	}

	bzero( &fl, sizeof(struct flock) );

	if ( save == True ) {
		if ( (fd = fopen( path , "r+" )) == NULL )
			return(NULL);
		fl.l_type = F_WRLCK;
	} else {
		if ( (fd = fopen( path , "r" )) == NULL )
			return(NULL);
		fl.l_type = F_RDLCK;
	}

	while( fcntl( fileno(fd), F_SETLK, &fl ) && --tm )
		(void)sleep(1);


	if ( ! tm ) {
		db_close(fd);
		return(NULL);
	}

	return(fd);
}


/*ARGSUSED*/
static void db_close( FILE *fd )
{

	/*
	 * Close and unlock the database.
	 */

	struct flock fl;
	int tmp = errno;

	if ( LCKfd != NULL )
		return;

	bzero( &fl, sizeof( struct flock ) );
	fl.l_type = F_UNLCK;

	(void)fcntl( fileno(fd), F_SETLKW, &fl );

	(void)fclose(fd);
	errno = tmp;
}


/*ARGSUSED*/
static int db_truncate( FILE *fd )
{

	/*
	 * Truncate the database.
	 */
#if defined(F_FREESP)
	struct flock fl;
#endif

	if ( fseek ( fd, 0L, SEEK_SET ) )
		return(-1);

	(void)fflush(fd);

#if defined(F_FREESP)
	bzero( &fl, sizeof(struct flock) );
	fl.l_whence = 1;
	fl.l_start  = 0;
	fl.l_len    = 0;
	if ( fcntl( fileno(fd), F_FREESP, &fl ) )
		return(-1);
#else
	if ( ftruncate( fileno(fd), 0L ) < 0 )
		return(-1);
#endif

	return(0);
}


/*ARGSUSED*/
static void db_header( FILE *fd, int cnt )
{
	/*
	 * Update the database header.
	 */
	time_t  clock_val;
	char dbuff0[1024];
	char dbuff1[1024];
	char dbuff2[1024];
	char dbuff3[1024];

	(void)time( &clock_val );

	(void)strftime( dbuff0, sizeof(dbuff0),
			"%B %d %Y", localtime( &clock_val ) );

	(void)sprintf( dbuff1, "%s<%06d>", DBTOT, cnt );
	(void)sprintf( dbuff2, DBCLASS, GetFolderClass() );
	(void)sprintf( dbuff3, DBREV, Folder, dbuff0 );
	(void)sprintf( Revision, "%s, Rev. %s", Folder, dbuff0);
	(void)fprintf( fd, "%s\n%s\n%s\n", dbuff1, dbuff2, dbuff3 );
}

char *db_foldername(void)
{
	/*
	 * Return folder name.
	 */
	return( Folder );
}

char *db_folderrev(void)
{
	/*
	 * Return revision of folder.
	 */
	return( Revision );
}

int db_folderitems(void)
{

	/*
	 * Return current No. of db items.
	 */
	char dbuff[1024];
	FILE *fd;

	if ( (fd = db_open( Folderpath, False )) == NULL )
		return(0);

	(void)fgets( dbuff, sizeof(dbuff), fd );
	db_close(fd);

	return( atoi( &dbuff[sizeof(DBTOT)] ) );
}

char *db_class(void)
{
	/*
	 * Return db class.
	 */
	return( Class );
}

Boolean db_iswritable(void)
{
	/*
	 * Return Writable status.
	 */
	return( Writable );
}
 
/*ARGSUSED*/
int db_lock(Boolean stat)
{

	FILE *fd;

	/*
	 * Keep the database locked Explicitly.
	 */
	if ( stat == True && LCKfd == NULL ) {
		if ( (LCKfd = db_open( Folderpath, True )) == NULL )
			return(-1);

	} else if ( LCKfd != NULL ) {

		fd = LCKfd;
		LCKfd = NULL;
		db_close(fd);
	}

	return(0);
}
