/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */

/* $Id: note.c,v 1.1.1.1 1996/04/16 15:12:04 leon Exp $ */

#include <string.h>
#include <time.h>
#include <X11/Intrinsic.h>
#include <Xm/MessageB.h>
#include <Xm/BulletinB.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include "Day.h"
#include "Year.h"
#include "agenda.h"
#include "main.h"
#include "noteD.h"
#include "easym.h"
#include "note.h"
#include "misc.h"
#include "error.h"
#include "spin.h"
#include "alarm.h"
#include "actions.h"


#define XtDefineCursor(widget, cursor)\
    XDefineCursor(XtDisplay(widget), XtWindow(widget), cursor)
#define XtUndefineCursor(widget)\
    XUndefineCursor(XtDisplay(widget), XtWindow(widget))
#define XtFlush(widget) XFlush(XtDisplay(widget))

/* state given when openning the dialog, and current state */
static Appointment first;
AppointmentRec current;
static CalendarId currentCal;


static void
SetDialogState(Appointment ap, Date *date);

/* note destruction is confirmed. do it.
 * set dialog state to "new note", keeping destroyed appointment's date.
 * returns: 
 */
static void
DoDestroyNoteCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    Date date;
    date = first->date;
    RemoveAlarmsFor(first);
    AgendaRemoveAppointment(agenda, currentCal, first);
    SetDialogState(0, &date);
    RedrawAll();
}



/* ask confirmation for note destruction
 * 
 * returns: 
 */
void
DestroyNoteCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    static Widget destroyD = NULL;


    if(NULL == destroyD) {
	destroyD = XmCreateQuestionDialog(toplevel, "destroyD", 0, 0);
	XtAddCallback(destroyD, XmNokCallback, DoDestroyNoteCB, 0);
	XtUnmanageChild(XmMessageBoxGetChild(destroyD, XmDIALOG_HELP_BUTTON));
    }
    XtManageChild(destroyD);
}


/* called when clicked on a hour-up or hour-down spin button
 * see spin.c for spin behavior description
 * hour will be incremented/decremented with the global minuteIncrement value
 * returns: 
 */
static void
ModifyHourCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    int h, m;
    char buffer[128];

    m = current.date.minute + (w == noteD_hour_up ? +Globals.minuteIncrement : -Globals.minuteIncrement);
    h = current.date.hour;

    if(m >= 60) {
	m -= 60;
	h++;
    }
    else 
    if(m < 0) {
	m += 60;
	h--;
    }
    if(h == 24) 
	h = 0;
    else if(h < 0) 
	    h = 23;
	    
    current.date.minute = m;
    current.date.hour = h;

    DayGetDateString(day, current.date, buffer);
    EmSetLabelString(noteD_hour, buffer);
}





static void
SetNewDateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    YearCallbackStruct *cs = (YearCallbackStruct *)call_data;
    char buffer[128];

    int h;

    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, DateCB, NULL);

    h = current.date.hour;
    current.date = cs->date;
    current.date.hour = h;

    YearGetDateString(year, current.date, buffer);
    EmSetLabelString(noteD_date, buffer);
    DayGetDateString(day, current.date, buffer);
    EmSetLabelString(noteD_hour, buffer);

    XtUndefineCursor(toplevel);
    XtUndefineCursor(noteD);
    UnfreezeBriefHelp();
    XtFlush(toplevel);
}

static void
NewDateCB(Widget w, XtPointer client_data, XtPointer call_data)
{

    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, SetNewDateCB, NULL);
    XtDefineCursor(toplevel, Globals.selectCursor);
    XtDefineCursor(noteD, Globals.selectCursor);
    FreezeBriefHelp(MSG[M_NEW_DAY]);
    XtFlush(toplevel);
}


/* called after FirstDayCB, when the user click on a day
 * get first date, and unfreeze interface
 * returns: 
 */
static void
SetFirstDayCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    YearCallbackStruct *cs = (YearCallbackStruct *)call_data;
    char buf[128];

    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, DateCB, NULL);

    current.start = cs->date;
    YearGetDateString(year, cs->date, buf);
    XmTextSetString(noteD_fdateT, buf);
    XtUndefineCursor(toplevel);
    XtUndefineCursor(noteD);
    UnfreezeBriefHelp();
    XtFlush(toplevel);
}


/* called when clicked on FirstDate,the user should now click on a day.
 * remove year's callbacks, and add SetFirstDayCB, display help and freeze
 * interface.
 * returns: 
 */
static void
FirstDayCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, SetFirstDayCB, NULL);
    XtDefineCursor(toplevel, Globals.selectCursor);
    XtDefineCursor(noteD, Globals.selectCursor);
    FreezeBriefHelp(MSG[M_FIRST_DATE]);
    XtFlush(toplevel);
}


/* called after LastDayCB, when the user click on a day
 * get last date and unfreeze interface
 * returns: 
 */
static void
SetLastDayCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    YearCallbackStruct *cs = (YearCallbackStruct *)call_data;
    char buf[128];

    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, DateCB, NULL);

    current.end = cs->date;
    YearGetDateString(year, cs->date, buf);
    XmTextSetString(noteD_ldateT, buf);
    XtUndefineCursor(toplevel);
    XtUndefineCursor(noteD);
    UnfreezeBriefHelp();
    XtFlush(toplevel);

}

/* called when clicked on LastDate,the user should now click on a day.
 * remove year's callbacks, and add SetLastDayCB, display help and freeze
 * interface.
 * returns: 
 */
static void
LastDayCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    XtRemoveAllCallbacks(year, XtNdateCallback);
    XtAddCallback(year, XtNdateCallback, SetLastDayCB, NULL);
    XtDefineCursor(toplevel, Globals.selectCursor);
    XtDefineCursor(noteD, Globals.selectCursor);
    FreezeBriefHelp(MSG[M_LAST_DATE]);
    XtFlush(toplevel);
}




/* maintain dialog consistency for note repeat.
 * if note repeat id NONE, disable the first and last buttons.
 * returns: 
 */
static void
RepeatCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    int sensitive;
    XmToggleButtonCallbackStruct *cs = 
	(XmToggleButtonCallbackStruct *)call_data;
    if(cs->set == True) {
	current.repeat = (Repeat)client_data;
	sensitive = (current.repeat != R_NONE);
	XtSetSensitive(noteD_fdate, sensitive);
	XtSetSensitive(noteD_ldate, sensitive);
	XtSetSensitive(noteD_fnone, sensitive);
	XtSetSensitive(noteD_lnone, sensitive);
    }
}



/* initialize dialog and global variables according to the displayed 
 * appointment. 
 * If ap==NULL (new note mode), we are displaying a new note: initialize
 * to defaults, else initialize with ap (view note mode).
 * returns: 
 */
static void
SetDialogState(Appointment ap, Date *date)
{
    char buffer[256];
    if(NULL == ap){
	/* new note state */
	first = NULL;
	current.alarm.days = 0;
	current.alarm.hours = 0;
	current.alarm.minutes = 0;
	current.note = NULL;
	current.date = *date;
	current.user = cuserid(NULL);
	current.timestamp = time(NULL);
	current.timeout = 0;
	XmToggleButtonSetState(noteD_rnone, True, True);
	XmTextSetString(noteD_note, "");
	sprintf(buffer, "%s: %s", current.user, ctime(&current.timestamp));
	XmTextSetString(noteD_origin, buffer);
	
	YearGetDateString(year, *date, buffer);
	EmSetLabelString(noteD_date, buffer);
	DayGetDateString(day, *date, buffer);
	EmSetLabelString(noteD_hour, buffer);
	XmTextSetString(noteD_fdateT, "");
	XmTextSetString(noteD_ldateT, "");
	XtSetSensitive(noteD_ok, True);
	XtSetSensitive(noteD_delete, False);
	XmToggleButtonSetState(noteD_private, False, False);
    }
    else {
	/* view note state */
	first = ap;
	current = *first;
	
	if((first->private) && (strcmp(first->user, cuserid(NULL)))) {
	    XtSetSensitive(noteD_ok, False);
	    XtSetSensitive(noteD_delete, False);
	}
	else {
	    XtSetSensitive(noteD_ok, True);
	    XtSetSensitive(noteD_delete, True);
	}
	
	
	switch(ap->repeat) {
	case R_YEARLY:
	    XmToggleButtonSetState(noteD_ryearly, True, True); break;
	case R_MONTHLY:
	    XmToggleButtonSetState(noteD_rmonthly, True, True); break;
	case R_WEEKLY:
	    XmToggleButtonSetState(noteD_rweekly, True, True); break;
	case R_DAYLY:
	    XmToggleButtonSetState(noteD_rdayly, True, True); break;
	default:
	    XmToggleButtonSetState(noteD_rnone, True, True); break;
	}
	
	EmOptionMenuSelectNamedItem(noteD_calendar, 
				    agenda->calendars[currentCal]->name);
	sprintf(buffer, "%s: %s", ap->user, ctime(&ap->timestamp));
	XmToggleButtonSetState(noteD_private, current.private, False);
	XmTextSetString(noteD_note, ap->note);
	XmTextSetString(noteD_origin, buffer);
	
	YearGetDateString(year, ap->date, buffer);
	EmSetLabelString(noteD_date, buffer);
	DayGetDateString(day, ap->date, buffer);
	EmSetLabelString(noteD_hour, buffer);
	
	buffer[0] = '\0';
	if(R_NONE != ap->repeat) {
	    YearGetDateString(year, ap->start, buffer);
	}
	XmTextSetString(noteD_fdateT, buffer);
	buffer[0] = '\0';
	if(R_NONE != ap->repeat) {
	    YearGetDateString(year, ap->end, buffer);
	}
	XmTextSetString(noteD_ldateT, buffer);
    }
}



/* called when we click on a note.
 * Builds up the note dialog in view note mode.
 * returns: 
 */
void
NewNoteCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    DayCallbackStruct *cs = (DayCallbackStruct *)call_data;

    if(0 == agenda->numCals) {
	DisplayError(MSG[M_CREATE_GROUP]);
	return;
    }
    if(NULL == noteD) {
	BuildNoteDialog();
    }
    SetDialogState(0, &cs->date);
    XtManageChild(noteD);
}



/* called when an hour is clicked. Builds up the note dialog in new note mode.
 * 
 * returns: 
 */
void
ViewNoteCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    DayCallbackStruct *cs = (DayCallbackStruct *)call_data;
    
    currentCal = cs->cal;
    SetDialogState(cs->id, 0);
    XtManageChild(noteD);
}


/* apply modifications on the selected appointment.
 * if we were in the new note mode, adds the new note to the current calendar,
 * else, in view note mode, remove the selected appointment from the calendar
 * and add a new one.
 * returns: 
 */
void
RegisterNoteCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    current.note = XmTextGetString(noteD_note);
    if((strlen(current.note) > 0))
       {
	AppointmentId aid;
	CalendarId cid;
	Widget scal;

	current.private = XmToggleButtonGetState(noteD_private);
	current.start.hour = current.end.hour = current.date.hour;
	XtVaGetValues(noteD_calendar, XmNmenuHistory, &scal, NULL, 0);
	aid =
	    AgendaRegisterAppointment(agenda, XtName(scal), 
				      &current, &cid);
	if(NULL != first) {
	    AgendaRemoveAppointment(agenda, currentCal, first);
	}
	first = aid;
	DayAddAppointment(day, current.date.hour, current.note, cid, aid, 
			  current.repeat, True);
	InstallAlarmsFor(aid);
	SetDialogState(first, 0);
	YearScanAppointments(year);
	DayScanAppointments(day);
	ClearWidget(year);
	ClearWidget(day);
    }
    else {
	/* no note typed in */
	DisplayError(MSG[M_TYPE_NOTE]);
    }
}


/* set first/last repeat date to none.
 * 
 * returns: 
 */
void
NoneCB(Widget w, XtPointer client_data, XtPointer call_data)
{
    if(w == noteD_fnone) {
	XmTextSetString(noteD_fdateT, "");
	memset(&current.start, sizeof(Date), 0);
    }
    else
    if(w == noteD_lnone) {
	XmTextSetString(noteD_ldateT, "");
	memset(&current.end, sizeof(Date), 0);
    }
	
}

/* Builds up the note dialog.
 * destroy the noteD_calendar widget (used to get resources) and add the option
 * menu instead. attach callbacks and spin behavior.
 * side effect: noteD will contain the dialog widget.
 * returns: 
 */
void
BuildNoteDialog()
{
    Widget button;


    noteD = Build_noteD_1(toplevel);
    XtDestroyWidget(noteD_calendar);
    
    button = EmCreateOptionMenu(noteD, "noteD_calendar", 0, 0);
    noteD_calendar = button;
    XtVaGetValues(button, XmNsubMenuId, &noteD_calendarsM, NULL, 0);
    XtManageChild(button);
    XtAddCallback(noteD_rnone, 
		  XmNvalueChangedCallback, RepeatCB, (XtPointer)R_NONE);
    XtAddCallback(noteD_rdayly, 
		  XmNvalueChangedCallback, RepeatCB, (XtPointer)R_DAYLY);
    XtAddCallback(noteD_rweekly, 
		  XmNvalueChangedCallback, RepeatCB, (XtPointer)R_WEEKLY);
    XtAddCallback(noteD_rmonthly, 
		  XmNvalueChangedCallback, RepeatCB, (XtPointer)R_MONTHLY);
    XtAddCallback(noteD_ryearly, 
		  XmNvalueChangedCallback, RepeatCB, (XtPointer)R_YEARLY);
    XtAddCallback(noteD_fdate, XmNactivateCallback, FirstDayCB, 0);
    XtAddCallback(noteD_ldate, XmNactivateCallback, LastDayCB, 0);
    XtAddCallback(noteD_ok, XmNactivateCallback, RegisterNoteCB, 0);
    XtAddCallback(noteD_delete, XmNactivateCallback, DestroyNoteCB, 0);
    XtVaSetValues(noteD, XmNdefaultButton, noteD_cancel, 0);
    XtAddCallback(noteD_fnone, XmNactivateCallback, NoneCB, 0);
    XtAddCallback(noteD_lnone, XmNactivateCallback, NoneCB, 0);
    XtAddCallback(noteD_cancel, XmNactivateCallback, EmUnmanageCB, noteD);
    XtAddCallback(noteD_date, XmNactivateCallback, NewDateCB, 0);
    XtAddCallback(noteD_hour_up, XmNarmCallback, ModifyHourCB, 0);
    XtAddCallback(noteD_alarms, XmNactivateCallback, AlarmCB, 0);
    InstallSpinBehavior(noteD_hour_up, 500, 0);
    XtAddCallback(noteD_hour_down, XmNarmCallback, ModifyHourCB, 0);
    InstallSpinBehavior(noteD_hour_down, 500, 0);
}
