/*
 * FILE:
 * Sheet.c
 *
 * FUNCTION:
 * create reports 
 *
 * DESIGN:
 * Filters phtml files through an ePerl (Embedded Perl)
 * interpreter.  Sets up variables before filtering.
 *
 * Invokes ePerl interpreter to handle perl embedded in html
 * files. The resulting final html is returned.  Communications
 * with eperl are through fork/exec/pipes.
 *
 * More work needs to be done:
 *  o We need some way of defining which routine gets invoked 
 *    depending on which link gets pressed.  But first, we need
 *    a url-encoding decoder.
 *
 * HISTORY:
 * Created by Linas Vepstas October 1998
 */

/********************************************************************\
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
\********************************************************************/


#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "Account.h"
#include "File.h"
#include "Group.h"
#include "util.h"

extern AccountGroup *topgroup;

static pid_t interp_pid = 0;

/* ======================================================= */

static FILE *
StartUpFile (void) 
{
  int fd;
  char tmpfilename[200];
  FILE *fh;
  
  /* create a tmp file with a bit of privacy */
  strcpy (tmpfilename, "/tmp/gncXXXXXX");
  mktemp (tmpfilename);
  strcat (tmpfilename, ".html");
  fd = open (tmpfilename, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
  if (0 > fd) {
    ERROR ();
    fprintf (stderr, "Error: can't open file %s\n", tmpfilename);
    return NULL;
  }

  fh = fdopen (fd, "w+");
  if (!fh) {
    ERROR ();
    fprintf (stderr, "Error: can't open file %s\n", tmpfilename);
    return NULL;
  }

  /* The unlink makes sure that file is deleted even if we crash.
   * It also affords a bit of privacy; no-one can snoop on our
   * temp file, because without the directory entry, it becomes
   * much harder to find and open.
   */
  unlink (tmpfilename);  

  return fh;
}
  
/* ======================================================= */

#define EPERL_PATH "/usr/bin/eperl"
#define EPERL_NAME "eperl"

static FILE *
StartUpInterp (FILE * out_fh) 
{
  int out_fd;
  int in_fd[2];
  FILE *to_perl;
  struct stat statbuf;
  int rc;

  /* don't bother if the interpreter is not present */
  rc = stat (EPERL_PATH, &statbuf);
  if (rc) {
    printf ("Error: StartUpInterp(): Can't find interpreter %s\n", EPERL_PATH);
    return NULL;
  }

  /* create the pipe that will be duped to stdin */
  pipe (in_fd);

  /* stdout will be dumped into a file */
  out_fd = fileno (out_fh);

  /* interpreter will run in a sub-process */
  interp_pid = fork ();
  if (!interp_pid) {
    /* attach the pipes to stdin and stdout */
    dup2 (in_fd[0], 0);
    close (in_fd[1]);
    dup2 (out_fd, 1);

#ifdef ECHO_DEBUG_STDIN_OUT
    do {
       int c, rc;
       rc = feof (stdin);
       printf ("rc=%d\n", rc);
       c = fgetc (stdin);
       if (EOF == c) {printf ("bye\n"); fflush (stdout); exit (0); }
       printf ("%c", c);
       fflush (stdout);
    } while (1);
#endif /* ECHO_DEBUG_STDIN_OUT */

    execl (EPERL_PATH, EPERL_NAME, "-", NULL);
    {
       int norr = errno;
       fprintf (stderr, "Error: couldn't exec: \n");
       fprintf (stderr, "\t%s (%d) %s\n", EPERL_PATH, norr, strerror (norr));
       exit (1);
    }
  }

  /* assign FILE structures to the fds, close the duped fds */
  to_perl = fdopen (in_fd[1], "w");
  close (in_fd[0]);

  return to_perl;
}

/* ======================================================= */

static char * 
FinishUp (FILE *fh) 
{
  int fd, rc;
  char * text = NULL;
  size_t size;

  /* cleanup zombie process, make sure that interpreter has written
   * to the output file, and properly flushed its buffers, etc.  
   * If we don't wait, then the output buffer might be empty or only
   * partially written.
   */
  waitpid (interp_pid, NULL, 0);

  /* make sure that file buffers are fluished ... Although, 
   * technically these should be empty since we aren't writing to them */
  fflush (fh);

  /* Find size: */
  fd = fileno (fh);
  size = lseek( fd, 0, SEEK_END );
  lseek( fd, 0, SEEK_SET );

  text = (char *) malloc ((size+1)*sizeof (char));

  /* read in file */
  rc = read (fd,text,size);
  if (-1 == rc) {
    free (text);
    text = NULL;
  }
  text[size] = 0x0;
  fclose (fh);
  return text;
}

/* ======================================================= */

#define OUT fprintf (to_perl,

/* ======================================================= */
static int PrtAcc (FILE *to_perl, Account *acc, int i, int lvl);

static int 
PrtGrp (FILE *to_perl, AccountGroup *grp, int idx, int lvl)
{
  int i;
  Account * acc;

  if (!grp) return idx;

  /* copy report data to perl vars */
  i=0;
  acc = xaccGroupGetAccount (grp, i);
  while (acc) {
    idx = PrtAcc (to_perl, acc, idx, lvl);
    i++; acc = xaccGroupGetAccount (grp, i);
  }

  return idx;
}

/* ======================================================= */

static int
PrtAcc (FILE *to_perl, Account *acc, int i, int lvl)
{
   int type;
   double baln;
   AccountGroup *children;

   if (!acc) return i;

   /* look for children,do them first */
   children = xaccAccountGetChildren (acc);
   i = PrtGrp (to_perl, children, i, (lvl+1));

   type = xaccAccountGetType(acc);
   baln = xaccAccountGetBalance(acc);

   /* reverse the sign on income and expense accounts */
   if ((EXPENSE == type) || (INCOME  == type )) {
     baln = -baln;
   }

   OUT "<: $accname[%d] = \"%s\"; "
       "   $acctype[%d] = \"%s\"; "
       "   $accbaln[%d] = %11.2f; "
       "   $acclvl[%d] = %d;  :>\n", 
           i, xaccAccountGetName (acc), 
           i, xaccAccountGetTypeStr (type),
           i, baln,
           i, lvl);

  i++;
  return i;
}

/* ======================================================= */

char * 
gncReport (char * const source_file) 
{
  char * text;
  FILE *from_perl, *to_perl;

  if (!source_file) return NULL; 
  text = gncReadFile (source_file);
  if (!text) return NULL;

  /* open temp staging file for our html output */
  from_perl = StartUpFile ();
  if (!from_perl) return NULL;

  /* initilaize the interpreter, attach its output 
   * to the temp file */
  to_perl = StartUpInterp (from_perl);
  if (!to_perl) {
    fclose (from_perl);
    return NULL;
  }

  /* copy report data to perl vars */
  PrtGrp (to_perl, topgroup, 0, 1);

  /* feed in the parsed-html template */
  OUT "%s", text);

  fflush (to_perl);
  fclose (to_perl);

  text = FinishUp (from_perl);
  return text;
}

/* =============================  END OF FILE ===================== */
