/* AutoCAD DXF file to DKB Data File Converter */
/* Version 1.0 By Aaron A. Collins.  Written 8/13/90 */
/* This program is released to the public domain. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <LL.h>

typedef struct {
  float           x;
  float           y;
  float           z;
}               t_Vertex;


typedef struct {
  t_LL            PolyVertices;
}               t_Polygon;

typedef struct {
  float           x;
  float           y;
  float           z;
  float           radius;
  float		  startangle;
  float		  endangle;
}               t_Sphere;


#define BUFSIZE	2048
#define FABS(x) ((x<0.0)?-x:x)

#ifdef _NO_PROTO 
int getline ();
void writeobj ();
void finishobj ();
void lookupcolor ();
int checkdegen();
void OutputGF();
#else
int getline ();
void writeobj ();
void finishobj (int color);
void lookupcolor (int color, float *red, float *green, float *blue);
int checkdegen(int a, int b, int c);
void OutputGF(FILE *FpOut);
#endif

int groupcode;
char inname[80], outname[80],gfname[80];
char linbuf[BUFSIZE];
FILE *FpIn, *FpOut;
long primitives;
long degenerates;
char curobj[80];
int curcolor;
float curthick;
float xcoords[10];
float ycoords[10];
float zcoords[10];
float floats[10];
float angles[10];
int ints[10];
float max_x, max_y, max_z, min_x, min_y, min_z;
t_LL Polygons;
t_LL Lines;
t_LL Circles;
t_LL Arcs;
t_Vertex Point;
t_Sphere Circle;
t_Polygon Poly;


int
main(argc, argv)
	int argc;
	char **argv;
{
        primitives = 0L;
        degenerates = 0L;
	Polygons=ConsLL();
	Circles=ConsLL();
	Arcs=ConsLL();
        Lines=ConsLL();

	

	printf("\n\nAutoCad DXF to Xmgf gf Data file Translator\n");
	printf("\n Converted from AutoCad DXF to DKB by Aaron A. Collins\n");
	printf("\n dxf2gf by Paul Hoad 16/7/93\n");

  	if (argc < 2) {
    	fprintf (stderr, "Usage: dxf2gf infile.[dxf] outfile\n");
    	exit (0);
  	}

  	if ((FpIn = fopen (argv[1], "r")) == NULL) {
    	fprintf (stderr, "File %s Does not exist\n", argv[1]);
    	exit (-1);
  	}

  	if ((FpOut = fopen (argv[2], "w")) == NULL) {
    	fprintf (stderr, "File %s Does not exist\n", argv[2]);
    	exit (-1);
  	}

	printf("\nPlease wait; Processing...");

	curobj[0] = '\0';	/* not working on any object currently */
	curcolor = 7;		/* and it also doesn't have a color yet... */
	max_x = max_y = max_z = -10000000.0;	/* init bounding limits */
	min_x = min_y = min_z =  10000000.0;

find:	while (!feof(FpIn))	/* run file up to the "ENTITIES" section */
	{
		if (getline())		/* get a group code and a line */
			goto stopit;
		if (groupcode == 0)	/* file section mark */
		{
			if (strstr(linbuf, "EOF"))
				goto stopit;
			if (strstr(linbuf, "SECTION"))
			{
				if (getline())
					goto stopit;
				if (groupcode != 2)
					continue;
				if (strstr(linbuf, "ENTITIES"))
					break;
			}
		}
	}
	while (!feof(FpIn))		/* scan ENTITIES section */
	{
		if (getline())		/* get a group code and a line */
			break;
		if (groupcode < 10)	/* cardinal group codes */
		{
			switch(groupcode)
			{
				case 0: /* start of entity, table, file sep */
					if (strstr(linbuf, "EOF"))
					{
						writeobj(); /* dump object */
						goto stopit;
					}
					if (strstr(linbuf, "ENDSEC"))
					{
						writeobj(); /* dump object */
						goto find;
					}
					writeobj();	/* dump old object */
					curobj[0] = '\0'; /* reset object */
					curcolor = 7;
					strcpy(curobj, linbuf);	/* get new */
					break;
				case 1:	/* primary text value for entity (?)*/
					break;
				case 2: /* block name, attribute tag, etc */
				case 3:	/* other names */
				case 4:
					break;
				case 5:	/* entity handle (hex string) */
					break;
				case 6: /* line type name */
					break;
				case 7: /* text style name */
					break;
				case 8: /* layer name */
					break;
				case 9: /* variable name ID (only in header)*/
					break;
			}
		}
		else if (groupcode >= 10 && groupcode < 19) /* Some X coord */
		{
			sscanf(linbuf, "%f", &(xcoords[groupcode-10]));
			if (xcoords[groupcode-10] > max_x)
				max_x = xcoords[groupcode-10];
			if (xcoords[groupcode-10] < min_x)
				min_x = xcoords[groupcode-10];
		}
		else if (groupcode >= 20 && groupcode < 29) /* Some Y coord */
		{
			sscanf(linbuf, "%f", &(ycoords[groupcode-20]));
			if (ycoords[groupcode-20] > max_y)
				max_y = ycoords[groupcode-20];
			if (ycoords[groupcode-20] < min_y)
				min_y = ycoords[groupcode-20];
		}
		else if (groupcode >= 30 && groupcode < 38) /* Some Z coord */
		{
			sscanf(linbuf, "%f", &(zcoords[groupcode-30]));
			if (zcoords[groupcode-30] > max_z)
				max_z = zcoords[groupcode-30];
			if (zcoords[groupcode-30] < min_z)
				min_z = zcoords[groupcode-30];
		}
		else if (groupcode == 38) /* entity elevation if nonzero */
		{
		}
		else if (groupcode == 39) /* entity thickness if nonzero */
		{
		}
		else if (groupcode >= 40 && groupcode < 49) /* misc floats */
		{
			sscanf(linbuf, "%f", &(floats[groupcode-40]));
		}
		else if (groupcode == 49) /* repeated value groups */
		{
		}
		else if (groupcode >= 50 && groupcode < 59) /* misc angles */
		{
			sscanf(linbuf, "%f", &(angles[groupcode-50]));
		}
		else if (groupcode == 62) /* Color number */
		{
			sscanf(linbuf, "%6d", &curcolor);
		}
		else if (groupcode == 66) /* "entities follow" flag */
		{
		}
		else if (groupcode >= 70 && groupcode < 79) /* misc ints */
		{
			sscanf(linbuf, "%f", &(ints[groupcode-70]));
		}
		else if (groupcode == 210 || groupcode == 220 || groupcode == 230)
		{	/* X, Y, Z components of extrusion direction */
		}
	}
stopit: fclose(FpIn);

	/* My Output Bits */
	OutputGF(FpOut);
	printf("Finished.\n\nTotal objects written to output file: %ld\n\n", primitives);
	printf("Total degenerate triangles removed from scene: %ld\n\n", degenerates);
	printf ("X bounding values range from %f to %f\n", min_x, max_x);
	printf ("Y bounding values range from %f to %f\n", min_y, max_y);
	printf ("Z bounding values range from %f to %f\n", min_z, max_z);
	exit(0);
}


int getline()		/* read a group code and the next line from FpIn */
{
	fgets(linbuf, BUFSIZE, FpIn);	/* get a line from .DXF */
	if (feof(FpIn))
		return(1);
	sscanf(linbuf, "%3d", &groupcode);  /* scan out group code */
	fgets(linbuf, BUFSIZE, FpIn);	/* get a line from .DXF */
	if (feof(FpIn))
		return(1);
	return(0);
}

void writeobj()	/* dump out current object we should have all info on */
{
	if ((!strstr(curobj, "LINE")) && strstr(curobj,"POLYLINE")) /* One Line */
	{
		Poly.PolyVertices = ConsLL ();
		Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
		InsLastLL (Poly.PolyVertices, Point);
		Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
		InsLastLL (Poly.PolyVertices, Point);
		InsLastLL (Lines, Poly);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "POINT"))	/* an itty, bitty sphere! */
	{
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
		InsLastLL (Polygons, Poly);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "CIRCLE"))	/* a VERY short cylinder! */
	{
                Circle.x = xcoords[0]; Circle.y = ycoords[0]; Circle.z = zcoords[0];
		Circle.radius = floats[0];
                InsLastLL (Circles, Circle);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "ARC"))		/* not implemented for now */
	{
                Circle.x = xcoords[0]; Circle.y = ycoords[0]; Circle.z = zcoords[0];
                Circle.radius = floats[0];
		Circle.startangle = angles[0];
		Circle.endangle = angles[1];
                InsLastLL (Arcs, Circle);
		return;
	}
	else if (strstr(curobj, "TRACE"))	/* 2 back-to-back triangles */
	{
		if (checkdegen(0, 1, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
		Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
        	InsLastLL (Poly.PolyVertices, Point);
		Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
        	InsLastLL (Poly.PolyVertices, Point);
		Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
        	InsLastLL (Poly.PolyVertices, Point);
        	InsLastLL (Polygons, Poly);
		finishobj(curcolor);

		if (checkdegen(0, 3, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Polygons, Poly);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "SOLID"))	/* 1 or 2 triangles */
	{
		if (checkdegen(0, 1, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Polygons, Poly);
		if (xcoords[2] == xcoords[3] && ycoords[2] == ycoords[3] && zcoords[2] == zcoords[3])
			return;	/* one triangle was enough... */

		if (checkdegen(0, 3, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Polygons, Poly);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "TEXT"))	/* not implemented for now */
	{
		return;
	}
	else if (strstr(curobj, "SHAPE"))	/* these look very hard */
	{
		return;
	}
	else if (strstr(curobj, "BLOCK"))	/* these look very hard */
	{
		return;
	}
	else if (strstr(curobj, "ENDBLK"))	/* these look very hard */
	{
		return;
	}
	else if (strstr(curobj, "INSERT"))	/* these look very hard */
	{
		return;
	}
	else if (strstr(curobj, "ATTDEF"))	/* not implemented for now */
	{
		return;
	}
	else if (strstr(curobj, "ATTRIB"))	/* not implemented for now */
	{
		return;
	}
	else if (strstr(curobj, "POLYLINE"))	/* these look fairly hard */
	{
        Poly.PolyVertices = ConsLL ();
		return;
	}
	else if (strstr(curobj, "VERTEX"))	/* these look fairly hard */
	{
	Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
        InsLastLL (Poly.PolyVertices, Point);
		return;
	}
	else if (strstr(curobj, "SEQEND"))	/* these look fairly hard */
	{
        InsLastLL (Polygons, Poly);
		return;
	}
	else if (strstr(curobj, "3DLINE"))	/* a VERY skinny triangle! */
	{
		if (xcoords[0] == xcoords[1] && ycoords[0] == ycoords[1] && zcoords[0] == zcoords[1])
		{
			degenerates++;
			return;
		}
                Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Lines, Poly);
		finishobj(curcolor);

		return;
	}
	else if (strstr(curobj, "3DFACE"))	/* 1 or 2 triangles */
	{
		if (checkdegen(0, 1, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Polygons, Poly);
		finishobj(curcolor);

		if (xcoords[2] == xcoords[3] && ycoords[2] == ycoords[3] && zcoords[2] == zcoords[3])
			return;	/* one triangle was enough... */

		if (checkdegen(0, 3, 2))
		{
			degenerates++;
			return;
		}
		Poly.PolyVertices = ConsLL ();
                Point.x = xcoords[0]; Point.y = ycoords[0]; Point.z = zcoords[0];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[1]; Point.y = ycoords[1]; Point.z = zcoords[1];
                InsLastLL (Poly.PolyVertices, Point);
                Point.x = xcoords[2]; Point.y = ycoords[2]; Point.z = zcoords[2];
                InsLastLL (Poly.PolyVertices, Point);
                InsLastLL (Polygons, Poly);
		finishobj(curcolor);
		return;
	}
	else if (strstr(curobj, "DIMENSION"))	/* not implemented for now */
	{
		return;
	}
	return;	/* no current object defined... */
}

void finishobj(color)	/* conclude a DKB-style object definition */
int color;
{
	float red, green, blue;

	lookupcolor(color, &red, &green, &blue);
	printf(".");		/* activity echo (happy dots) */
	primitives++;		/* count another output file primitive */
}

void lookupcolor(color, red, green, blue) /* basic AutoCAD 9-color pallette */
int color;
float *red, *green, *blue;
{
	switch (color)
	{
		case 0:	/* black */
			*red = *green = *blue = 0.0;
			break;
		case 1: /* red */
			*red = 1.0;
			*blue = *green = 0.0;
			break;
		case 2: /* yellow */
			*red = *green = 1.0;
			*blue = 0.0;
			break;
		case 3:	/* green */
			*green = 1.0;
			*red = *blue = 0.0;
			break;
		case 4: /* cyan */
			*blue = *green = 1.0;
			*red = 0.0;
			break;
		case 5: /* blue */
			*blue = 1.0;
			*red = *green = 0.0;
			break;
		case 6: /* magenta */
			*blue = *red = 1.0;
			*green = 0.0;
			break;
		case 8:	/* dk. grey */
			*red = *green = *blue = 0.5;
			break;
		case 9: /* lt. grey */
			*red = *green = *blue = 0.75;
			break;
		case 7: /* white */
		default: /* make anything else white (?) */
			*red = *green = *blue = 1.0;
	}
	return;
}

int checkdegen(a, b, c)		/* check for degenerate triangle structure */
int a, b, c;
{
	if (
	(xcoords[a] == xcoords[b] &&
	 ycoords[a] == ycoords[b] &&
	 zcoords[a] == zcoords[b]) || 
	(xcoords[b] == xcoords[c] &&
	 ycoords[b] == ycoords[c] &&
	 zcoords[b] == zcoords[c]) || 
	(xcoords[a] == xcoords[c] &&
	 ycoords[a] == ycoords[c] &&
	 zcoords[a] == zcoords[c]))
		return(1);
	return(0);
}

void
OutputGF(FpOut)
FILE *FpOut;
{
  t_Polygon *OutPoly;
  t_Vertex *OutPoint;
  t_Sphere *OutSphere;
  char BaseName[100];

  strcpy(BaseName,"dx2gf");
  fprintf(stderr,"%d, Polygons\n",SizeLL(Polygons));

  fprintf (FpOut, "@Set %s\nFormat Polygon3D\n", BaseName);
  ForeachLL_M (Polygons, OutPoly) {
    fprintf (FpOut, "%d ", SizeLL (OutPoly->PolyVertices));
    ForeachLL_M (OutPoly->PolyVertices, OutPoint) {
    fprintf (FpOut, "%1.03f %1.03f %1.03f ", OutPoint->x, OutPoint->y, OutPoint->z);
    }
    fprintf (FpOut, "\n");
  }
  fprintf (FpOut, "@\n");

  fprintf(stderr,"%d, Circles\n",SizeLL(Circles));
  /* Circles */
  fprintf (FpOut, "@Set %s\nFormat Circle3D\n", BaseName);
  ForeachLL_M (Circles, OutSphere) {
      fprintf (FpOut, "%f %f %f %f\n", OutSphere->x, OutSphere->y, OutSphere->z,OutSphere->radius);
  }
  fprintf (FpOut, "@\n");

  fprintf(stderr,"%d, Lines\n",SizeLL(Lines));
  /* Lines */
  fprintf (FpOut, "@Set %s\nFormat Line3D\n", BaseName);
  ForeachLL_M (Lines, OutPoly) {
    ForeachLL_M (OutPoly->PolyVertices, OutPoint) {
    fprintf (FpOut, "%1.03f %1.03f %1.03f ", OutPoint->x, OutPoint->y, OutPoint->z);
    }
    fprintf (FpOut, "\n");
  }
  fprintf (FpOut, "@\n");

  fprintf(stderr,"%d, Arcs\n",SizeLL(Arcs));
  /* Arcs */
  fprintf (FpOut, "@Set %s\nFormat Arc3D\n", BaseName);
  ForeachLL_M (Arcs, OutSphere) {
      fprintf (FpOut, "%f %f %f %f %f %f\n", OutSphere->x, OutSphere->y, OutSphere->z,OutSphere->radius,OutSphere->startangle, OutSphere->endangle);
  }
  fprintf (FpOut, "@\n");


}

