/*	xspaceball - multi-player, multi-display space battles

	              (C) 1992, Kevin Laws

	This program may be copied from, hacked, changed, eaten, stepped on,
	or thrown away as you desire provided no profit is generated thereby.
	Enjoy.

	For those who want to get started right away:

	     xspaceball <display1> <display2> [<display3> [<display4>]]

	or

	     xspaceball <display1> <robotlvl1> [<robotlvl2> [<robotlvl3>]]

	Where <robotlvl#> is the level of skill of the robot you play against,
        with 1 as the best and 9 as the worst.

	For example:

	     xspaceball max:0.0 homer:0.0

	would start a game with two players, one on the display "max" and 
	the other on the display "homer".

	     xspaceball max:0.0 1 5

	would start a game with 1 player and two robots, one a very good
	player, and the other average.

	Anywhere from one to four can play.

	Commands:

	   If you have a keypad:

	       8 - Thrust
	       2 - Stop completely
	       6 - Rotate Right		9 - Rotate Right Quickly
	       4 - Rotate Left		7 - Rotate Left Quickly
	       5 - Fire

	   If you don't or don't like using it:
	       L - Thrust
	       / - Stop Completely
	       . - Rotate Right		> - Rotate Right Quickly
	       , - Rotate Left		, - Rotate Left Quickly
	       SPACE - Fire

	Configurations:

	    - (MINUS) - Decrease the time interval between updates
	    + (PLUS)  - Increase the time interval between updates

	         These are to allow the program to work with different
		 displays.  Sometimes, you'll find that the display is
		 very, very jerky.  In that case, you'll want to increase
		 the time interval to give the other displays time to
		 catch up.  If you find that play is too slow, then
		 decrease the time interval.  A good balance can be
		 found rather easily.

	    S - Toggle "relative" shots.  Normally, shots are fired
	        at the shot speed plus the speed the ship that fired
		the shot is travelling.  "S" can toggle between that
		and a "fixed speed" shot, always travelling at a 
		constant speed.

	    P - Toggle planet.  This adds (or removes) a planet in the
	        center of the screen.  If the planet is on and the player
		hits it, then the player will take damage just as
		if hit by a shot.

	    G - Increase gravity.  Initially, there is no gravity.  This
	        increases the gravity making a stronger pull towards the
		center.

	    g - Decrease gravity.  (See above).  Gravity can be decreased
	        to the point of negative gravity, with the center actually
		REPULSING objects approaching it!

	    A - Add asteroid.  This adds an asteroid to the game, copied
	        directly from X-asteroids.  It cannot be shot, but it can
		hit and damage a ship!

	    a - Remove asteroid.  Removes previously added asteroids.

	    B - Toggle bounce mode.  Initally, "bounce" is on, meaning
	        that when a ship hits the edge of the playing field, it
		bounces back from it.  If bounce is off, the player
		travels through to the other side.

	    Z - Add a ball to play (if previously removed).  See below for
	        the different rules in effect when the ball is in play.

	    z - Remove the ball from play and go to "shoot-em-up" mode.
	   

	The program was written loosely based on the SPACEWAR arcade game, 
	but using the vector graphics and rotation functions found in 
	"X-Asteroids" by Phil Goetz.

	Various other functions were inserted to make this more than
	a blast-em-up game.  There are two main "play-modes" to the game.
	Without the ball in play, the game resembles spacewar fairly closely.
	The options are there: planet, gravity, borders, etc.  In this game,
	every player tries to shoot every other player.  I don't remember
	exactly, but I believe I let each player be shot 4 times before they
	died.  When there is only one player left, the score counters are
	updated and everybody starts again.

	The other mode is "spaceball" mode.  In this mode, there is a small
	spaceball (actually a direct rip-off of the small asterod shape in
	Phil Goetz's X-asteroids).  The object of the game is to pick up the
	spaceball by running into it and then make a point by flying into your
	goal (the square of your color) with it.  You cannot shoot while
	holding the ball, and you travel more slowly.  Shooting somebody just
	screws up their ship for a short time, and if they have the ball, it
	will put the ball back into neutral for anybody to pick up.  Points
	are only awarded for getting the ball into your goal, not for shooting
	anybody...in this game, shooting another player is only the means
	to another end.
*/
	
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <math.h>
#include <time.h>
#include <limits.h>

#define	WINDOW_X	0
#define WINDOW_Y	0
#define WINDOW_WIDTH	1004
#define	WINDOW_HEIGHT	660

#define HUMAN_PLAYER	0
#define ROBOT_PLAYER	1
#define NO_PLAYER	2

#define	pi	3.1415926535897932384

#define SHIP		16
#define BALL		21
#define FIRST_AST	22
#define PLANET		31

#define SHAPE_SHIP	0
#define SHAPE_SHOT	1
#define SHAPE_AST	2
#define SHAPE_BALL	3
#define SHAPE_GOAL	4
#define SHAPE_PLANET	5

typedef struct {
    double angle;
    int length;
} PolarPair;

typedef struct {
    int shape;
    int pnum;
    int alive;
    int time;
    double x, y, xvel, yvel, rot, rotvel;
    int hits;
} ObjType;

struct timeval octime;
struct timeval stime;
struct timezone zone;

ObjType oldobj[PLANET + 1];
ObjType obj[PLANET + 1];
ObjType goal[4];

PolarPair shapes[5][12] = {
    {
	{0,0}, {5*pi/4,14}, {0,10},
	{pi/4,14}, {3*pi/4,14}, {pi,10},
	{7*pi/4,14}
    },
    {
	{0,0}, {0,10}
    },
    {
	{0,0}, {3*pi/2,20}, {0,10}, {pi/4,14}, {pi/2,20},
	{3*pi/4,14},{pi,20},{5*pi/4,14},{3*pi/2,20},
	{7*pi/4,14},{0,10},{pi/2,20}
    },
    {
	{0,0}, {3*pi/2,10}, {0,5}, {pi/4,7}, {pi/2,10},
	{3*pi/4,7},{pi,10},{5*pi/4,7},{3*pi/2,10},
	{7*pi/4,7},{0,5}, {pi/2,10}
    },
    {
	{0,0}, {pi,10}, {3*pi/2,10}, {0,20}, {pi/2,20},
	{pi,20}, {3*pi/2, 10}
    }
};

int numpairs[5] = {7,2,11,11, 7};
int shapesize[6] = {14*14, 1, 30*30, 14*14, 14*14, 28*28};

Display *display[4];
Window window[4];
int	robdelay[4];
int	player_stat[4];
int	curdelay = 0;
unsigned long tpv[4][4];
GC pgc[4][4];
GC ngc[4];
int score[4];
char **pname;
int ready[4];
int shots[4];
int relatives[4];
int alive;
int player_count;
int drawscale;
int speedscale;
int timint;
int maxhits;
int maxshots;
int maxspeed;
int carryballmax;
int gravity;
int bounce;
int planet;
int playball;
int numasts;
int shotspeed;
double twopi;

int
graphic_isqrt( square )
	long square;
{
    long a,b,c,as,bs;

    a = 1;
    b = 1;
    while (a<=square) {
	a = a <<2;
	b = b <<1;
    }
    as = 0;
    bs = 0;
    while ( (b>1) && (square>0) ) {
	a = a >> 2;
	b = b >> 1;
	c = square - (as | a);
	if (c >=0 ) {
	    square = c;
	    as |= (a<<1);
	    bs |= b;
	}
	as = as >> 1;
    }
    return( (int)bs );
}

InitializeConstants( argc, argv )
    int *argc;
    char **argv;
{
    twopi = (2.0 * pi);
    drawscale = 1;
    speedscale = 1;
    timint = 40000;
    maxshots = 3;
    shotspeed = 16;
    maxhits = 4;
    maxspeed = 30;
    carryballmax = 3;
    planet = False;
    gravity = 0;
    bounce = True;
    numasts = 1;
    playball = True;
    player_stat[0] = player_stat[1] = 
	player_stat[2] = player_stat[3] = NO_PLAYER;
}

int
Aplayer( i, argc, argv )
    int i;
    int argc;
    char **argv;
{
    int isdigits, j;

    if (strncmp( argv[i + 1], "robot", 5 ) == 0)
	return( False );
    else if (strncmp( argv[i + 1], "watch", 5 ) == 0)
	return( False );
    else {
	isdigits = True;
	for (j = 0; isdigits && (j < strlen( argv[i + 1] )); j++)
	    isdigits = (argv[i + 1][j] >= '0') && (argv[i+1][j] <= '9');
	return( !isdigits );
    }
}

int
AnyPlayers(argc, argv)
    int argc;
    char **argv;
{
    int isplayer, i;

    isplayer = False;

    for (i = 0; (isplayer == False) && (i <= argc); i++)
	isplayer = Aplayer( i, argc, argv );

    return( isplayer );
}

InitDisplay( argc, argv, dispname2, dispnum )
    int *argc;
    char **argv;
    char *dispname2;
    int dispnum;
{
    int screen;
    unsigned long back, fore;
    XSizeHints myhint;
    char dispname[100];
    int	i, isdigits;
    

    if (strncmp( dispname2, "watch", 5 ) == 0) {
	strcpy( dispname, "robot" );
	strcat( dispname, &dispname2[5] );
    } else
	strcpy( dispname, dispname2 );
    isdigits = True;
    for (i = 0; isdigits && (i < strlen( dispname )); i++)
	isdigits = (dispname[i] >= '0') && (dispname[i] <= '9');

    if ((strlen( dispname) != 0) && isdigits) {
	strcpy( dispname, "robot" );
	strcat( dispname, dispname2 );
    }

    if (strncmp( dispname, "robot", 5 ) == 0) {
	display[dispnum] == NULL;
	if (dispname[5] == NULL)
	    robdelay[dispnum] = 9;
	else
	    robdelay[dispnum] = dispname[5] - '1';

	switch( robdelay[dispnum] ) {
	case 0:
	    /* This is OK as 0 */
	    break;
	case 1:
	    robdelay[dispnum] = -7; /* Skips every seventh turn */
	    break;
	case 2:
	    robdelay[dispnum] = -6; /* Skips every sixth turn */
	    break;
	case 3:
	    robdelay[dispnum] = -5; /* Skips every fifth turn */
	    break;
	case 4:
	    robdelay[dispnum] = -4; /* Skips every fourth turn */
	    break;
	case 5:
	    robdelay[dispnum] = -3; /* Skips every third turn */
	    break;
	case 6:
	    robdelay[dispnum] = 2; /* Goes every other turn */
	    break;
	case 7:
	    robdelay[dispnum] = 4; /* Goes every third turn */
	    break;
	case 8:
	    robdelay[dispnum] = 6; /* Goes every fourth turn */
	    break;
	case 9:
	    robdelay[dispnum] = 8;
	    break;
	default:
	    printf( "Robot skill level must range from 1 (highest) to 9 (lowest)\n" );
	    exit( 0 );
	    break;
	}
	player_stat[dispnum] = ROBOT_PLAYER;
	if ((strncmp( dispname2, "watch", 5 ) != 0) && (! AnyPlayers(*argc, argv)))
	    return;
	else {
	    if (dispnum == 0)
		strcpy( dispname, "unix:0.0" );
	    else
		return;
	}
    } else {
	if (strchr( dispname, ':' ) == NULL)
	    strcat( dispname, ":0.0" );
    }
    display[dispnum] = XOpenDisplay( dispname );
    if (display[dispnum] == NULL) {
	printf( "Cannot open display %s\n", dispname );
	exit( 0 );
    }
    if (player_stat[dispnum] == NO_PLAYER)
	player_stat[dispnum] = HUMAN_PLAYER;
    screen = DefaultScreen( display[dispnum] );
    back = BlackPixel( display[dispnum], screen );
    fore = WhitePixel( display[dispnum], screen );

    myhint.x = WINDOW_X;
    myhint.y = WINDOW_Y;
    myhint.width = WINDOW_WIDTH;
    myhint.height = WINDOW_HEIGHT + 80;
    myhint.flags = PSize | PPosition;

    window[dispnum] = 
	XCreateSimpleWindow( display[dispnum],
			    DefaultRootWindow( display[dispnum] ),
			    myhint.x, myhint.y,
			    myhint.width, myhint.height,
			    5, fore, back );
    XSetStandardProperties( display[dispnum], window[dispnum],
			   "Integration", "Integration", None,
			   argv, argc, &myhint );
    XSelectInput( display[dispnum], window[dispnum],
		 KeyPressMask | ExposureMask );
    XMapRaised( display[dispnum], window[dispnum] );
}

InitializeGCs( pcount )
    int pcount;
{
    int curp;
    int myscreen;
    Colormap cmap;
    XColor exact, tcolor;
    unsigned long i;
    int pind;
    Font myfont;

    for (curp = 0; curp < pcount; curp++) {
	if (display[curp] == NULL)
	    continue;
	myscreen = DefaultScreen( display[curp] );
	cmap = XDefaultColormap( display[curp], myscreen );
	i = XAllocNamedColor( display[curp], cmap, "yellow",
			     &tcolor, &exact );
	tpv[curp][0] = tcolor.pixel;
	i = XAllocNamedColor( display[curp], cmap, "cyan",
			     &tcolor, &exact );
	tpv[curp][1] = tcolor.pixel;
	i = XAllocNamedColor( display[curp], cmap, "green",
			     &tcolor, &exact );
	tpv[curp][2] = tcolor.pixel;
	i = XAllocNamedColor( display[curp], cmap, "pink",
			     &tcolor, &exact );
	tpv[curp][3] = tcolor.pixel;

	myfont = XLoadFont( display[curp], "6x10" );

	for (pind = 0; pind < pcount; pind++) {
	    pgc[curp][pind] = XCreateGC( display[curp],
					window[curp], 0, 0 );
	    XSetFont( display[curp], pgc[curp][pind], myfont );
	    XSetBackground( display[curp], pgc[curp][pind],
			   BlackPixel( display[curp], myscreen ));
	    XSetForeground( display[curp], pgc[curp][pind],
			   tpv[curp][pind] );
	    XSetFunction( display[curp], pgc[curp][pind],
			 GXxor );
	}
	ngc[curp] = XCreateGC( display[curp], window[curp], 0, 0 );
	XSetFont( display[curp], ngc[curp], myfont );
	XSetBackground( display[curp], ngc[curp],
		       BlackPixel( display[curp], myscreen ));
	XSetForeground( display[curp], ngc[curp],
		       WhitePixel( display[curp], myscreen ));
	XSetFunction( display[curp], ngc[curp], GXxor );
    }
}

InitializeDisplays( argc, argv )
    int *argc;
    char **argv;
{
    if (*argc <= 1)
	InitDisplay( argc, argv, "unix:0.0", 0 );
    else
	InitDisplay( argc, argv, argv[1], 0 );

    if (*argc <= 2)
	InitDisplay( argc, argv, "robot", 1 );
    else
	InitDisplay( argc, argv, argv[2], 1 );

    if (*argc > 3)
	InitDisplay( argc, argv, argv[3], 2 );
    if (*argc > 4)
	InitDisplay( argc, argv, argv[4], 3 );

    player_count = *argc - 1;

    if (player_count < 2)
	player_count = 2;

    InitializeGCs( player_count );
}

InitializePlayers( argc, argv )
    int *argc;
    char **argv;
{
    int i;

    for (i = 0; i < player_count; i++) {
	score[i] = 0;
	relatives[i] = (player_stat[i] != ROBOT_PLAYER);
    }
    pname = (char **)malloc( 4 * sizeof( char *));
    if (*argc <= 1)
	pname[0] = (char *)strdup( "Player" );
    else {
	if (player_stat[0] == ROBOT_PLAYER)
	    pname[0] = (char *)strdup( "robot" );
	else
	    pname[0] = (char *)strdup( argv[1] );
    }

    if (*argc <= 2)
	pname[1] = (char *)strdup( "robot" );
    else {
	if (player_stat[1] == ROBOT_PLAYER)
	    pname[1] = (char *)strdup( "robot" );
	else
	    pname[1] = (char *)strdup( argv[2] );
    }

    if (*argc > 3) {
	if (player_stat[2] == ROBOT_PLAYER)
	    pname[2] = (char *)strdup( "robot" );
	else
	    pname[2] = (char *)strdup( argv[3] );
    }
    if (*argc > 4) {
	if (player_stat[3] == ROBOT_PLAYER)
	    pname[3] = (char *)strdup( "robot" );
	else
	    pname[3] = (char *)strdup( argv[4] );
    }
}

DrawPlanets()
{
    int i;

    for (i = 0; i < player_count; i++) {
	if (display[i] != NULL) {
	    XFillArc( display[i], window[i], pgc[i][i],
		     (WINDOW_WIDTH / 2) - 15,
		     (WINDOW_HEIGHT / 2) - 15,
		     30, 30,
		     0, 64*360);
	}
    }
}

DrawPlanet( i )
    int i;
{
    if (display[i] != NULL) {
	XFillArc( display[i], window[i], pgc[i][i],
		 (WINDOW_WIDTH / 2) - 15,
		 (WINDOW_HEIGHT / 2) - 15,
		 30, 30,
		 0, 64*360);
    }
}

DrawGoal( i )
{
    int j;

    if (display[i] == NULL)
	return;

    for (j = 0; j < player_count; j++)
	DrawObject( display[i], window[i], pgc[i][j],
		   SHAPE_GOAL, goal[j].x, goal[j].y,
		   goal[j].rot );
}

DrawGoals()
{
    int i;
    for (i = 0; i < player_count; i++)
	DrawGoal( i );
}

PrepareGame()
{
    int saveasts, i;

    for (i = 0; i < (SHIP + 15); i++)
	obj[i].alive = False;
    obj[PLANET].rot = pi;
    obj[PLANET].x = (WINDOW_WIDTH / 2);
    obj[PLANET].y = (WINDOW_HEIGHT / 2);

    for (i = 0; i < player_count; i++) {
	shots[i] = 0;
	if (player_stat[i] == ROBOT_PLAYER)
	    ready[i] == True;
	else
	    ready[i] = False;
	obj[SHIP + i].shape = SHAPE_SHIP;
	obj[SHIP + i].pnum = i;
	obj[SHIP + i].alive = True;
	obj[SHIP + i].time = -1;
	obj[SHIP + i].xvel = 0;
	obj[SHIP + i].yvel = 0;
	obj[SHIP + i].rotvel = 0;
	obj[SHIP + i].hits = 0;
	goal[i].shape = SHAPE_GOAL;
	goal[i].pnum = i;
	goal[i].alive = True;
	goal[i].time = -1;
	goal[i].xvel = 0;
	goal[i].yvel = 0;
	goal[i].rotvel = 0;
	goal[i].hits = 0;

	switch (i) {
	case 0:
	    obj[SHIP + i].x = 50.0;
	    obj[SHIP + i].y = (WINDOW_HEIGHT / 2.0) - 50.0;
	    obj[SHIP + i].rot = 0.0;
	    goal[i].x = 30.0;
	    goal[i].y = (WINDOW_HEIGHT / 2.0);
	    goal[i].rot = 0.0;
	    break;
	case 1:
	    obj[SHIP + i].x = WINDOW_WIDTH - 50.0;
	    obj[SHIP + i].y = (WINDOW_HEIGHT / 2.0) + 50.0;
	    obj[SHIP + i].rot = pi;
	    goal[i].x = WINDOW_WIDTH - 30.0;
	    goal[i].y = WINDOW_HEIGHT / 2.0;
	    goal[i].rot = pi;
	    break;
	case 2:
	    obj[SHIP + i].x = (WINDOW_WIDTH / 2.0) - 50.0;
	    obj[SHIP + i].y = 50.0;
	    obj[SHIP + i].rot = pi / 2.0;
	    goal[i].x = WINDOW_WIDTH / 2.0;
	    goal[i].y = 30.0;
	    goal[i].rot = pi / 2.0;
	    break;
	case 3:
	    obj[SHIP + i].x = (WINDOW_WIDTH / 2.0) + 50.0;
	    obj[SHIP + i].y = WINDOW_HEIGHT - 50.0;
	    obj[SHIP + i].rot = (3.0 * pi) / 2.0;
	    goal[i].x = WINDOW_WIDTH / 2.0;
	    goal[i].y = WINDOW_HEIGHT - 30.0;
	    goal[i].rot = (3.0 * pi) / 2.0;
	    break;
	}
    }
    saveasts = numasts;
    numasts = 0;
    for (i = 0; i < saveasts; i++)
	AddAsteroid();
    if (playball)
	StartBall();
}

WaitForExposes()
{
    int i;
    XEvent event;
    for (i = 0; i < player_count; i++) {
	if (display[i] != NULL) {
	    do {
		XNextEvent( display[i], &event );
	    } while (event.type != Expose );
	}
    }
}

DrawObject( disp, window, gc, shape, x, y, rot )
    Display *disp;
    Drawable window;
    GC gc;
    int shape;
    double x, y, rot;
{
    int line;
    XPoint figure[20];

    if (disp == NULL)
	return;

    figure[0].x = (int) x;
    figure[0].y = (int) y;
    for (line=1; line < numpairs[shape]; line++) {
	figure[line].x  = figure[line-1].x + (int)(shapes[shape][line].length *
	    cos(shapes[shape][line].angle + rot) * drawscale);
	figure[line].y  = figure[line-1].y + (int)(shapes[shape][line].length *
	    sin(shapes[shape][line].angle + rot) * drawscale);
    }
    XDrawLines (disp, window, gc, figure, numpairs[shape], CoordModeOrigin);
}

DrawAll( pcol, shape, x, y, rot )
    int pcol;
    int shape;
    double x, y, rot;
{
    int line;
    int j;
    XPoint figure[20];

    figure[0].x = (int) x;
    figure[0].y = (int) y;
    for (line=1; line < numpairs[shape]; line++) {
	figure[line].x  = figure[line-1].x + (int)(shapes[shape][line].length *
	    cos(shapes[shape][line].angle + rot) * drawscale);
	figure[line].y  = figure[line-1].y + (int)(shapes[shape][line].length *
	    sin(shapes[shape][line].angle + rot) * drawscale);
    }
    for (j = 0; j < player_count; j++)
	if (display[j] != NULL) {
	    if (pcol == -1) {
		XDrawLines (display[j], window[j], ngc[j],
			    figure, numpairs[shape],
			    CoordModeOrigin);
	    } else {
		XDrawLines (display[j], window[j], pgc[j][pcol],
			    figure, numpairs[shape],
			    CoordModeOrigin);
	    }
	}
}

DrawHitSingleSpace( i )
    int i;
{
    char *outstring = { "Press SPACE to begin" };

    if (display[i] != NULL) {
	XDrawImageString( display[i], window[i], pgc[i][i], 
			 (WINDOW_WIDTH / 2) - (strlen(outstring)/2),
			 (WINDOW_HEIGHT / 4) * 3,
			 outstring, strlen( outstring ));
    }
}

DrawHitSpace()
{
    int i;

    for (i = 0; i < player_count; i++)
	DrawHitSingleSpace( i );
}

DrawWaitingOther( i )
    int i;
{
    char *outstring = { "Waiting for other players..." };

    if (display[i] != NULL) {
	XDrawImageString( display[i], window[i], pgc[i][i], 
			 (WINDOW_WIDTH / 2) - (strlen(outstring)/2),
			 (WINDOW_HEIGHT / 4) * 3,
			 outstring, strlen( outstring ));
    }
}

EraseHitSpace()
{
    int i;

    char *outstring = { "                            " };

    for (i = 0; i < player_count; i++) {
	if (display[i] != NULL) {
	    XDrawImageString( display[i], window[i], pgc[i][i], 
			     (WINDOW_WIDTH / 2) - (strlen(outstring)/2),
			     (WINDOW_HEIGHT / 4) * 3,
			     outstring, strlen( outstring ));
	}
    }
}

DrawScreen( i )
    int i;
{
    int j;
    char outstring[100];

    if (display[i] == NULL)
	return;

    XClearWindow( display[i], window[i] );
    for (j = 0; j < player_count; j++) {
	sprintf( outstring, "%4d  Player %d   %s", score[j],
		j, pname[j] );
	XDrawImageString( display[i], window[i],
			 pgc[i][j], 15,
			 WINDOW_HEIGHT + (15 * (j + 1)) + 5,
			 outstring, strlen( outstring ));
	if (ready[j])
	    XDrawImageString( display[i], window[i],
			     pgc[i][j], 200,
			     WINDOW_HEIGHT + (15*(j+1))+5,
			     "Ready", strlen( "Ready" ));
	if (!obj[SHIP + j].alive)
	    XDrawImageString( display[i], window[i],
			     pgc[i][j], 300,
			     WINDOW_HEIGHT + (15*(j+1))+5,
			     "Dead", strlen( "Dead" ));
	DrawObject( display[i], window[i], pgc[i][j],
		   obj[SHIP + j].shape,
		   obj[SHIP + j].x,
		   obj[SHIP + j].y,
		   obj[SHIP + j].rot );
    }
    XDrawLine( display[i], window[i], pgc[i][i],
	      0, WINDOW_HEIGHT+2, WINDOW_WIDTH, WINDOW_HEIGHT+2);
    if (planet)
	DrawPlanet( i );
    for (j = FIRST_AST; j < FIRST_AST + numasts; j++)
	DrawObject( display[i], window[i], ngc[i],
		   obj[j].shape,
		   obj[j].x,
		   obj[j].y,
		   obj[j].rot );

    if (playball) {
	DrawGoal( i );
	if (obj[BALL].pnum == -1)
	    DrawObject( display[i], window[i], ngc[i],
		       obj[BALL].shape,
		       obj[BALL].x,
		       obj[BALL].y,
		       obj[BALL].rot );
	else
	    DrawObject( display[i], window[i], pgc[i][obj[BALL].pnum],
		       obj[BALL].shape,
		       obj[BALL].x,
		       obj[BALL].y,
		       obj[BALL].rot );
    }
}

DrawClearScreens()
{
    int i;

    for (i = 0; i < player_count; i++) {
	DrawScreen( i );
    }
}

WaitForPlayers()
{
    int i;
    XEvent event;
    int nready = 0;
    int kp, ck, j;
    char text[20];
    KeySym mykey;

    for (i = 0; i < player_count; i++)
	if (player_stat[i] == ROBOT_PLAYER) {
	    ready[i] = True;
	    nready++;
	} else {
	    ready[i] = False;
	}

    while (nready < player_count) {
	for (i = 0; i < player_count; i++) {
	    if ((player_stat[i] == ROBOT_PLAYER) && (display[i] != NULL)) {
		while (XEventsQueued( display[i], QueuedAfterFlush ) != 0 ) {
		    XNextEvent( display[i], &event );
		    if (event.type == Expose)
			DrawScreen( i );
		}
	    } else if (display[i] != NULL) {
		while (XEventsQueued( display[i], QueuedAfterFlush ) != 0) {
		    XNextEvent( display[i], &event );
		    switch( event.type ) {
		    case Expose:
			DrawScreen( i );
			if (ready[i])
			    DrawWaitingOther( i );
			else
			    DrawHitSingleSpace( i );
			break;

		    case KeyPress:
			kp = XLookupString( &event, text, 10,&mykey,0);
			for (ck = 0; ck < kp; ck++) {
			    switch( text[ck] ) {
			    case '0':
			    case ' ':
				if (!ready[i]) {
				    nready++;
				    ready[i] = True;
				    for (j=0; j<player_count;j++)
					if (display[j] != NULL) {
					    XDrawImageString(display[j],
							     window[j],
							     pgc[j][i],
							     200,
						    WINDOW_HEIGHT+(15*(i+1))+5,
							     "Ready",
						     strlen( "Ready" ));
					}
				}
				DrawWaitingOther( i );
				break;
	
			    case 'q':
			    case 'Q':
				exit( 0 );
				break;
			    }
			}
			break;
		    }
		}
	    }
	}
    }
}

FireShot( i )
    int i;
{
    int j;
    double shiprot;
    double cosrot;
    double sinrot;

    if (shots[i] < 0)
	shots[i] = 0;
    if (shots[i] < maxshots) {
	shots[i]++;
	j = 0;
	while (obj[j].alive) j++;
	obj[j].alive = True;
	shiprot = obj[SHIP + i].rot;
	cosrot = cos(shiprot); sinrot = sin(shiprot);
	obj[j].shape = SHAPE_SHOT;
	obj[j].pnum = i;
	obj[j].hits = 0;
	obj[j].x = obj[SHIP + i].x; obj[j].y = obj[SHIP + i].y;
	if (relatives[i]) {
	    obj[j].xvel = obj[SHIP + i].xvel + (shotspeed * cosrot);
	    obj[j].yvel = obj[SHIP + i].yvel + (shotspeed * sinrot);
	} else {
	    obj[j].xvel = (shotspeed * cosrot);
	    obj[j].yvel = (shotspeed * sinrot);
	}
	obj[j].rot = shiprot;
	obj[j].time = WINDOW_WIDTH / (speedscale * 11);
    }
}

FireBall( i )
    int i;
{
    obj[BALL].hits = 1;
    obj[BALL].xvel = (8 * cos(obj[SHIP + i].rot));
    obj[BALL].yvel = (8 * sin(obj[SHIP + i].rot));
    obj[BALL].x += (2 * obj[BALL].xvel);
    obj[BALL].y += (2 * obj[BALL].yvel);
}

AddAsteroid()
{
    if (numasts >= 8)
	return;
    obj[FIRST_AST+numasts].alive = True;
    obj[FIRST_AST+numasts].shape = SHAPE_AST;
    obj[FIRST_AST+numasts].pnum = -1;
    obj[FIRST_AST+numasts].time = -1;
    obj[FIRST_AST+numasts].x = (double)(WINDOW_WIDTH / 2);
    obj[FIRST_AST+numasts].y = (double)(WINDOW_HEIGHT / 2);
    obj[FIRST_AST+numasts].xvel = (double)(rand() % 7) + 1;
    obj[FIRST_AST+numasts].yvel = (double)(rand() % 7) + 1;
    obj[FIRST_AST+numasts].rot = (double)(rand() % 6) / 3.0;
    obj[FIRST_AST+numasts].rotvel = (double)(rand() % 60) / 10.0;
    obj[FIRST_AST+numasts].hits = 0;
    numasts++;
}

DeleteAsteroid()
{
    int i;

    if (numasts <= 0)
	numasts = 0;
    else {
	numasts--;
	for (i = 0; i < player_count; i++)
	    if (display[i] != NULL) {
		DrawObject( display[i], window[i], ngc[i],
			   SHAPE_AST, obj[FIRST_AST + numasts].x, 
			   obj[FIRST_AST + numasts].y,
			   obj[FIRST_AST + numasts].rot );
	    }
	obj[FIRST_AST + numasts].alive = False;
    }
}

AddBall()
{
    if (playball)
	return;
    DrawGoals();
    StartBall();
    playball = True;
}

StartBall()
{
    obj[BALL].alive = True;
    obj[BALL].shape = SHAPE_BALL;
    obj[BALL].pnum = -1;
    obj[BALL].time = -1;
    obj[BALL].x = (double)(WINDOW_WIDTH / 2);
    obj[BALL].y = (double)(WINDOW_HEIGHT / 2);
    obj[BALL].xvel = (double)(7 - ((rand() % 14) + 1));
    obj[BALL].yvel = (double)(7 - ((rand() % 14) + 1));
    obj[BALL].rot = (double)(rand() % 6);
    obj[BALL].rotvel = (double)(rand() % 60) / 10.0;
    obj[BALL].hits = 0;
}

DeleteBall()
{
    int i;

    if (playball) {
	DrawGoals();
	for (i = 0; i < player_count; i++)
	    if (display[i] != NULL) {
		GC usegc;
		if (obj[BALL].pnum == -1)
		    usegc = ngc[i];
		else
		    usegc = pgc[i][obj[BALL].pnum];
		DrawObject( display[i], window[i], usegc,
			   SHAPE_BALL, obj[BALL].x, obj[BALL].y,
			   obj[BALL].rot );
	    }
	obj[BALL].alive = False;
	playball = False;
    }
}

double
CalcAngle( i, checkobj )
    int i;
    ObjType *checkobj;
{
    double retang;

    retang = ( atan2( (double)(checkobj->y - obj[i].y), 
		   (double)(checkobj->x - obj[i].x)));

/*    if (retang < 0)
	retang += twopi; */

    return( retang );
}

int
CalcXYDist( i, x, y )
    int i;
    int x, y;
{
    int xdist, ydist;

    xdist = obj[i].x - x;
    ydist = obj[i].y - y;

    return( graphic_isqrt( (long)((long)xdist * (long)xdist) + 
			  (long)((long)ydist * (long)ydist)));
}

int
CalcDist( i, j )
    int i, j;
{
    int xdist, ydist;

    xdist = obj[i].x - obj[j].x;
    ydist = obj[i].y - obj[j].y;
    return( graphic_isqrt( (long)((long)xdist * (long)xdist) + 
			  (long)((long)ydist * (long)ydist)));
}

int
WhereMeet( i, j, x, y )
    int i, j;
    int *x, *y;
{
    int meets;
    double m1, m2, b1, b2;
    double tx, ty;

    m1 = tan(obj[i].rot);
    b1 = -((m1*obj[i].x) - obj[i].y);

    if (obj[j].xvel == 0) {
	m2 = .0001;
	b2 = -obj[j].y;
    } else {
	m2 = ((double)obj[j].yvel / (double)obj[j].xvel);
	b2 = -((m1*obj[j].x) - obj[j].y);
    }

    if ((m1 - m2) < .0001)
	return( False );
    else {
	tx = (b2 - b1) / (m2 - m1);
	ty = (m1 * tx) + b1;

	if ((fabs( tx ) > (double)INT_MAX) ||
	    (fabs( ty ) > (double)INT_MAX))
	    return( False );
	else {
	    *x = floor( tx );
	    *y = floor( ty );

	    if (((*x < -5) || (*x > WINDOW_WIDTH + 5)) ||
		((*y < -5) || (*y > WINDOW_WIDTH + 5)))
		return( False );
	    else
		return( True );
	}
    }
}

int
GetRobotMove( i, text )
    int i;
    char *text;
{
    ObjType *checkobj;
    int j, k;
    int x, y, d;
    double t1, t2;
    double lowang, highang;

    text[0] = NULL;

    /* If the delay is negative, then it means the robot can go
       EXCEPT every ROBDELAYth turn.  If it is positive, it means
       the robot can only go on the ROBDELAYth turn.  Thus a 2 is
       equal to -2, -3 means only every third turn must he skip,
       3 means every third turn he gets to go, etc. */
    if (robdelay[i] != 0) {
	if (robdelay[i] < 0) {
	    if ((curdelay % -robdelay[i]) == 0)
		return( False );
	} else if ((curdelay % robdelay[i]) != 0)
	    return( False );
    }

    if ((shots[i] < maxshots) && !((playball && obj[BALL].pnum==i) || 
	  ((CalcDist( SHIP + i, BALL)<30) && (obj[BALL].pnum == -1)))) {
	for (j = 0; (j < player_count) && (text[0] != '5'); j++) {
	    if ((i != j) &&
		(!((obj[BALL].pnum != -1) && (obj[BALL].pnum != j)))) {
		if ((CalcDist( SHIP + i, SHIP + j ) < 30) || 
		    (((obj[SHIP + j].xvel == 0) && 
		      (obj[SHIP + j].yvel == 0)) && 
		     (fabs( CalcAngle( SHIP + i, &obj[SHIP + j] )
			  - obj[SHIP + i].rot) < 0.4 )))
		    strcat( text, "5" );
		else {
		    if (WhereMeet( SHIP + i, SHIP + j, &x, &y )) {
			d = CalcXYDist( SHIP + i, x, y );
			t1 = (double)d / (double)shotspeed;
			if (obj[SHIP + j].xvel != 0) {
			    t2 = (double)(x - obj[SHIP + j].x) / (double)obj[SHIP + j].xvel;
			    if (fabs( t1 - t2 ) <= 10.1) {
				strcat( text, "5" );
			    }
			}
		    }
		}
	    }
	}
    }

    if (text[0] != NULL)
	return( True );

    if (playball) {
	if ((obj[BALL].pnum != i) && (obj[BALL].hits != 0)) {
	    if (CalcXYDist( SHIP + i, (int)goal[obj[BALL].pnum].x,
			   (int)goal[obj[BALL].pnum].y ) > 50)
		checkobj = &goal[obj[BALL].pnum];
	    else
		checkobj = &obj[BALL];
	} else {
	    if ((obj[BALL].pnum != i) ||
		((obj[BALL].pnum == i) && (obj[BALL].hits != 0)))
		checkobj = &obj[BALL];
	    else
		checkobj = &goal[i];
	}
    } else {
	d = 32767;
	j = -1;
	for (k = 0; k < player_count; k++)
	    if ((i != k) && obj[SHIP + k].alive) {
		int td;
		td = CalcDist( SHIP + i, SHIP + k );
		if (td < d) {
		    d = td;
		    j = SHIP + k;
		}
	    }
	checkobj = &obj[j];
    }

    lowang = CalcAngle( SHIP + i, checkobj );
    highang = lowang + twopi;

    if (fabs(obj[SHIP + i].rot - lowang) > fabs(highang - obj[SHIP + i].rot))
	lowang = highang;


    if ((obj[SHIP + i].yvel > 4.0) || (obj[SHIP + i].xvel > 4.0)) {
	highang = atan2( (double)obj[SHIP+i].yvel, (double)obj[SHIP+i].xvel );
	if (highang < 0)
	    highang += twopi;
	if (fabs(lowang-highang) > 0.8 ) {
	    strcat( text, "2" );
	    return( True );
	}
    }

/*    printf( "xvel: %lf, yvel: %lf, rot: %lf, lowang: %lf, comp1: %lf > 0.8, comp2: %lf > 0.3\n", obj[SHIP + i].xvel,
	   obj[SHIP + i].yvel, obj[SHIP + i].rot, lowang,
	   fabs( lowang - highang ),
	   fabs( lowang - obj[SHIP + i].rot )); */

    if (fabs(lowang - obj[SHIP + i].rot) > 0.3) {
	if (lowang < obj[SHIP + i].rot)
	    strcat( text, "4" );
	else
	    strcat( text, "6" );
    } else {
	if ((robdelay[i] < 5) && 
	    ((playball && (obj[BALL].pnum == i && obj[BALL].hits == 0)) &&
	     (fabs(lowang - obj[SHIP + i].rot) < 0.05)))
	    strcat( text, "5" );
	else
	    strcat( text, "8" );
    }

    return( text[0] != NULL );
}

EvalKeypress( i, text, watcher, event, done, tmaxspeed )
    int i;
    char *text;
    int watcher;
    XEvent *event;
    int *done;
    int tmaxspeed;
{
   int kp, ck;
   KeySym mykey;
 
   if ((player_stat[i] == HUMAN_PLAYER) || watcher)
       kp = XLookupString( event, text, 10, &mykey, 0 );
   else
       kp = strlen( text );

   for (ck = 0; ck < kp; ck++) {
       switch( text[ck] ) {
       case 'q':
       case 'Q':
	   *done = True;
	   exit( 0 );
	   break;
       case '8':
       case 'l':
       case 'L':
	   obj[SHIP + i].xvel += cos(obj[SHIP + i].rot);
	   obj[SHIP + i].yvel += sin(obj[SHIP + i].rot);
	   if (obj[SHIP + i].xvel > tmaxspeed)
	       obj[SHIP + i].xvel = tmaxspeed;
	   if (obj[SHIP + i].yvel > tmaxspeed)
	       obj[SHIP + i].yvel = tmaxspeed;
	   if (obj[SHIP + i].xvel < -tmaxspeed)
	       obj[SHIP + i].xvel = -tmaxspeed;
	   if (obj[SHIP + i].yvel < -tmaxspeed)
	       obj[SHIP + i].yvel = -tmaxspeed;
	   break;

       case '5':
       case ' ':
	   if (obj[SHIP + i].alive) {
	       if (playball) {
		   if (i != obj[BALL].pnum || obj[BALL].hits > 0)
		       FireShot( i );
		   else
		       FireBall( i );
	       } else {
		   FireShot( i );
	       }
	   }
	   break;

       case '4':
       case ',':
	   obj[SHIP + i].rot -= 0.2;
	   if (obj[SHIP+i].rot < 0)
	       obj[SHIP+i].rot += twopi;
	   break;

       case '6':
       case '.':
	   obj[SHIP + i].rot += 0.2;
	   if (obj[SHIP+i].rot > twopi)
	       obj[SHIP+i].rot -= twopi;
	   break;

       case '7':
       case '<':
	   obj[SHIP + i].rot -= pi / 4;
	   if (obj[SHIP+i].rot < 0)
	       obj[SHIP+i].rot += twopi;
	   break;

       case '9':
       case '>':
	   obj[SHIP + i].rot += pi / 4;
	   if (obj[SHIP+i].rot > twopi)
	       obj[SHIP+i].rot -= twopi;
	   break;

       case '2':
       case '/':
       case '?':
	   if (obj[SHIP + i].rotvel == 0) {
	       obj[SHIP + i].xvel = 0;
	       obj[SHIP + i].yvel = 0;
	   }
	   break;

       case '-':
       case '_':
	   if (i == 0)
	       timint -= 5000;
	   break;

       case '+':
       case '=':
	   if (i == 0)
	       timint += 5000;
	   break;

       case 's':
       case 'S':
	   if (relatives[i])
	       relatives[i] = False;
	   else
	       relatives[i] = True;
	   break;

       case 'P':
       case 'p':
	   if (i == 0) {
	       DrawPlanets();
	       if (planet)
		   planet = False;
	       else
		   planet = True;
	   }
	   break;

       case 'G':
	   if (i == 0)
	       gravity = gravity + 100;
	   break;

       case 'g':
	   if (i == 0)
	       gravity = gravity - 100;
	   break;

       case 'A':
	   if (i == 0)
	       AddAsteroid();
	   break;

       case 'a':
	   if (i == 0)
	       DeleteAsteroid();
	   break;

       case 'B':
       case 'b':
	   if (i == 0) {
	       if (bounce)
		   bounce = False;
	       else
		   bounce = True;
	   }
	   break;

       case 'Z':
	   if (i == 0)
	       AddBall();
	   break;

       case 'z':
	   if (i == 0)
	       DeleteBall();
	   break;

       case '[':
	   shotspeed--;
	   if (shotspeed == 0)
	       shotspeed--;
	   break;
       case ']':
	   shotspeed++;
	   if (shotspeed == 0)
	       shotspeed++;
	   break;
       case '}':
	   drawscale++;
	   break;
       case '{':
	   if (drawscale > 1)
	       drawscale--;
	   break;
       }
   }
}

int GetPlayerInput( i )
    int i;
{
    int tmaxspeed;
    int done;
    char text[20];
 
    XEvent event;

    done = False;
    if ((playball) && ((obj[BALL].pnum == i) && obj[BALL].hits == 0))
	tmaxspeed = carryballmax;
    else
	tmaxspeed = maxspeed;
    if (player_stat[i] == HUMAN_PLAYER)
	XNextEvent( display[i], &event );
    else {
	if (display[i] != NULL) {
	    while (XEventsQueued( display[i], QueuedAfterReading )) {
		XNextEvent( display[i], &event );
		if (event.type == Expose)
		    DrawScreen( i );
		else if (event.type == KeyPress)
		    EvalKeypress( i, text, True, &event, &done, tmaxspeed );
	    }
	}
	if (GetRobotMove( i, text ))
	    event.type = KeyPress;
	else
	    event.type = Expose;
    }

    switch( event.type ) {
    case Expose:
	if (player_stat[i] == HUMAN_PLAYER)
	    DrawScreen( i );
	break;

    case MappingNotify:
	XRefreshKeyboardMapping( &event );
	break;

    case KeyPress:
	EvalKeypress( i, text, False, &event, &done, tmaxspeed );
	break;
    }
    return( done );
}

Collide( i, j )
    int i, j;
{
    int diff, xd, yd;

    xd = obj[i].x - obj[j].x;
    yd = obj[i].y - obj[j].y;
    diff = (xd*xd) + (yd*yd);
    return( diff < (shapesize[obj[i].shape] + shapesize[obj[j].shape])
	   *(drawscale*drawscale)) ? 1 : 0;
}

CheckGoalCollide( i, j )
    int i, j;
{
    int diff, xd, yd;

    xd = obj[i].x - goal[j].x;
    yd = obj[i].y - goal[j].y;
    diff = (xd*xd) + (yd*yd);
    return( diff < (shapesize[obj[i].shape] + shapesize[goal[j].shape])
	   *(drawscale*drawscale)) ? 1 : 0;
}

GoalCollide( i )
{
    if (obj[i].pnum == -1)
	return ( 0 );
    else
	return( CheckGoalCollide( i, obj[i].pnum ));
}

PrintDead( i )
    int i;
{
    int j;

    for (j = 0; j < player_count; j++) {
	if (display[j] != NULL) {
	    XDrawImageString( display[j], window[j],
			     pgc[j][i], 300,
			     WINDOW_HEIGHT + (15*(i+1))+5,
			     "Dead", strlen( "Dead" ));
	}
    }
		
}

StoreObjects()
{
    int i;
    for (i = 0; i < (FIRST_AST + numasts + 2); i++) {
	if (obj[i].alive)
	    oldobj[i] = obj[i];
	else
	    oldobj[i].alive = False;
    }
}

HitShip( i, j )
    int i, j;
/* Ship i was hit by object j */
{
    double srot, trot;
    int spd;

    if (!playball) {
	if (obj[SHIP + i].shape == SHAPE_SHIP)
	    obj[SHIP + i].hits++;
	if ((j < SHIP) && (SHIP + i < FIRST_AST))
	    score[obj[j].pnum]++;
    }
    if (obj[SHIP + i].hits >= maxhits) {
	obj[SHIP + i].alive = False;
	alive--;
	PrintDead( i );
    } else {
	obj[SHIP + i].rotvel = (2.0 * pi);
	srot = obj[SHIP + i].rot;
	obj[SHIP + i].rot = obj[j].rot;
	obj[SHIP + i].xvel = 5 * cos( obj[j].rot + 
				     ((double)(50 -(rand() % 100))/30));
	obj[SHIP + i].yvel = 5 * sin( obj[j].rot +
				     ((double)(50 -(rand() % 100))/30));
	if (j >= FIRST_AST) {
	    trot = (obj[SHIP + i].rot - srot);
	    if (trot < (pi/3.0) && (trot > -(pi/3.0)))
		srot = obj[SHIP + i].rot + (pi/2.0);
	    if (srot > twopi)
		srot -= twopi;
	    obj[j].rot = srot;
	    obj[j].xvel = 5 * cos( srot );
	    obj[j].yvel = 5 * sin( srot );
	    obj[j].x += 5*obj[j].xvel;
	    obj[j].y += 5*obj[j].yvel;
	}
	if (playball) {
	    if ((obj[BALL].pnum == i) && (obj[BALL].hits == 0)){
		obj[BALL].pnum = -1;
		obj[BALL].rot = obj[j].rot;
		obj[BALL].xvel = -obj[SHIP + i].xvel;
		obj[BALL].yvel = -obj[SHIP + i].yvel;
/*		spd = ((rand() % 5) + 2);
		obj[BALL].xvel = (double)spd * cos( obj[j].rot - (pi/3));
		obj[BALL].yvel = (double)spd * sin( obj[j].rot - (pi/3));
*/
	    }
	}
    }
}

isqrt( square, root)
    long square, *root;
{
    register int a,b,c,as,bs;

    a = 1;
    b = 1;
    while (a<=square) {
	a = a <<2;
	b = b <<1;
    }
    as = 0;
    bs = 0;
    while ( (b>1) && (square>0) ) {
	a = a >> 2;
	b = b >> 1;
	c = square - (as | a);
	if (c >=0 ) {
	    square = c;
	    as |= (a<<1);
	    bs |= b;
	}
	as = as >> 1;
    }
    *root = (long)bs;
}

/* In order for the asteroid not to leave debris, we need to move the
   asteroid before the ship.  However, things are set up in a certain
   way, so we need to actually translate it at this point */
int
TranslateTI( ti )
    int ti;
{
    if (ti < SHIP)
	return( ti );
    else if ((ti - SHIP) < (numasts+2))
	return( ti + (FIRST_AST - SHIP));
    else
	return( ti - (numasts + 2));
}

MoveObjects()
{
    int ti, i, j;
    double temp;
    long xdist, ydist, factor, dist;
    double pull;
    int tmaxspeed;

    tmaxspeed = carryballmax;

    for (ti = 0; ti < FIRST_AST + numasts + 2; ti++) {
	i = TranslateTI( ti );
	if (obj[i].alive) {
	    if (gravity) {
		xdist = obj[i].x;
		xdist -= (long)(WINDOW_WIDTH / 2);
		ydist = obj[i].y;
		ydist -= (long)(WINDOW_HEIGHT / 2);
		factor = (xdist * xdist) + (ydist * ydist);
		isqrt( factor, &dist );
		dist = dist / 5;;
		if (dist != 0) {
		    if (dist < 1)
			dist = 1;
		    pull = (double)(gravity) / (double)(dist*dist);
		    if (pull > 0.8)
			pull = 0.8;
		    obj[i].xvel -= (double)(pull * (double)((double)xdist/
							    (double)dist));
		    obj[i].yvel -= (double)(pull * (double)((double)ydist/
							    (double)dist));
		    if (obj[i].xvel > maxspeed)
			obj[i].xvel = maxspeed;
		    if (obj[i].yvel > maxspeed)
			obj[i].yvel = maxspeed;
		    if (obj[i].xvel < -maxspeed)
			obj[i].xvel = -maxspeed;
		    if (obj[i].yvel < -maxspeed)
			obj[i].yvel = -maxspeed;
		}
	    }
	    temp = obj[i].x;
	    temp += obj[i].xvel * speedscale;
	    if (bounce) {
		if ((temp < 0) || (temp > WINDOW_WIDTH)) {
		    if (obj[i].shape == SHAPE_SHOT)
			obj[i].time = 0;
		    else {
			obj[i].xvel = (-fabs(obj[i].xvel))*(temp / fabs(temp));
			temp = obj[i].x +
			    (obj[i].xvel *
			     speedscale);
			if (i == BALL) {
			    obj[i].hits = 0;
			    obj[i].pnum = -1;
			}
		    }
		}
	    } else {
		while (temp < 0)
		    temp += (double)WINDOW_WIDTH;
		while (temp > WINDOW_WIDTH)
		    temp -= (double)WINDOW_WIDTH;
	    }
	    obj[i].x = temp;

	    temp = obj[i].y;
	    temp += obj[i].yvel * speedscale;
	    if (bounce) {
		if ((temp < 0) || (temp > WINDOW_HEIGHT)) {
		    if (obj[i].shape == SHAPE_SHOT)
			obj[i].time = 0;
		    else {
			obj[i].yvel = (-fabs(obj[i].yvel))*(temp/fabs(temp));
			temp = obj[i].y +
			    (obj[i].yvel * speedscale);
			if (i == BALL) {
			    obj[i].hits = 0;
			    obj[i].pnum = -1;
			}
		    }
		}
	    } else {
		while (temp < 0)
		    temp += (double)WINDOW_HEIGHT;
		while (temp > WINDOW_HEIGHT)
		    temp -= (double)WINDOW_HEIGHT;
	    }
	    obj[i].y = temp;
	    
	    if (i == BALL) {
		if ((obj[i].pnum != -1) && (obj[i].hits == 0)) {
		    obj[i].x = obj[SHIP + obj[i].pnum].x;
		    obj[i].y = obj[SHIP + obj[i].pnum].y;
		}
	    }
	    obj[i].rot += obj[i].rotvel;
	    if (obj[i].rot < 0)
		obj[i].rot += twopi;
	    else if (obj[i].rot > twopi)
		obj[i].rot -= twopi;
	    if ((obj[i].rotvel != 0) && (obj[i].shape != SHAPE_AST)){
		obj[i].rotvel -= .2;
		if (obj[i].rotvel < 0)
		    obj[i].rotvel = 0;
	    }
	    /* Check for collisions */
	    if ((i < SHIP) || (i >= FIRST_AST)) {
		for (j = 0; (j < player_count) && obj[i].alive; j++) {
		    if ((j != obj[i].pnum) &&
			((obj[SHIP + j].alive) && Collide( i, (SHIP + j) ))) {
			HitShip( j, i );
			if (obj[i].shape == SHAPE_SHOT) {
			    obj[i].alive = False;
			    shots[obj[i].pnum]--;
			}
		    }
		    if ( playball && ((((i < SHIP) && (obj[i].alive)) &&
				       ((obj[BALL].pnum == -1) ||
					(obj[BALL].hits != 0))) &&
				      (Collide( i, BALL)))) {
			shots[obj[i].pnum]--;
			obj[i].alive = False;
			obj[BALL].hits = 0;
			obj[BALL].pnum = -1;
			obj[BALL].xvel = obj[BALL].xvel + (obj[i].xvel / 4);
			obj[BALL].yvel = obj[BALL].yvel + (obj[i].yvel / 4);
		    }
		}
	    } else if (planet) {
		if (Collide( i, PLANET ))
		    HitShip( i - SHIP, PLANET );
	    }

	    if (i == BALL) {
		if ((obj[i].pnum) == -1 || (obj[i].hits != 0)) {
		    for (j = 0; j < player_count; j++) {
			if (obj[SHIP + j].alive && 
			    (obj[SHIP + j].rotvel == 0.0)) {
			    if ((obj[i].pnum != j) && Collide( SHIP + j, i )) {
				obj[i].pnum = j;
				obj[i].hits = 0;
				if (obj[SHIP + j].xvel > tmaxspeed)
				    obj[SHIP + j].xvel = tmaxspeed;
				if (obj[SHIP + j].yvel > tmaxspeed)
				    obj[SHIP + j].yvel = tmaxspeed;
				if (obj[SHIP + j].xvel < -tmaxspeed)
				    obj[SHIP + j].xvel = -tmaxspeed;
				if (obj[SHIP + j].yvel < -tmaxspeed)
				    obj[SHIP + j].yvel = -tmaxspeed;
			    }
			}
		    }
		} else {
		    if (obj[SHIP + obj[i].pnum].rotvel != 0.0) {
			obj[i].pnum = -1;
			obj[i].hits = 0;
		    }
		}
		if ((obj[i].pnum != -1) && GoalCollide( i )) {
		    score[obj[i].pnum]++;
		    alive = 0;
		    obj[i].hits = 0;
		}
	    }
	    if (obj[i].time >= 0) {
		obj[i].time--;
		if (obj[i].time <= 0) {
		    if (obj[i].alive) {
			obj[i].alive = False;
			shots[obj[i].pnum]--;
		    }
		}
	    }

	    if (obj[i].alive)
		DrawAll( obj[i].pnum, obj[i].shape,
			obj[i].x, obj[i].y,
			obj[i].rot );
	    if (oldobj[i].alive)
		DrawAll( oldobj[i].pnum, oldobj[i].shape,
			oldobj[i].x, oldobj[i].y,
			oldobj[i].rot );
	}
    }
}

SetWait( i )
    int i;
{
    /* stime = stop time */
    gettimeofday( &stime, &zone );
    stime.tv_usec += i;
    stime.tv_sec += (stime.tv_usec / 1000000);
    stime.tv_usec %= 1000000;
}

FinishWait()
{
    do {
	gettimeofday( &octime, &zone );
    } while ((stime.tv_sec > octime.tv_sec) ||
	     ((stime.tv_sec == octime.tv_sec) &&
	      (stime.tv_usec > octime.tv_usec)));
}

FinishGame()
{
}

main (argc, argv)
    int argc;
    char **argv;
{
    int done;
    int i;

    if (argc == 1) {
	printf( "Usage: %s <player> [<player> [<player> [ <player]]]\n", 
	       argv[0] );
	printf("Where each <player> is either a display or a number from 1 to 9 indicating\n");
	printf("a skill level for a robot player, with 1 being the best and 9 the worst.\n" );
	printf( "Example: A two player game with one user on host 'chao' and another on 'twix'\n" );
	printf( "       %s chao twix\n", argv[0] );
	printf( "For the same game, but with a robot in it of skill 4:\n" );
	printf( "       %s chao twix 4\n", argv[0] );
	printf( "Or, just to play against a robot of level 7:\n" );
	printf( "       %s unix 7\n", argv[0] );
	printf( "\n\nFor a good demonstration, type:\n" );
	printf( "       %s 1 2 3 4\n", argv[0] );
	printf( "to see four robots fight.\n" );
	exit( 0 );
    }

    gettimeofday( &octime, &zone );

    srand( octime.tv_usec );

    InitializeConstants( &argc, argv );
    InitializeDisplays( &argc, argv );
    InitializePlayers( &argc, argv );

    WaitForExposes();
    do {
	PrepareGame();
	DrawClearScreens();
	DrawHitSpace();
	WaitForPlayers();
	EraseHitSpace();
	/* Play the game */
	alive = player_count;
	while (alive > 1) {
	    curdelay++;
	    if (curdelay == INT_MAX)
		curdelay = 0;
	    SetWait( timint );
	    done = False;
	    StoreObjects();
	    for (i = 0; i < player_count; i++) {
		if (player_stat[i] == HUMAN_PLAYER) {
		    XFlush( display[i] );
		    while (XEventsQueued( display[i], QueuedAfterReading )) {
			done = GetPlayerInput( i );
			if (done)
			    exit( 0 );
		    }
		} else {
		    if (display[i] != NULL)
			XFlush( display[i] );
		    done = GetPlayerInput( i );
		}
	    }
	    MoveObjects();
	    FinishWait();
	}
    } while (!done);

    FinishGame();
}
