/* 
 * Copyright (C) 1993 Alain Nissen & Raphael Quinet
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors :
 *    Raphael Quinet <quinet@montefiore.ulg.ac.be>
 *                   <quinet@server.montefiore.ulg.ac.be>
 *    Alain Nissen   <nissen@montefiore.ulg.ac.be>
 *                   <nissen@server.montefiore.ulg.ac.be>
 */

#include <string.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/AsciiText.h>
#include "xsession.h"

extern char    *program_name;
extern Widget   ascii_text;
extern Boolean  widget_ok;
extern Boolean  child_died;

extern void     grimreaper();

/* The four following functions (strip_non_print, text_length, text_replace  */
/* and text_append) were taken and munged from "xconsole.c".   (C) 1990 MIT  */


/*****************************************************************************
 *                            strip_non_print                                *
 *---------------------------------------------------------------------------*
 * Strips any non-printing characters from the input buffer "old" and        *
 * places the stripped text in the same buffer.  If the text contains BEL    *
 * characters (007), this function returns True.                             *
 *****************************************************************************/

#ifdef __STDC__
static Boolean strip_non_print(char *old)
#else
static Boolean strip_non_print(old)
     char *old;
#endif
{
  char    *new;
  Boolean  ring_bell;

  ring_bell = False;
  new = old;
  while(*old != '\0')
    {
      if (isalnum(*old) || ispunct(*old) || isspace(*old) && *old != '\r')
        {
          if (new != old)
            *new = *old;
          new++;
        }
      else
	if (*old == '\7')
	  ring_bell = True;
      old++;
    }
  *new = '\0';
  return ring_bell;
}


/*****************************************************************************
 *                               text_length                                 *
 *---------------------------------------------------------------------------*
 * Returns the length of the text string in the AsciiText widget "w".        *
 *****************************************************************************/

#ifdef __STDC__
static long text_length(Widget w)
#else
static long text_length(w)
     Widget  w;
#endif
{
  return XawTextSourceScan(XawTextGetSource(w), (XawTextPosition) 0, XawstAll,
                           XawsdRight, 1, TRUE);
}


/*****************************************************************************
 *                               text_replace                                *
 *---------------------------------------------------------------------------*
 * Replaces a block of text in the AsciiText widget "w" by another block.    *
 *****************************************************************************/

#ifdef __STDC__
static void text_replace(Widget w, int start, int end, XawTextBlock *block)
#else
static void text_replace(w, start, end, block)
     Widget	   w;
     int           start, end;
     XawTextBlock *block;
#endif
{
  Arg		  arg;
  Widget	  source;
  XawTextEditType edit_mode;

  source = XawTextGetSource(w);
  XtSetArg(arg, XtNeditType, &edit_mode);
  XtGetValues(source, &arg, ONE);
  XtSetArg(arg, XtNeditType, XawtextEdit);
  XtSetValues(source, &arg, ONE);
  XawTextReplace(w, start, end, block);
  XtSetArg(arg, XtNeditType, edit_mode);
  XtSetValues(source, &arg, ONE);
}


/*****************************************************************************
 *                               text_append                                 *
 *---------------------------------------------------------------------------*
 * Appends a text string "s" to the end of the text block in the AsciiText   *
 * widget "w".  If the insertion point was at the end of the text block, it  *
 * is moved to the new end.                                                  *
 *****************************************************************************/

#ifdef __STDC__
static void text_append(Widget w, char *s, int len)
#else
static void text_append(w, s, len)
     Widget  w;
     char   *s;
     int     len;
#endif
{
  long	       last, current;
  XawTextBlock block;

  current = XawTextGetInsertionPoint(w);
  last = text_length(w);
  block.ptr = s;
  block.firstPos = 0;
  block.length = len;
  block.format = FMT8BIT;
  text_replace(w, last, last, &block);
  if (current == last)
    XawTextSetInsertionPoint(w, last + block.length);
}


/*****************************************************************************
 *                               console_log                                 *
 *---------------------------------------------------------------------------*
 * Prints the message "s" in the "ascii_text" widget if possible, or on the  *
 * standard error output (stderr).                                           *
 *****************************************************************************/

#ifdef __STDC__
void console_log(char *s)
#else
void console_log(s)
     char *s;
#endif
{
  if (*s == '\0')
    return;
  if (widget_ok)
    {
      text_append(ascii_text, s, strlen(s));
      notify();
    }
  else
    fprintf(stderr, "%s: %s", program_name, s);
}


/*****************************************************************************
 *                                print_motd                                 *
 *---------------------------------------------------------------------------*
 * Prints the contents of the "/etc/motd" file, if available.                *
 *****************************************************************************/

void print_motd()
{
  FILE *motd;
  char  buffer[1025];
  int   l;

  motd = fopen("/etc/motd", "r");
  if (motd != NULL)
    {
      while ((l = fread(buffer, sizeof(char), 1024, motd)) > 0)
        {
          buffer[l] = '\0';
          strip_non_print(buffer);
          console_log(buffer);
        }
      console_log("\n");
      fclose(motd);
    }
}


/*****************************************************************************
 *                            input_ready_callback                           *
 *---------------------------------------------------------------------------*
 * Prints the text coming from "input"; also calls the "grimreaper" function *
 * function if a child process has died (hack, hack...).                     *
 *****************************************************************************/

#ifdef __STDC__
void input_ready_callback(XtPointer client_data, int *input, XtInputId *input_id)
#else
void input_ready_callback(client_data, input, input_id)
     XtPointer  client_data;
     int       *input;
     XtInputId *input_id;
#endif
{
  char buffer[1025];
  int  n;

  while (child_died)  /* Don't change this ! */
    grimreaper();
  n = read(*input, buffer, 1024);
  if (n > 0)
    {
      buffer[n] = '\0';
      strip_non_print(buffer);
      console_log(buffer);
    }
}


/*****************************************************************************
 *                                 clear_proc                                *
 *---------------------------------------------------------------------------*
 * Clears the "ascii_text" widget.                                           *
 *****************************************************************************/

#ifdef __STDC__
void clear_proc(Widget w, XEvent *event, String params, Cardinal *num_params)
#else
void clear_proc(w, event, params, num_params)
     Widget    w;
     XEvent   *event;
     String    params[];
     Cardinal *num_params;
#endif
{
  long	       last;
  XawTextBlock block;

  last = text_length(ascii_text);
  block.ptr = "";
  block.firstPos = 0;
  block.length = 0;
  block.format = FMT8BIT;
  text_replace(ascii_text, 0, last, &block);
}
