/* xrt.c - Display a text box in the root window. (C) Richard K. Lloyd 1993.
  FindRootWindow() routine extracted from xroach.
  Colour code contributed by Rik Turnbull. */

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>

#include "patchlevel.h"

#ifndef SYSV
/* SunOS include files DON'T define these next 2 lines. Ho hum. */
extern char *optarg;
extern int optind;
#endif

typedef unsigned long Pixel;

Window rootWin;
Display *display;
Colormap colourmap;
int screen,width,height,fheight,fwidth;
GC gc;
XGCValues xgcv;
Font myfont;
XFontStruct *fstruct;
int numlines=0;
int longest=0;
int rw,tw,rx,ry;
int showdate=0,showfile=0,showtitle=0,reverse=0,centre=0,ascii=0,horiz=0;
#ifdef TIME_ZONE
int set_timezone=0;
char zoneset[255];
#endif

#define MAXLINES 255    /* Maximum number of lines displayed (rest ignored) */
#define MAXCHARS 255    /* Maximum number of chars per line */
#define MAXFILES 16     /* Maximum number of READABLE files if -h supplied */
#define PERCENTAGE 50   /* %age down screen where centre of box is placed */
#define DEFAULT_FONT "9x15" /* Default font (preferably non-proportional) */
#define DEFAULT_BG "beige"  /* Default background colour */
#define DEFAULT_FG "navyblue"  /* Default foreground (text/border) colour */
#define LINEWIDTH 3     /* Default line width around box */
#define LINEGAP 1       /* Extra pixel gap between lines */

int percent=PERCENTAGE,linewidth=LINEWIDTH,linegap=LINEGAP,maxlines,
    boxwidth=0,numfiles=0;
int filepos[MAXFILES];
char title[MAXCHARS+1];

char *msg[MAXLINES];
char fontname[255];
char bgcol[255],fgcol[255];

usage()
/* Tell the user what to type next time... */
{
   fprintf(stderr,"xrt V%s - X Root Text Displayer (C) Richard K. Lloyd 1993\n\n",XRT_VERSION);
   fprintf(stderr,"Syntax: xrt [-?] [-a] [-b colour] [-c] [-d] [-f font] [-g gap] [-h]\n");
   fprintf(stderr,"        [-l width] [-n] [-p percent] [-r] [-s colour] [-t title] [-w width]\n");
   fprintf(stderr,"        [filename [filename...]]\n\n");
   fprintf(stderr,"-? displays this syntax message.\n");
   fprintf(stderr,"-a only displays characters between ASCII 32 and 126.\n");
   fprintf(stderr,"-b sets the background box colour (default is %s.\n",bgcol);
   fprintf(stderr,"-c centres each line of text in the box (useful for proportional fonts)\n");
   fprintf(stderr,"-d adds the time and date (of file or current time) at the top of box.\n");
   fprintf(stderr,"-f chooses a different font (default is %s).\n",DEFAULT_FONT);
   fprintf(stderr,"-g specifies the number of pixels gap between each text line (default is %d).\n",LINEGAP);
   fprintf(stderr,"-h splits each readable file by a horizontal line.\n");
   fprintf(stderr,"-l sets the line width of the box border (default is %d).\n",LINEWIDTH);
   fprintf(stderr,"-n adds the filename at the top of box (<stdin> if standard input).\n");
   fprintf(stderr,"-p specifies the percentage down from the top of the screen where the vertical\n");
   fprintf(stderr,"   centre of the box is located (default is %d%%).\n",PERCENTAGE);
   fprintf(stderr,"-r switches to reverse video.\n");
   fprintf(stderr,"-s sets the foreground text/border colour (default is %s).\n",fgcol);
   fprintf(stderr,"-t adds the specified title to the top of the box. Overrides the -n option.\n\n");
   fprintf(stderr,"-w specifies the width of the box (by default, it is dynamically calculated).\n");
   fprintf(stderr,"If a filename is omitted, then standard input is used instead.\n");
#ifdef TIME_ZONE
   fprintf(stderr,"Fallback time zone if TZ not set: %s\n",TIME_ZONE);
#endif
   exit(1);
}

/*
   Find the root or virtual root window. Stolen from xroach.
*/
Window
FindRootWindow()
{
    Window rootWindow;
    Window realRoot;
    Atom swmVroot;
    Window rootReturn, parentReturn, *children;
    unsigned int numChildren;
    int cx;
    Atom actualType;
    Atom actualFormat;
    unsigned long nItems;
    unsigned long bytesAfter;
    Window *newRoot;
    
    /*
       Get real root window.
    */
    realRoot = rootWindow = RootWindow(display, screen);
    
    /*
       Get atom for virtual root property.  If the atom doesn't
       exist, we can assume the corresponding property does not
       exist. 
    */
    swmVroot = XInternAtom(display, "__SWM_VROOT", True);
    
    if (swmVroot == None)
	return rootWindow;
    
    /*
       Run children of root, looking for a virtual root.
    */
    XQueryTree(display, rootWindow, &rootReturn, &parentReturn, 
		    &children, &numChildren);
    for (cx=0; cx<numChildren; cx++) {
	newRoot = NULL;
	nItems = 0;
	if (XGetWindowProperty(display, children[cx], swmVroot, 0L, 1L, False,
	    XA_WINDOW, &actualType, (int *)&actualFormat, &nItems,
	    &bytesAfter, (unsigned char **)&newRoot) == Success && actualFormat != None) {
		if (nItems >= 1) {
		    rootWindow = *newRoot;
		}
		if (newRoot) {
#ifdef XFREE_CHAR
		    XFree((char *)newRoot);
#else
		    XFree(newRoot);
#endif
		}
	}
	if (rootWindow != realRoot) break;
    }
#ifdef XFREE_CHAR
    XFree((char *)children);
#else
    XFree(children);
#endif
    
    return rootWindow;
}

int set_colour(isfg,col)
int isfg;
char *col;
{
   XColor real_col;
   if (!XParseColor(display,colourmap,col,&real_col)) return(0);
   if (!XAllocColor(display,colourmap,&real_col)) return(0);
   if (isfg) XSetForeground(display,gc,real_col.pixel);
      else XSetBackground(display,gc,real_col.pixel);
   return(1);
}

int set_bg(col)
char *col;
{
   return(set_colour(0,col));
}

int set_fg(col)
char *col;
{
   return(set_colour(1,col));
}

int get_string_width(str)
char *str;
{
   return(XTextWidth(fstruct,str,strlen(str)));
}

free_memory(first,last)
int first,last;
{
   int a;
   for (a=first;a<last;a++)
   free((void *)msg[a]);
}

int read_lines(fd)
FILE *fd;
{
   char thestring[1024];
   int size,pixels,best=0,start=numlines;

   while (fgets(thestring,MAXCHARS+1,fd)!=NULL && numlines<maxlines)
   {
      int a,b=0,c;
      int maxstr;
      char outbuf[1024];
      if (boxwidth) maxstr=boxwidth; else maxstr=width-linewidth*2-1;
      maxstr=maxstr-fwidth*2;
      size=strlen(thestring)-1;
      if (size>MAXCHARS) size=MAXCHARS;
      thestring[size]='\0';
      for (a=0;a<size;a++)
      {
         int thechar=(int)thestring[a];
         if (thechar==9)
         {
            c=b+7-(b % 8);
            for (;b<=c;) outbuf[b++]=' ';
         }
         else
         if (!ascii || (thechar>=32 && thechar<=126))
            outbuf[b++]=thestring[a];
      }
      if (centre)
      {
         /* Yes, the next lot of code isn't efficient, but it's easy
            to understand ! */
         for (a=0;a<b;a++)
         {
            if (outbuf[a]!=' ') break;
         }
         for (c=b-1;c>a;c--)
         {
            if (outbuf[c]!=' ') break;
         }
         if (a>0 || c<b-1)
            for (b=0;b<c-a+1;b++) outbuf[b]=outbuf[a+b];
      }
      if (b>MAXCHARS) size=MAXCHARS; else size=b;
      outbuf[size]='\0';
      while ((pixels=get_string_width(outbuf))>maxstr)
      { outbuf[--size]='\0'; }
      if (pixels>longest) longest=pixels;
      if (pixels>best) best=pixels;
      msg[numlines]=(char *)malloc(size+1);
      strcpy(msg[numlines++],outbuf);
   }
   if (!best)
   {
      /* Backtrack because file was blank */
      free_memory(start,numlines);
      numlines=start;
   }
   return(best);
}

init_x()
{
   display=XOpenDisplay(NULL);
   if (display==(Display *)NULL)
   {
      fprintf(stderr,"Could not open display\n");
      exit(1);
   }
   screen=DefaultScreen(display);
   colourmap=DefaultColormap(display,screen);
   width=DisplayWidth(display,screen);
   height=DisplayHeight(display,screen);
   rootWin=FindRootWindow();
   if (reverse)
   {
      char temp[255];
      strcpy(temp,bgcol); strcpy(bgcol,fgcol); strcpy(fgcol,temp);
   }
   gc=XCreateGC(display, rootWin, 0L, &xgcv);
   fstruct=XLoadQueryFont(display,fontname);
   if (fstruct==(XFontStruct *)NULL)
   {
      fprintf(stderr,"Warning: Could not load %s font - using %s instead\n",
              fontname,DEFAULT_FONT);
      fstruct=XLoadQueryFont(display,DEFAULT_FONT);
   }           
   myfont=fstruct->fid;
   XSetFont(display,gc,myfont);
   fheight=fstruct->max_bounds.ascent+fstruct->max_bounds.descent+linegap;
   fwidth=fstruct->max_bounds.width;
   maxlines=(int)((height-linewidth*2-1)/fheight)-1;
   if (title[0]) maxlines--;
   if (maxlines>MAXLINES) maxlines=MAXLINES;
}

draw_rect()
{
   int rh=(numlines+1)*fheight,tx,ty;
   if (boxwidth) rw=boxwidth; else rw=longest+fwidth*2;
   rx=(width-rw)/2;
   ry=height*percent/100-rh/2;
   if (title[0])
   {
       ty=ry-fheight;
       if (ty<linewidth) { ry=ry+linewidth-ty; ty=linewidth; }
   }
   if (ry<linewidth) ry=linewidth;
   if (ry+rh+linewidth>height)
   { 
      if (title[0]) ry=fheight+linewidth; else ry=linewidth;
      ty=ry-fheight;
   }
   if (rx<linewidth) rx=linewidth;
   if (linewidth>0)
   {
      set_fg(fgcol);
      XFillRectangle(display,rootWin,gc,rx-linewidth,ry-linewidth,rw+linewidth*2,rh+linewidth*2);
   }
   if (title[0])
   {
      tw=get_string_width(title)+fwidth;
      tx=rx+(rw-tw)/2;
      if (linewidth>0)
      {
         set_fg(fgcol);
         XFillRectangle(display,rootWin,gc,tx-linewidth,ty-linewidth,tw+linewidth*2,fheight+linewidth*2);
      }
      set_fg(bgcol);
      XFillRectangle(display,rootWin,gc,tx,ty,tw,fheight);
   }
   set_fg(bgcol);
   XFillRectangle(display, rootWin,gc,rx,ry,rw,rh);
}

draw_text()
{
   int a,add=fwidth,b=0,ypos=ry+1.25*fheight;
   if (horiz && numfiles>1)
           XSetLineAttributes(display,gc,1,LineSolid,CapNotLast,JoinMiter);
   set_fg(fgcol);
   set_bg(bgcol);
   if (title[0])
      XDrawString(display,rootWin,gc,rx+(rw+fwidth-tw)/2-1,ry,title,strlen(title));
   for (a=0;a<numlines;a++)
   {
      int doline=0;
      if (horiz && b<numfiles && numfiles>1)
      {
         if (filepos[b]==a)
         {
            XDrawLine(display,rootWin,gc,rx,ypos-fheight/2,rx+rw-1,ypos-fheight/2);
            doline=1; b++;
         }
      }
      if (!doline)
      {
         if (centre) add=(rw-get_string_width(msg[a]))/2;
         XDrawString(display,rootWin,gc,rx+add,ypos,msg[a],strlen(msg[a]));
      }
      ypos+=fheight;
   }
   XUnloadFont(display,myfont);
}

get_options(argc,argv)
int argc;
char **argv;
/* Parse the standard qualifiers */
{
   int param;
   title[0]='\0'; strcpy(fontname,DEFAULT_FONT);
   strcpy(bgcol,DEFAULT_BG);
   strcpy(fgcol,DEFAULT_FG);
   while ((param = getopt(argc,argv,"ab:cdf:g:hl:np:rs:t:w:?")) != EOF)
   {
      switch (param)
      {
         case '?': usage(); /* Bad option */
                   break;
         case 'a': ascii=1;
                   break;
         case 'b': strcpy(bgcol,optarg);
                   break;
         case 'c': centre=1;
                   break;
         case 'd': showdate=1;
                   break;
         case 'f': strcpy(fontname,optarg);
                   break;
         case 'g': linegap=atoi(optarg);
                   break;
         case 'h': horiz=1;
                   break;
         case 'l': linewidth=atoi(optarg);
		   break;
         case 'n': showfile=1;
                   break;
         case 'p': percent=atoi(optarg);
                   break;
         case 'r': reverse=1;
                   break;
         case 's': strcpy(fgcol,optarg);
                   break;
         case 't': strcpy(title,optarg);
                   showtitle=1;
                   break;
         case 'w': boxwidth=atoi(optarg);
                   break;
      }
   }
}

add_date(newtime)
long newtime;
{
   char newtitle[64],curday[8],curmon[8],curdate[8];
   int hrs,mins,secs,year;
#ifdef TIME_ZONE
   if (!set_timezone)
   {
      if (getenv("TZ")==(char *)NULL)
      {
         sprintf(zoneset,"TZ=%s",TIME_ZONE);
         putenv(zoneset);
      }
      set_timezone=1;
   }
#endif
   sscanf(ctime(&newtime),"%s %s %s %d:%d:%d %d",
          curday,curmon,curdate,&hrs,&mins,&secs,&year);
   sprintf(newtitle," (%2d:%02d %3s %2s %3s %d)",
           hrs,mins,curday,curdate,curmon,year);
   strcat(title,newtitle);
}

tidy_up()
{
   int a;
   XCloseDisplay(display);
   free_memory(0,numlines);
}

int main(argc,argv)
int argc;
char **argv;
{
   char name[255];
   FILE *filedes;
   long savetime=0;

   get_options(argc,argv);
   init_x();
   if (optind==argc)
   {
      if (showfile && !showtitle) strcpy(title,"<stdin>");
      savetime=time((time_t *)0);
      (void)read_lines(stdin);
   }
   else
   {
      for (;optind<argc && (numfiles<MAXFILES || !horiz);optind++)
      {
         strcpy(name,argv[optind]);
         if ((filedes=fopen(name,"r"))==NULL)
            fprintf(stderr,"Warning: File %s skipped - could not open\n",name);
         else
         {
           struct stat filestats;
           stat(name,&filestats);
           if (showfile && !showtitle) strcpy(title,name);
           if (filestats.st_mtime>savetime) savetime=filestats.st_mtime;
           if (read_lines(filedes))
              if (horiz) filepos[numfiles++]=numlines++;
           fclose(filedes);
         }
      }
      if (horiz && numfiles) numlines--;
   }
   if (longest)
   {
      if (showdate) add_date(savetime);
      draw_rect();
      draw_text();
   }
   tidy_up();
   return(0);
}
