/**
*** XPG - Graphical User Interface for Postgres
*** Copyright (C) 1993  Ranen Goren (ranen@cs.huji.ac.il).

*** This program is free software; you can redistribute it and/or modify
*** it under the terms of the GNU General Public License as published by
*** the Free Software Foundation; either version 2 of the License, or
*** (at your option) any later version.

*** This program is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*** GNU General Public License for more details.

*** You should have received a copy of the GNU General Public License
*** along with this program; if not, write to the Free Software
*** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**/


/* This file is based on Krys Kochut's work, in the Alberi
   package. Krys' e-mail is kochut@cs.uga.edu . */


#include "xpg.h"
#include "lib.h"


#define String JUST_A_DUMB_STRING
#include "tmp/c.h"

#include "tmp/simplelists.h"
#include "tmp/libpq-fe.h"
#include "tmp/fastpath.h"

#include "fmgr.h"
#include "utils/exception.h"
#undef String


extern char *PQhost;
extern char *PQport;
extern char *PQtty;
extern char *PQoption;
extern char PQdatabase[];
extern int PQportset;
extern int PQxactid;

extern toplevel;


char *execAndWarn();
char *xpgPQexec();
extern char *skipSpaces();
void dummy();



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
char *execAndWarn(parent, query, msg)
  Widget parent;
  char *query;   /* query cmd to send to PQexec() */
  char *msg;     /* msg to display if the cmd fails */
{
    char *pgResult, *origResult;
    char buf[BUFSIZ];
    int len, i;
    char *ptr, *errorHdr, *errorMsg;
    
    origResult = xpgPQexec(query);
    /* remove the nagging '\n's from the tail */
    len = strlen(origResult);
    for (;  origResult[len-1]=='\n';  len--)
	origResult[len-1] = '\0';
    pgResult = strdup(origResult);
    if (pgResult[0] == 'R')
    {
	/* the following looks for the 5th ':' symbol and
	   turns it into a '\0'. This divides the line into
	   two, in the place where the error 'header' and the
	   error message itself meet */
	for (ptr=pgResult, i=0;  i<5;  i++, ptr++)
	    if ((ptr = strchr(ptr, ':')) == NULL)
		break;
	errorHdr = pgResult+1;
	errorMsg = "";
	if (ptr != NULL)
	{
	    *(ptr-1) = '\0';
	    errorMsg = skipSpaces(ptr);
	}
	if (msg)
	    warn(parent, "%s\n\n%s\n%s", msg, errorMsg, errorHdr);
    }
    XtFree(origResult);
    return origResult;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
char *xpgPQexec(query)
  char *query;
{
    char id[2];
    char errormsg[error_msg_length];
    char command[command_length];
    static char PQcommand[command_length+1];
    void xpg_EstablishComm(), xpg_read_remark_notice();
    char *xpg_process_portal();
    static char buff[1024];
    static Boolean backendReset = False;
    char *msg;
    
    /* If the communication is not established, establish it. */
    if (!PQportset)
	xpg_EstablishComm();
    
    /* Send a query to backend. */
    pq_putnchar("Q", 1);
    pq_putint(PQxactid, 4);
    pq_putstr(query);
    pqdebug("The query sent to the backend: %s", query);
    pq_flush();
    
    /* forever (or at least until we get something besides a notice) */
    for (;;) {  
	
    	/* Process return values from the backend. 
	 * The code in this function is the implementation of
	 * the communication protocol.
	 */
    	id[0] = '?';
	
    	/* Get the identifier. */
    	pq_getnchar(id,0,1); 

    	xpg_read_remark_notice(id);
    	pqdebug("The Identifier is: %c", (char *)id[0]);
	
    	/* Read in the transaction id. */
    	PQxactid = pq_getint(4);
    	pqdebug("The Transaction Id is: %d", (char *)PQxactid);
	
    	switch (id[0]) {
	  case 'I':
	    return("I");
	    
	  case 'E':
	    backendReset = False;
	    /* An error, return 0. */
	    pq_getstr(errormsg, error_msg_length);
	    pqdebug("%s error encountered.", errormsg);
	    
	    buff[0] = 'R';	/* set up the answer string "R errmsg" */
	    strcpy(buff+1, errormsg);
	    return buff;	/* and return it back */
	    
	  case 'N': /* print notice and go back to processing return values */
	    /*
	     * If we get an EOF (i.e. backend quickdies) return an R to the fe
	     */
	    if (pq_getstr(errormsg, error_msg_length) == EOF) {
		buff[0] = 'R';
		strcpy(buff+1, errormsg);
		return buff;
	    }
	    pqdebug("%s notice encountered.", errormsg);
	    buff[0] = 'R';
	    strcpy(buff+1, errormsg);
	    return buff;
	    
	  case 'A': {
	      char relname[16];
	      extern int PQAsyncNotifyWaiting;
	      int pid;
	      PQAsyncNotifyWaiting = 0;
	      
	      backendReset = False;
	      /* Asynchronized portal. */
	      /* No tuple transfer at this stage. */
	      pqdebug("%s portal encountered.", "Asynchronized");
	      /* Processed the same way as synchronized portal. */
	      /*	    return
			    process_portal(1);*/
	      pq_getstr(relname,16);
	      pid =pq_getint(4);
	      PQappendNotify(relname,pid);
	  }
	    break;
	  case 'P':
	    backendReset = False;
	    /* Synchronized (normal) portal. */
	    return
		xpg_process_portal(0);
	    
	  case 'C':
	    backendReset = False;
	    /* Query executed successfully. */
	    pq_getstr (command, command_length);
	    pqdebug ("Query command: %s", command);
	    sprintf (PQcommand, "C%s", command);
	    return
		PQcommand;
	    
	  case 'B':
	    backendReset = False;
	    /* Copy command began successfully - it is sending stuff back... */
	    return "BCOPY";
	    
	  case 'D':
	    backendReset = False;
	    /* Copy command began successfully - it is waiting to receive... */
	    return "DCOPY";
	    
	  default:
	    /* The backend violates the protocol. */
	    if (id[0] == '?')
		msg = "No response from the backend!";
	    else
		msg = "Unexpected response from the backend!";
	    if (!backendReset)     /* try to reset a single time and */
	    {                      /* executed the query again       */
		backendReset = True;
		PQreset();
		xpg_EstablishComm();
		fprintf(stderr, "xpg: %s -- trying to reset once\n", msg);
		return xpgPQexec(query);     /* repeat last command */
	    }
	    else
	    {
		/* execution has never reached this stage, but anyway */
		sprintf(errormsg, "%s\nxpg already reset the backend, but there are still problems.\nYou can try to continue (expect trouble!) or exit xpg now.", msg);
		if (yesNo(toplevel, errormsg, "Exit", "Continue", 1) == 1)
		    exit(1);
		backendReset = False;    /* give the continue cmd a chance */
	    }
    	}
    }
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
void xpg_EstablishComm()
{
    if (!PQportset) { 
	read_initstr();
	
	if (pq_connect(PQdatabase, cuserid(NULL), PQoption, PQhost, PQtty,
		       (char *) NULL, (short)atoi(PQport) ) == -1 ) {
	    libpq_raise(&ProtocolError,
		form((int)"Failed to connect to backend (host=%s, port=%s)",
		     PQhost, PQport));
	}
	
	pq_flush();
	PQportset = 1;
    }
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
void xpg_read_remark_notice(id)
  char id[];
{
    char remarks[remark_length];
    char errormsg[error_msg_length];
    
    while (id[0] == 'R') {
	pq_getstr(remarks, remark_length);
	if (pq_getnchar(id, 0, 1) == EOF) 
	    return;
    }
    while(id[0] == 'N') {
        pq_getstr(errormsg,error_msg_length);
#ifdef XPG_DEBUG
	printf("XPG: %s", errormsg+4);
        printf("XPG: Postgres backend returned a notice\n");
#endif
	logText(True,  "** postgres notice **");
	logText(False, "\n%s", errormsg+4);
        if (pq_getnchar(id, 0, 1) == EOF)
	    return;
    }
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* ----------------
 *	process_portal
 *	
 * 	Process portal queries. 
 * 	Return values are the same as PQexec().
 * ----------------
 */

char *xpg_process_portal(rule_p)
  int rule_p;
{
    char pname[portal_name_length];
    char id[2];
    char errormsg[error_msg_length];
    char command[command_length];
    char PQcommand[portal_name_length+1];
    static char retbuf[portal_name_length + 1];
    char buff[1024];
    
    /* Read in the portal name. */
    pq_getstr(pname, portal_name_length);
    pqdebug("Portal name = %s", pname);
    
    /*
     * This for loop is necessary so that NOTICES out of portal processing
     * stuff are handled properly.
     */
    
    for (;;) {
        /* Read in the identifier following the portal name. */
        pq_getnchar(id, 0, 1);
        read_remark(id);
        pqdebug("Identifier is: %c", (char *)id[0]);
	
        switch (id[0]) {
	  case 'E':
	    /* An error, return 0. */
	    pq_getstr(errormsg, error_msg_length);
	    pqdebug("%s error encountered.", errormsg);
	    /* get past gunk at front of errmsg */
	    
	    buff[0] = 'R';	/* set up the answer string "R errmsg" */
	    strcpy(buff+1, errormsg+4);
	    return buff;
	    
	  case 'N': /* print notice and go back to processing return values */
	    /*
	     * If we get an EOF (i.e. backend quickdies) return an R to the fe
	     */
	    if (pq_getstr(errormsg, error_msg_length) == EOF) {
		buff[0] = 'R';
		strcpy(buff+1, errormsg);
		return buff;
	    }
	    pqdebug("%s notice encountered.", errormsg);
	    buff[0] = 'R';
	    strcpy(buff+1, errormsg+4);
	    return buff;
	    
	    break;
	    
	  case 'T':
	    /* Tuples are returned, dump data into a portal buffer. */
	    if (dump_data(pname, rule_p) == -1)
	    {
		return("R");
	    }
	    sprintf(PQcommand, "P%s", pname);
	    strcpy(retbuf, PQcommand);
	    return(retbuf);
	    
	    /* Pending data inquiry - return nothing */
	  case 'C':
	    /*
	     * Portal query command (e.g., retrieve, close),
	     * no tuple returned.
	     */
	    PQxactid = pq_getint (4);
	    pqdebug("Transaction Id is: %d", (char *)PQxactid);
	    pq_getstr(command, command_length);
	    pqdebug("Query command: %s", command);
	    
	    /* Process the portal commands. */
	    if (strcmp(command, "retrieve") == 0) {
	        /* Allocate a portal buffer, if portal table is full, error. */
	        pbuf_setup(pname);
	        return
		    "Cretrieve";
	    } 
	    else if (strcmp (command, "close") == 0) 
	        return
		    "Cclose";
	    else {
	        sprintf(retbuf, "C%s", command);
	        return
		    retbuf;
	    }
	    
	  default:
	{
	    char s[45];
	    
	    PQreset();
	    sprintf(s, "Unexpected identifier in process_portal: %c", id[0]);
	    libpq_raise(&ProtocolError, form ((int)s));
	}
	}
    }
}
