/* equasion.c
 *	Handles equasion dialog setup,
 *	Equasion parsing, and anything else mathematically related
 *
 *	complex number handling code modified from "fractv2.1",
 *	by Massimo Casal
 *	[which was taken from a program by jesjones@u.washington.edu]
 *	I would normally put the complex stuff in a separate file,
 *	but I want to keep them "inline" if possible
 */


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <Intrinsic.h>
#include <StringDefs.h>
#include <Xaw/Label.h>
#include <Xaw/AsciiText.h>
#include <Xaw/Dialog.h>

#include "equasion.h"
#include "globals.h"

Widget mandelvalues;

static char *mandelbrotEquasion = "Z^2 +C  ";

/* this stores the ascii representation for the current equasion */
static char equasionText[100];


struct runEquasion * MainEquasion = NULL;

int parseEquasion(char *);
void UpdateEquasion();
/* why the  !##!!! do I have to have this prototype here??!!!
 * Stupid Gcc won't work without it...
 */
void freeEquasion(struct runEquasion * killthis);

/* skipwhitespace():
 *	moves pointer to skip over blanks..
 *	Returns 1 if more non-blank string left,
 *	Returns 0 otherwise
 */
int skipwhitespace(char **blank){
	do {
		if(isspace(**blank)){
			(*blank)++;
		} else {
			break;
		}
	} while(**blank != '\0');

	if(**blank != '\0')
		return 1;
	else
		return 0;
}

/* SetPairdoubleValue(),SetPairintValue():
 *	Given a value, and the index to the WidgetPairs array,
 *	will set the value, at a particular level of precision
 */
void SetPairdoubleValue(int widget_index,double value){
	char stringbuf[100];
	sprintf(stringbuf,"%1.20G",value);
	XtVaSetValues(WidgetPairs[WIDGETVALUE][widget_index],
		XtNstring,stringbuf,
		NULL);
}

void SetPairintValue(int widget_index,int value){
	char stringbuf[100];
	sprintf(stringbuf,"%d",value);
	XtVaSetValues(WidgetPairs[WIDGETVALUE][widget_index],
		XtNstring,stringbuf,
		NULL);
}

/* GetPairdoubleValue():
 *	returns numerical value of string,
 *	in "double" type. There is no GetPairintValue().
 */
 double GetPairdoubleValue(int widget_index){
	String str;
	double ret_value;
	XtVaGetValues(WidgetPairs[WIDGETVALUE][widget_index],
		      XtNstring,&str,NULL);
	ret_value = atof(str);
	return (ret_value);
}

/* ResetCallback():
 *	reset displayed values to what they were the last time we
 *	rendered something
 */
void ResetCallback(Widget w,XtPointer call_data,XtPointer client_data){
	char stringbuf[100];
	int valuecount;

	/* MUST SET stringbuf FOR EVERY ENTRY!! */
	for(valuecount=0;valuecount<NUM_OF_WIDGETS;valuecount++){
		switch(valuecount){
			case EQUASION:
				/* we SHOULD read in encoding here,
				 * but we currently don't USE encoding
				 */
				strcpy(stringbuf,equasionText);
				XtVaSetValues(WidgetPairs[WIDGETVALUE][EQUASION],
					      XtNstring,stringbuf,
					      NULL);
				break;
			case DEPTH:
				SetPairintValue(valuecount,iterate);
				break;
			case MANDELCENTERX:
				SetPairdoubleValue(valuecount,viewcenterx);
				break;
			case MANDELCENTERY:
				SetPairdoubleValue(valuecount,viewcentery);
				break;
			case MANDELWIDTH:
				SetPairdoubleValue(valuecount,viewwidth);
				break;
			case MANDELHEIGHT:
				SetPairdoubleValue(valuecount,viewheight);
				break;
			case ESC_RADIUS:
				SetPairdoubleValue(valuecount,esc_radius);
				break;
#ifdef DEBUG
			default:
				puts("HEY?!! there's a widget in the list you forgot to update resetCalback for!!");
				break;
#endif
		}
	}
}

/* only called for ButtonPress */
void MandelHandler(Widget widget,XtPointer client_data,
		   XEvent *Buttonevent,Boolean *continue_dispatch)
{
	XButtonEvent *event = (XButtonEvent *) Buttonevent;
	/* assume event is ButtonPress */
	ComplexNum Icoord;
	double tempwidth,tempheight;

	updatescaling();
	switch(event->button){
		/* set ZOOM level, and also fall through to
		 * coordinate center set.
		 */
		case 2:
			tempwidth  =GetPairdoubleValue(MANDELWIDTH);
			tempheight =GetPairdoubleValue(MANDELHEIGHT);

			tempwidth /= 2;
			tempheight /= 2;
			SetPairdoubleValue(MANDELWIDTH,tempwidth);
			SetPairdoubleValue(MANDELHEIGHT,tempheight);
			/* fallthrough.. */

		/* Set a new coordinate center */
		case 1:
			Icoord = coordtocomplex(event->x,event->y);
			SetPairdoubleValue(MANDELCENTERX,Icoord.real_part);
			SetPairdoubleValue(MANDELCENTERY,Icoord.imag_part);

			break;
		case 3:
			tempwidth = GetPairdoubleValue(MANDELWIDTH);
			tempheight = GetPairdoubleValue(MANDELHEIGHT);

			tempwidth *= 2;
			tempheight*= 2;
			SetPairdoubleValue(MANDELWIDTH,tempwidth);
			SetPairdoubleValue(MANDELHEIGHT,tempheight);
			break;
		
	}
		
}

/*ReadValues():
 *	sets up all drawing values based on values displayed 
 */
int ReadValues(){
	int valueloop;
	
	for(valueloop=0;valueloop<NUM_OF_WIDGETS;valueloop++){

		/* NOTE!!
		* Each case MUST SET VALUE FOR stringbuffer!!
		*/
		switch(valueloop){
			case EQUASION:
				UpdateEquasion();

				break;
			case DEPTH:
				iterate = GetPairdoubleValue(DEPTH);
				SetPairintValue(DEPTH,iterate);
				break;
			case MANDELCENTERX:
				viewcenterx = GetPairdoubleValue(MANDELCENTERX);
				SetPairdoubleValue(MANDELCENTERX,viewcenterx);
				break;
			case MANDELCENTERY:
				viewcentery = GetPairdoubleValue(MANDELCENTERY);
				SetPairdoubleValue(MANDELCENTERY,viewcentery);
				break;
			case MANDELWIDTH:
				viewwidth = GetPairdoubleValue(MANDELWIDTH);
				SetPairdoubleValue(MANDELWIDTH,viewwidth);
				break;
			case MANDELHEIGHT:
				viewheight = GetPairdoubleValue(MANDELHEIGHT);
				SetPairdoubleValue(MANDELHEIGHT,viewheight);
				break;
			case ESC_RADIUS:
				esc_radius = GetPairdoubleValue(ESC_RADIUS);
				SetPairdoubleValue(ESC_RADIUS,esc_radius);
				break;
#ifdef DEBUG
			default:
				puts("HEY?!! there's a widget in the list you forgot to update ReadValues for!!");
				break;
#endif
		}
	}
	return 1;
}

/* DialogOK():
 *	ShortCut for <return> acelerator.
 *	Now it's basically just a way to ignore Return
 */
void DialogOK(Widget w, XEvent *event, String *params,Cardinal *num_params)
{
	if(w != WidgetPairs[WIDGETVALUE][EQUASION])
		return;
	UpdateEquasion();

}


/* MakeCustomDialog():
 *	make all the text/label widgetpairs for the display. yeauch.
 */
void MakeCustomDialog(){
	int widgetcounter;
	for(widgetcounter=0;widgetcounter<NUM_OF_WIDGETS;widgetcounter++){
		Widget *WidgetLabel,*WidgetValue;
		char widgetname[100];
		WidgetLabel = &WidgetPairs[WIDGETLABEL][widgetcounter];
		WidgetValue = &WidgetPairs[WIDGETVALUE][widgetcounter];

		/* first set "label" widget.. */
		sprintf(widgetname,"%sL",widgetnames[widgetcounter]);
		*WidgetLabel = XtCreateManagedWidget(
				widgetname,labelWidgetClass,mandelvalues,
				NULL,0);

		sprintf(widgetname,"%s:",widgetnames[widgetcounter]);
		XtVaSetValues(*WidgetLabel,
			XtNlabel,widgetname,
			XtNborderWidth,0,
			NULL);
		if(widgetcounter != 0){
			XtVaSetValues(*WidgetLabel,
			XtNfromVert,WidgetPairs[WIDGETVALUE][widgetcounter-1],
			NULL);/* yes, WIDGETVALUE, since the label has no border */
		}

		/*** and now the "value" part of the pair.. ****/
		sprintf(widgetname,"%sV",widgetnames[widgetcounter]);
		*WidgetValue = XtVaCreateManagedWidget(
				widgetname,asciiTextWidgetClass,mandelvalues,
				XtNeditType,XawtextEdit,
				XtNfromHoriz,*WidgetLabel,
				XtNstring,values[widgetcounter],
				XtNwidth,150,
				NULL);

		/* make Return key work.. */
		XtOverrideTranslations(*WidgetValue,ReturnAccelerator);
		if(widgetcounter != 0){
			XtVaSetValues(*WidgetValue,
				XtNfromVert,WidgetPairs[WIDGETVALUE][widgetcounter-1],
				NULL);
		}
	}
}

/*MakeEquasionDialog():
*	sets up all the widgets pertaining to equasions.
 *	That's almost all of them!!
 */
void MakeEquasionDialog(){

	mandelvalues = XtVaCreateManagedWidget(
			"mandelvalues",formWidgetClass,equasionForm,
			XtNfromVert,buttonBox,
			XtNborderWidth,2,
			NULL);
/* And NOW.. to make an EXTREMELY long fake dialog widget */
	MakeCustomDialog();

	strcpy(equasionText,mandelbrotEquasion);
	XtVaSetValues(WidgetPairs[WIDGETVALUE][EQUASION],
#ifdef MANDELONLY
		      XtNeditType,XawtextRead,
#else
		      XtNeditType,XawtextEdit,
#endif
		      XtNlabel,equasionText,
		      NULL);
}

/*******************************************************************
 *     And now.. Finally... the internal equasion coding routines  *
 *******************************************************************/

/* freeEquasion():
 *	safe freeing method.
 *	does not try to free NULL pointers,
 *	and halts program if out of memory.
 */
void freeEquasion(struct runEquasion * killthis){
	if(killthis==NULL) return;
	if(killthis->nextStep != NULL)
		freeEquasion((struct runEquasion *) killthis->nextStep);
	if(killthis != NULL)
		free(killthis);
}
/* newEquasion():
 *	allocs a new node for use in the equasion string,
 *	with the default type of 'DONE'
 */
struct runEquasion * 
newEquasion(){
	struct runEquasion * newEq;
	newEq = (struct runEquasion * ) malloc(sizeof (struct runEquasion));
	if(newEq == NULL){
		fprintf(stderr,"Sorry... out of memory. Quitting...\n");
		quit();
	}
	newEq->nextStep = NULL;
	newEq->operation = DONE;
	return newEq;
}

/* UpdateEquasion():
 *	Can be called at any time.
 *	Updates the encoding of the equasion, IF text string
 *	  has been changed.
 *	Copy of last ok equasion is storedin equasionText,
 *	  in case new one is not acceptable.
 */
void 
UpdateEquasion() {
	String value;
	Widget w = WidgetPairs[WIDGETVALUE][EQUASION];

	if(!XawAsciiSourceChanged(w)){
#ifdef DEBUG
		puts("Equasion has not changed");
#endif
		return;
	}

#ifdef DEBUG
	puts("Equasion value has changed.");
#endif
	XtVaGetValues(w, XtNstring,&value,  NULL);

	if(parseEquasion(value)){
#ifdef DEBUG
		puts("new Equasion is okay");
#endif
		strcpy(equasionText,value);
	} else {
#ifdef DEBUG
		puts("New equasion FAILED!!");
#endif
	}
	XtVaSetValues(w,XtNstring,equasionText,NULL);
}


void EqAddPart(	struct runEquasion ** currentEq,int value){
	(*currentEq)->operation = value;
	(*currentEq)->nextStep = newEquasion();
	*currentEq = (*currentEq)->nextStep;
}

/* parseEquasion():
 *	Passed a string to be used as the equasion for mandelbrotting.
 *	This is basically a portable lex function:
 *		breaks up string into tokens.
 *	Changes "MainEquasion" IFF new equasion is okay.
 *	Constructs the MainEquasion list for evaluating.
 *	returns 1 if string is acceptable,
 *	returns 0 if not acceptable.
 */
int parseEquasion(char *equasionString){
	struct runEquasion * EqStructP,* headofnew;
	ComplexNum tempcomplex;

	headofnew = EqStructP=newEquasion();

	if(!skipwhitespace(&equasionString)){
		freeEquasion(headofnew);
		return 0;
	}

	while(skipwhitespace(&equasionString)){
		/* get operation, then operand */
		switch(equasionString[0]){
			case '^':
				if(equasionString[1] != '2'){
					freeEquasion(headofnew);
					return 0;
				}
				equasionString++; 
				EqAddPart(&EqStructP,POWER2);
				equasionString++;
				continue;
				break;
			case '*':
				EqAddPart(&EqStructP,MUL);

				break;
			case '+':
				EqAddPart(&EqStructP,ADD);
				break;
			case '-':
				EqAddPart(&EqStructP,SUB);
				break;
			case 'Z':
			case 'z':
				EqAddPart(&EqStructP,STARTZ);
				break;
			case 'C':
			case 'c':
				EqAddPart(&EqStructP,CONSTANT);
				break;
			case '(':
				EqAddPart(&EqStructP,STARTPAREN);
				break;
			case ')':
				EqAddPart(&EqStructP,ENDPAREN);
				break;
			default:
				freeEquasion(headofnew);
				return 0;
		}
		equasionString++;
			
	} /* while another non-whitespace char.. */
	EqStructP = headofnew;
	if(Equate(&tempcomplex,tempcomplex,tempcomplex,&EqStructP)== 0)
		return 0;

	freeEquasion(MainEquasion);
	MainEquasion = headofnew;
	return 1;
}


/* Equate():
 *	Subroutine evaluate "complex" equasion
 *	given initial values of variables,
 *	and start of equasion struct list.
 *	(used so we can do recursive () matching)
 */
int
Equate(ComplexNum *EqValue,ComplexNum C,ComplexNum startZ,struct runEquasion **Equasion){
	ComplexNum accumilateZ,accum2;
	struct runEquasion *parseEqPtr = *Equasion;
	
	switch(parseEqPtr->operation){
		case STARTZ:
			accumilateZ = startZ;
			break;
		case CONSTANT:
			accumilateZ = C;
			break;
		default:
			return 0;
	}
	parseEqPtr = parseEqPtr->nextStep;

	while((parseEqPtr->operation != DONE)
	      && (parseEqPtr->operation != ENDPAREN)) {
		ComplexNum (*OpFunction)(ComplexNum,ComplexNum);

		switch(parseEqPtr->operation){
			case POWER2:
				accumilateZ = ComplexSquare(accumilateZ);
				parseEqPtr= parseEqPtr->nextStep;
				continue;
				break;
			case ADD:
				OpFunction = ComplexAdd;
				break;
			case SUB:
				OpFunction = ComplexSubtract;
				break;
			case MUL:
				OpFunction = ComplexMultiply;
				break;
			case DONE:
			case ENDPAREN:
				break;
			default:
				/* error */
				return 0;
				break;
		}
		parseEqPtr= parseEqPtr->nextStep;
		switch(parseEqPtr->operation){
			case STARTPAREN:
				parseEqPtr= parseEqPtr->nextStep;
				Equate(&accum2,C,startZ,&parseEqPtr);
				accumilateZ = (*OpFunction)(accumilateZ,accum2);
				
				break;
			case ENDPAREN:
				break;
			case STARTZ:
				accumilateZ = (*OpFunction)(accumilateZ,startZ);
				break;
			case CONSTANT:
				accumilateZ = (*OpFunction)(accumilateZ,C);
				break;
			default:
				/* error */
				return 0;
				break;
		}
		parseEqPtr=parseEqPtr->nextStep;
	}/* while parseEqPtr != DONE */


	
	*Equasion = parseEqPtr;
	*EqValue = accumilateZ;
	return 1;
}

/* calculatePoint(), fastcalculatePoint()
 *	calculates INDEX into gc_colors[]
 *	for each point given.
 *	"x" is real coord, "y" is imaginary coord
 *
 *	assinging color values is still somewhat experimental..
 */

int calculatePoint(ComplexNum C){
	ComplexNum accumilateZ;
	int colorindex,partitionsize;
	int iterateloop = iterate;

	if(iterate == 0) return 0;
#ifndef MANDELONLY
	if(MainEquasion == NULL){
		if(parseEquasion(equasionText)== 0){
			fprintf(stderr,"Xmandel: compiled with bad equasion\n");
			exit(1);
		}
	}
#endif
	/* color[0] doesn't count! */
	partitionsize = (iterate / (maxcolor-1)) ;
	/* rounding error.. but we shall err on the favour
	 * of red. There will be more red than natural, for
	 * iterations not evenly divisible.
	 *   (Too many partitions is desirable.
	 *	 Red gets the extra one)
	 */
#ifdef TOOFEWPARTITIONS
	if( (iterate % (maxcolor-1)) != 0)
		partitionsize+=1;
#endif

	accumilateZ = C;

	while(iterateloop-- >0){
#ifndef MANDELONLY
		struct runEquasion *parseEqPtr = MainEquasion;

		Equate(&accumilateZ,C,accumilateZ,&parseEqPtr);

#else	/* MANDELONLY */
		accumilateZ = ComplexSquare(accumilateZ);
		accumilateZ = ComplexAdd(accumilateZ,C);
#endif /* MANDELONLY */

#ifdef RELATIVEMAGNITUDE
		if( (((accumilateZ.real_part- viewcenterx) *
		      (accumilateZ.real_part- viewcenterx)) + 
		     ((accumilateZ.imag_part -viewcentery) *
		      (accumilateZ.imag_part -viewcentery)))
		    > radius_squared )
#else
		if( (accumilateZ.real_part * accumilateZ.real_part) +
		    (accumilateZ.imag_part* accumilateZ.imag_part) 
		    > radius_squared )
#endif
		{
			colorindex = iterateloop /partitionsize;
			/* skip black.. */
			colorindex+=1;
			
			/* damn rounding errors.. */
			if(colorindex>=maxcolor){
				colorindex = maxcolor-1;
			}
			return (colorindex);
		}
	}
	return (0);
}

/* fastcalculatePoint()
 *	See calculatePoint() for comments.
 *	This is selected as the calculation base DYNAMICALLY,
 *	depending on the "Real/Fast" toggle button
 */
int fastcalculatePoint(ComplexNum startZ){
	ComplexNum accumilateZ;
	ComplexNum C;
	int colorindex,partitionsize;
	int iterateloop = iterate;

	if(iterate == 0) return 0;
	/* color[0] doesn't count! */
	
	partitionsize = (iterate / (maxcolor-1)) ;
	C = startZ;
	
	while(iterateloop-- >0){
		accumilateZ = ComplexSquare(startZ);
		startZ = accumilateZ = ComplexAdd(accumilateZ,C);

		if((fabs(accumilateZ.real_part ) > esc_radius)
		    ||(fabs(accumilateZ.imag_part) > esc_radius) ){

			colorindex = iterateloop /partitionsize;
			/* skip black.. */
			colorindex+=1;
			if(colorindex>=maxcolor){
				colorindex = maxcolor-1;
			}
			return (colorindex);
		    }
	}
	return (0);
}
