/* template.c -- create feature templates				*/
/*
 * Copyright (c) 1994  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "xdatplot.h"
#include "rfiles.h"

#define	TPL_VERSION		"0.1"
#define	TPL_MAGIC_FORMAT	"! tpl-%s\n"
#define	MAXITER	10			/* max alignment iterations	*/
#define	TOL	(0.001)			/* alignment fit tolerance	*/

#ifdef	__STDC__
static	void	view_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	create_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	cancel_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	help_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	set_scale(double, Widget);
static	void	w_arrow_cb(Widget, int, XmArrowButtonCallbackStruct *);
static	void	do_template(void);
static	void	template_file(XtCallbackProc);
static	void	template_save_ok(Widget, XtPointer,
				 XmFileSelectionBoxCallbackStruct *);
static	void	template_view_ok(Widget, XtPointer,
				 XmFileSelectionBoxCallbackStruct *);
static	String	write_template(String);
static	String	get_string_resource(XrmDatabase, String, String);
static	String	points_to_string(POINT *);
static	POINT	*string_to_points(String);
static	String	align_template(FILTER_VAL *, TIME, TIME, int);
static	void	condition(FILTER_VAL *, FILTER_VAL *, TIME, TIME);
static	double	ssr(TIME, FILTER_VAL *, TIME, TIME);
#else	/* __STDC__ */
static	void	view_cb();
static	void	create_cb();
static	void	cancel_cb();
static	void	help_cb();
static	void	set_scale();
static	void	w_arrow_cb();
static	void	do_template();
static	void	template_file();
static	void	template_save_ok();
static	void	template_view_ok();
static	String	write_template();
static	String	get_string_resource();
static	String	points_to_string();
static	POINT	*string_to_points();
static	String	align_template();
static	void	condition();
static	double	ssr();
#endif	/* __STDC__ */

static	XtResource	tt_resource = {
    "templateType",
    "TemplateType",
    XtRTemplateType,
    sizeof(TEMPL_TYPE),
    XtOffsetOf(TEMPL, type),
    XtRString,
    "INTERPOLATED",
};

static	XtResource	ti_resources[] = {
    {
	"templateType",
	"TemplateType",
	XtRTemplateType,
	sizeof(TEMPL_TYPE),
	XtOffsetOf(TEMPL_INTER, type),
	XtRString,
	"INTERPOLATED"
    },
    {
	"interTemplate.weight",
	"Template.Weight",
	XtRDouble,
	sizeof(double),
	XtOffsetOf(TEMPL_INTER, weight),
	XtRString,
	"1.0"
    },
    {
	"interTemplate.normalization",
	"Template.Normalization",
	XtRDouble,
	sizeof(double),
	XtOffsetOf(TEMPL_INTER, norm),
	XtRString,
	"1.0"
    },
    {
	"interTemplate.numPoints",
	"Template.NumPoints",
	XtRInt,
	sizeof(int),
	XtOffsetOf(TEMPL_INTER, npoints),
	XmRImmediate,
	(XtPointer) 0
    },
    {
	"interTemplate.points",
	"Template.Points",
	XtRPointList,
	sizeof(POINT *),
	XtOffsetOf(TEMPL_INTER, points),
	XtRString,
	""
    },
};

static	Widget	template_save_pu = NO_WIDGET;
static	Widget	left_width = NO_WIDGET;
static	Widget	right_width = NO_WIDGET;
static	Widget	middle_mark = NO_WIDGET;
static	Widget	norm_toggle = NO_WIDGET;
static	Widget	align_toggle = NO_WIDGET;
static	TEMPL	*the_templ = NULL;
static	String	scr = NULL;

Widget
make_template_dialog()
{
    Widget		template_sh;
    char		lbuf[LLEN];
    String		s;
    Widget		template_rowcol;
    Widget		action_frame;
    Widget		action_form;
    Widget		view_button;
    Widget		create_button;
    Widget		cancel_button;
    Widget		help_button;
    Widget		size_frame;
    Widget		size_form;
    Widget		where_frame;
    Widget		where_form;
    Widget		label;
    Widget		arrow1;
    Widget		arrow2;

    /*
     * create the dialog shell and a form widget to go in it
     */
    XtVaGetValues(toplevel, XmNtitle, &s, NULL);
    sprintf(lbuf, "%s Create Template", s);
    template_sh = XtVaAppCreateShell("templateShell", "TemplateShell",
	topLevelShellWidgetClass, XtDisplay(toplevel),
	XtNtitle, lbuf,
	XmNiconName, "template",
	XmNiconPixmap, app_data.icon,
	XmNdeleteResponse, XmUNMAP,
    NULL);
    template_rowcol = XtVaCreateWidget("templateRowCol",
	xmRowColumnWidgetClass, template_sh,
	XmNorientation, XmVERTICAL,
	XmNpacking, XmPACK_TIGHT,
	XmNnumColumns, 1,
	XmNresizeHeight, True,
	XmNresizeWidth, True,
	XmNspacing, 0,
	XmNmarginHeight, 0,
	XmNmarginWidth, 0,
    NULL);
    /*
     * create the widget for the dimensions of the template
     */
    size_frame = XtVaCreateManagedWidget("templateSizeFrame",
	xmFrameWidgetClass, template_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    size_form = XtVaCreateManagedWidget("templateSizeForm",
	xmFormWidgetClass, size_frame,
    NULL);
    left_width = XtVaCreateManagedWidget("leftWidth",
	xmScaleWidgetClass, size_form,
    NULL);
    right_width = XtVaCreateManagedWidget("rightWidth",
	xmScaleWidgetClass, size_form,
    NULL);
    set_scales(app_data.template_left_width, app_data.template_right_width,
	      right_width, left_width);
    arrow1 = XtVaCreateManagedWidget("wUpArrow",
	xmArrowButtonWidgetClass, size_form,
	XmNarrowDirection, XmARROW_UP,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, left_width,
	XmNrightWidget, left_width,
	XmNtopWidget, right_width,
	XmNbottomWidget, right_width,
    NULL);
    AddCallback(arrow1, XmNactivateCallback, w_arrow_cb, 1);
    arrow2 = XtVaCreateManagedWidget("wDownArrow",
	xmArrowButtonWidgetClass, size_form,
	XmNarrowDirection, XmARROW_DOWN,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, arrow1,
	XmNrightWidget, arrow1,
	XmNtopWidget, arrow1,
	XmNbottomWidget, arrow1,
    NULL);
    AddCallback(arrow2, XmNactivateCallback, w_arrow_cb, -1);
    label = XtVaCreateManagedWidget("wArrowLabel",
	xmLabelWidgetClass, size_form,
	XmNleftWidget, arrow2,
	XmNrightWidget, arrow2,
	XmNtopWidget, arrow2,
	XmNbottomWidget, arrow2,
    NULL);
    XtManageChild(size_form);
    /*
     * Create the widget for saying how to find template
     */
    where_frame = XtVaCreateManagedWidget("templateWhereFrame",
	xmFrameWidgetClass, template_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    where_form = XtVaCreateManagedWidget("templateWhereForm",
	xmFormWidgetClass, where_frame,
    NULL);
    label = XtVaCreateManagedWidget("middleMarkLabel",
	xmLabelWidgetClass, where_form,
    NULL);
    middle_mark = XtVaCreateManagedWidget("middleMarkField",
	xmTextFieldWidgetClass, where_form,
	XmNtopWidget, label,
	XmNleftWidget, label,
	XmNrightWidget, label,
	XmNbottomWidget, label,
    NULL);
    AddCallback(middle_mark, XmNactivateCallback, create_cb, template_sh);
    norm_toggle = XtVaCreateManagedWidget("normToggle",
	xmToggleButtonWidgetClass, where_form,
    NULL);
    align_toggle = XtVaCreateManagedWidget("alignToggle",
	xmToggleButtonWidgetClass, where_form,
    NULL);
    XtManageChild(where_form);
    /*
     * create the action area
     */
    action_frame = XtVaCreateManagedWidget("templateActionFrame",
	xmFrameWidgetClass, template_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNbottomAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    action_form = XtVaCreateWidget("templateActionForm",
	xmFormWidgetClass, action_frame,
    NULL);
    view_button = XtVaCreateManagedWidget("templateViewButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(view_button, XmNactivateCallback, view_cb, template_sh);
    create_button = XtVaCreateManagedWidget("templateCreateButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(create_button, XmNactivateCallback, create_cb, template_sh);
    cancel_button = XtVaCreateManagedWidget("templateCancelButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(cancel_button, XmNactivateCallback, cancel_cb, template_sh);
    help_button = XtVaCreateManagedWidget("templateHelpButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(help_button, XmNactivateCallback, help_cb, template_sh);
    XtManageChild(action_form);
    /*
     * done: manage the whole form and return
     */
    XtManageChild(template_rowcol);
    return(template_sh);
}

static void
w_arrow_cb(arrow, dir, cbs)
Widget				arrow;
int				dir;
XmArrowButtonCallbackStruct	*cbs;
{
    bump_scales(dir, left_width, right_width);
}

static void
view_cb(button, template_sh, cbs)
Widget				button;
Widget				template_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("view_cb (template.c)\n");
    template_file((XtCallbackProc) template_view_ok);
}

static void
create_cb(button, template_sh, cbs)
Widget				button;
Widget				template_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("create_cb (template.c)\n");
    do_template();
}

static void
cancel_cb(button, template_sh, cbs)
Widget				button;
Widget				template_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("cancel_cb (template.c)\n");
    XtPopdown(template_sh);
}

static void
help_cb(button, template_sh, cbs)
Widget				button;
Widget				template_sh;
XmPushButtonCallbackStruct	*cbs;
{
    help_window("template.html");
}

#define	get_scale_val(sc, adm)\
{						\
    int			val;			\
    short		dp;			\
						\
    XtVaGetValues(sc,				\
	XmNvalue, &val,				\
	XmNdecimalPoints, &dp,			\
    NULL);					\
    app_data.adm = (double) val;		\
    app_data.adm /= exp10((double) dp);		\
}

static void
do_template()
{
    char		lbuf[LLEN];
    register TIME	t;
    String		s;
    int			middle;
    TIME		tm;
    MARK		*mp;
    int			weight;
    FILTER_VAL		sum;
    FILTER_VAL		vmin;
    FILTER_VAL		vmax;
    FILTER_VAL		*template0;
    FILTER_VAL		*template;
    TIME		left;
    TIME		right;
    TEMPL_INTER		*ti;

    /*
     * add up all the features
     */
    get_scale_val(left_width, template_left_width);
    get_scale_val(right_width, template_right_width);
    s = XmTextFieldGetString(middle_mark);
    middle = *s;
    Free(s);
    left = floor(-app_data.template_left_width / TIME_to_t_sf + 0.5);
    right = floor(app_data.template_right_width / TIME_to_t_sf + 0.5);
    if ((left >= 0) && (right <= 0)) {
	PU_error("template must be > 1 point wide", "template.html");
	return;
    }
    template0 = (FILTER_VAL *) XtCalloc(right - left + 1, sizeof(FILTER_VAL));
    template = template0 - left;
    for(t = left; t <= right; t++) {
	template[t] = 0.0;
    }
    weight = 0;
    for(
	mp = next_mark(Map_TIME_to_MARK(TMIN), middle);
	NULL != mp;
	mp = next_mark(mp->t + 1, middle)
    ) {
	tm = Map_MARK_to_TIME(mp->t);
	if ((tm + left < TMIN) || (tm + right >= TMAX))
	    continue;
	for(t = left; t <= right; t++) {
	    template[t] += Vf(tm + t);
	}
	weight++;
    }
    /*
     * Align if checked
     */
    if (XmToggleButtonGetState(align_toggle)) {
	s = align_template(template, left, right, middle);
	if (
	    (NULL != s) &&
	    !ask_user("Failed to converge: save anyway?")
	) {
	    Free(template0);
	    PU_error(s, "template.html");
	    Free(s);
	    return;
	}
    }
    Nfree(s);
    /*
     * displace so sum is zero
     */
    sum = template[left];
    for(t = left+1; t <= right; t++) {
	sum += template[t];
    }
    sum /= right - left + 1;
    for(t = left; t <= right; t++) {
	template[t] -= sum;
    }
    /*
     * Normalize if it checked
     */
    if (XmToggleButtonGetState(norm_toggle)) {
	vmin = vmax = template[left];
	for(t = left+1; t <= right; t++) {
	    if (template[t] < vmin) vmin = template[t];
	    if (template[t] > vmax) vmax = template[t];
	}
	if (vmin >= vmax) {
	    Free(template0);
	    PU_error("marked features are flat", "template.html");
	    return;
	}
	vmax -= vmin;
	for(t = left; t <= right; t++) {
	    template[t] /= vmax;
	}
    }
    else {
	vmax = 1.0;
    }
    /*
     * turn it into a TEMPL structure
     */
    destroy_templ(the_templ);
    the_templ = XtNew(TEMPL);
    the_templ->type = INTERPOLATED;
    ti = &the_templ->templ_inter;
    ti->weight = weight;
    ti->norm = vmax;
    ti->npoints = right - left + 1;
    ti->points = (POINT *) XtCalloc(ti->npoints + 1, sizeof(POINT));
    for(t = left; t <= right; t++) {
	ti->points[t-left].time = t * TIME_to_t_sf;
	ti->points[t-left].val = template[t];
    }
    ti->points[right - left + 1].time = signaling_nan(0);
    ti->points[right - left + 1].val = INVALID_FILTER_VAL;
    Free(template0);
    /*
     * pop up the file selection box to write it out
     */
    template_file((XtCallbackProc) template_save_ok);
}

static void
template_file(callback)
XtCallbackProc	callback;
{
    char		lbuf[LLEN];
    String		s;

    if (NO_WIDGET == template_save_pu) {
	template_save_pu = XmCreateFileSelectionDialog(toplevel,
						       "templateSaveSB",
						       NULL, 0);
	XtSetSensitive(
	    XmSelectionBoxGetChild(template_save_pu, XmDIALOG_HELP_BUTTON),
	    FALSE
	);
	XtVaGetValues(toplevel, XmNtitle, &s, NULL);
	sprintf(lbuf, "%s Template Save/View", s);
	XtVaSetValues(template_save_pu,
	    XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
	    XtVaTypedArg, XmNdialogTitle, XmRString,
	    lbuf, strlen(lbuf)+1,
	NULL);
    }
    XtRemoveAllCallbacks(template_save_pu, XmNokCallback);
    AddCallback(template_save_pu, XmNokCallback, callback, NULL);
    XtManageChild(template_save_pu);
    XtPopup(XtParent(template_save_pu), XtGrabNone);
}

static void
template_save_ok(widget, data, cbs)
Widget					widget;
XtPointer				data;
XmFileSelectionBoxCallbackStruct	*cbs;
{
    char		lbuf[LLEN];
    String		fname;
    String		s;

    if (!XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &fname))
	error("template_save_ok: can't happen");
    dprintf1("template_save_ok: %s\n", fname);
    if (NULL != (s = write_template(fname))) {
	PU_error(s, "template.html");
	Free(s);
    }
    Free(fname);
    destroy_templ(the_templ);
    the_templ = NULL;
    XtUnmanageChild(widget);
    XtPopdown(XtParent(widget));
}

static String
write_template(fname)
String	fname;
{
    char		lbuf[LLEN];
    FILE		*tpl;
    String		s;

    if (NULL == (tpl = fopen(fname, "w"))) {
	sprintf(lbuf, "unable to open %s", fname);
	return(XtNewString(lbuf));
    }
    fprintf(tpl, TPL_MAGIC_FORMAT, TPL_VERSION);
    switch(the_templ->type) {
    case INTERPOLATED:
	{
	    TEMPL_INTER		*ti = &the_templ->templ_inter;

	    s = write_resource_stream(tpl, ti, toplevel, "*",
				      ti_resources, XtNumber(ti_resources));
	    if (NULL != s) {
		fclose(tpl);
		return(s);
	    }
	}
	break;

    default:
	fclose(tpl);
	sprintf(lbuf, "invalid template type %d", the_templ->type);
	return(XtNewString(lbuf));
    }
    fclose(tpl);
    return(NULL);
}

static void
template_view_ok(widget, data, cbs)
Widget					widget;
XtPointer				data;
XmFileSelectionBoxCallbackStruct	*cbs;
{
    static int		*pids = NULL;
    static int		npids = -1;

    register int	i, j;
    char		lbuf[LLEN];
    String		fname;
    String		s;
    int			gfd;
    int			gpid;		
    FILE		*graph;

    /*
     * clean up any old processes lying around
     */
    if (NULL == pids) {
	pids = XtNew(int);
	npids = 0;
    }
    for(i=j=0; i<npids; i++) {
	if ((EOF == kill(pids[i], 0)) && (ESRCH == errno)) {
	    wait();
	}
	else {
	    pids[j++] = pids[i];
	}
    }
    npids = j;
    /*
     * start a new one, then ram the template down its throat
     */
    if (!XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &fname))
	error("template_view_ok: can't happen");
    dprintf1("template_view_ok: %s\n", fname);
    the_templ = XtNew(TEMPL);
    bzero(the_templ, sizeof(TEMPL));
    if (NULL != (s = read_template(fname, the_templ))) {
	PU_error(s, "template.html");
	Free(s);
	Free(fname);
	goto quit;
    }
    Free(fname);
    if (app_data.read_commands) {
	sprintf(lbuf, "%s 1>&2", app_data.graph_command);
    }
    else {
	sprintf(lbuf, "%s", app_data.graph_command);
    }
    if (EOF == (gpid = run1(lbuf, &gfd))) {
	sprintf(lbuf, "Unable to open pipe to %s", app_data.graph_command);
	PU_error(lbuf, "template.html");
	goto quit;
    }
    pids = (int *) Realloc(pids, (npids+1) * sizeof(int));
    pids[npids++] = gpid;
    graph = fdopen(gfd, "w");
    switch(the_templ->type) {
    case INTERPOLATED:
	{
	    TEMPL_INTER		*ti = &the_templ->templ_inter;
	    register int	i;

	    for(i=0; i<ti->npoints; i++) {
		fprintf(graph, "%.17g %.17g\n",
			ti->points[i].time, ti->points[i].val);
	    }
	}
	break;

    default:
	PU_error("unknown template type", "template,html");
	break;
    }
    fclose(graph);

    /*
     * clean up and quit
     */
quit:
    destroy_templ(the_templ);
    the_templ = NULL;
    XtUnmanageChild(widget);
    XtPopdown(XtParent(widget));
}

void
destroy_templ(t)
TEMPL	*t;
{
    if (NULL == t) return;
    switch(t->type) {
    case INTERPOLATED:
	{
	    TEMPL_INTER	*ti = &t->templ_inter;

	    Free(ti->points);
	}
	break;

    default:
	break;
    }
    Free(t);
}

/* canned from Xt ref manual						*/
#define	done(type, value)\
{									\
    if (to->addr != NULL) {						\
	if (to->size < sizeof(type)) {					\
	    to->size = sizeof(type);					\
	    return(False);						\
	}								\
	*((type *) to->addr) = (value);					\
    }									\
    else {								\
	static type	static_val;					\
	static_val = (value);						\
	to->addr = (XtPointer) &static_val;				\
    }									\
    to->size = sizeof(type);						\
    return(True);							\
}

Boolean
CvtTemplateTypeToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    char		lbuf[LLEN];
    String		s;
    int			l;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtTemplateTypeToString",
		     "XtToolkitError", "TemplateType to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    switch(*((TEMPL_TYPE *) from->addr)) {
    case INTERPOLATED:
	s = "INTERPOLATED";
	break;

    default:
	XtWarningMsg("wrongParameters", "cvtTemplateTypeToString",
		     "XtToolkitError", "TemplateType to String\
		     invalid template type",
		     NULL, NULL);
	s = "INVALID";
	break;
    }
    l = strlen(s);
    if (NULL == to->addr) {
	to->addr = (caddr_t) s;
	to->size = l + 1;
	return(True);
    }
    else if (to->size <= l) {
	to->size = l + 1;
	return(False);
    }
    else {
	strcpy((String) to->addr, s);
	to->size = l + 1;
	return(True);
    }
}

Boolean
CvtStringToTemplateType(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    TEMPL_TYPE		tt;
    String		s;
    String		t, u;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToTemplateType",
		     "XtToolkitError", "String to TemplateType\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    s = XtMalloc(from->size + 1);
    bcopy(from->addr, s, from->size);
    s[from->size] = '\0';
    for(t=s; isspace(*t); t++);
    if (0 == strcmp(t, "INTERPOLATED")) {
	tt = INTERPOLATED;
    }
    else {
	XtDisplayStringConversionWarning(d, (char *) from->addr,
					 XtRTemplateType);
	tt = INVALID;
    }
    done(TEMPL_TYPE, tt);
}

static POINT *
string_to_points(pl)
String	pl;
{
    register int	i;
    POINT		*pp;
    int			n;
    String		s;
    String		t;

    n = -1;
    for(s=pl-1; NULL != s; s = STRCHR(s+1, '\n')) n++;
    pp = (POINT *) XtCalloc(n+1, sizeof(POINT));
    s = pl;
    for(i=0; i<n; i++) {
	pp[i].time = strtod(s, &t);
	if (t == s) break;
	pp[i].val = strtod(t, &s);
	if (t == s) break;
	while(isspace(*s) && ('\n' != *s)) s++;
	if ('\n' == *s) s++;
    }
    if (i < n) {
	Free(pp);
	return(NULL);
    }
    pp[n].time = signaling_nan(0);
    pp[n].val = INVALID_FILTER_VAL;
    return(pp);
}

Boolean
CvtStringToPointList(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    POINT		*pp;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToPointList",
		     "XtToolkitError", "String to PointList\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    pp = string_to_points((String) from->addr);
    done(POINT *, pp);
}

static String
points_to_string(points)
POINT	*points;
{
    char		lbuf[LLEN];
    String		s = XtNewString("");
    int			len = 0;

    while(!isnan(points->val)) {
	sprintf(lbuf, "%.17g %.17g\n", points->time, points->val);
	s = append_string(s, lbuf, &len);
	points++;
    }
    return(s);
}

Boolean
CvtPointListToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    String		fc;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtPointListToString",
		     "XtToolkitError", "PointList to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    fc = points_to_string(*(POINT **) from->addr);
    if (NULL == to->addr) {
	Nfree(scr);
	scr = fc;
	to->addr = (caddr_t) scr;
	to->size = strlen(fc) + 1;
	return(True);
    }
    else if (to->size <= strlen(fc)) {
	to->size = strlen(fc) + 1;
	Free(fc);
	return(False);
    }
    else {
	strcpy((String) to->addr, fc);
	to->size = strlen(fc) + 1;
	Free(fc);
	return(True);
    }
}

String
read_template(fname, tpl)
String	fname;
TEMPL	*tpl;
{
    char		lbuf[LLEN];
    String		s;
    XrmDatabase		db;

    bzero(tpl, sizeof(*tpl));
    db = XrmGetFileDatabase(fname);
    s = read_resource_db(db, (XtPointer) tpl, toplevel, "template",
			 "Template", &tt_resource, 1);
    if (NULL != s) goto quit;
    switch(tpl->type) {
    case INTERPOLATED:
	{
	    TEMPL_INTER		*ti = &tpl->templ_inter;
	    register int	i;

	    s = read_resource_db(db, (XtPointer) ti, toplevel, "template",
				 "Template", ti_resources,
				 XtNumber(ti_resources));
	    for(i=0; i<ti->npoints; i++) {
		if (isnan(ti->points[i].time)) break;
	    }
	    if (
		(i < ti->npoints) ||
		!isnan(ti->points[ti->npoints].time)
	    ) {
		sprintf(lbuf, "inconsistent numPoints and Points in %s", fname);
		s = XtNewString(lbuf);
		goto quit;
	    }
	}
	break;

    default:
	sprintf(lbuf, "Unknown template type in file %s", fname);
	s = XtNewString(lbuf);
	break;
    }

 quit:
    if (NULL != s) tpl->type = INVALID;
    XrmDestroyDatabase(db);
    return(s);
}

/* cook_template -- translate template to filter values
 *
 * TEMPL	*tpl;
 * double	tscale;
 * FILTER_VAL	*vals;
 * int		nvals;
 * int		tzero;
 * String	s;
 * s = cook_template(tpl, tscale, &vals, &nvals, &tzero);
 *
 * cook_template cooks a template into a series of equally spaced
 * values.  The time values in the template are multiplied by tscale
 * to get cooked time values (where the spacing between points is one
 * time unit).  The values themselves are returned in vals, and the
 * number of them in nvals.  vals[tzero] is the value at time zero.
 * Note that this point may not actually exist.  For instance, time
 * zero might be before the start of the template, in which case tzero
 * will be negative.  The values are displaced so that they add up to
 * zero.
 *
 * cook_template returns NULL if there is no error: other wise it
 * returns an error message in space allocated from the heap.
 */
String
cook_template(tpl, tscale, vp, nvp, tzp)
TEMPL		*tpl;
double		tscale;
FILTER_VAL	**vp;
int		*nvp;
int		*tzp;
{
    register int	i;
    FILTER_VAL		*vals;
    int			nvals;

    *vp = vals = NULL;
    switch(tpl->type) {
    case INTERPOLATED:
	{
	    TEMPL_INTER	*ti = &tpl->templ_inter;
	    TIME	left;
	    TIME	right;
	    TIME	t;
	    double	r;
	    POINT	*p = tpl->templ_inter.points;

	    if (ti->npoints <= 1) {
		nvals = 0;
		break;
	    }
	    left = ceil(ti->points[0].time * tscale);
	    right = floor(ti->points[ti->npoints - 1].time * tscale);
	    *nvp = nvals = right - left + 1;
	    *vp = vals = (FILTER_VAL *) XtCalloc(nvals, sizeof(FILTER_VAL));
	    *tzp = -left;
	    r = (p[1].val - p[0].val) / (p[1].time - p[0].time);
	    for(t = left; t <= right; t++) {
		if (t/tscale > p[1].time) {
		    while(t/tscale > p[1].time) p++;
		    r = (p[1].val - p[0].val) / (p[1].time - p[0].time);
		}
		*vals++ = p[0].val + r * (t/tscale - p[0].time);
	    }
	    vals = *vp;
	}
	break;

    default:
	return(XtNewString("unknown template type"));
    }
    if (0 >= nvals) {
	Nfree(*vp);
	return(XtNewString("Template too short"));
    }
    return(NULL);
}

static INLINE double
ssr(tm, tp, left, right)
TIME		tm;
FILTER_VAL	*tp;
TIME		left;
TIME		right;
{
    register TIME	t;
    FILTER_VAL		val;
    double		sum = 0.0;
    double		ss = 0.0;
    double		svt = 0.0;

    for(t = left; t <= right; t++) {
	val = Vf(tm + t);
	sum += val;
	ss += val * val;
	svt += val * tp[t];
    }
    return(ss - sum * sum / (right - left + 1) - svt * svt);
}

static String
align_template(template, left, right, middle)
FILTER_VAL	*template;
TIME		left;
TIME		right;
int		middle;
{
    register int	i;
    FILTER_VAL		*tp0 = NULL;
    FILTER_VAL		*tp;
    MARK		*mp;
    int			m = 0;
    int			*deltas;
    int			anchor;
    TIME		tm;

    tp0 = (FILTER_VAL *) XtCalloc(right-left+1, sizeof(FILTER_VAL));
    tp = tp0 - left;
    /*
     * count the marks
     */
    for(
	m = 0, mp = next_mark(Map_TIME_to_MARK(TMIN), middle);
	NULL != mp;
	mp = next_mark(mp->t + 1, middle)
    ) {
	tm = Map_MARK_to_TIME(mp->t);
	if ((tm + left < TMIN) || (tm + right >= TMAX))
	    continue;
	m++;
    }
    deltas = (int *) XtCalloc(m, sizeof(int));

    anchor = -1;
    for(i=0; i<MAXITER; i++) {
	Bool		changed;
	TIME		t;
	TIME		tmd;
	double		ssrm, ssrl, ssrr;
	double		maxdiff = 0.0;
	int		nmd = -1;

	condition(template, tp, left, right);
	changed = False;
	/*
	 * For each point, try to improve its alignment
	 */
	for(
	    m = 0, mp = next_mark(Map_TIME_to_MARK(TMIN), middle);
	    NULL != mp;
	    mp = next_mark(mp->t + 1, middle)
	) {
	    tm = Map_MARK_to_TIME(mp->t);
	    if ((tm + left < TMIN) || (tm + right >= TMAX))
		continue;
	    if (m == anchor) {
		m++;
		continue;
	    }
	    ssrm = ssrr = ssrl = quiet_nan(0);
	    for(;;) {
		tmd = tm + deltas[m];
		if (isnan(ssrm)) {
		    ssrm = ssr(tmd, tp, left, right);
		}
		/*
		 * It's just a step to the left...
		 */
		if (isnan(ssrl)) {
		    if ((tmd + left > TMIN) && (deltas[m] > left)) {
			ssrl = ssr(tmd-1, tp, left, right);
		    }
		    else {
			ssrl = infinity();
		    }
		}
		/*
		 * ...and then a jump to the right...
		 */
		if (isnan(ssrr)) {
		    if ((tmd + right < TMAX-1) && (deltas[m] < right)) {
			ssrr = ssr(tmd+1, tp, left, right);
		    }
		    else {
			ssrr = infinity();
		    }
		}

		if ((ssrm <= ssrl) && (ssrm <= ssrr)) break;
		/*
		 * Let's do the TimeWarp again!
		 */
		if (ssrl < ssrr) {
		    deltas[m]--;
		    if ((ssrm - ssrl)/ssrm > TOL) changed = True;
		    ssrr = ssrm;
		    ssrm = ssrl;
		    ssrl = quiet_nan(0);
		}
		else {
		    deltas[m]++;
		    if ((ssrm - ssrr)/ssrm > TOL) changed = True;
		    ssrl = ssrm;
		    ssrm = ssrr;
		    ssrr = quiet_nan(0);
		}
	    }
	    /*
	     * in first iteration, look for anchor
	     */
	    if (
		(-1 == anchor) &&
		(ssrl - ssrm > maxdiff) &&
		(ssrr - ssrm > maxdiff)
	    ) {
		maxdiff = min(ssrr - ssrm, ssrl - ssrm);
		nmd = m;
	    }
	    m++;
	}
	/*
	 * If nothing changed, we're finished.  If there was a change,
	 * recompute the template and try again.
	 */
	if (!changed) break;
	bzero(template + left, (right - left + 1) * sizeof(FILTER_VAL));
	for(
	    m = 0, mp = next_mark(Map_TIME_to_MARK(TMIN), middle);
	    NULL != mp;
	    mp = next_mark(mp->t + 1, middle)
	) {
	    tm = Map_MARK_to_TIME(mp->t);
	    if ((tm + left < TMIN) || (tm + right >= TMAX))
		continue;
	    tmd = tm + deltas[m];
	    for(t = left; t <= right; t++) {
		template[t] += Vf(tmd + t);
	    }
	    m++;
	}
	if (-1 == anchor) anchor = nmd;
    }
    Free(deltas);
    Free(tp0);
    if (i >= MAXITER) {
	char		lbuf[LLEN];

	sprintf(lbuf,
		"Least squares alignment didn't converge in %d iterations",
		i);
	return(XtNewString(lbuf));
    }
    return(NULL);
}

static void
condition(tpin, tpout, left, right)
FILTER_VAL	*tpin;
FILTER_VAL	*tpout;
TIME		left;
TIME		right;
{
    register TIME	t;
    int			n = right - left + 1;
    FILTER_VAL		sum = 0.0;
    FILTER_VAL		ss;

    tpin += left;
    tpout += left;
    for(t = 0; t < n; t++) {
	sum += tpin[t];
    }
    sum /= n;
    for(t = 0; t < n; t++) {
	tpout[t] = tpin[t] - sum;
    }
    ss = dot_product(tpin, tpin, n);
    if (0.0 >= ss) return;
    ss = sqrt(ss);
    for(t = 0; t < n; t++) {
	tpout[t] /= ss;
    }
}
