/* peaks.c -- find peaks in data					*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

/*
 * parameters used in searching for peaks
 */
typedef	enum	HTM {			/* how to mark			*/
    HTM_NONE = 0,			/* don't mark			*/
    HTM_PEAK = 1,			/* mark peak only		*/
    HTM_BOUND = 2,			/* mark bounds			*/
    HTM_ALL = 3,			/* mark bounds and peak		*/
}		HTM;

typedef	struct	PeakParams {
    FILTER_VAL	ph;			/* peak height			*/
    TIME	phw;			/* peak half width		*/
    String	pm;			/* peak marks			*/
    HTM		htmp;			/* how to mark peaks		*/
    FILTER_VAL	td;			/* trough depth			*/
    TIME	thw;			/* trough half-width		*/
    String	tm;			/* trough marks			*/
    HTM		htmt;			/* how to mark troughs		*/
}		PeakParams;

typedef struct	Peak {
    TIME	tl, tm, tr;
    FILTER_VAL	vl, vm, vr;
}		Peak;

#ifdef	__STDC__
static	void	t_arrow_cb(Widget, int, XmArrowButtonCallbackStruct *);
static	void	v_arrow_cb(Widget, int, XmArrowButtonCallbackStruct *);
static	void	ok_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	search_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	cancel_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	help_cb(Widget, Widget, XmPushButtonCallbackStruct *);
static	void	do_search(void);
static	void	find_peaks(TIME, TIME, PeakParams *);
static	TIME	peaktest(TIME, PeakParams *, Peak *);
static	TIME	troughtest(TIME, PeakParams *, Peak *);
static	void	mark_peak(Peak *, PeakParams *);
static	void	mark_trough(Peak *, PeakParams *);
static	void	mark_point(TIME, int, String);
#else	/* __STDC__ */
static	void	t_arrow_cb();
static	void	v_arrow_cb();
static	void	ok_cb();
static	void	search_cb();
static	void	cancel_cb();
static	void	help_cb();
static	void	do_search();
static	void	find_peaks();
static	TIME	peaktest();
static	TIME	troughtest();
static	void	mark_peak();
static	void	mark_trough();
static	void	mark_point();
#endif	/* __STDC__ */

static	int	scales[] = {		/* max scale sizes		*/
    100, 200, 500
};

static	Widget	peak_half_width = NO_WIDGET;
static	Widget	peak_height = NO_WIDGET;
static	Widget	peak_marks = NO_WIDGET;
static	Widget	trough_half_width = NO_WIDGET;
static	Widget	trough_depth = NO_WIDGET;
static	Widget	trough_marks = NO_WIDGET;
static	Widget	begin_mark = NO_WIDGET;
static	Widget	end_mark = NO_WIDGET;

Widget
make_peaks_dialog()
{
    Widget		peaks_sh;
    char		lbuf[LLEN];
    String		s;
    Widget		peaks_form;
    Widget		thres_frame;
    Widget		thres_form;
    Widget		arrow1;
    Widget		arrow2;
    Widget		marks_frame;
    Widget		marks_form;
    Widget		label;
    Widget		action_frame;
    Widget		action_form;
    Widget		ok_button;
    Widget		search_button;
    Widget		cancel_button;
    Widget		help_button;

    /*
     * create the dialog shell and a form widget to go in it
     */
    XtVaGetValues(toplevel, XmNtitle, &s, NULL);
    sprintf(lbuf, "%s Analyze Peaks", s);
    peaks_sh = XtVaAppCreateShell("peaksShell", "PeaksShell",
	topLevelShellWidgetClass, XtDisplay(toplevel),
	XtNtitle, lbuf,
	XmNiconName, "peaks",
	XmNiconPixmap, app_data.icon,
	XmNdeleteResponse, XmUNMAP,
    NULL);
    peaks_form = XtVaCreateWidget("peaksForm",
	xmFormWidgetClass, peaks_sh,
    NULL);
    /*
     * create the action area
     */
    action_frame = XtVaCreateManagedWidget("peaksActionFrame",
	xmFrameWidgetClass, peaks_form,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNbottomAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    action_form = XtVaCreateWidget("peaksActionForm",
	xmFormWidgetClass, action_frame,
    NULL);
    ok_button = XtVaCreateManagedWidget("peaksOkButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(ok_button, XmNactivateCallback, ok_cb, peaks_sh);
    search_button = XtVaCreateManagedWidget("peaksSearchButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(search_button, XmNactivateCallback, search_cb, peaks_sh);
    cancel_button = XtVaCreateManagedWidget("peaksCancelButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(cancel_button, XmNactivateCallback, cancel_cb, peaks_sh);
    help_button = XtVaCreateManagedWidget("peaksHelpButton",
	xmPushButtonWidgetClass, action_form,
    NULL);
    AddCallback(help_button, XmNactivateCallback, help_cb, peaks_sh);
    XtManageChild(action_form);
    /*
     * Create the widget for saying how to mark peaks
     */
    marks_frame = XtVaCreateManagedWidget("peaksMarksFrame",
	xmFrameWidgetClass, peaks_form,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, action_frame,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    marks_form = XtVaCreateManagedWidget("peaksMarksForm",
	xmFormWidgetClass, marks_frame,
    NULL);
    peak_marks = XtVaCreateManagedWidget("peakMarksField",
	xmTextFieldWidgetClass, marks_form,
    NULL);
    AddCallback(peak_marks, XmNactivateCallback, search_cb, peaks_sh);
    label = XtVaCreateManagedWidget("peakMarksLabel",
	xmLabelWidgetClass, marks_form,
	XmNtopWidget, peak_marks,
	XmNleftWidget, peak_marks,
	XmNrightWidget, peak_marks,
	XmNbottomWidget, peak_marks,
    NULL);
    trough_marks = XtVaCreateManagedWidget("troughMarksField",
	xmTextFieldWidgetClass, marks_form,
    NULL);
    AddCallback(trough_marks, XmNactivateCallback, search_cb, peaks_sh);
    label = XtVaCreateManagedWidget("troughMarksLabel",
	xmLabelWidgetClass, marks_form,
	XmNtopWidget, trough_marks,
	XmNleftWidget, trough_marks,
	XmNrightWidget, trough_marks,
	XmNbottomWidget, trough_marks,
    NULL);
    begin_mark = XtVaCreateManagedWidget("beginMarkField",
	xmTextFieldWidgetClass, marks_form,
    NULL);
    AddCallback(begin_mark, XmNactivateCallback, search_cb, peaks_sh);
    label = XtVaCreateManagedWidget("beginMarkLabel",
	xmLabelWidgetClass, marks_form,
	XmNtopWidget, begin_mark,
	XmNleftWidget, begin_mark,
	XmNrightWidget, begin_mark,
	XmNbottomWidget, begin_mark,
    NULL);
    end_mark = XtVaCreateManagedWidget("endMarkField",
	xmTextFieldWidgetClass, marks_form,
    NULL);
    AddCallback(end_mark, XmNactivateCallback, search_cb, peaks_sh);
    label = XtVaCreateManagedWidget("endMarkLabel",
	xmLabelWidgetClass, marks_form,
	XmNtopWidget, end_mark,
	XmNleftWidget, end_mark,
	XmNrightWidget, end_mark,
	XmNbottomWidget, end_mark,
    NULL);
    XtManageChild(marks_form);
    /*
     * create the threshold form
     */
    thres_frame = XtVaCreateManagedWidget("peakThresholdsFrame",
	xmFrameWidgetClass, peaks_form,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, marks_frame,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    thres_form = XtVaCreateManagedWidget("peakThresholdsForm",
	xmFormWidgetClass, thres_frame,
    NULL);
    peak_half_width = XtVaCreateManagedWidget("peakHalfWidth",
	xmScaleWidgetClass, thres_form,
    NULL);
    trough_half_width = XtVaCreateManagedWidget("troughHalfWidth",
	xmScaleWidgetClass, thres_form,
    NULL);
    peak_height = XtVaCreateManagedWidget("peakHeight",
	xmScaleWidgetClass, thres_form,
    NULL);
    trough_depth = XtVaCreateManagedWidget("troughDepth",
	xmScaleWidgetClass, thres_form,
    NULL);
    set_scales(app_data.peak_half_width, app_data.trough_half_width,
	       peak_half_width, trough_half_width);
    set_scales(app_data.peak_height, app_data.trough_depth,
	       peak_height, trough_depth);
    arrow1 = XtVaCreateManagedWidget("vUpArrow",
	xmArrowButtonWidgetClass, thres_form,
	XmNarrowDirection, XmARROW_UP,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, peak_height,
	XmNrightWidget, peak_height,
	XmNtopWidget, peak_half_width,
	XmNbottomWidget, peak_half_width,
    NULL);
    AddCallback(arrow1, XmNactivateCallback, v_arrow_cb, 1);
    arrow2 = XtVaCreateManagedWidget("vDownArrow",
	xmArrowButtonWidgetClass, thres_form,
	XmNarrowDirection, XmARROW_DOWN,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, arrow1,
	XmNrightWidget, arrow1,
	XmNtopWidget, arrow1,
	XmNbottomWidget, arrow1,
    NULL);
    AddCallback(arrow2, XmNactivateCallback, v_arrow_cb, -1);
    label = XtVaCreateManagedWidget("vArrowLabel",
	xmLabelWidgetClass, thres_form,
	XmNleftWidget, arrow2,
	XmNrightWidget, arrow2,
	XmNtopWidget, arrow2,
	XmNbottomWidget, arrow2,
    NULL);
    arrow1 = XtVaCreateManagedWidget("tUpArrow",
	xmArrowButtonWidgetClass, thres_form,
	XmNarrowDirection, XmARROW_UP,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, peak_height,
	XmNrightWidget, peak_height,
	XmNtopWidget, peak_half_width,
	XmNbottomWidget, peak_half_width,
    NULL);
    AddCallback(arrow1, XmNactivateCallback, t_arrow_cb, 1);
    arrow2 = XtVaCreateManagedWidget("tDownArrow",
	xmArrowButtonWidgetClass, thres_form,
	XmNarrowDirection, XmARROW_DOWN,
	XmNmultiClick, XmMULTICLICK_KEEP,
	XmNleftWidget, arrow1,
	XmNrightWidget, arrow1,
	XmNtopWidget, arrow1,
	XmNbottomWidget, arrow1,
    NULL);
    AddCallback(arrow2, XmNactivateCallback, t_arrow_cb, -1);
    label = XtVaCreateManagedWidget("tArrowLabel",
	xmLabelWidgetClass, thres_form,
	XmNleftWidget, arrow2,
	XmNrightWidget, arrow2,
	XmNtopWidget, arrow2,
	XmNbottomWidget, arrow2,
    NULL);
    XtManageChild(thres_form);
    /*
     * done: manage the whole form and return
     */
    XtManageChild(peaks_form);
    return(peaks_sh);
}

static void
v_arrow_cb(arrow, dir, cbs)
Widget				arrow;
int				dir;
XmArrowButtonCallbackStruct	*cbs;
{
    bump_scales(dir, peak_height, trough_depth);
}

static void
t_arrow_cb(arrow, dir, cbs)
Widget				arrow;
int				dir;
XmArrowButtonCallbackStruct	*cbs;
{
    bump_scales(dir, peak_half_width, trough_half_width);
}

void
bump_scales(dir, scale1, scale2)
int		dir;
Widget		scale1;
Widget		scale2;
{
    int			i = 0;
    int			max;
    int			val;
    short		dp;
    int			fac;
    int			lfac;
    int			newmax;
    int			g;

    XtVaGetValues(scale1,
	XmNmaximum, &max,
	XmNvalue, &val,
	XmNdecimalPoints, &dp,
    NULL);
    for(
	fac=1, lfac = 0;
	fac < MAXLONG / 10 / scales[XtNumber(scales) - 1];
	fac *= 10, lfac++
    ) {
	for(i=0; i<XtNumber(scales); i++) {
	    if (max <= fac * scales[i]) break;
	}
	if (i < XtNumber(scales)) break;
    }
    if (i >= XtNumber(scales)) {
	if (dir >= 0) Bellret;
	lfac--;
	fac /= 10;
    }
    while((dp > 0) && (lfac > 0)) {
	dp--;
	lfac--;
	fac /= 10;
    }
    if (0 > dir) {
	if (--i < 0) {
	    if (lfac > 0) {
		lfac--;
		fac /= 10;
	    }
	    else {
		dp++;
	    }
	    i = XtNumber(scales) - 1;
	}
    }
    else {
	if (++i >= XtNumber(scales)) {
	    if (dp <= 0) {
		lfac++;
		fac *= 10;
	    }
	    else {
		dp--;
	    }
	    i = 0;
	}
    }	
    newmax = scales[i] * fac;
    g = gcf(newmax, max);
    val *= newmax/g;
    val /= max/g;
    XtVaSetValues(scale1,
	XmNmaximum, newmax,
	XmNscaleMultiple, newmax/10,
	XmNvalue, val,
	XmNdecimalPoints, dp,
    NULL);
    XtVaGetValues(scale2,
	XmNvalue, &val,
    NULL);
    val *= newmax/g;
    val /= max/g;
    XtVaSetValues(scale2,
	XmNmaximum, newmax,
	XmNscaleMultiple, newmax/10,
	XmNvalue, val,
	XmNdecimalPoints, dp,
    NULL);
}

void
set_scales(v1, v2, scale1, scale2)
double		v1;
double		v2;
Widget		scale1;
Widget		scale2;
{
    double		v = max(v1, v2);
    short		dp = 0;
    double		fac = 1.0;	/* = 10^-dp			*/
    int			i = 0;
    int			max;
    int			val;

    if (v <= 0.0) {
	Warning("scale parameters must be positive");
	v = 1.0;
    }
    while(v < fac * scales[i]) {
	if (i-- <= 0) {
	    i = XtNumber(scales) - 1;
	    dp++;
	    fac /= 10;
	}
    }
    while(v > fac * scales[i]) {
	if (++i >= XtNumber(scales)) {
	    i = 0;
	    dp--;
	    fac *= 10;
	}
    }
    max = scales[i];
    while(dp < 0) {
	dp++;
	if (max > MAXINT / 10) break;
	max *= 10;
    }
    if (dp <= 0) {
	dp = 0;
	fac = 1.0;
    }
    if (v1 / fac > max) val = max;
    else val = nint(v1 / fac);
    XtVaSetValues(scale1,
	XmNminimum, 0,
	XmNmaximum, max,
	XmNscaleMultiple, max/10,
	XmNvalue, val,
	XmNdecimalPoints, dp,
    NULL);
    if (v2 / fac > max) val = max;
    else val = nint(v2 / fac);
    XtVaSetValues(scale2,
	XmNminimum, 0,
	XmNmaximum, max,
	XmNscaleMultiple, max/10,
	XmNvalue, val,
	XmNdecimalPoints, dp,
    NULL);
}

/* gcf -- greatest common factor of two integers			*/
int
gcf(i1, i2)
int		i1;
int		i2;
{
    int			t;

    if (i1 < i2) {
	t = i1; i1 = i2; i2 = t;
    }
    while(0 != (t = i1 % i2)) {
	i1 = i2;
	i2 = t;
    }
    return(i2);
}

static void
ok_cb(button, peaks_sh, cbs)
Widget				button;
Widget				peaks_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("ok_cb (peaks.c)\n");
    do_search();
    XtPopdown(peaks_sh);
}

static void
search_cb(button, peaks_sh, cbs)
Widget				button;
Widget				peaks_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("search_cb (peaks.c)\n");
    do_search();
}

static void
cancel_cb(button, peaks_sh, cbs)
Widget				button;
Widget				peaks_sh;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("cancel_cb (peaks.c)\n");
    XtPopdown(peaks_sh);
}

static void
help_cb(button, peaks_sh, cbs)
Widget				button;
Widget				peaks_sh;
XmPushButtonCallbackStruct	*cbs;
{
    help_window("peaks.html");
}

#define	get_scale_val(sc)\
{						\
    int			val;			\
    short		dp;			\
						\
    XtVaGetValues(sc,				\
	XmNvalue, &val,				\
	XmNdecimalPoints, &dp,			\
    NULL);					\
    app_data.sc = (double) val;			\
    app_data.sc /= exp10((double) dp);		\
}

static void
do_search()
{
    int			begin;
    TIME		tmin;
    TIME		tmax;
    TIME		tb;
    int			end;
    TIME		te;
    String		s;
    MARK		*mp;
    PeakParams		pp;

    /*
     * get all the parameters from the dialog widgets
     */
    get_scale_val(peak_height);
    get_scale_val(trough_depth);
    get_scale_val(peak_half_width);
    get_scale_val(trough_half_width);
    s = XmTextFieldGetString(begin_mark);
    begin = *s;
    Free(s);
    s = XmTextFieldGetString(end_mark);
    end = *s;
    Free(s);
    pp.ph = Map_v_to_VOLTAGE_c(app_data.peak_height);
    pp.phw = Map_t_to_TIME_c(app_data.peak_half_width);
    pp.pm = XmTextFieldGetString(peak_marks);
    pp.htmp = min(strlen(pp.pm), HTM_ALL);
    pp.td = Map_v_to_VOLTAGE_c(app_data.trough_depth);
    pp.thw = Map_t_to_TIME_c(app_data.trough_half_width);
    pp.tm = XmTextFieldGetString(trough_marks);
    pp.htmt = min(strlen(pp.tm), HTM_ALL);
    /*
     * work out intervals and search each one
     */
    if (IS_CURSOR && IS_CURSOR_END) {
	tmin = CURSOR;
	tmax = CURSOR_END;
    }
    else {
	tmin = TMIN;
	tmax = TMAX;
    }
    mp = next_mark(Map_TIME_to_MARK(tmin), begin);
    if ((NULL == mp) || (mp->t >= tmax)) tb = tmin;
    else tb = Map_MARK_to_TIME(mp->t);
    for(;;) {
	mp = next_mark(Map_TIME_to_MARK(tb+1), end);
	if ((NULL == mp) || (mp->t >= tmax)) te = tmax;
	else te = Map_MARK_to_TIME(mp->t);
	find_peaks(tb, te, &pp);
	mp = next_mark(Map_TIME_to_MARK(te), begin);
	if ((NULL == mp) || (mp->t >= tmax)) break;
	else tb = Map_MARK_to_TIME(mp->t);
    }	
    /*
     * clean up and finish
     */
    Free(pp.pm);
    Free(pp.tm);
    replot_whole();
}

static void
mark_point(t, val, comment)
TIME		t;
int		val;
String		comment;
{
    MARK		*mp;

    mp = find_mark(Map_TIME_to_MARK(t), val);
    Nfree(mp->comment);
    mp->comment = XtNewString(comment);
}

static void
mark_peak(peak, pp)
Peak		*peak;
PeakParams	*pp;
{
    int			i = 0;
    char		lbuf[LLEN];
    double		sum = 0.0;
    FILTER_VAL		h;
    TIME		t;
    MARK		*mp;

    for(t = peak->tl+1; t < peak->tr; t++) sum += Vf(t);
    sum -= (peak->tr - peak->tl - 1) * (peak->vl + peak->vr) / 2.0;
    sum *= VOLTAGE_to_v_sf * TIME_to_t_sf;
    h = peak->vm - (peak->vl + peak->vr) / 2.0;
    switch(pp->htmp) {
    case HTM_NONE:
	return;

    case HTM_PEAK:
	sprintf(lbuf, app_data.mark_comment_format_3,
		Map_TIME_to_t_c(peak->tm), Map_VOLTAGE_to_v_c(peak->vm),
		sum, h * VOLTAGE_to_v_sf);
	mark_point(peak->tm, (int) pp->pm[i], lbuf); i++;
	return;

    case HTM_BOUND:
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tl), Map_VOLTAGE_to_v_c(peak->vl));
	mark_point(peak->tl, (int) pp->pm[i], lbuf); i++;
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tr), Map_VOLTAGE_to_v_c(peak->vr));
	mark_point(peak->tr, (int) pp->pm[i], lbuf); i++;
	return;

    case HTM_ALL:
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tl), Map_VOLTAGE_to_v_c(peak->vl));
	mark_point(peak->tl, (int) pp->pm[i], lbuf); i++;
	sprintf(lbuf, app_data.mark_comment_format_3,
		Map_TIME_to_t_c(peak->tm), Map_VOLTAGE_to_v_c(peak->vm),
		sum, h * VOLTAGE_to_v_sf);
	mark_point(peak->tm, (int) pp->pm[i], lbuf); i++;
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tr), Map_VOLTAGE_to_v_c(peak->vr));
	mark_point(peak->tr, (int) pp->pm[i], lbuf); i++;
	return;
    }
}

static void
mark_trough(peak, pp)
Peak		*peak;
PeakParams	*pp;
{
    int			i = 0;
    char		lbuf[LLEN];
    double		sum = 0.0;
    FILTER_VAL		h;
    TIME		t;
    MARK		*mp;

    for(t = peak->tl+1; t < peak->tr; t++) sum += Vf(t);
    sum -= (peak->tr - peak->tl - 1) * (peak->vl + peak->vr) / 2.0;
    sum *= -VOLTAGE_to_v_sf * TIME_to_t_sf;
    h = (peak->vl + peak->vr) / 2.0 - peak->vm;
    switch(pp->htmt) {
    case HTM_NONE:
	return;

    case HTM_PEAK:
	sprintf(lbuf, app_data.mark_comment_format_3,
		Map_TIME_to_t_c(peak->tm), Map_VOLTAGE_to_v_c(peak->vm),
		sum, h * VOLTAGE_to_v_sf);
	mark_point(peak->tm, (int) pp->tm[i], lbuf); i++;
	return;

    case HTM_BOUND:
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tl), Map_VOLTAGE_to_v_c(peak->vl));
	mark_point(peak->tl, (int) pp->tm[i], lbuf); i++;
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tr), Map_VOLTAGE_to_v_c(peak->vr));
	mark_point(peak->tr, (int) pp->tm[i], lbuf); i++;
	return;

    case HTM_ALL:
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tl), Map_VOLTAGE_to_v_c(peak->vl));
	mark_point(peak->tl, (int) pp->tm[i], lbuf);  i++;
	sprintf(lbuf, app_data.mark_comment_format_3,
		Map_TIME_to_t_c(peak->tm), Map_VOLTAGE_to_v_c(peak->vm),
		sum, h * VOLTAGE_to_v_sf);
	mark_point(peak->tm, (int) pp->tm[i], lbuf); i++;
	sprintf(lbuf, app_data.mark_comment_format_2,
		Map_TIME_to_t_c(peak->tr), Map_VOLTAGE_to_v_c(peak->vr));
	mark_point(peak->tr, (int) pp->tm[i], lbuf); i++;
	return;
    }
}

static INLINE TIME
peaktest(tm, pp, peak)
TIME		tm;
PeakParams	*pp;
Peak		*peak;
{
    TIME		t;
    FILTER_VAL		val;
    FILTER_VAL		p = Vf(tm);	/* peak value			*/
    FILTER_VAL		ml, mr;		/* min on left and right	*/
    TIME		tml, tmr;	/* times at min			*/

    /*
     * is rise fast enough?
     */
    ml = p; tml = tm;
    for(t = tm-1; t >= tm - pp->phw; t--) {
	if (
	    (IS_INVALID_FILTER_VAL(val = Vf(t))) ||
	    (val > ml + pp->ph)
	) break;
	if (val < ml) {
	    ml = val; tml = t;
	}
    }
    if (pp->ph > p - ml) return(EOF);
    /*
     * is fall fast enough?
     */
    mr = p; tmr = tm;
    for(t = tm+1; t <= tm + pp->phw; t++) {
	if (
	    (IS_INVALID_FILTER_VAL(val = Vf(t))) ||
	    (val > mr + pp->ph)
	) break;
	if (val < mr) {
	    mr = val; tmr = t;
	}
    }
    if (pp->ph > p - mr) return(EOF);
    /*
     * all conditions satisfied: return it
     */
    peak->tl = tml;
    peak->tm = tm;
    peak->tr = tmr;
    peak->vl = ml;
    peak->vm = p;
    peak->vr = mr;
    return(tmr);
}

static INLINE TIME
troughtest(tm, pp, peak)
TIME		tm;
PeakParams	*pp;
Peak		*peak;
{
    TIME		t;
    FILTER_VAL		val;
    FILTER_VAL		p = Vf(tm);	/* peak value			*/
    FILTER_VAL		ml, mr;		/* max on left and right	*/
    TIME		tml, tmr;	/* times at max			*/

    /*
     * is fall fast enough?
     */
    ml = p; tml = tm;
    for(t = tm-1; t >= tm - pp->thw; t--) {
	if (
	    (IS_INVALID_FILTER_VAL(val = Vf(t))) ||
	    (val < ml - pp->td)
	) break;
	if (val > ml) {
	    ml = val; tml = t;
	}
    }
    if (pp->td > ml - p) return(EOF);
    /*
     * is fall fast enough?
     */
    mr = p; tmr = tm;
    for(t = tm+1; t <= tm + pp->thw; t++) {
	if (
	    (IS_INVALID_FILTER_VAL(val = Vf(t))) ||
	    (val < mr - pp->td)
	) break;
	if (val > mr) {
	    mr = val; tmr = t;
	}
    }
    if (pp->td > mr - p) return(EOF);
    /*
     * all conditions satisfied: mark it
     */
    peak->tl = tml;
    peak->tm = tm;
    peak->tr = tmr;
    peak->vl = ml;
    peak->vm = p;
    peak->vr = mr;
    return(tmr);
}

#define	PTBLK	16
#define	add(p)\
(									\
    (									\
	(npt < ptlist_size) ||						\
	(								\
	    ptlist_size += PTBLK,					\
	    ptlist = (Peak *) Realloc(ptlist,				\
				      ptlist_size * sizeof(Peak))	\
        )								\
    ),									\
    (bcopy((p), ptlist + npt, sizeof(Peak))),				\
    npt++								\
)

static void
find_peaks(tb, te, pp)
TIME		tb;
TIME		te;
PeakParams	*pp;
{
    int			i;
    register TIME	t;
    TIME		tr;
    register FILTER_VAL	val;
    register FILTER_VAL	max, min;
    register TIME	tmax, tmin;
    FILTER_VAL		ptt = min(pp->ph, pp->td);
    Peak		peak;
    Peak		*ptlist;
    int			ptlist_size;
    int			npt;
    Bool		is_peak;
    Bool		starts_with_peak;
    FILTER_VAL		psum, tsum;
    int			np, nt;

    ptlist = (Peak *) XtMalloc(PTBLK * sizeof(Peak));
    ptlist_size = PTBLK;
    npt = 0;
    for(;;) {
	/*
	 * find first peak or trough
	 */
	if (npt > 0) {			/* have one from last time	*/
	    if (ptlist[npt-1].vm < ptlist[npt-1].vr) {
		tmin = ptlist[npt-1].tm;
		min = ptlist[npt-1].vm;
		tmax = ptlist[npt-1].tr;
		max = ptlist[npt-1].vr;
	    }
	    else {
		tmin = ptlist[npt-1].tr;
		min = ptlist[npt-1].vr;
		tmax = ptlist[npt-1].tm;
		max = ptlist[npt-1].vm;
	    }
	    tb = ptlist[npt-1].tr;
	}
	else {
	    tmax = tmin = tb;
	    max = min = Vf(tb);
	}
	for(
	    t = tb+1;
	    (t < te) && (max-min < ptt);
	    t++
	) {
	    val = Vf(t);
	    if (val < min) {
		min = val; tmin = t;
	    }
	    if (val > max) {
		max = val; tmax = t;
	    }
	}
	if (t >= te) break;

	/*
	 * The business of this loop is to find a series of
	 * overlapping peaks and troughs.  It continues until a peak
	 * or trough fails the criteria in pp.  When it exits, ptlist
	 * is a series of npt peaks.
	 */
	is_peak = (tmin < tmax);
	starts_with_peak = (npt % 2 == 0) ? is_peak : !is_peak;
	for(;;) {
	    if (is_peak) {
		/*
		 * max comes after min: look for a peak
		 */
		min = max; tmin = tmax;
		for(
		    t=tmax+1;
		    (t < te) && (max-min < ptt);
		    t++
		) {
		    val = Vf(t);
		    if (val > max) {
			min = max = val; tmin = tmax = t;
		    }
		    if (val < min) {
			min = val; tmin = t;
		    }
		}
		if (t >= te) {
		    t = EOF;
		    tb = te - 1;
		    break;
		}
		tb = tmax;
		t = tmin = peaktest(tmax, pp, &peak);
		if (EOF == tmin) break;
		if ((npt > 0) && (peak.tl >= ptlist[npt-1].tr)) break;
		min = peak.vr;
		add(&peak);
		is_peak = FALSE;
	    }

	    else {
		/*
		 * min comes after max: look for a trough
		 */
		max = min; tmax = tmin;
		for(
		    t = tmax+1;
		    (t < te) && (max-min < ptt);
		    t++
		) {
		    val = Vf(t);
		    if (val < min) {
			max = min = val; tmax = tmin = t;
		    }
		    if (val > max) {
			max = val; tmax = t;
		    }
		}
		if (t >= te) {
		    t = EOF;
		    tb = te - 1;
		    break;
		}
		tb = tmin;
		t = tmax = troughtest(tmin, pp, &peak);
		if (EOF == tmax) break;
		if ((npt > 0) && (peak.tl >= ptlist[npt-1].tr)) break;
		max = peak.vr;
		add(&peak);
		is_peak = TRUE;
	    }
	}
	/*
	 * Now we decide whether we're going to believe in the peaks
	 * or the troughs.
	 */
	if (npt == 0) {
	    continue;
	}
	else if (npt == 1) {
	    if (starts_with_peak)
		mark_peak(ptlist, pp);
	    else
		mark_trough(ptlist, pp);
	    npt = 0;
	    if (EOF != t) add(&peak);
	    continue;
	}
	else {
	    psum = tsum = 0.0;
	    np = nt = 0;
	    is_peak = starts_with_peak;
	    for(i=0; i<npt; i++) {
		if (is_peak) {
		    psum += 2.0 * ptlist[i].vm;
		    psum -= Vf((TIME) (ptlist[i].tl + ptlist[i].tm)/2);
		    psum -= Vf((TIME) (ptlist[i].tr + ptlist[i].tm)/2);
		    np++;
		    is_peak = FALSE;
		}
		else {
		    tsum -= 2.0 * ptlist[i].vm;
		    tsum += Vf((TIME) (ptlist[i].tl + ptlist[i].tm)/2);
		    tsum += Vf((TIME) (ptlist[i].tr + ptlist[i].tm)/2);
		    nt++;
		    is_peak = TRUE;
		}
	    }
					/* peaks bigger			*/
	    if (psum / np > tsum / nt) {
		i = (starts_with_peak ? 0 : 1);
		for(; i < npt; i += 2) {
		    mark_peak(&ptlist[i], pp);
		}
	    }
	    else {			/* troughs bigger		*/
		i = (starts_with_peak ? 1 : 0);
		for(; i < npt; i += 2) {
		    mark_trough(&ptlist[i], pp);
		}
	    }
	    npt = 0;
	    if (EOF != t) add(&peak);
	    continue;
	}
    }
    Free(ptlist);
}
