/* xdp.c -- routines for reading and writing xdp files			*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"
#include <sys/stat.h>

#define	XDP_MAGIC_FORMAT	"! xdp-%s\n"
#define	XDP_MAGIC		"! xdp-"

/* append a char to a line, allocating new space if necessary		*/
#define	addc(l, s, len, n, c)\
(									\
    (len++ < n) ? (*s++ = c) : (len--, faddc(&l, &s, &len, &n, c))	\
)

#ifdef	__STDC__
static	String	get_application_resource(String);
static	Bool	read_xdp_stream(FILE *);
static	String	read_xdp_line(FILE *, int *);
static	int	escape(FILE *, int *);
static	Bool	parse_xdp_line(String, String *, String *, String *);
static	void	faddc(String *, String *, int *, int *, int);
#else	/* __STDC__ */
static	String	get_application_resource();
static	Bool	read_xdp_stream();
static	String	read_xdp_line()
static	int	escape();
static	Bool	parse_xdp_line();
static	void	faddc();
#endif	/* __STDC__ */

/* write_xdp -- write resources to an xdp file
 *
 * String	file_name;
 * String	save_list;
 * if (write_xdp(file_name, save_list)) error...
 *
 * save_list is the name of a whitespace-separated list of resources.
 * Their values are fetched and written to the indicated file in X
 * resources format.
 */
Bool
write_xdp(fname, list)
String		fname;
String		list;
{
    FILE		*xdp;
    Bool		ret;

    if (NULL == (xdp = fopen(fname, "w"))) return(TRUE);
    ret = write_xdp_stream(xdp, list);
    fclose(xdp);
    return(ret);
}

Bool
write_xdp_stream(xdp, list)
FILE		*xdp;
String		list;
{
    char		lbuf[LLEN];
    String		warn = XtNewString("");
    int			ncwarn = 0;
    String		s;
    int			len;
    String		t = XtNewString("");
    int			tlen = 0;
    String		res = XtNewString("");
    String		wid = XtNewString("");
    String		u;
    String		v;
    Widget		widget;
    String		val;
    Bool		ret = FALSE;

    fprintf(xdp, XDP_MAGIC_FORMAT, VERSION);
    for(
	s = list;
	'\0' != s[strspn(s, WHITE)];
	s += len
    ) {
	s += strspn(s, WHITE);
	len = strcspn(s, WHITE);
	if (len > tlen) {
	    t = (String) XtRealloc(t, len+1);
	    res = (String) XtRealloc(res, len+1);
	    wid = (String) XtRealloc(wid, len+1);
	    tlen = len;
	}
	strncpy(t, s, len);
	t[len] = '\0';			/* fully qualified resource	*/
	u = STRRCHR(t, '.');
	v = STRRCHR(t, '*');
	if ((NULL != u) || (NULL != v)) {
	    if (v > u) u = v;
	    strncpy(wid, t, u-t);
	    wid[u-t] = '\0';		/* widget name			*/
	    strcpy(res, u+1);		/* resource name		*/
	}
	else {
	    strcpy(wid, "*");
	    strcpy(res, u);
	}
	if (strlen(wid) == strspn(wid, "*.")) {
	    widget = toplevel;
	}
	else {
	    if (NO_WIDGET == (widget = XtNameToWidget(toplevel, wid))) {
		sprintf(lbuf, "unknown widget %s in saveList\n", wid);
		warn = append_string(warn, lbuf, &ncwarn);
		ret = TRUE;
		continue;
	    }
	}
	if (
	    (widget != toplevel) ||
	    (NULL == (val = get_application_resource(res)))
	) {
	    val = NULL;
	    XtVaGetValues(widget,
		XtVaTypedArg, res, XtRString,
		&val, sizeof(val),
	    NULL);
	    if (NULL == val) {
		sprintf(lbuf, "unknown resource %s.%s in saveList\n",
			wid, res);
		warn = append_string(warn, lbuf, &ncwarn);
		ret = TRUE;
		continue;
	    }
	}
	put_string(xdp, t);
	putc(':', xdp);
	putc('\t', xdp);
	put_string(xdp, val);
	putc('\n', xdp);
    }
    Free(t);
    Free(res);
    Free(wid);
    if (0 < ncwarn) PU_error(warn, NULL);
    Free(warn);
    return(ret);
}

/* get_application_resource -- return app resource value as String	*/
static String
get_application_resource(res)
String		res;
{
    XtResource		*appres = resources;
    int			i;
    XrmValue		from;
    XrmValue		to;
    String		type;
    Bool		ret;

    for(i=0; i<num_app_resources; i++) {
	if (
	    (0 == strcmp(res, appres[i].resource_name)) ||
	    (0 == strcmp(res, appres[i].resource_class))
	) break;
    }
    if (i >= num_app_resources) return(NULL);
    from.addr = (caddr_t) &app_data;
    from.addr += appres[i].resource_offset;
    from.size = appres[i].resource_size;
    type = appres[i].resource_type;
    if (0 == strcmp(type, XtRString)) {
	return(*((String *) from.addr));
    }
    else {
	to.addr = NULL;
	ret = XtConvertAndStore(toplevel, type, &from, XtRString, &to);
	return(ret ? (String) to.addr : NULL);
    }
}

/* set_application_resource -- set app resource value from String	*/
static Bool
set_application_resource(res, val)
String		res;
String		val;
{
    XtResource		*appres = resources;
    int			i;
    XrmValue		from;
    XrmValue		to;
    String		type;
    Bool		ret;

    for(i=0; i<num_app_resources; i++) {
	if (
	    (0 == strcmp(res, appres[i].resource_name)) ||
	    (0 == strcmp(res, appres[i].resource_class))
	) break;
    }
    if (i >= num_app_resources) return(TRUE);
    to.addr = (caddr_t) &app_data;
    to.addr += appres[i].resource_offset;
    to.size = appres[i].resource_size;
    type = appres[i].resource_type;
    if (0 == strcmp(type, XtRString)) {
	if (free_res[i]) Nfree(*((String *) to.addr));
	*((String *) to.addr) = XtNewString(val);
	free_res[i] = True;
	ret = FALSE;
    }
    else {
	from.addr = val;
	from.size = strlen(val)+1;
	ret = !XtConvertAndStore(toplevel, XtRString, &from, type, &to);
    }
    return(ret);
}

void
put_string(xdp, s)
FILE		*xdp;
String		s;
{
    int c;

    while('\0' != (c = *s++)) {
	if ('\n' == c) fprintf(xdp, "\\n\\\n");
	else if (isprint(c)) putc(c, xdp);
	else fprintf(xdp, "\\%03o", c);
    }
}

/* xdp_get_datafile -- get datafile name from an xdp file
 * xdp_get_values -- get widget values from an xdp file
 *
 * String	xdp_file_name;
 * String	datafile_name;
 * datafile_name = xdp_get_datafile(xdp_file_name);
 * if (NULL == datafile_name) error...
 *
 *  ...open datafile...
 *
 * if (xdp_get_values(xdp_file_name) error...
 *
 * There are two steps to reading an xdp file.  First, call
 * xdp_get_datafile to get the name of the datafile refered to by the
 * xdp file.  It returns a pointer to the datafile name, which should
 * eventually be freed with XtFree.  If the file isn't an xdp file or
 * if the datafile can't be found, xdp_get_datafile returns NULL.
 *
 * The second step comes after the datafile has been loaded.  Call
 * xdp_get_vales with the same xdp file name to set widget and
 * application resource values.  xdp_get_values should only be called
 * after a successful call to xdp_get_datafile, since it doesn't check
 * that the file is an xdp file.  If it encounters an error it pops up
 * a warning message and returns TRUE; otherwise it returns FALSE.
 */
String
xdp_get_datafile(fname)
String		fname;
{
    FILE		*xdp;
    Bool		ret;
    char		lbuf[LLEN];
    XrmDatabase		db;
    String		class;
    String		name;
    String		type;
    XrmValue		where;
    struct stat		stat_buf;

    if (NULL == (xdp = fopen(fname, "r"))) return(NULL);
    ret = (1 != fread(lbuf, strlen(XDP_MAGIC), 1, xdp)) ||
	  (0 != strncmp(lbuf, XDP_MAGIC, strlen(XDP_MAGIC)));
    fclose(xdp);
    if (ret) return(NULL);
    if (NULL == (db = XrmGetFileDatabase(fname))) return(NULL);
    sprintf(lbuf, "%s.%s", CLASS, "DataFile");
    class = XtNewString(lbuf);
    sprintf(lbuf, "%s.%s", XtName(toplevel), "dataFile");
    name = XtNewString(lbuf);
    ret = (!XrmGetResource(db, name, class, &type, &where)) ||
	  (0 != strcmp(XtRString, type));
    Free(name);
    Free(class);
    if (!ret) strncpy(lbuf, (String) where.addr, LLEN);
    XrmDestroyDatabase(db);
    if (ret) return(NULL);
    for(
	name = lbuf;
	((String) NULL + 1) != name;
	name = 1 + STRCHR(name, DIRSEP)
    ) {
	if ((EOF != stat(name, &stat_buf)) && S_ISREG(stat_buf.st_mode))
	    break;
    }
    if (((String) NULL + 1) == name) return(XtNewString(lbuf));
    return(XtNewString(name));
}

Bool
xdp_get_values(fname)
String		fname;
{
    FILE		*xdp;
    int			ret;

    if (NULL == (xdp = fopen(fname, "r"))) return(TRUE);
    ret = read_xdp_stream(xdp);
    fclose(xdp);
    return(ret);
}

static Bool
read_xdp_stream(xdp)
FILE		*xdp;
{
    char		lbuf[LLEN];
    String		warn = XtNewString("");
    int			ncwarn = 0;
    int			lineno = 0;
    String		line;
    String		wid;
    String		res;
    String		val;
    Widget		widget;
    Bool		ret = FALSE;

    while(NULL != (line = read_xdp_line(xdp, &lineno))) {
	if (parse_xdp_line(line, &wid, &res, &val)) {
	    sprintf(lbuf, "parse error in xdp file near line %d\n", lineno);
	    warn = append_string(warn, lbuf, &ncwarn);
	    ret = TRUE;
	    continue;
	}
	Free(line);
	if (NULL == wid) continue;
	if (strlen(wid) == strspn(wid, "*.")) {
	    widget = toplevel;
	}
	else {
	    if (NO_WIDGET == (widget = XtNameToWidget(toplevel, wid))) {
		sprintf(lbuf, "unknown widget %s in xdp file near line %d\n",
			wid, lineno);
		warn = append_string(warn, lbuf, &ncwarn);
		ret = TRUE;
		continue;
	    }
	}
	if (
	    (widget != toplevel) ||
	    (set_application_resource(res, val))
	) {
	    XtVaSetValues(widget,
		XtVaTypedArg, res, XtRString,
		val, strlen(val)+1,
	    NULL);
	}
	Free(wid);
	Free(res);
	Free(val);
    }
    if (0 < ncwarn) PU_error(warn, NULL);
    Free(warn);
    return(ret);
}

static String
read_xdp_line(xdp, lineno)
FILE		*xdp;
int		*lineno;
{
    String		line = XtMalloc(LLEN);
    String		s;
    int			len = 0;
    int			nc = LLEN;
    int			c;
    Bool		comment = FALSE;

    s = line;
    while((EOF != (c = getc(xdp))) && ('\n' != c)) {
	if ('!' == c) comment = TRUE;
	if ('\\' == c) c = escape(xdp, lineno);
	if ((EOF != c) && !comment) addc(line, s, len, nc, c);
    }
    addc(line, s, len, nc, '\0');
    if ((EOF == c) && ('\0' == line[0])) {
	Free(line);
	line = NULL;
    }
    else {
	line = XtRealloc(line, len);
	if ('\n' == c) (*lineno)++;
    }
    return(line);
}

static int
escape(xdp, lineno)
FILE		*xdp;
int		*lineno;
{
    int			c;
    int			d;

    switch(c = getc(xdp)) {
    case 'x':				/* hex escape			*/
	c = 0;
	while(isxdigit(d = getc(xdp))) {
	    c *= 0x10;
	    c += isdigit(d) ? (d - '0') : (tolower(d) - 'a' + 0xa);
	}
	ungetc(d, xdp);
	break;

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
	ungetc(c, xdp);
	c = 0;
	while(('0' <= (d = getc(xdp))) && ('7' >= d)) {
	    c *= 010;
	    c += isdigit(d) ? (d - '0') : (tolower(d) - 'a' + 0xa);
	}
	ungetc(d, xdp);
	break;

    case 'a':
	c = '\a';
	break;

    case 'b':
	c = '\b';
	break;

    case 'f':
	c = '\f';
	break;

    case 'n':
	c = '\n';
	break;

    case 'r':
	c = '\r';
	break;

    case 't':
	c = '\t';
	break;

    case 'v':
	c = '\v';
	break;

    case '\n':
	c = EOF;
	(*lineno)++;
	break;

    case EOF:
    default:
	break;
    }
    return(c);
}

static Bool
parse_xdp_line(line, wid, res, val)
String		line;
String		*wid;
String		*res;
String		*val;
{
    String		colon;
    String		s;
    String		t;

    if (line[strspn(line, WHITE)] == '\0') {
	*wid = *res = *val = NULL;	/* blank (or comment) line	*/
	return(FALSE);
    }
    if (NULL == (colon = STRCHR(line, ':'))) return(TRUE);
    *val = XtNewString(colon + 1 + strspn(colon+1, WHITE));
    *wid = (String) XtMalloc(colon - line + 1);
    strncpy(*wid, line, colon - line);
    (*wid)[colon-line] = '\0';
    s = STRRCHR(*wid, '.');
    t = STRRCHR(*wid, '*');
    if ((NULL != s) || (NULL != t)) {
	if (t > s) s = t;
	*s++ = '\0';
	*res = XtNewString(s);
    }
    else {
	*res = *wid;
	*wid = XtNewString("*");
    }
    return(FALSE);
}

/* append_string -- append to a string, using XtRealloc			*/
String
append_string(base, cat, len)
String		base;
String		cat;
int		*len;
{
    String		s;

    s = (String) XtRealloc(base, strlen(cat) + *len + 1);
    strcpy(s + *len, cat);
    *len += strlen(cat);
    return(s);
}

void
faddc(l, s, len, n, c)
String		*l;
String		*s;
int		*len;
int		*n;
int		c;
{
    int			d = (*s) - (*l);

    if (*len < *n) {
	*(*s)++ = c;
	(*len)++;
	return;
    }
    *l = (String) XtRealloc(*l, *n + LLEN);
    *n += LLEN;
    *s = *l + d;
    *(*s)++ = c;
    (*len)++;
}
