/* xdatplot.h -- data formats for xdatplot				*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#ifndef	_XDATPLOT_H
#define	_XDATPLOT_H

#define	VERSION	"0.1"

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <signal.h>

#include <Xm/Xm.h>
#include <X11/keysym.h>
#include <X11/StringDefs.h>
#include <Xm/MainW.h>
#include <Xm/ScrolledW.h>
#include <Xm/ScrollBar.h>
#include <Xm/DrawingA.h>
#include <Xm/PushB.h>
#include <Xm/Form.h>
#include <Xm/DialogS.h>
#include <Xm/Frame.h>
#include <Xm/ToggleB.h>
#include <Xm/Label.h>
#include <Xm/FileSB.h>
#include <Xm/SelectioB.h>
#include <Xm/MessageB.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/Protocols.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/ArrowB.h>

#include "config.h"
#include "Vt.h"
#include "conversion.h"
#include "filter.h"

#define	CLASS	"XDatplot"
/*
 * Make some X calls a little easier:
 */
#define	Error	error			/* until I write an X error	*/
#define	Warning(msg)	XtAppWarning(app, msg)
#define	Free(p)	XtFree((XtPointer) (p))
#define	Realloc(p, s)	XtRealloc((XtPointer) (p), (s))
#define	AddCallback(w, n, c, d) \
    XtAddCallback((w), (n), (XtCallbackProc) (c), (XtPointer) (d))
#define	AppAddInput(a, s, c, p, d) \
    XtAppAddInput((a), (s), (XtPointer) (c), (XtInputCallbackProc) (p),	\
		  (XtPointer) (d))
#define	SetTypeConverter(f, t, c, a, n, ct, d) \
    XtSetTypeConverter((f), (t), (XtTypeConverter) (c),			\
		       (XtConvertArgList) (a), (n), (XtCacheType) (ct),	\
		       (XtDestructor) (d))
#define	GetSize(widget, w, h) \
    XtVaGetValues((widget), XmNwidth, &(w), XmNheight, &(h), NULL)
#define	SetSize(widget, w, h) \
    XtVaSetValues((widget), XmNwidth, (Dimension) (w), XmNheight,	\
		  (Dimension) (h), NULL)

#define	NO_WIDGET ((Widget) NULL)
#define	NO_WORKID	((XtWorkProcId) NULL)
#define	NO_INPUTID	((XtInputId) NULL)

#define	ZOOMFAC	(1.5)			/* zoom factors			*/
#define	BIGZOOMFAC (5.0)
#define	SBINC	(25)			/* scrollbar increment		*/
#define	Bell	XBell(XtDisplay(toplevel), 0) /* convenience macro	*/
#define	Bellret	{Bell; return;}
#define	Nfree(p)	((NULL == (p)) || (XtFree((XtPointer) p), (p) = NULL))
#define	DIRSEP	('/')			/* separates directories	*/
#define	EXTSEP	('.')			/* separates extension		*/
#define	WHITE	" \t\n"			/* whitespace chars		*/
#define	LWHITE	" \t"			/* same w/o newline		*/

#ifdef	unix				/* default buffer size		*/
# define	V_BUFSIZ	0x10000
#else	/* unix */
# define	V_BUFSIZ	0x1000
#endif	/* unix */

#ifdef	HAS_FUNC_ATTR
/*
 * In the GNU C dialect, these decls will allow common subexpression
 * elimination on the results of calls to these functions.
 */
int	isnan(double)	__attribute__ ((const));
double	quiet_nan(long)	__attribute__ ((const));
double	signaling_nan(long) __attribute__ ((const));
int	nint(double)	__attribute__ ((const));
double	floor(double)	__attribute__ ((const));
double	ceil(double)	__attribute__ ((const));
double	sqrt(double)	__attribute__ ((const));
double	exp(double)	__attribute__ ((const));
double	expm1(double)	__attribute__ ((const));
double	exp10(double)	__attribute__ ((const));
double	log(double)	__attribute__ ((const));
double	log10(double)	__attribute__ ((const));
double	pow(double, double) __attribute__ ((const));
#endif	/* HAS_FUNC_ATTR */

#ifdef	min
#undef	min
#endif
#define	min(x, y)	(((x) < (y)) ? (x) : (y))
#ifdef	max
#undef	max
#endif
#define	max(x, y)	(((x) > (y)) ? (x) : (y))

#define	Vf(t)	F(C_FILTER, ((TIME) t))

typedef	struct TICS {			/* specify where tic marks go	*/
    int		tcmin;			/* min number tic marks		*/
    double	lo, hi;			/* interval to be plotted	*/
    int		n;			/* actual number tics		*/
    int		first;			/* first tic mark / space	*/
    double	space;			/* spacing of tics		*/
}		TICS;

typedef	struct MARK {
    TIME	t;			/* time				*/
    int		val;			/* mark value			*/
    String	comment;		/* comment			*/
}		MARK;

typedef	struct MARKLIST {
    int		nmalloc;		/* # marks allocated		*/
    int		nmarks;			/* # actually present		*/
    MARK	*marks;			/* the marks themselves		*/
}		MARKLIST;

typedef	enum ANSWER {
    YES = 1,
    NO = 0,
    DUNNO = -1,
}		ANSWER;

typedef struct AppData {
    Bool	debug;
    String	xdp_file;		/* XDP file name		*/
    String	save_list;		/* resources to save in xdp	*/
    String	data_file;		/* name of actual data file	*/
    String	file_convert;		/* file conversion filters	*/
    String	uncompress;		/* uncompression filters	*/
    int		xd_bufsiz;
    Cardinal	print_limit;		/* max number points to print	*/
    String	print_cmd;		/* for printing plots		*/
    String	help_print_cmd;		/* for printing help		*/
    String	help_viewer;		/* for reading help files	*/
    String	help_URL;		/* where to find them		*/
    int		help_pid;		/* Mosaic process pid		*/
    String	help_tmp;		/* help command temp file	*/
    int		help_wait;		/* time viewer needs to start up*/
    double	t_time;			/* time to be centered on open	*/
    String	t_units;		/* units of time		*/
    int		t_tics;			/* minimum t-axis tics		*/
    int		t_ticlen;		/* tic length in pixels		*/
    double	t_multiplier;		/* file conversion factor	*/
    double	t_gain;			/* time gain			*/
    double	t_width;		/* interval to display on open	*/
    String	v_units;		/* units of voltage		*/
    int		v_tics;			/* minimum v-axis tics		*/
    int		v_ticlen;		/* tic length in pixels		*/
    double	v_multiplier;		/* file conversion factor	*/
    double     	v_gain;			/* voltage gain			*/
    double	v_height;		/* interval to display on open	*/
    double	peak_height;		/* peak detection defaults	*/
    double	peak_half_width;
    double	trough_depth;
    double	trough_half_width;
    int		top_margin;		/* plot region margins		*/
    int		bottom_margin;
    int		right_margin;
    int		left_margin;
    Font	plot_font;		/* for labeling axes		*/
    Font	plot_heading_font;	/* for heading			*/
    Pixel	cursor_color;		/* cursor color			*/
    Pixel	mark_color;		/* default color of marks	*/
    Pixel	mark_color_2;		/* alternative colors		*/
    Pixel	mark_color_3;		/* alternative colors		*/
    Pixel	mark_color_4;		/* alternative colors		*/
    String	color_2_marks;		/* marks in those colors	*/
    String	color_3_marks;		/* marks in those colors	*/
    String	color_4_marks;		/* marks in those colors	*/
    int		mark_size;		/* size of mark crosses		*/
    int		mark_thickness;		/* thickness of lines used	*/
    Bool	autoscale_always;	/* autoscale voltage axis	*/
    String	tfname;			/* temp file name, NULL if none	*/
    FILTER	df_fil;			/* filter from which file read	*/
    FILTER	*c_filter;		/* current output filter	*/
    MARKLIST	mark_list;		/* current mark list		*/
    String	mark_comment_format_2;	/* for reporting numbers	*/
    String	mark_comment_format_3;	/* for reporting numbers	*/
    TIME	tl;			/* time at left edge		*/
    double	tlfrac;			/* used to adjust scrolling	*/
    TIME	tr;			/* time at right edge		*/
    FILTER_VAL	vb;			/* voltage at bottom		*/
    FILTER_VAL	vt;			/* voltage at top		*/
    Dimension	pww;			/* plot window width (pixels)	*/
    Dimension	pwh;			/* plot window height		*/
    Dimension	prx;			/* region used for plot		*/
    Dimension	pry;
    Dimension	prw;
    Dimension	prh;
    double	t_con;			/* time conversion factor	*/
    double	v_con;			/* voltage conversion factor	*/
    GC		plot_gc;		/* GC for plotting		*/
    GC		axes_gc;		/* GC for axes			*/
    XFontStruct	*pfinfo;		/* info about plot font		*/
    XFontStruct	*phfinfo;		/* info about plot heading font	*/
    Bool	is_cursor;		/* cursor exists		*/
    TIME	cursor;			/* where it is			*/
    Bool	is_cursor_end;		/* cursor region exists		*/
    TIME	cursor_end;		/* where it ends		*/
    Pixmap	icon;			/* the xdatplot icon		*/
} AppData;

#ifdef	__STDC__
/*
 * library functions
 */
double	strtod(const char *, char **);
long	strtol(char *, char **, int);
String	getenv(String);
/*
 * my library functions
 */
long	filelen(FILE *);
void	error(char *f);
void	dprintf(char *f);
void	dprintf1(char *f, char *a);
VOIDST	ealloc(size_t);
VOIDST	erealloc(VOIDST, size_t);
String	estrdup(String);
/*
 * from xdatplot.c
 */
void	exit_program(void);
void	exit_cleanup(void);
void	unmap_shell(Widget, Widget, XmAnyCallbackStruct *);
void	PU_error(String, String);
Bool	ask_user(String);
/*
 * from mainw.c
 */
void	create_main_window(Widget);
/*
 * from fildialog.c
 */
Widget	make_filter_dialog(void);
void	make_current_filters(void);
String	**save_filter_params(FILTER *);
void	restore_filter_params(FILTER *, String **);
void	discard_filter_params(String **);
Boolean	CvtFilterChainToString(Display *, XrmValuePtr,
			       Cardinal *, XrmValuePtr, XrmValuePtr);
Boolean	CvtStringToFilterChain(Display *, XrmValuePtr,
			       Cardinal *, XrmValuePtr, XrmValuePtr);
/*
 * from control.c
 */
Widget	create_control(Widget);
void	set_auto_always(Bool);
void	v_zoom_in(double);
void	v_zoom_out(double);
void	t_zoom_in(double);
void	t_zoom_out(double);
void	t_full_scale(void);
void	v_full_scale(void);
void	t_is_full_scale(Bool);
void	v_is_full_scale(Bool);
void	set_zoom_fields(void);
void	set_comment(String);
String	get_comment(void);
/*
 * from plot.c
 */
void	create_plot_window(Widget);
void	cancel_redraw(void);
void	redo_whole(void);
Bool	set_auto_scale(void);
void	adjust_sb(void);
void	scroll_plot(TIME);
void	replot_part(int, int, int, int);
void	replot_whole(void);
void	find_tics(TICS *);
void	expose_whole(void);
void	set_cursor(TIME);
void	set_cursor_end(TIME);
void	reset_cursor(void);
/*
 * from mif.c
 */
void	MIFplot(String);
void	MIFplot_stream(FILE *);
/*
 * from ps.c
 */
extern	String	ps_formats[];
extern	int	num_ps_formats;
void	PSplot(String, int, Bool);
void	PSplot_stream(FILE *, int, Bool);
/*
 * from files.c
 */
void	new_file(String);
void	close_file(FILTER *);
void	kill_filters(void);
void	init_plot_params(Bool);
double	default_tmul(void);
double	default_vmul(void);
void	set_tmul(double);
void	set_vmul(double);
String	default_xdp_file(void);
/*
 * from marks.c
 */
void	clear_all_marks(void);
void	clear_all_marks_in_list(MARKLIST *);
int	lookup_marks(TIME, MARK **);
int	lookup_marks_in_list(TIME, MARK **, MARKLIST *);
int	lookup_interval(TIME, TIME, MARK **);
int	lookup_interval_in_list(TIME, TIME, MARK **, MARKLIST *);
MARK	*find_mark(TIME, int);
MARK	*find_mark_in_list(TIME, int, MARKLIST *);
void	delete_marks(TIME);
void	delete_marks_in_list(TIME, MARKLIST *);
void	delete_interval(TIME, TIME);
void	delete_interval_in_list(TIME, TIME, MARKLIST *);
void	delete_mark(TIME, int);
void	delete_mark_in_list(TIME, int, MARKLIST *);
int	next_marks(TIME, MARK **);
int	next_marks_in_list(TIME, MARK **, MARKLIST *);
int	prev_marks(TIME, MARK **);
int	prev_marks_in_list(TIME, MARK **, MARKLIST *);
MARK	*next_mark(TIME, int);
MARK	*next_mark_in_list(TIME, int, MARKLIST *);
MARK	*prev_mark(TIME, int);
MARK	*prev_mark_in_list(TIME, int, MARKLIST *);
Bool	write_marks(String);
Bool	write_marks_in_list(String, MARKLIST *);
MARKLIST *merge_marklists(MARKLIST *, MARKLIST *);
Bool	read_marks(String);
Bool	read_marks_in_list(String, MARKLIST *);
Bool	filter_marks(String);
Bool	filter_marks_in_list(String, MARKLIST *);
Boolean	CvtStringToMarkList(Display *, XrmValuePtr,
			    Cardinal *, XrmValuePtr, XrmValuePtr);
Boolean	CvtMarkListToString(Display *, XrmValuePtr,
			    Cardinal *, XrmValuePtr, XrmValuePtr);
Widget	make_marks_filter_dialog(void);
void	marks_filter_drop(void);
void	marks_filter_end(void);
/*
 * from help.c
 */
void	help_window(String);
void	rm_help_tmp(void);
/*
 * from units.c
 */
Widget	make_units_dialog(void);
void	copy_resources_into_fields(void);
/*
 * from xdp.c
 */
Bool	write_xdp(String, String);
String	xdp_get_datafile(String);
Bool	xdp_get_resources(String);
String	append_string(String, String, int *);
void	put_string(FILE *, String);
/*
 * from peaks.c
 */
Widget	make_peaks_dialog(void);
#ifdef	HAS_FUNC_ATTR
int	gcf(int, int)	__attribute__ ((const));
#else
int	gcf(int, int);
#endif
/*
 * from run.c
 */
int	run(char *, int *);
#else
/*
 * library functions
 */
double	strtod();
long	strtol();
String	getenv();
/*
 * my library functions
 */
long	filelen();
void	error();
void	dprintf();
void	dprintf1();
VOIDST	ealloc();
VOIDST	erealloc();
String	estrdup();
/*
 * from xdatplot.c
 */
void	exit_program();
void	exit_cleanup();
void	unmap_shell();
void	PU_error();
Bool	ask_user();
/*
 * from mainw.c
 */
void	create_main_window();
/*
 * from fildialog.c
 */
Widget	make_filter_dialog();
void	make_current_filters();
String	**save_filter_params();
void	restore_filter_params();
void	discard_filter_params();
Boolean	CvtFilterChainToString();
Boolean	CvtStringToFilterChain();
/*
 * from control.c
 */
Widget	create_control();
void	set_auto_always();
void	v_zoom_in();
void	v_zoom_out();
void	t_zoom_in();
void	t_zoom_out();
void	t_full_scale();
void	v_full_scale();
void	t_is_full_scale();
void	v_is_full_scale();
void	set_zoom_fields();
void	set_comment();
String	get_comment();
/*
 * from plot.c
 */
void	create_plot_window();
void	cancel_redraw();
void	redo_whole();
Bool	set_auto_scale();
void	adjust_sb();
void	scroll_plot();
void	replot_part();
void	replot_whole();
void	find_tics();
void	expose_whole();
void	set_cursor();
void	set_cursor_end();
void	reset_cursor();
/*
 * from mif.c
 */
void	MIFplot();
void	MIFplot_stream();
/*
 * from ps.c
 */
extern	String	ps_formats[];
extern	int	num_ps_formats;
void	PSplot();
void	PSplot_stream();
/*
 * from files.c
 */
void	new_file();
void	close_file();
void	kill_filters();
void	init_plot_params();
double	default_tmul();
double	default_vmul();
void	set_tmul();
void	set_vmul();
String	default_xdp_file();
/*
 * from marks.c
 */
void	clear_all_marks();
void	clear_all_marks_in_list();
int	lookup_marks();
int	lookup_marks_in_list();
int	lookup_interval();
int	lookup_interval_in_list();
MARK	*find_mark();
MARK	*find_mark_in_list();
void	delete_marks();
void	delete_marks_in_list();
void	delete_mark_in_list();
void	delete_mark();
void	delete_interval_in_list();
void	delete_interval();
int	next_marks();
int	next_marks_in_list();
int	prev_marks();
int	prev_marks_in_list();
MARK	*next_mark();
MARK	*next_mark_in_list();
MARK	*prev_mark();
MARK	*prev_mark_in_list();
Bool	write_marks();
Bool	write_marks_in_list();
MARKLIST *merge_marklists();
Bool	read_marks();
Bool	read_marks_in_list();
Bool	filter_marks();
Bool	filter_marks_in_list();
Boolean	CvtStringToMarkList();
Boolean	CvtMarkListToString();
Widget	make_marks_filter_dialog();
void	marks_filter_drop();
void	marks_filter_end();
/*
 * from help.c
 */
void	help_window();
void	rm_help_tmp();
/*
 * from units.c
 */
Widget	make_units_dialog();
void	copy_resources_into_fields();
/*
 * from xdp.c
 */
Bool	write_xdp();
String	xdp_get_datafile();
Bool	xdp_get_resources();
String	append_string();
void	put_string();
/*
 * from peaks.c
 */
Widget	make_peaks_dialog();
int	gcf();
/*
 * from run.c
 */
int	run();
#endif

extern	XtResource	resources[];	/* application resources	*/
extern	Bool		free_res[];	/* whether val is from heap	*/
extern	int		num_app_resources; /* how many there are	*/
extern	String		PROGNAME;	/* for error messages		*/
extern	Status		ret;		/* return status		*/
extern	XtAppContext	app;		/* the application context	*/
extern	int		debug;		/* produce debugging output	*/
extern	int		indent;		/* debug msg indentation	*/

/*
 * defaults and default values that can be changed interactively
 */
extern	AppData		app_data;	/* application defaults		*/
#define	PRINT_CMD	(app_data.print_cmd)
#define	HELP_PRINT_CMD	(app_data.help_print_cmd)
#define	T_UNITS		(app_data.t_units)
#define	T_TICS		(app_data.t_tics)
#define	T_GAIN		(app_data.t_gain)
#define	V_UNITS		(app_data.v_units)
#define	V_TICS		(app_data.v_tics)
#define	V_GAIN		(app_data.v_gain)
#define	TOP_MARGIN	(app_data.top_margin)
#define	BOTTOM_MARGIN	(app_data.bottom_margin)
#define	RIGHT_MARGIN	(app_data.right_margin)
#define	LEFT_MARGIN	(app_data.left_margin)
#define	AUTOSCALE_ALWAYS (app_data.autoscale_always)

/*
 * defaults we don't want to change (but do need to convert)
 */
#define	T_TIME		(app_data.t_time)
#define	T_TIME_DEF	(isnan(T_TIME))
#define	T_MULTIPLIER	(app_data.t_multiplier)
#define	T_MUL_DEF	(isnan(T_MULTIPLIER))
#define	T_WIDTH		(app_data.t_width)
#define	V_MULTIPLIER	(app_data.v_multiplier)
#define	V_MUL_DEF	(isnan(V_MULTIPLIER))
#define	V_HEIGHT	(app_data.v_height)
#define	V_HEIGHT_DEF	(isnan(V_HEIGHT))
#define	T_TICLEN	(app_data.t_ticlen)
#define	V_TICLEN	(app_data.v_ticlen)
#define	PLOT_FONT	(app_data.plot_font)
#define	PLOT_HEADING_FONT (app_data.plot_heading_font)
#define	CURSOR_COLOR	(app_data.cursor_color)
#define	MARK_COLOR	(app_data.mark_color)
#define	MARK_SIZE	(app_data.mark_size)
#define	MARK_THICKNESS	(app_data.mark_thickness)
/*
 * File information:
 */
#define	DATA_FILE	(app_data.data_file)
#define	TFNAME		(app_data.tfname)
#define	XD_BUFSIZ	(app_data.xd_bufsiz)
#define	DF_FIL		(app_data.df_fil)
#define	C_FILTER	(app_data.c_filter)

/*
 * Things we need to know to plot data in the scrolled window:
 */
#define	TL		(app_data.tl + app_data.tlfrac)
#define	TR		(app_data.tr)
#define	VB		(app_data.vb)
#define	VT		(app_data.vt)
#define	PWW		(app_data.pww)
#define	PWH		(app_data.pwh)
#define	PRX		(app_data.prx)
#define	PRY		(app_data.pry)
#define	PRW		(app_data.prw)
#define	PRH		(app_data.prh)
#define	T_CON		(app_data.t_con)
#define	V_CON		(app_data.v_con)
#define	PLOT_GC		(app_data.plot_gc)
#define	AXES_GC		(app_data.axes_gc)
#define	PFINFO		(app_data.pfinfo)
#define	PHFINFO		(app_data.phfinfo)
#define	IS_CURSOR	(app_data.is_cursor)
#define	IS_CURSOR_END	(app_data.is_cursor_end)
#define	CURSOR		(app_data.cursor)
#define	CURSOR_END	(app_data.cursor_end)

extern	Widget		toplevel;	/* main window Shell widget	*/
extern	Widget		plot;		/* Drawing Area for plot	*/
extern	Widget		vsb;		/* vertical scroll bar		*/
extern	Widget		hsb;		/* horizontal scroll bar	*/

/*
 * Widgets with values we need to set or read
 */
extern	Widget		t_field;	/* t text field for cursor	*/
extern	Widget		v_field;	/* v text field for cursor	*/

/*
 * scale factors and mapping functions
 */
#define	EPS	(1.0e-7)
#define	TMIN	(filter_tmin(C_FILTER))
#define	TMAX	(filter_tmax(C_FILTER))
#define	VMIN	(filter_fmin(C_FILTER))
#define	VMAX	(filter_fmax(C_FILTER))
#define	TIME_to_t_sf		(T_CON)
#define	TIME_to_Dim_sf		(((double) PRW) / ((double) (TR - app_data.tl)))
#define	Map_TIME_to_Dim_c(t)	(PRX + TIME_to_Dim_sf * ((t) - TL))
#define	Map_TIME_to_Dim(t)	(PRX + 					\
				 floor(EPS + TIME_to_Dim_sf * ((t) - TL)))
#define	Map_Dim_to_TIME(d)	(TL + ceil(-EPS +			\
				       ((d) - PRX) / TIME_to_Dim_sf	\
				       ))
#define	Map_TIME_to_t(t)	((t) * T_CON)
#define	Map_TIME_to_t_c(t)	((t) * T_CON)
#define	Map_t_to_TIME(t)	(nint((t) / T_CON))
#define	Map_t_to_TIME_c(t)	((t) / T_CON)
#define	Map_t_to_Dim(t)		Map_TIME_to_Dim((t) / T_CON)
#define	Map_t_to_Dim_c(t)	Map_TIME_to_Dim_c((t) / T_CON)
#define	t_to_Dim_sf		(TIME_to_Dim_sf / T_CON)
#define	VOLTAGE_to_v_sf		(V_CON)
#define	VOLTAGE_to_Dim_sf	(((double) PRH) / ((double) (VT - VB)))
#define	Map_VOLTAGE_to_Dim_c(v)	(PRY + VOLTAGE_to_Dim_sf * (VT - (v)))
#define	Map_VOLTAGE_to_Dim(v)	(PRY + nint(VOLTAGE_to_Dim_sf * (VT - (v))))
#define	Map_Dim_to_VOLTAGE(d)	(nint(VT - ((d) - PRY) / VOLTAGE_to_Dim_sf))
#define	Map_VOLTAGE_to_v(v)	((v) * V_CON)
#define	Map_VOLTAGE_to_v_c(v)	((v) * V_CON)
#define	Map_v_to_VOLTAGE(v)	(nint((v) / V_CON))
#define	Map_v_to_VOLTAGE_c(v)	((v) / V_CON)
#define	Map_v_to_Dim(v)		Map_VOLTAGE_to_Dim((v) / V_CON)
#define	Map_v_to_Dim_c(v)	Map_VOLTAGE_to_Dim_c((v) / V_CON)
#define	v_to_Dim_sf		(VOLTAGE_to_Dim_sf / V_CON)
#define	MARK_to_TIME_sf		(filter_tmul(&DF_FIL) / filter_tmul(C_FILTER))
#define	Map_MARK_to_TIME(m)	(nint((m) * MARK_to_TIME_sf))
#define	Map_TIME_to_MARK(t)	(nint((t) / MARK_to_TIME_sf))
#define	Map_MARK_to_Dim(m)	Map_TIME_to_Dim(Map_MARK_to_TIME(m))
#define	Map_MARK_to_Dim_c(m)	Map_TIME_to_Dim_c(Map_MARK_to_TIME(m))

/*
 * utility macro: force a range to fit in another.  (This has to be a
 * macro because it may be used on integers of different widths, and
 * uses two of the integers as lvalues.)  Arguments are evaluated
 * multiple times, so avoid expressions with side-effects.
 */
#define	FORCE_RANGE(s, e, min, max)					\
			(						\
			 (						\
			  ((e) - (s) >= (max) - (min)) &&		\
			  ((s) = (min), (e) = (max), TRUE)		\
			 ) ||						\
			 (						\
			  ((s) < (min)) &&				\
			  ((e) += (min) - (s), (s) = (min), TRUE)	\
			 ) ||						\
			 (						\
			  ((e) > (max)) &&				\
			  ((s) -= (e) - (max), (e) = (max), TRUE)	\
			 )						\
			)

#endif	_XDATPLOT_H
