/* filter.h -- header file for filter package				*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#ifndef	FILTER_H
#define	FILTER_H

#include <math.h>
#include "Vt.h"
#include "config.h"

typedef	double		FILTER_VAL;
#ifndef	Bool
typedef	int		Bool;
#endif

typedef	enum	FILTER_TYPE {
    F_INACTIVE = -1,			/* inactive filter		*/
    F_EMPTY = 0,			/* empty filter			*/
    F_DF = 1,				/* input from datafile		*/
    F_HIPASS_TAU = 2,			/* hi pass filter, tau given	*/
    F_HIPASS_F = 3,			/* hi pass filter, freq given	*/
    F_LOPASS_TAU = 4,			/* lo pass filter, tau given	*/
    F_LOPASS_F = 5,			/* lo pass filter, freq given	*/
    F_MINIMAX = 6,			/* minimax compression		*/
    F_SCALE = 7,			/* scale output values		*/
    F_PICK = 8				/* pick regular values		*/
}		FILTER_TYPE;
#define	F_NUMBER	9		/* number of filters		*/

typedef	struct	DF_DATA {
    FILTER_TYPE	type;
    FILE	*stream;
    char	*fname;
    unsigned	bufsiz;
}		DF_DATA;

typedef	struct	HP_TAU_DATA {
    FILTER_TYPE	type;
    double	tau;
    double	tol;
}		HP_TAU_DATA;

typedef	struct	HP_F_DATA {
    FILTER_TYPE	type;
    double	f;
    double	tol;
}		HP_F_DATA;

typedef	struct	LP_TAU_DATA {
    FILTER_TYPE	type;
    double	tau;
    double	tol;
}		LP_TAU_DATA;

typedef	struct	LP_F_DATA {
    FILTER_TYPE	type;
    double	f;
    double	tol;
}		LP_F_DATA;

typedef	struct	MM_DATA {
    FILTER_TYPE	type;
    int		n;
}		MM_DATA;

typedef	struct	SC_DATA {
    FILTER_TYPE	type;
    double	factor;
}		SC_DATA;

typedef	struct	PK_DATA {
    FILTER_TYPE	type;
    int		st;
    int		n;
}		PK_DATA;

typedef	union	FILTER_DATA {
    FILTER_TYPE	type;
    DF_DATA	df_data;
    HP_TAU_DATA	hp_tau_data;
    HP_F_DATA	hp_f_data;
    LP_TAU_DATA	lp_tau_data;
    LP_F_DATA	lp_f_data;
    MM_DATA	mm_data;
    SC_DATA	sc_data;
    PK_DATA	pk_data;
}		FILTER_DATA;

typedef	struct	FILTER_CACHE {
    struct FILTER_CACHE	*f_nxt;		/* if there's more than one	*/
    TIME		f_t0;		/* first cached time		*/
    TIME		f_n;		/* # of cached values		*/
    TIME		f_sp;		/* spacing between them		*/
    int			f_nv;		/* # vals storage available for	*/
    FILTER_VAL		*f_vals;	/* the actual values		*/
}		FILTER_CACHE;

typedef	struct	FILTER {
    struct FILTER	*f_src;		/* where input comes from	*/
    struct FILTER	*f_dst;		/* where output goes		*/
    FILTER_TYPE		f_type;		/* filter type			*/
    char		*f_name;	/* mnemonic filter name		*/
    FILTER_DATA		*f_data;	/* filter-specific data		*/
    TIME		f_tmin;		/* first time point		*/
    TIME		f_tmax;		/* last time + 1		*/
    FILTER_VAL		f_fmin;		/* minimum output		*/
    FILTER_VAL		f_fmax;		/* maximum output		*/
    double		f_tmul;		/* time scale			*/
    double		f_fmul;		/* output scale factor		*/
    VOIDST		*f_priv;	/* private data			*/
    FILTER_CACHE	f_cv;		/* cached values		*/
    FILTER_CACHE	f_cm;		/* cached memories		*/
    FILTER_VAL		(*f_func)();	/* the real filter		*/
    Bool		(*f_make)();	/* create filter		*/
    Bool		(*f_chg)();	/* change parameters		*/
    Bool		(*f_nsrc)();	/* change source		*/
    Bool		(*f_kill)();	/* destroy filter		*/
}		FILTER;

/*
 * In a valid filter f_cv.f_nxt is NULL and f_cv.f_sp is 1, i.e. the
 * cache contains a single complete contiguous range of values.  This
 * cache is used by the F macro to avoid unnecessary function calls.
 * f_cm, in contrast, may be used to cache a few, possibly widely
 * space values.  Its usefulness comes from the dependence of a
 * filtered value at TIME t on all previous values: by storing a few
 * filtered values, one may not need to backtrack very far.
 *
 * f_src and f_dst are used for hooking filters to each other.  Each
 * filter takes its input from the f_src filter: this happens
 * transparently once the filters have been hooked together.  f_dst is
 * used only when a filter's parameters change; it is used to clear
 * the caches of all filters that get their input from the changed
 * filter, since those caches are no longer valid.
 */

#ifdef	min
#undef	min
#endif
#define	min(a, b)	(((a) < (b)) ? (a) : (b))
#ifdef	max
#undef	max
#endif
#define	max(a, b)	(((a) > (b)) ? (a) : (b))

#define	F(f, t)	(							\
		 (							\
		  ((t) >= (f)->f_cv.f_t0) &&				\
		  ((t) - (f)->f_cv.f_t0 < (f)->f_cv.f_n)		\
		 ) ?							\
		 (f)->f_cv.f_vals[(t) - (f)->f_cv.f_t0] :		\
		 ((f)->f_func)((FILTER *) (f), (TIME) (t))		\
		)

extern	const	char	*filter_names[F_NUMBER];
#ifdef	__STDC__
Bool	make_filter(FILTER_TYPE, FILTER_DATA *, FILTER *);
Bool	connect_filters(FILTER *, FILTER *, FILTER *);
Bool	disconnect_filter(FILTER *);
Bool	delete_filter(FILTER *);
Bool	force_filter_tmul(FILTER *, double);
Bool	force_filter_fmul(FILTER *, double);
#else	/* __STDC__ */
Bool	make_filter();
Bool	connect_filters();
Bool	disconnect_filter();
Bool	delete_filter();
Bool	force_filter_tmul();
Bool	force_filter_fmul();
#endif	/* __STDC__ */

#define	INVALID_FILTER_VAL	(signaling_nan(0))
#define	IS_INVALID_FILTER_VAL(v) (isnan(v))
#define	change_filter(fp, data)	((*(fp)->f_chg)((fp), (data)))
#define	kill_filter(fp)		((*(fp)->f_kill)(fp))
#define	is_active_filter(fp)	(F_INACTIVE != (fp)->f_type)
#define	inactivate_filter(fp)	((fp)->f_type = F_INACTIVE)
#define	filter_data(fp)		((fp)->f_data)
#define	filter_name(fp)		((fp)->f_name)
#define	filter_type(fp)		((fp)->f_type)
#define	source_filter(fp)	((fp)->f_src)
#define	destination_filter(fp)	((fp)->f_dst)
#define	filter_tmin(fp)		((fp)->f_tmin)
#define	filter_tmax(fp)		((fp)->f_tmax)
#define	filter_fmin(fp)		((fp)->f_fmin)
#define	filter_fmax(fp)		((fp)->f_fmax)
#define	filter_tmul(fp)		((fp)->f_tmul)
#define	filter_fmul(fp)		((fp)->f_fmul)

#endif	/* not FILTER_H */
