/* conversion.c -- define Xt type conversions for xdatplot		*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

#ifdef	__STDC__
static	Boolean	CvtStringToDouble(Display *, XrmValuePtr, Cardinal *,
				  XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToTIME(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToTime(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToFreq(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToVOLTAGE(Display *, XrmValuePtr, Cardinal *,
				   XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToVoltage(Display *, XrmValuePtr, Cardinal *,
				   XrmValuePtr, XrmValuePtr);
static	Boolean	CvtDoubleToString(Display *, XrmValuePtr, Cardinal *,
				  XrmValuePtr, XrmValuePtr);
static	Boolean	CvtTIMEToString(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtTimeToString(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtFreqToString(Display *, XrmValuePtr, Cardinal *,
				XrmValuePtr, XrmValuePtr);
static	Boolean	CvtVOLTAGEToString(Display *, XrmValuePtr, Cardinal *,
				   XrmValuePtr, XrmValuePtr);
static	Boolean	CvtVoltageToString(Display *, XrmValuePtr, Cardinal *,
				   XrmValuePtr, XrmValuePtr);
static	Boolean	CvtIntToString(Display *, XrmValuePtr, Cardinal *,
			       XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringTableToString(Display *, XrmValuePtr, Cardinal *,
				       XrmValuePtr, XrmValuePtr);
static	Boolean	CvtStringToStringTable(Display *, XrmValuePtr, Cardinal *,
				       XrmValuePtr, XrmValuePtr);
#else	/* __STDC__ */
static	Boolean	CvtStringToDouble();
static	Boolean	CvtStringToTIME();
static	Boolean	CvtStringToTime();
static	Boolean	CvtStringToFreq();
static	Boolean	CvtStringToVOLTAGE();
static	Boolean	CvtStringToVoltage();
static	Boolean	CvtDoubleToString();
static	Boolean	CvtTIMEToString();
static	Boolean	CvtTimeToString();
static	Boolean	CvtFreqToString();
static	Boolean	CvtVOLTAGEToString();
static	Boolean	CvtVoltageToString();
static	Boolean	CvtStringTableToString();
static	Boolean	CvtStringToStringTable();
#endif	/* __STDC__ */

static	String	s = NULL;		/* scratch for Cvt...ToString	*/

void
add_converters(fp)
FILTER		*fp;
{
    XtConvertArgRec	arg;
    XtConvertArgList	args;
    Cardinal		nargs;

    if (NULL == fp) {
	args = NULL;
	nargs = 0;
    }
    else {
	arg.address_mode = XtAddress;
	arg.address_id = (XtPointer) fp;
	arg.size = sizeof(FILTER);
	args = &arg;
	nargs = 1;
    }
    SetTypeConverter(XtRString, XtRDouble, CvtStringToDouble, NULL, 0,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRTIME, CvtStringToTIME, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRTime, CvtStringToTime, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRFreq, CvtStringToFreq, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRVOLTAGE, CvtStringToVOLTAGE, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRVoltage, CvtStringToVoltage, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRDouble, XtRString, CvtDoubleToString, NULL, 0,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRTIME, XtRString, CvtTIMEToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRTime, XtRString, CvtTimeToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRFreq, XtRString, CvtFreqToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRVOLTAGE, XtRString, CvtVOLTAGEToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRVoltage, XtRString, CvtVoltageToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRInt, XtRString, CvtIntToString, NULL, 0,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRStringTable, XtRString, CvtStringTableToString,
		     NULL, 0, XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRStringTable, CvtStringToStringTable,
		     NULL, 0, XtCacheNone, NULL);
    SetTypeConverter(XtRFilterChain, XtRString, CvtFilterChainToString,
		     NULL, 0, XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRFilterChain, CvtStringToFilterChain,
		     NULL, 0, XtCacheNone, NULL);
    SetTypeConverter(XtRMarkList, XtRString, CvtMarkListToString,
		     NULL, 0, XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRMarkList, CvtStringToMarkList,
		     NULL, 0, XtCacheNone, NULL);
}

void
chg_converters(fp)
FILTER		*fp;
{
    XtConvertArgRec	arg;
    XtConvertArgList	args;
    Cardinal		nargs;

    if (NULL == fp) {
	args = NULL;
	nargs = 0;
    }
    else {
	arg.address_mode = XtAddress;
	arg.address_id = (XtPointer) fp;
	arg.size = sizeof(FILTER);
	args = &arg;
	nargs = 1;
    }
    SetTypeConverter(XtRString, XtRTIME, CvtStringToTIME, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRTime, CvtStringToTime, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRFreq, CvtStringToFreq, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRVOLTAGE, CvtStringToVOLTAGE, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRString, XtRVoltage, CvtStringToVoltage, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRTIME, XtRString, CvtTIMEToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRTime, XtRString, CvtTimeToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRFreq, XtRString, CvtFreqToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRVOLTAGE, XtRString, CvtVOLTAGEToString, args, nargs,
		     XtCacheNone, NULL);
    SetTypeConverter(XtRVoltage, XtRString, CvtVoltageToString, args, nargs,
		     XtCacheNone, NULL);
}

/* canned from Xt ref manual						*/
#define	done(type, value)\
{									\
    if (to->addr != NULL) {						\
	if (to->size < sizeof(type)) {					\
	    to->size = sizeof(type);					\
	    return(False);						\
	}								\
	*((type *) to->addr) = (value);					\
    }									\
    else {								\
	static type	static_val;					\
	static_val = (value);						\
	to->addr = (XtPointer) &static_val;				\
    }									\
    to->size = sizeof(type);						\
    return(True);							\
}

#define	get_filter(type1, type2)\
{									\
    if (0 == *nargs) fp = C_FILTER;					\
    else if (1 == *nargs) fp = (FILTER *) args->addr;			\
    else {								\
	char	mbuf1[LLEN];						\
	char	mbuf2[LLEN];						\
									\
	sprintf(mbuf1, "cvt%sTo%s", type1, type2);			\
	sprintf(mbuf2, "Too many arguments to %s to %s conversion",	\
		type1, type2);						\
	XtWarningMsg("wrongParameters", mbuf1, "XtToolkitError",	\
		     mbuf2, NULL, NULL);				\
	fp = C_FILTER;							\
    }									\
}

static Boolean
CvtStringToDouble(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    double		f = 0.0;
    String		t, u;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToDouble",
		     "XtToolkitError", "String to Double\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    Nfree(s);
    s = XtMalloc(from->size + 1);
    bcopy(from->addr, s, from->size);
    s[from->size] = '\0';
    for(t=s; isspace(*t); t++);
    if ('\0' != *t) {
	if (
	    ('n' == tolower(t[0])) &&
	    ('a' == tolower(t[1])) &&
	    ('n' == tolower(t[2]))
	) {
	    f = quiet_nan(0);
	    u = t + 3;
	}
	else{
	    f = strtod(t, &u);
	}
	while(isspace(*u)) u++;
	if ('\0' != *u)
	    XtDisplayStringConversionWarning(d, (char *) from->addr, XtRDouble);
    }
    done(double, f);
}

static Boolean
CvtStringToTIME(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    TIME		t = 0;
    XrmValue		dtv;
    FILTER		*fp;

    dtv.addr = NULL;
    if (!XtConvertAndStore(toplevel, XtRString, from, XtRDouble, &dtv)) {
	XtDisplayStringConversionWarning(d, (char *) from->addr, XtRTIME);
	return(False);
    }
    get_filter(XtRString, XtRTIME);
    t = nint(*((double *) dtv.addr) * T_GAIN / filter_tmul(fp));
    done(TIME, t);
}

static Boolean
CvtStringToTime(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    FILTER		*fp;
    double		t;
    int			zero = 0;
    Boolean		ret;

    ret = CvtStringToDouble(d, NULL, &zero, from, to);
    if (ret) {
	get_filter(XtRString, XtRTime);
	*((double *) to->addr) =
	    *((double *) to->addr) * T_GAIN / filter_tmul(fp);
    }
    return(ret);
}

static Boolean
CvtStringToFreq(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    FILTER		*fp;
    int			zero = 0;
    Boolean		ret;

    ret = CvtStringToDouble(d, NULL, &zero, from, to);
    if (ret) {
	get_filter(XtRString, XtRFreq);
	*((double *) to->addr) =
	    *((double *) to->addr) * filter_tmul(fp) / T_GAIN;
    }
    return(ret);
}

static Boolean
CvtStringToVOLTAGE(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    FILTER		*fp;
    VOLTAGE		v = 0;
    XrmValue		dtv;

    dtv.addr = NULL;
    if (!XtConvertAndStore(toplevel, XtRString, from, XtRDouble, &dtv)) {
	XtDisplayStringConversionWarning(d, (char *) from->addr, XtRVOLTAGE);
	return(False);
    }
    get_filter(XtRString, XtRVOLTAGE);
    v = nint(*((double *) dtv.addr) * V_GAIN / filter_fmul(fp));
    done(VOLTAGE, v);
}

static Boolean
CvtStringToVoltage(d, args, nargs, from, to)
Display		*d;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    FILTER		*fp;
    Boolean		ret;
    int			zero = 0;

    ret = CvtStringToDouble(d, NULL, &zero, from, to);
    if (ret) {
	get_filter(XtRString, XtRVoltage);
	*((double *) to->addr) =
	    *((double *) to->addr) * V_GAIN / filter_fmul(fp);
    }
    return(ret);
}

static Boolean
CvtDoubleToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    char		lbuf[LLEN];
    int			l;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtDoubleToString",
		     "XtToolkitError", "Double to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    sprintf(lbuf, "%.10g", *((double *) from->addr));
    l = strlen(lbuf);
    if (NULL == to->addr) {
	Nfree(s);
	s = XtNewString(lbuf);
	to->addr = (caddr_t) s;
	to->size = l + 1;
	return(True);
    }
    else if (to->size <= l) {
	to->size = l + 1;
	return(False);
    }
    else {
	strcpy((String) to->addr, lbuf);
	to->size = l + 1;
	return(True);
    }
}

static Boolean
CvtIntToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    char		lbuf[LLEN];
    int			l;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtIntToString",
		     "XtToolkitError", "Int to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    sprintf(lbuf, "%d", *((int *) from->addr));
    l = strlen(lbuf);
    if (NULL == to->addr) {
	Nfree(s);
	s = XtNewString(lbuf);
	to->addr = (caddr_t) s;
	to->size = l + 1;
	return(True);
    }
    else if (to->size <= l) {
	to->size = l + 1;
	return(False);
    }
    else {
	strcpy((String) to->addr, lbuf);
	to->size = l + 1;
	return(True);
    }
}

static Boolean
CvtTIMEToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			zero = 0;
    double		d;
    XrmValue		dtv;
    FILTER		*fp;

    get_filter(XtRTIME, XtRString);
    d = *((TIME *) from->addr) * filter_tmul(fp) / T_GAIN;
    dtv.addr = (caddr_t) &d;
    dtv.size = sizeof(d);
    return(CvtDoubleToString(disp, NULL, &zero, &dtv, to));
}

static Boolean
CvtTimeToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			zero = 0;
    double		d;
    XrmValue		dtv;
    FILTER		*fp;

    get_filter(XtRTime, XtRString);
    d = *((double *) from->addr) * filter_tmul(fp) / T_GAIN;
    dtv.addr = (caddr_t) &d;
    dtv.size = sizeof(d);
    return(CvtDoubleToString(disp, NULL, &zero, &dtv, to));
}

static Boolean
CvtFreqToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			zero = 0;
    double		d;
    XrmValue		dtv;
    FILTER		*fp;

    get_filter(XtRFreq, XtRString);
    d = *((double *) from->addr) * T_GAIN / filter_tmul(fp);
    dtv.addr = (caddr_t) &d;
    dtv.size = sizeof(d);
    return(CvtDoubleToString(disp, NULL, &zero, &dtv, to));
}

static Boolean
CvtVOLTAGEToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			zero = 0;
    double		d;
    XrmValue		dtv;
    FILTER		*fp;

    get_filter(XtRVOLTAGE, XtRString);
    d = *((VOLTAGE *) from->addr) * filter_fmul(fp) / V_GAIN;
    dtv.addr = (caddr_t) &d;
    dtv.size = sizeof(d);
    return(CvtDoubleToString(disp, NULL, &zero, &dtv, to));
}

static Boolean
CvtVoltageToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			zero = 0;
    double		d;
    XrmValue		dtv;
    FILTER		*fp;

    get_filter(XtRVoltage, XtRString);
    d = Map_VOLTAGE_to_v_c(*((double *) from->addr));
    dtv.addr = (caddr_t) &d;
    dtv.size = sizeof(d);
    return(CvtDoubleToString(disp, NULL, &zero, &dtv, to));
}

static Boolean
CvtStringTableToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			l;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringTableToString",
		     "XtToolkitError", "StringTable to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    l = strlen(*((String *) from->addr));
    if (NULL == to->addr) {
	Nfree(s);
	s = XtNewString((String) from->addr);
	to->addr = (caddr_t) s;
	to->size = l + 1;
	return(True);
    }
    else if (to->size <= l) {
	to->size = l + 1;
	return(False);
    }
    else {
	strcpy((String) to->addr, *((String *) from->addr));
	return(True);
    }
}

static Boolean
CvtStringToStringTable(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    String		t;

    t = (String) from->addr;
    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToStringTable",
		     "XtToolkitError", "String to StringTable\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    done(String *, &t);
}
