/*
  Module: xcursor.c
  Author: Richard Neitzel
  Date:   21 Apr 1993

revision history
----------------
1.0,12may89,rekn      Written.
2.0,25may89,rekn      Added support for named windows and selection by 
                      pointer.
3.0,8june89,rekn      Added color and bitmap support.
4.0,14apr93,rekn      Made path name a define.
4.1,21apr93,rekn      Added the -s option.
  This is a quick program that reads the header file defining cursors
  and lets one change the cursor for a specified window. Window
  defaults to root, cursor to what's in xcursor.h.
*/

#include "xcursor.h"

/* Text for select by pointer case */
char *point_txt[] = {
"           ====> Please select the window",
"           ====> whose cursor you wish to",
"           ====> change by clicking the",
"           ====> pointer in that window.",
0};


main(argc,argv)
int argc;
char **argv;
{
    register int opt;		/* Parsed option letter */
    char *id_str = 0;		/* Window id in ASCII */
    char *disp_str = 0;		/* Display name */
    char *name_str = 0;		/* Window name */
    register int uflag = 0;	/* Source specified */
    register int sizeflg = 0;	/* Tell best cursor size. */
    register char **ptr;	/* Fast pointer */
    register char *sizes;	/* User query size. */
    extern int optind;
    extern char *optarg;
    extern int opterr;

    opterr = 0;			/* No error msgs from getopt */

    while ((opt = getopt(argc,argv,"lvb:f:d:n:i:rps:")) != -1)
      {
	  switch(opt)
	    {
	      case 'l':		/* Just list cursors */
		lflg++;
		break;

	      case 'v':		/* Print version */
		print_vers();
		break;

	      case 'f':		/* Foreground color */
		fcolor = argv[optind++];
		if (*fcolor == '-')
		  local_help();
		break;

	      case 'b':		/* Background color */
		bcolor = argv[optind++];
		if (*bcolor == '-')
		  local_help();
		break;

	      case 'd':		/* Get display name */
		disp_str = argv[optind++];
		if (*disp_str == '-') /* Check to see if user skipped */
				      /* required data */
		  local_help();
		break;

	      case 'n':		/* Get window name */
		name_str = argv[optind++];
		if (*name_str == '-')
		  local_help();
		uflag++;
		break;

	      case 'i':		/* Get window id */
		id_str = argv[optind++];
		if (*id_str == '-')
		  local_help();
		uflag++;
		break;
		
	      case 'r':		/* Set root cursor */
		uflag++;
		break;

	      case 'p':		/* Change to named bitmap */
		source = argv[optind++];
		mask = argv[optind++];
		if (*source == '-' || *mask == '-')
		  local_help();
		break;

	      case 's':		/* Print the maximum size of cursor. */
		sizeflg++;
		sizes = optarg;
		break;

	      case '?':		/* Ooops! */
		fprintf(stderr,"xcursor - bad option\n\n");
		local_help();
		break;
	    }
      }

    if ((display = XOpenDisplay(disp_str)) == NULL)
      {
	  fprintf(stderr,"Cannot open display %s\n",disp_str);
	  exit(1);
      }

    screen = DefaultScreen(display);

    if (sizeflg)
      bestSize(sizes);
    else
      {
	  if (name_str != NULL)
	    {
		if ((window = window_by_name(DefaultRootWindow(display),
					     name_str)) == 0)
		  {
		      fprintf(stderr,"No window named %s found.\n",name_str);
		      exit(2);
		  }
	    }
	  else if (id_str != NULL)
	    window = (Window)strtol(id_str,(char **)NULL,0);
	  else if (uflag)
	    window = DefaultRootWindow(display);
	  else if (!lflg)
	    {
		for (ptr = point_txt; *ptr; ptr++)
		  printf("%s\n",*ptr);

		window = point_to_window(); /* use pointer to get window */
	    }

	  if (optind < argc)		/* check for user requested cursor */
	    find_cursor = argv[optind];
	  else
	    find_cursor = default_cursor;
	  
	  if (source)
	    pixCursor();
	  else
	    work();
      }
    exit(0);
}

void print_vers()
{
    fprintf(stderr,"xcursor version %s\n",version_number);
    exit(0);
}


char *help_text[] = {
"xcursor is a small tool to set the cursor for a specified window.",
"The syntax is:",
"",
"    xcursor [-l] [-v] [-r] [-display display][-name name] [-id id]",
"            [-fg color] [-bg color] [-p cursorfile maskfile] [cursor]",
"",
"the -l flag lists all known cursors; the -name option sets the cursor for",
"the named window; the -id option sets the cursor for the window with the",
"specified id; the -display option causes the named display to be used",
"instead of the default display. The -v option prints the current version",
"number of xcursor. The -r option causes the root window to be effected.",
"Cursor is the name (XC_ prefix optional) of the cursor you want (defaults",
"to whatever the local builder desired). If none of -r, -id or -name are",
"used, xcursor prompts you to use the pointer to select a window. Colors may",
"be selected by using -bg and -fg. The defaults are a white background and",
"a black foreground. The cursor may be specified as a pixmap using -p. This",
"takes two file names, one for the cursor and one for the mask. The hot spot",
"is taken from the cursor file.",
0};

void local_help()
{
    register char **ptr;

    for (ptr = help_text; *ptr; ptr++)
      fprintf(stderr,"%s\n",*ptr);

    exit(3);
}


void work()
{
    register FILE *fp;		/* Source file */
    register char *start;	/* Cursor name */
    register char *middle;	/* Value in ASCII */
    register Cursor curs;	/* Selected cursor */
    register unsigned int cursor; /* Converted middle */
    register int flag = 0;	/* OK flag */
    char buffer[BUFSIZE];	/* Read buffer */

    if ((fp = fopen(cursor_file,"r")) == (FILE *)NULL)
      {
	  perror("xcursor");
	  exit(1);
      }

    while (fgets(buffer,BUFSIZE,fp))
      {
	  if ((start = strchr(buffer,'X')) != NULL)
	    {
		if (!strncmp(start,"XC_",3))
		  {
		      middle = strchr(start,' ');
		      *middle++ = 0;
		      cursor = (unsigned int)strtol(middle,(char **)NULL,0);

		      if (lflg)
			{
			    flag++;
			    if (strcmp(start,"XC_num_glyphs"))
			      printf("%s\n",start);
			}
		      else if (!strcmp(start, find_cursor) ||
			       !strcmp(&start[3], find_cursor))
			{
			    curs = XCreateFontCursor(display,cursor);
			    XDefineCursor(display,window,curs);
			    flag++;
			    break;
			}
		  }
	    }
      }
    if (!flag)
      fprintf(stderr,"xcursor - Cannot find %s\n",find_cursor);
    else if (bcolor || fcolor)	/* Only if needed */
      {
	  setCursorColor();
	  XRecolorCursor(display,curs,&fg,&bg);
      }

    XCloseDisplay(display);
    fclose(fp);
}

void setCursorColor()
{
    Colormap cmp;

    if (!bcolor)
      bcolor = "white";
    
    if (!fcolor)
      fcolor = "black";

    cmp = DefaultColormap(display,screen);

    if (!XParseColor(display,cmp,bcolor,&bg))
      {
	  fprintf(stderr,"Cannot locate color %s in color database.\n",bcolor);
	  return;
      }

    if (!XParseColor(display,cmp,fcolor,&fg))
      {
	  fprintf(stderr,"Cannot locate color %s in color database.\n",fcolor);
	  return;
      }
}

void pixCursor()
{
    Pixmap psource;		/* Cursor pixmap */
    Pixmap pmask;		/* Mask pixmap */
    int width;			/* Guess */
    int height;			/*   " */
    int xhot;			/* X location of hot spot */
    int yhot;			/* Y location of hot spot */
    Cursor cursor;		/* Our new cursor */
    int status;			/* Error return */
    int dummy;			/* Junk holder */
    Colormap cmp;		/* Ibid. */

/* Read the bitmap files and check for errors*/
    status = XReadBitmapFile(display,window,source,&width,&height,
			     &psource,&xhot,&yhot);

    switch(status)
      {
	case BitmapOpenFailed:
	  fprintf(stderr,"Sorry, but cannot open file %s\n",source);
	  return;
	  break;

	case BitmapFileInvalid:
	  fprintf(stderr,"Sorry, but file %s is not is bitmap format\n",
		  source);
	  return;
	  break;

	case BitmapNoMemory:
	  fprintf(stderr,"Sorry, but the server ran out of memory\n");
	  return;
	  break;
      }

    status = XReadBitmapFile(display,window,mask,&width,&height,
			     &pmask,&dummy,&dummy);

    switch(status)
      {
	case BitmapOpenFailed:
	  fprintf(stderr,"Sorry, but cannot open file %s\n",mask);
	  return;
	  break;

	case BitmapFileInvalid:
	  fprintf(stderr,"Sorry, but file %s is not is bitmap format\n",
		  mask);
	  return;
	  break;

	case BitmapNoMemory:
	  fprintf(stderr,"Sorry, but the server ran out of memory\n");
	  return;
	  break;
      }

    setCursorColor();		/* Need some color */

    if ((cursor = XCreatePixmapCursor(display,psource,pmask,&fg,&bg,xhot,yhot))
	== 0)
      {
	  fprintf(stderr,"Sorry, cannot create your cursor\n");
	  return;
      }

    XDefineCursor(display,window,cursor);
}

/* Code for the next two routines was lifted from dsimple.c in 
   xwininfo */

Window window_by_name(wdw,name)
Window wdw;
char *name;
{
    Window *offspring;		/* Any children */
    Window junk;		/* Just that */
    Window w = 0;		/* Found window */
    int count;			/* Number of kids */
    int loop;			/* Loop counter */
    char *wdw_name;		/* Returnewd name */

    if (XFetchName(display,wdw,&wdw_name) && !strcmp(wdw_name,name))
      return(wdw);
    
    if (!XQueryTree(display,wdw,&junk,&junk,&offspring,&count))
      return(0);

    for (loop = 0; loop < count; loop++)
      {
	  w = window_by_name(offspring[loop],name);
	  if (w)
	    break;
      }

    if (offspring)
      XFree(offspring);

    return(w);
}
    
Window point_to_window()
{
    int status;
    Cursor cursor;
    XEvent event;
    Window target_win = None;
    int buttons = 0;
    
    /* Make the target cursor */
    cursor = XCreateFontCursor(display, XC_crosshair);
    
    /* Grab the pointer using target cursor, letting it room all over */
    status = XGrabPointer(display, RootWindow(display, screen), False,
			  ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			  GrabModeAsync, None, cursor, CurrentTime);
    if (status != GrabSuccess)
      {
	  fprintf(stderr,"Can't grab the mouse.");
	  exit(4);
      }
    
    /* Let the user select a window... */
    while ((target_win == None) || (buttons != 0)) /* allow one more event */
      {
	  XAllowEvents(display, SyncPointer, CurrentTime);
	  XWindowEvent(display, RootWindow(display, screen),
		       ButtonPressMask|ButtonReleaseMask, &event);
	  switch (event.type) 
	    {
	      case ButtonPress:
		if (target_win == None) 
		  {
		      /* window selected */
		      target_win = event.xbutton.subwindow; 
		      if (target_win == None)
			target_win = RootWindow(display,screen);
		  }
		buttons++;
		break;

		/* there may have been some down before we started */
	      case ButtonRelease:
		if (buttons > 0) 
		  buttons--;
		break;
	    }
      } 
    
    XUngrabPointer(display, CurrentTime);      /* Done with pointer */
    
    return(target_win);
}

void bestSize(char *sizes)
{
    Window rootwdw = DefaultRootWindow(display);
    int qw, qh;			/* Query size. */
    int bw, bh;			/* Answer. */
    char *sh = strchr(sizes,'x');

    if (sh == NULL)
      {
	  fprintf(stderr,
		  "Sorry but %s is malformed. Size is of the form nnxmm.\n",
		  sizes);
	  exit(1);
      }
    
    *sh = 0;
    sh++;

    qw = atoi(sizes);
    qh = atoi(sh);
    
    XQueryBestCursor(display,rootwdw,qw,qh,&bw,&bh);

    printf("Best size is %dx%d.\n",bw,bh);
}
