/*
 * Programm XBLAST V1.00 or higher
 * (C) by Oliver Vogel (e-mail: vogel@ikp.uni-koeln.de)
 * January 2nd 1993
 * started August 1993
 *
 * File: blastmain.c 
 * gameplay and main programm
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include "blastincl.h"
#include "blasttypes.h"
#include "blastconst.h"
#include <sys/time.h>
#include <time.h>
#include "patchlevel.h"

  /* some constants */
#define OTHER_RANDOM_MODE TRUE


#define INTRO_LENGTH (5*CHAR_ANIME)
#define FRAME_TIME 56000 /* 56000 */
#define BOMB_STEP 2
#define TIME_STEP 48 /* 48 */
#define DEAD_TIME 8
#define GAME_TIME (60*TIME_STEP + DEAD_TIME)
#define SCORE_TIME 16
#define MIN_RANGE 2



/* extern variables */

extern int Block_A[CHAR_ANIME][CHARH][CHARW];  
extern int Block_B[CHAR_ANIME][CHARH][CHARW];  
extern int Block_L[CHAR_ANIME][CHARH][CHARW];
extern int Block_M[CHAR_ANIME][CHARH][CHARW];  
extern int Block_O[CHAR_ANIME][CHARH][CHARW];  
extern int Block_S[CHAR_ANIME][CHARH][CHARW];  
extern int Block_T[CHAR_ANIME][CHARH][CHARW];  
extern char big_mask[SIZE_OF_BIG];
extern char big_bits[MAX_PLAYER][SIZE_OF_BIG];

  /* global variables */

static BMPlayer player_stat[MAX_PLAYER+1] = {
  { SPRITE_UP, SPRITE_LEFT, 0, GoStop,GoStop,GoDown,
      NEW_INVINCIBLE, 0, 0, 0, MAX_LIVES, 0, MIN_RANGE, 1, 0, 
      "Player_1",0 },
  { SPRITE_UP, SPRITE_RIGHT, 0, GoStop,GoStop,GoDown,
      NEW_INVINCIBLE, 0, 0, 0, MAX_LIVES, 0, MIN_RANGE, 1, 0, 
      "Player_2",0 },
  { SPRITE_DOWN, SPRITE_RIGHT, 0, GoStop,GoStop,GoDown,
      NEW_INVINCIBLE, 0, 0, 0, MAX_LIVES, 0, MIN_RANGE, 1, 0, 
      "Player_3",0 },
  { SPRITE_DOWN, SPRITE_LEFT, 0, GoStop,GoStop,GoDown,
      NEW_INVINCIBLE, 0, 0, 0, MAX_LIVES, 0, MIN_RANGE, 1, 0, 
      "Player_4",0 },
};

static struct
  {
    int num,alpha;
  }
control[MAX_PLAYER];



static int game_time, game_next, time_led,time_flag;
static int num_player,num_disp,max_lives;
static int pause_status = -1;
static int random_mode;
static int active_player;
static int level, start_level;
static int num_victories = 0;
static int max_victories;
static int pause_text_flag = FALSE;
static char *display[MAX_PLAYER];
static int sound_flag;


extern void do_bell();
extern void no_bell();

static void (*sound_function)();

void (*special_init_function)();
void (*special_revive_function)();
void (*special_game_function)();
void (*special_extra_function)();
void (*special_key_function)();


#include "blastfunc.h"



  /* local function bm_fatal */

static void bm_fatal(text,prog)
     char *text,*prog;
{
  fprintf(stderr,"ERROR : %s.\n",text);
  fprintf(stderr,"Type %s -? for help\n",prog); 
  exit(2);
}



  /* local function print_options */

static void print_options(text,name)
     char *text,*name;
{
  static char format[]="%-24s%-s\n";
  static char format2[]="%-12s%-12s%-s\n";

  if (text != NULL)
    fprintf(stderr,"Unknown Option %s\n\n",text);

  printf("Usage %s [-D] [ -option [argument] ] ... [-S] \n",name);
  printf("\n");
  printf(format,"-D[efault]","Load default settings from ~/.xblast");
  printf(format,"-S[save]","Save settings as default in ~/.xblast");
  printf("\n");
  printf(format,"-p[layer] n:m","n # of players, m # of displays");
  printf(format,"-d[isplay] <display>","specify your display");
  printf(format,"-l[evel] n","Start at level n");
  printf(format,"-o[ther] <list>","give a list of other players' displays");
  printf(format,"-n[ame] <list>","give a list of all players' names");
  printf(format,"-q[uiet]","no (bell) sound");
  printf(format,"-v[ictories] n","n number of victories to win the game");
  printf(format,"-L[ifes] n","n number of lifes");
  printf(format,"-h[elp] or -?","this message");
  printf("\n");
  printf(format2,"","Controls","");
  printf(format2,"Player1","","Player2");
  printf(format2,"2,4,6,8","Walk","t,f,h,b/v");
  printf(format2,"5","Stop","g");
  printf(format2,"0","Bomb","SPACE");
  printf(format2,"+","Special","TAB");
  printf("\n");
}



  /* local function check_commandline */

static void check_commandline(argc, argv)
      int argc;
      char **argv;
{
  int i,j;
  char msg[80];

  /* Set all defaults */
  start_level =0;
  num_player = -1;
  num_disp = -1;
  max_victories = 3;
  random_mode = 0;
  max_lives = MAX_LIVES;
  for (i=0; i<MAX_PLAYER; i++)
    {
      display[i] =  NULL;
    }
  sound_function = do_bell;


  for (i=1; i<argc; i++)
    {
      if (argv[i][0] != '-')
        print_options(argv[i],argv[0]);
      else
        switch(argv[i][1])
          {
	  case 'D':
	    if ( (strcmp(argv[i]+1,"Default") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
	        if (load_setup(&num_player, &num_disp, display, player_stat,
			&max_victories, &random_mode, &start_level,
			&sound_flag, &max_lives) )
		  bm_fatal("setup file does not exist",argv[0]);

		if (sound_flag)
		  sound_function = do_bell;
		else
		  sound_function = no_bell;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;

	  case 'L':
	    if ( (strcmp(argv[i]+1,"Lifes") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
		if ( (argc-i) <= 1)
		  bm_fatal("Option -L no number of lives given",argv[0]);
		sscanf(argv[i+1],"%d",&max_lives);
		if (max_lives < 1)
		  bm_fatal("Use positive number of lives",argv[0]);
                if (max_lives > MAX_LIVES)
		  bm_fatal("Too many lives",argv[0]);
		i++;
	      }
	    break;

	  case 'S':
	    if ( (strcmp(argv[i]+1,"Save") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
		if ( num_player < 0 )
		  bm_fatal("Define number of players first",argv[0]);
		if ( num_disp < 0 ) 
		  bm_fatal("Define number of displays first",argv[0]);
		save_setup(num_player, num_disp, display, player_stat,
			   max_victories, random_mode, start_level,
			   (sound_function == do_bell), max_lives );
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'd':
	    if ( (strcmp(argv[i]+1,"display") == 0)
		|| (strlen(argv[i]) == 2) )
	      /* read 1st Display */
	      {
		if ( (argc-i ) <= 1)
		  bm_fatal("Option -d no Display specified",argv[0]);
		display[0] = argv[i+1];
		i ++;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'l':
	    if ( (strcmp(argv[i]+1,"level")==0 )
		|| (strlen(argv[i])==2) )
	      /* read start_level */
	      {
		if ( (argc-i ) <= 1)
		  bm_fatal("Option -l no level specified",argv[0]);
		sscanf(argv[i+1],"%d",&start_level);
		i++;
		start_level--;
		if (start_level<0)
		  bm_fatal("Use positive level number",argv[0]);
		if (start_level >= LEVEL_MAX)
		  {
		    sprintf(msg,"Sorry only %d levels avaiable",LEVEL_MAX);
		    bm_fatal(msg,argv[0]);
		  }
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'h':
	  case '?':
	    print_options(NULL,argv[0]);
	    exit(0);
	    break;
	    
	  case 'n':
	    if ( (strcmp(argv[i]+1,"name") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
		if ( num_player < 0 )
		  bm_fatal("Define number of players first",argv[0]);
		if ( (argc-i ) <= num_player)
		  bm_fatal("Option -n list of names not complete",argv[0]);
		for (j=0; j<num_player; j++)
		  player_stat[j].name = argv[i+j+1];
		i += num_player;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'o':
	    if ( (strcmp(argv[i]+1,"other") == 0)
		|| (strlen(argv[i]) == 2) )
	      /* lese andere Displays */
	      {
		if ( num_disp < 0 ) 
		  bm_fatal("Define number of displays first",argv[0]);
		if ( (argc-i ) < num_disp)
		  bm_fatal("Option -o list of other displays not complete",
			   argv[0]);
		for (j=1; j<num_disp; j++)
		  display[j] = argv[i+j];
		i += num_disp-1;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'p':
	    if ( (strcmp(argv[i]+1,"player") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
		if ( (argc-i ) <= 1)
		  bm_fatal("Option -p number of players:displays is missing",
			   argv[0]);
		sscanf(argv[++i],"%d:%d",&num_player,&num_disp);
#if DEVELOP
		if (num_player <= 0)
		  bm_fatal("Use positive number of players",argv[0]);
#else
		if (num_player <= 1)
		  bm_fatal("Sorry at least 2 players",argv[0]);
#endif
		if  (num_player > MAX_PLAYER)
		  bm_fatal("Too many players",argv[0]);
		if (num_disp > (num_player))
		  bm_fatal("Too many displays",argv[0]);
		if (2*num_disp < num_player )
		  bm_fatal("Not enough displays",argv[0]);
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'q':
	    if ( (strcmp(argv[i]+1,"quiet") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
#if DEVELOP
		printf("No sound\n");
#endif
		sound_function = no_bell;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;

	  case 'r':
	    if ( (strcmp(argv[i]+1,"random") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
#if DEVELOP
		printf("Random Mode\n");
#endif		
		random_mode = TRUE;
		start_level = rand() % LEVEL_MAX;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  case 'v':
	    if ( (strcmp(argv[i]+1,"victories") == 0)
		|| (strlen(argv[i]) == 2) )
	      {
		if ( (argc-i ) <= 1)
		  bm_fatal("Option -d number of victories not specified",
			   argv[0]);
		sscanf(argv[i+1],"%d",&max_victories);
		if (max_victories < 1)
		  bm_fatal("Use positive number of victories",argv[0]);
		if (max_victories > 5)
		  bm_fatal("Sorry only up to 5 victories"),argv[0]; 
#if DEVELOP
		printf("MV: %d\n",max_victories);
#endif
		i++;
	      }
	    else
	      print_options(argv[i],argv[0]);
	    break;
	    
	  default :
	    print_options(argv[i],argv[0]);
	    break;
	  }
    }
}




/* public function print_frames */

#if DEVELOP
static frame_count = 0;
static int del_yes = 0;
static int del_no = 0;

void print_frames()
{
  fprintf(stderr,"Total Frames = %d\n",frame_count);
  fprintf(stderr,"Total Delays = %d\n",del_yes);
  fprintf(stderr,"Nondelayed   = %d\n",del_no);
}
#endif




/* local function randomize_levels */

static void randomize_levels(feld)
     int feld[LEVEL_MAX];
{
  int i, j;
  int hilf;
  int old_end;
  
  old_end = feld[LEVEL_MAX-1];

  for (i=0; i < LEVEL_MAX; i++)
    {
      j = (rand()>>4) % LEVEL_MAX;
      hilf = feld[i];
      feld[i] = feld[j];
      feld[j] = hilf;
    }

  if (old_end == feld[0])
    {
      j = 1+(rand()>>4) % (LEVEL_MAX-1);
      hilf = feld[0];
      feld[0] = feld[j];
      feld[j] = hilf;
    }
}

/* local function game_expose */

static int game_expose(player, sendevent)
     int player, sendevent;
{
#if DEVELOP
  frame_count ++;
#endif

  if ( !sendevent || (pause_status == -1))
    {
      update_maze(player);
      draw_sprites(player);
      if (time_flag)
        draw_time_led(player, time_led, 0);
    }
  flush_pixmap(player, num_disp, sendevent);

  return(sendevent);
}



static int intro_expose(player, sendevent)
     int player, sendevent;
{
#if DEVELOP
  frame_count ++;
#endif
  if ( !sendevent || (pause_status == -1))
    {
      update_expl(player);
      draw_sprites(player);
      if (time_flag)
        draw_time_led(player, time_led, 0);
    }
  flush_pixmap(player, num_disp, sendevent);

  return(sendevent);
}



/* local function wait_expose */

static int wait_expose(player, sendevent)
     int player, sendevent;
{
  if (!sendevent)
    flush_pixmap(player, num_disp, sendevent);

  return(0);
}



/* local function wait_expose */

static int fade_out_expose(player, sendevent)
     int player, sendevent;
{
  if (!sendevent)
    flush_pixmap(player, num_disp, sendevent);
  fade_out_window(player, num_disp);
  return(0);
}



static int fade_in_expose(player, sendevent)
     int player, sendevent;
{
  fade_in_window(player, num_disp);
  return(0);
}


static int radius = 0;

static circle_expose(player, dummy)
     int player, dummy;
{
  draw_circle_from_pixmap(player, PIXW/2, PIXH/2, radius);
}


/* local function wait_key */

static wait_key(player, key)
     int player;
     char key;
{
  return((int)(key == ' '));
}



/* local function void_key */

static void_key(player, key)
     int player;
     char key;
{
  return(FALSE);
}



/* local function drop_bomb */

static void drop_bomb(player, ps)
     int player;
     BMPlayer *ps;
{
  if (ps->bombs !=0 && ps->illness!=IllEmpty)
    {
      if (ps->lives >0)
        {
          new_explosion(player, (ps->x + BLOCK_WIDTH/2)/BLOCK_WIDTH,
                        (ps->y + BLOCK_HEIGHT + BLOCK_HEIGHT/2)
                        /BLOCK_HEIGHT,
                        (ps->illness == IllMini) ? 1 : ps->range,
			(ps->remote_control > 0) );
          ps->bombs --;
        }
    }
}



  /* local function game key */

static int game_key(player, key)
     int player;
     char key;
{
  int disp;
  BMPlayer *ps1,*ps2;

  ps1 = player_stat + control[player].num;
  ps2 = player_stat + control[player].alpha;

  switch(key)
    {
    case 'p':
      if ( pause_status  == player )
        {
          /* Pause off */
          pause_status = -1;
          for (disp = 0; disp < num_disp; disp ++)
            draw_text(disp, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                      get_level_name(level), FALSE, TRUE, 404, 32);
	  pause_text_flag = TRUE;
        }
      else
        if ( pause_status == -1)
          {
            /* Pause on */
            pause_status = player;
            for (disp = 0; disp < num_disp; disp ++)
              draw_text(disp, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                        "Game Paused", FALSE, TRUE, 404, 32);
	    pause_text_flag = TRUE;
          }
      break;

      /* bombing */
    case '0':
      drop_bomb(control[player].num, ps1);
      break;

      /* down */
    case '2':
      ps1->d_soll = GoDown;
      if (ps1->d_ist == GoUp)
        ps1->d_ist = GoDown;
      break;

      /* left */
    case '4':
      ps1->d_soll = GoLeft;
      if (ps1->d_ist == GoRight)
        ps1->d_ist = GoLeft;
      break;

      /* stop */
    case '5':
      ps1->d_soll = GoStop;
      break;

      /* right */
    case '6':
      ps1->d_soll = GoRight;
      if (ps1->d_ist == GoLeft)
        ps1->d_ist = GoRight;
      break;

      /* up */
    case '8':
      ps1->d_soll = GoUp;
      if (ps1->d_ist == GoDown)
        ps1->d_ist = GoUp;
      break;

      /* special */
    case '+':
      (*special_key_function)(ps1, control[player].num);
      break;


      /* bomb */
    case ' ':
      drop_bomb(control[player].alpha, ps2);
      break;

      /* down */
    case 'b':
    case 'v':
      ps2->d_soll = GoDown;
      if (ps2->d_ist == GoUp)
        ps2->d_ist = GoDown;
      break;

      /* left */
    case 'f':
      ps2->d_soll = GoLeft;
      if (ps2->d_ist == GoRight)
        ps2->d_ist = GoLeft;
      break;

      /* stop */
    case 'g':
      ps2->d_soll = GoStop;
      break;

      /* right */
    case 'h':
      ps2->d_soll = GoRight;
      if (ps2->d_ist == GoLeft)
        ps2->d_ist = GoRight;
      break;

      /* up */
    case 't':
      ps2->d_soll = GoUp;
      if (ps2->d_ist == GoDown)
        ps2->d_ist = GoUp;
      break;

      /* special */
    case 9: /* TAB */
      (*special_key_function)(ps2, control[player].alpha);
      break;

    }
}



  /* local fucntion walk_stop */

static int walk_stop(ps)
     BMPlayer *ps;
{
  switch(ps->d_look)
    {
    case GoDown:
      ps->anime = 0;
      break;
    case GoUp:
      ps->anime = 3;
      break;
    case GoLeft:
      ps->anime = 9;
      break;
    case GoRight:
      ps->anime = 6;
      break;
    }

  return(FALSE);
}




  /* local function walk_up */

static int walk_up(ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex, mazey-1)))
    ps->y -= STEP_VERT;
  else
    {
      ps->d_ist = GoStop;
      look_to_wall = TRUE;
    }

  switch(ps->y % (STEP_VERT*4))
    {
    case 0:
    case (2*STEP_VERT):
      ps->anime = 3;
      break;
    case STEP_VERT:
      ps->anime = 4;
      break;
    case (STEP_VERT*3):
      ps->anime = 5;
      break;
    }
  return(look_to_wall);
}



  /* local function walk_left */

static int walk_left(ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex -1, mazey) ))
    ps->x -= STEP_HORI;
  else
    {
      ps->d_ist = GoStop;
      look_to_wall = TRUE;
    }

  switch(ps->x % (STEP_HORI*4))
    {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 9;
      break;
    case STEP_HORI:
      ps->anime = 10;
      break;
    case (STEP_HORI*3):
      ps->anime = 11;
      break;
    }
  return(look_to_wall);
}



  /* local function walk_down */

static int walk_down(ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex, mazey+1) ))
    ps->y += STEP_VERT;
  else
    {
      look_to_wall = TRUE;
      ps->d_ist = GoStop;
    }

  switch(ps->y % (STEP_VERT*4))
    {
    case 0:
    case (STEP_VERT*2):
      ps->anime = 0;
      break;
    case STEP_VERT:
      ps->anime = 1;
      break;
    case (STEP_VERT*3):
      ps->anime = 2;
      break;
    }
  return(look_to_wall);
}



  /* local function walk_right */

static int walk_right(ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex +1, mazey)))
    ps->x += STEP_HORI;
  else
    {
      look_to_wall = TRUE;
      ps->d_ist = GoStop;
    }

  switch(ps->x % (STEP_HORI*4))
    {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 6;
      break;
    case STEP_HORI:
      ps->anime = 7;
      break;
    case (STEP_HORI*3):
      ps->anime = 8;
      break;
    }
  return(look_to_wall);
}



  /* local function walk_and_bomb */

static void do_walk(player)
      int player;
{
  int flag;
  int look_to_wall = FALSE;
  int mazex,mazey;
  int i,x ;
  int xalt,yalt;
  int disp;

  BMPlayer *ps;

  ps = player_stat + player;
  xalt = ps->x;
  yalt = ps->y;

  if ( (ps->illness == IllSlow) && (ps->illtime & 0x01) )
    {
    }
  else
    for (i=0; i<=(ps->illness == IllRun); i++)
      {
        flag = FALSE;

        mazex = ps->x / BLOCK_WIDTH;
        mazey = ps->y / BLOCK_HEIGHT + 1;

        if ( ( (ps->x % BLOCK_WIDTH) == 0)
            && ((ps->y % BLOCK_HEIGHT) == 0) )
          {
            flag = TRUE;
            ps->d_ist = ps->d_soll;
            if (ps->d_ist != GoStop)
              ps->d_look = ps->d_ist;
          }

        switch(ps->d_ist)
          {
          case GoStop:
            look_to_wall = walk_stop(ps);
            break;

          case GoUp:
            look_to_wall = walk_up(ps, flag, mazex, mazey );
            break;

          case GoLeft:
            look_to_wall = walk_left(ps, flag, mazex, mazey );
            break;

          case GoDown:
            look_to_wall = walk_down(ps, flag, mazex, mazey );
            break;

          case GoRight:
            look_to_wall = walk_right(ps, flag, mazex, mazey );
            break;
          }

        switch(ps->d_ist)
          {
            case GoLeft:
              if ( ((ps->x % BLOCK_WIDTH) == (STEP_HORI*BOMB_STEP) )
                 && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1) )
		ps->x += STEP_HORI;
            break;

            case GoRight:
              if ( ( (ps->x % BLOCK_WIDTH)
                     == (BLOCK_WIDTH - STEP_HORI*BOMB_STEP) )
                 && check_bomb(ps->x/BLOCK_WIDTH+1, ps->y/BLOCK_HEIGHT+1) )
		ps->x -= STEP_HORI;
            break;

            case GoUp:
              if ( ( (ps->y % BLOCK_HEIGHT) == (STEP_VERT*BOMB_STEP) )
                 && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1))
		ps->y += STEP_VERT;
                  break;

            case GoDown:
              if ( ( (ps->y % BLOCK_HEIGHT)
                     == (BLOCK_HEIGHT - STEP_VERT*BOMB_STEP) )
                 && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+2))
		ps->y -= STEP_VERT;
                break;
          }
      }



  if (look_to_wall || ps->d_ist != GoStop || ps->invincible > 0
      || ps->dying > 0)
    mark_maze( (int)((MIN(ps->x,xalt)+ SPRITE_X_OFF )/ BLOCK_WIDTH),
              (int)((MIN(ps->y,yalt) + SPRITE_Y_OFF) / BLOCK_HEIGHT),
              (int)((MAX(ps->x,xalt) + SPRITE_X_OFF + SPRITE_WIDTH -1)
                    / BLOCK_WIDTH),
              (int)((MAX(ps->y,yalt) + SPRITE_Y_OFF + SPRITE_HEIGHT -1)
                    / BLOCK_HEIGHT) );

  if (ps->invincible > 0)
    add_player_to_sprite_list(player, ps->x, ps->y, ps->anime,
                              ( ps->invincible-- & 0x01 ));
  else
    add_player_to_sprite_list(player, ps->x, ps->y, ps->anime, TRUE);

  if (ps->illness != Healthy)
    {
      if ( (ps->illtime --) == 0)
	{
	  ps->illness = Healthy;
	  x = (2*player + (player > 1 ? 7 : 0));
	  for (disp = 0; disp < num_disp; disp ++)
	    draw_score_block(disp, x, SBPlayer + player);
	  mark_maze(x, MAZE_H, x, MAZE_H);
	}
#if !OTHER_RANDOM_BOMB
      if (ps->illness == IllBomb)
        {
          if ( (rand() % BOMB_PROB) == 0)
            drop_bomb(player, ps);
        }
#endif
    }

  if ( (ps->x % BLOCK_WIDTH == 0) && (ps->y % BLOCK_HEIGHT == 0) )
    {
#if OTHER_RANDOM_BOMB
      if (ps->illness == IllBomb)
        {
          if ( ( (rand()>>4) % 4) != 0)
            drop_bomb(player, ps);
        }
#endif
      switch(get_extra(ps->invincible, ps->x / BLOCK_WIDTH ,
                       ps->y / BLOCK_HEIGHT + 1 ))
        {
        case BTBomb:
          ps->bombs ++;
          break;
        case BTRange:
	  if (ps->range < MAX_RANGE)
	    ps->range ++;
          break;
        case BTSick:
          ps->illtime = ILLTIME;
          ps->illness = (rand()>>4)%MAX_ILL+1;

	  x = (2*player + (player > 1 ? 7 : 0));
	  for (disp = 0; disp < num_disp; disp ++)
	    draw_score_block(disp, x, SBSick + player);
	  mark_maze(x, MAZE_H, x, MAZE_H);
          break;
	case BTSpecial:
	  (*special_extra_function)(ps, player, num_disp);
	  break;
        }
    }
}



static void infect_other_players()
{
  BMPlayer *ps1,*ps2;
  int player, x, disp;

  for (ps1 = player_stat; ps1 < player_stat+num_player; ps1 ++)
    for (ps2= ps1+1; ps2 < player_stat+num_player; ps2 ++)
      if ( (ps1->illness != ps2->illness)
	  && (ABS(ps1->x - ps2->x) < ILL_X)
	  && (ABS(ps1->y - ps2->y) < ILL_Y))
	{
	  if ( (ps2->invincible==0)
	      &&  (ps1->illtime > ps2->illtime) )
	    {
	      ps2->illness = ps1->illness;
	      ps2->illtime = ILLTIME;
	      
	      player = ps2 - player_stat;
	      x = (2*player + (player > 1 ? 7 : 0));
	      for (disp = 0; disp < num_disp; disp ++)
		draw_score_block(disp, x, SBSick + player);
	      mark_maze(x, MAZE_H, x, MAZE_H);
	    }
	  else if ( (ps1->invincible==0)
		   &&  (ps2->illtime > ps1->illtime) )
	    {
	      ps1->illness = ps2->illness;
	      ps1->illtime = ILLTIME;
	      
	      player = ps1 - player_stat;
	      x = (2*player + (player > 1 ? 7 : 0));
	      for (disp = 0; disp < num_disp; disp ++)
		draw_score_block(disp, x, SBSick + player);
	      mark_maze(x, MAZE_H, x, MAZE_H);
	    }
	}
}


static print_player_lives(player, flag)
     int player, flag;
{
  BMPlayer *ps;
  int disp;
  int x;

  ps = player_stat + player;


  x = (2*player + 1 + (player > 1 ? 7 : 0));
  mark_maze(x, MAZE_H, x, MAZE_H);

  if (flag)
    {
      for (disp = 0; disp < num_disp; disp ++)
        draw_score_block(disp, x, SBLives + ps->lives);

      if (ps->lives == 0)
        {
          mark_maze(x-1, MAZE_H, x-1, MAZE_H);
          for (disp = 0; disp < num_disp; disp ++)
            draw_score_block(disp, x-1, SBDead + player);
        }
    }
  else
    {
      for (disp = 0; disp < num_disp; disp ++)
        draw_score_block(disp, x, SBLives + ps->victories);

      mark_maze(x-1, MAZE_H, x-1, MAZE_H);
      for (disp = 0; disp < num_disp; disp ++)
        draw_score_block(disp, x-1, SBPlayer + player);
    }
}



static void revive_player(player)
     int player;
{
  BMPlayer *ps;
  int disp, x;

  ps = player_stat + player;


  ps->lives --;
  if ( ps->lives == 0)
    active_player --;
  ps->invincible = NEW_INVINCIBLE;
  ps->dying = 0;
  ps->illness = Healthy;
  ps->illtime = 0;

  /* very important */
  if (ps->remote_control > 0)
    ignite_players_bombs(player);

  ps->remote_control = 0;

  print_player_lives(player, TRUE);

  mark_maze( (int)((ps->x + SPRITE_X_OFF)/ BLOCK_WIDTH),
            (int)((ps->y +SPRITE_Y_OFF)/ BLOCK_HEIGHT),
            (int)((ps->x + SPRITE_X_OFF + SPRITE_WIDTH -1) / BLOCK_WIDTH),
            (int)((ps->y + SPRITE_Y_OFF + SPRITE_HEIGHT -1) / BLOCK_HEIGHT) );

  if (ps->lives != 0)
    {
      x = (2*player + (player > 1 ? 7 : 0));
      for (disp = 0; disp < num_disp; disp ++)
	draw_score_block(disp, x, SBPlayer + player);
      mark_maze(x, MAZE_H, x, MAZE_H);

      (*special_revive_function)(player);
    }
}



static void do_die(player)
     int player;
{

  if (player_stat[player].lives > 1)
    add_player_to_sprite_list
      (player,player_stat[player].x,player_stat[player].y,
       13,TRUE);
  else
    add_player_to_sprite_list
      (player,player_stat[player].x,player_stat[player].y,
       12,TRUE);

  if (player_stat[player].dying == DEAD_TIME)
    mark_maze( (int)((player_stat[player].x + SPRITE_X_OFF)/ BLOCK_WIDTH),
              (int)((player_stat[player].y + SPRITE_Y_OFF)/ BLOCK_HEIGHT),
              (int)((player_stat[player].x + SPRITE_X_OFF + SPRITE_WIDTH -1)
                    /BLOCK_WIDTH),
              (int)((player_stat[player].y + SPRITE_Y_OFF + SPRITE_HEIGHT-1)
                    /BLOCK_HEIGHT)
              );
  player_stat[player].dying -- ;
}



void add_player_num_bomb(player)
     int player;
{
  player_stat[player].bombs ++;
}


static struct timeval tv1,tv2,tvgame;
static struct timezone tzp;



static void init_delta_time()
{
  gettimeofday(&tv1,&tzp);
  tvgame = tv1;
}



static int get_delta_time()

{
  int rueck;

  gettimeofday(&tv2,&tzp);

  rueck = ( 1000000*(tv2.tv_sec-tv1.tv_sec)+(tv2.tv_usec-tv1.tv_usec));
  tv1 = tv2;
  return(rueck);
}




static int wait_micsec(i)
     int i;
{
  static struct timeval tv;

  if (i>0)
    {
#if DEVELOP
      del_yes ++;
#endif
      tv.tv_sec = 0;
      tv.tv_usec = i;
      select(0,NULL,NULL,NULL,&tv);
    }
#if DEVELOP
  else
    del_no++;
#endif
  return(i>0 ? i : 0);
}



/* local function fade_in */

static void fade_in_out(in)
     int in;
{
  int player;
  int count,last_delay;
  count = FADE_STEP;

  init_delta_time();
  last_delay = 0;
  while (count > 1)
    {
      init_fade(count);
      for (player =0; player < num_disp; player ++)
        wait_event(player, in ? fade_in_expose : fade_out_expose , void_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);

      count /= 2;
    }
}



/* local function fade_in */

static void circle_in()
{
  int player;
  int count,last_delay;
  count = FADE_STEP;

  init_delta_time();
  last_delay = 0;
  radius = 16;
  while (radius*radius < (PIXW*PIXW + PIXH*PIXH)/4)
    {
      init_fade(count);
      for (player =0; player < num_disp; player ++)
        wait_event(player, circle_expose , void_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);

      radius +=16;
    }
}




/* local function wait_for_space */

static void wait_two_text(text1, text2)
     char *text1, *text2;
{
  int event, player;
  int last_delay;
  int count;

  mark_maze(4, MAZE_H, 10, MAZE_H);
  set_redraw_rectangles();
  for (player = 0; player < num_disp; player ++)
    {
      draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                text1, FALSE, TRUE, 404, 32);
      flush_pixmap(player, num_disp, TRUE);
    }
  clear_redraw_map();

  event = 0;
  init_delta_time();
  last_delay = 0;
  count = 0;
  while (event == 0)
    {
      count ++;
      if (count & 0x1f)
        {
          mark_maze(4, MAZE_H, 10, MAZE_H);
          if (count & 0x20)
            for (player = 0; player < num_disp; player ++)
              draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                       text2, FALSE, TRUE, 404, 32);
          else
            for (player = 0; player < num_disp; player ++)
              draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                       text1, FALSE, TRUE, 404, 32);

          set_redraw_rectangles();
          for (player = 0; player < num_disp; player ++)
            flush_pixmap(player, num_disp, TRUE);
          clear_redraw_map();
        }

      for (player = 0; player < num_disp; player ++)
        event |= wait_event(player, wait_expose, wait_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);
    }
}



extern BMPoint pointx[SIZE_OF_X];

/* local function do_intro */

static void do_intro()
{
  static char disp_name[]="X\0";
  int disp, event = 0;
  int count;
  int last_delay;
  int player;


  for (disp = 0; disp < num_disp; disp ++)
    {
      init_block(disp, BLScoreFloor, BTFree);
      init_explosion_blocks(disp);
      clear_pixmap(disp);
      draw_text(disp, 0, 304, "On a Workstation",FALSE, FALSE, PIXW, 32);
      draw_text(disp, 0, 352, "not far away",FALSE, FALSE, PIXW, 32);
    }

  set_fade_max(PIXH+SCOREH);
  fade_in_out(TRUE);

  for (disp = 0; disp < num_disp; disp ++)
    flush_pixmap(disp, num_disp, FALSE);

  init_delta_time();
  last_delay = 0;
  while (event ==  0)
    {
      for (disp = 0; disp < num_disp; disp ++)
	event |= wait_event(disp, wait_expose, wait_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);
    }

  for (disp = 0; disp < num_disp; disp ++)
    {
      clear_pixmap(disp);
      draw_polygon(disp, 5*BLOCK_WIDTH, BLOCK_HEIGHT, 
		   BLOCK_WIDTH*5, BLOCK_HEIGHT*6, pointx, SIZE_OF_X, TRUE);
    }

  fade_in_out(TRUE);
  for (disp = 0; disp < num_disp; disp ++)
    flush_pixmap(disp, num_disp, FALSE);


  init_delta_time();
  last_delay = 0;

  
  for (count = 0; count < INTRO_LENGTH; count ++)
    {
      if (count < CHAR_ANIME)
	copy_expl_block( 0, 8, Block_B[count]);
      else if (count < (CHAR_ANIME*2))
	copy_expl_block( 3, 8, Block_L[count-CHAR_ANIME]);
      else if (count < (CHAR_ANIME*3))
	copy_expl_block( 6, 8, Block_A[count-CHAR_ANIME*2]);
      else if (count < (CHAR_ANIME*4))
	copy_expl_block( 9, 8, Block_S[count-CHAR_ANIME*3]);
      else
	copy_expl_block(12, 8, Block_T[count-CHAR_ANIME*4]);
      
      if (count==(INTRO_LENGTH-1))
	{
	  

	  for (disp = 0; disp < num_disp; disp ++) 
	    {
	      init_block(disp, BLIronFloor, BTFree);
	      init_block(disp, BLControlNum, 1);
	      init_block(disp, BLControlAlpha, 2);
	      init_block(disp, BLDisplay, 3);
	      draw_text(disp, 3*BLOCK_WIDTH, PIXH+12, c_string,FALSE, TRUE, 
			9*BLOCK_WIDTH-1, 40);
	    }
	  mark_maze(3, 13, 11, 14);
	  
	  clear_sprite_list();
	  
	  for (player = 0; player < num_player; player ++)
	    switch(player)
	      {
	      case 0:
		add_player_to_sprite_list(player, 5*BLOCK_WIDTH, -4, 0, TRUE);
		for (disp = 0; disp < num_disp; disp ++)
		  {
		    draw_text(disp, BLOCK_WIDTH, 1*BLOCK_HEIGHT, 
			      player_stat[player].name, FALSE, TRUE,
			      3*BLOCK_WIDTH-1, 40);
		    /* Floor */
		    draw_block(disp, 5, 1, 0);
		    /* Display */
		    draw_block(disp, 1, 0, 3);
		    disp_name[0]='1';
		    draw_text(disp, BLOCK_WIDTH+16, 16, 
			      disp_name, TRUE, FALSE,
			      28, 20);
		    if ((player % num_disp) == disp)
		      {
			/* Alpha Control */
			if (player >= (num_player-num_disp))
			  draw_block(disp, 2, 0, 2);
			/* Num Control */
			if (player < num_disp)
			  draw_block(disp, 3, 0, 1);
		      }
		  }
		mark_maze(5,0,5,1);
		mark_maze(1,0,4,1);
		break;
	      case 1:
		add_player_to_sprite_list(player, 9*BLOCK_WIDTH, -4, 0, TRUE);
		for (disp = 0; disp < num_disp; disp ++)
		  {
		    draw_text(disp, 11*BLOCK_WIDTH, 1*BLOCK_HEIGHT, 
			      player_stat[player].name, FALSE, TRUE,
			      3*BLOCK_WIDTH-1, 40);
		    /* Floor */
		    draw_block(disp, 9, 1, 0);
		    /* Display */
		    draw_block(disp, 11, 0, 3);
		    disp_name[0]=(player < num_disp) ? '2':'1';
		    draw_text(disp, 11*BLOCK_WIDTH+16, 16, 
			      disp_name, TRUE, FALSE,
			      28, 20);
		    if ((player % num_disp) == disp)
		      {
			/* Alpha Control */
			if (player >= (num_player-num_disp))
			  draw_block(disp, 12, 0, 2);
			/* Num Control */
			if (player < num_disp)
			  draw_block(disp, 13, 0, 1);
		      }
		  }
		mark_maze(9,0,9,1);
		mark_maze(11,0,13,1);
		break;
	      case 2:
		add_player_to_sprite_list(player, 9*BLOCK_WIDTH, 
					  5*BLOCK_HEIGHT -4, 
					  0, TRUE);
		for (disp = 0; disp < num_disp; disp ++)
		  {
		    draw_text(disp, 11*BLOCK_WIDTH, 6*BLOCK_HEIGHT, 
			      player_stat[player].name, FALSE, TRUE,
			      3*BLOCK_WIDTH-1, 40);
		    /* Floor */
		    draw_block(disp, 9, 6, 0);
		    /* Display */
		    draw_block(disp, 11, 5, 3);
		    disp_name[0] = (player < num_disp) ? '3':'1';
		    draw_text(disp, 11*BLOCK_WIDTH+16, 5*BLOCK_HEIGHT+16, 
			      disp_name, TRUE, FALSE,
			      28, 20);
		    if ((player % num_disp) == disp)
		      {
			/* Alpha Control */
			if (player >= (num_player-num_disp))
			  draw_block(disp, 12, 5, 2);
			/* Num Control */
			if (player < num_disp)
			  draw_block(disp, 13, 5, 1);
		      }
		  }
		mark_maze(9,5,9,6);
		mark_maze(11,5,13,6);
		break;
	      case 3:
		add_player_to_sprite_list(player, 5*BLOCK_WIDTH, 
					  5*BLOCK_HEIGHT -4, 
					  0, TRUE);
		for (disp = 0; disp < num_disp; disp ++)
		  {
		    draw_text(disp, BLOCK_WIDTH, 6*BLOCK_HEIGHT, 
			      player_stat[player].name, FALSE, TRUE,
			      3*BLOCK_WIDTH-1, 40);
		    /* Floor */
		    draw_block(disp, 5, 6, 0);
		    /* Display */
		    draw_block(disp, 1, 5, 3);
		    disp_name[0] = (player < num_disp) 
		      ? '4':'1'+(player-num_disp);
		    draw_text(disp, 1*BLOCK_WIDTH+16, 5*BLOCK_HEIGHT+16, 
			      disp_name, TRUE, FALSE,
			      28, 20);
		    if ((player % num_disp) == disp)
		      {
			/* Alpha Control */
			if (player >= (num_player-num_disp))
			  draw_block(disp, 2, 5, 2);
			/* Num Control */
			if (player < num_disp)
			  draw_block(disp, 3, 5, 1);
		      }
		  }
		mark_maze(5,5,5,6);
		mark_maze(1,5,4,6);
		break;
	      }
	  
	  for (disp = 0; disp < num_disp; disp ++)
	    draw_sprites(disp);
	}

      

      set_redraw_rectangles();
      for (disp = 0; disp < num_disp; disp ++)
	wait_event(disp, intro_expose, void_key);
      clear_redraw_map();
    }


  event = 0;
  init_delta_time();
  last_delay = 0;
  while (event ==  0)
    {
      for (disp = 0; disp < num_disp; disp ++)
	event |= wait_event(disp, wait_expose, wait_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);
    }
  fade_in_out(FALSE);

  set_fade_max(PIXH);

  for (disp = 0; disp < num_disp; disp ++)
    {
      free_block(disp, BTFree);
      free_explosion_blocks(disp);
    }

}



/* public function level_start  */

static void level_start()
{
  int player;

  for (player =0; player < num_disp; player ++)
    flush_score_board(player, num_disp);

  fade_in_out(TRUE);
}



/* after one game */

static char *msg_oot  = "Out Of Time";
static char *msg_draw = "Draw Game";
static char msg_win[80];
static int last_player;



static void level_end()
{
  int disp;
  char *Message;
  
  if (game_time >= (GAME_TIME-1))
    {
      Message = msg_oot;
      last_player = MAX_PLAYER;
    }
  else
    {
      if (active_player != 0)
	for (last_player=0; (last_player<num_player)
	     && (player_stat[last_player].lives == 0); last_player ++);
      else
	last_player = MAX_PLAYER;
      if ( (last_player >= num_player) ||
          (player_stat[last_player].dying > 0
           && (player_stat[last_player].lives == 1)))
	{
	  Message = msg_draw;
	  last_player = MAX_PLAYER;
	}
      else
        {
          player_stat[last_player].victories ++;
          num_victories = MAX(num_victories,
                              player_stat[last_player].victories);
          sprintf(msg_win,"%s wins",player_stat[last_player].name);
          Message = msg_win;
        }
    }

  clear_sprite_list();
  if (last_player < MAX_PLAYER)
    {
      add_player_to_sprite_list(last_player, 
				player_stat[last_player].x,
				player_stat[last_player].y,
				WINNER_ANIME, TRUE );
      mark_maze( (int)((player_stat[last_player].x + WINNER_X_OFF)
		       /BLOCK_WIDTH),
		(int)((player_stat[last_player].y + WINNER_Y_OFF)
		      /BLOCK_HEIGHT),
		(int)((player_stat[last_player].x 
		       + WINNER_X_OFF + WINNER_WIDTH)
		      /BLOCK_WIDTH),
		(int)((player_stat[last_player].y 
		       + WINNER_Y_OFF + WINNER_HEIGHT)
		      /BLOCK_HEIGHT)
		);
    }

  for (disp = 0; disp < num_disp; disp ++)
    {
      update_maze(disp);
      draw_sprites(disp);
      flush_pixmap(disp, num_disp, TRUE);
    }
  clear_sprite_list();

  wait_two_text(Message,"Press Space");

  fade_in_out(FALSE);


}



/* local function status_board */

static void status_board()
{
  int player,disp , i;
  int count;
  int event, last_delay;

  /* load score board maze */
  load_maze(player_stat, LEVEL_MAX, num_player, num_disp, max_lives);

  /* draw maze in pixmap */
  for (player = 0; player < num_disp; player ++)
    {
      draw_maze(player);
      draw_score_board(player, num_player);
      draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                get_level_name(LEVEL_MAX), FALSE, TRUE, 404, 32);
    }


  /* draw audience */

  for (i = 0; i < (3 * PIXW-128); i += 128)
    add_player_to_sprite_list((rand()>>4)%MAX_PLAYER, i/3, 0,0, TRUE);
  for (i = 64; i < (3 * PIXW-128); i += 128)
    add_player_to_sprite_list((rand()>>4)%MAX_PLAYER, i/3, -20,0, TRUE);

  /* draw player positions */
  for (player = 0; player < num_player; player ++)
    {
      print_player_lives(player, FALSE);
      for (disp = 0; disp < num_disp; disp ++)
        draw_text(disp, BLOCK_WIDTH/4, BLOCK_HEIGHT*(5+player*2) + 8,
                  player_stat[player].name, FALSE, TRUE,
                  (5*BLOCK_WIDTH) / 2 , 32);

      if (player == last_player)
	player_stat[player].anime = WINNER_ANIME;
      else if (player_stat[player].victories == num_victories)
	player_stat[player].anime = 0;
      else
	player_stat[player].anime = 14;


      add_player_to_sprite_list(player,
                                player_stat[player].x, player_stat[player].y,
                                player_stat[player].anime, TRUE);
      for (i = 0; i < player_stat[player].victories; i ++)
        {
          add_trophy_to_sprite_list( i, player);
	}
    }


  for (player = 0; player < num_disp; player ++)
    draw_sprites(player);
  clear_sprite_list();

  level_start();

  for (player = 0; player < num_disp; player ++)
    flush_pixmap(player, num_disp, FALSE);


  last_delay = 0;
  for (count = 0; count < SCORE_TIME ; count ++)
    {
      for (player =0; player < num_disp; player ++)
	wait_event(player, wait_expose, void_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);
    }

  event = 0;
  while (event == 0)
    {
      for (player = 0; player < num_disp; player ++)
        event |= wait_event(player, wait_expose, wait_key);
      last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);
    }


  fade_in_out(FALSE);

}




/* local function winning the game */

static void winning_the_game()

{
  static char Message[80];
  int disp,player;
  int i;


  for (disp = 0; disp < num_disp; disp++)
    clear_window(disp);

  /* load score board maze */
  load_maze(player_stat, LEVEL_MAX+1, num_player, num_disp, max_lives);

  /* draw maze in pixmap */
  for (player = 0; player < num_disp; player ++)
    {
      draw_maze(player);
      draw_score_board(player, num_player);
      draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                get_level_name(LEVEL_MAX), FALSE, TRUE, 404, 32);
    }

  for (i = 0; i < (3 * PIXW-128); i += 128)
    {
      player = (rand()>>4)%MAX_PLAYER;
      add_player_to_sprite_list(player, i/3, 0,
				(player == last_player)? 15 : 14 , TRUE);
    }

  for (i = 64; i < (3 * PIXW-128); i += 128)
    {
      player = (rand()>>4)%MAX_PLAYER;
      add_player_to_sprite_list(player, i/3, -20,
				(player == last_player)? 15 : 14 , TRUE);
    }



  /* draw player positions */
  for (player = 0; player < num_player; player ++)
    {
      print_player_lives(player, FALSE);
#if 0
      for (disp = 0; disp < num_disp; disp ++)
        draw_text(disp, BLOCK_WIDTH/4, BLOCK_HEIGHT*(5+player*2) + 8,
                  player_stat[player].name, FALSE, TRUE,
                  (5*BLOCK_WIDTH) / 2 , 32);
#endif
      if (player == last_player)
	{
	}
      else 
	{
	  player_stat[player].anime = 14;
	  add_player_to_sprite_list(player,
				    player_stat[player].x, 
				    player_stat[player].y,
				    player_stat[player].anime, TRUE);
	}
    }

  for (disp = 0; disp < num_disp; disp ++)
    {
      draw_sprite_from_bitmap(disp, 
			      (PIXW-BIG_WIDTH)/2,(PIXH-BIG_HEIGHT)/2-24,
			      BIG_WIDTH, BIG_HEIGHT, 
			      big_mask, big_bits[last_player]);
      draw_text(disp, 6*BLOCK_WIDTH+16, BLOCK_HEIGHT*8 + 8,
		player_stat[last_player].name, FALSE, TRUE,
		(5*BLOCK_WIDTH) / 2 , 32);
    }
 
  for (player = 0; player < num_disp; player ++)
    draw_sprites(player);
  clear_sprite_list();




  circle_in();
  for (disp = 0; disp < num_disp; disp ++)
    {
      flush_pixmap(disp, num_disp, FALSE);
    }
  
  sprintf(Message, "The Winner is %s\0",player_stat[last_player].name);
  wait_two_text("CONGRATULATIONS!","Press Space");
  
  fade_in_out(FALSE);
}




  /* main programm what else */

main(argc, argv)
     int argc;
     char *argv[];
{
  int player, disp;
  int last_delay;
  int level_index;
  int level_field[LEVEL_MAX];

  printf("%s\n",c_string);
  printf("Report any bugs to: vogel@ikp.uni-koeln.de\n");

  /* Initialize Random */
  srand(getpid());

  if (1 != argc)
    {
      check_commandline(argc, argv);
      if (num_player <= 0)
	bm_fatal("Number of players not defined",argv[0]);
      if (num_disp <= 0)
	bm_fatal("Number of displays not defined",argv[0]);
    }
  else
    {
      interactive_setup(&num_player, &num_disp, display, player_stat,
			&max_victories, &random_mode, &start_level,
			&sound_flag, &max_lives);
      if (sound_flag)
	sound_function = do_bell;
      else
	sound_function = no_bell;
	
    }

  /* set status board */
  set_victories(max_victories);

  /* set control */
  for (player = 0; player < num_disp; player ++)
    control[player].num = player;
  for (player = 0; player < (num_player - num_disp) ; player ++)
    control[player].alpha = player + num_disp;
  for (player = (num_player - num_disp); player  < num_disp; player ++ )
    control[player].alpha = player;


  /* Initialize Graphics */
  for (player=0; player < num_disp; player++)
    init_graphics(player, display[player]);

  /* Title Screen */
  do_intro();

  /* the game */

  for (level_index = 0; level_index < LEVEL_MAX; level_index ++)
    level_field[level_index] = level_index;

  if (random_mode)
    {
      level_index = 0;
      randomize_levels(level_field);
    }
  else
    level_index = start_level;

  while( num_victories < max_victories )
    {
      level = level_field[level_index];

      if ( (++ level_index) == LEVEL_MAX )
	{
	  level_index = 0;
	  if (random_mode)
	    randomize_levels(level_field);
	}
#if DEVELOP
      printf("Level %d\n",level);
#endif

      /* load maze from data */
      load_maze(player_stat, level, num_player, num_disp, max_lives);
      active_player = num_player;

      /* exec special_init_function */
      (*special_init_function)();

      /* draw maze in pixmap */
      for (player = 0; player < num_disp; player ++)
        {
          draw_maze(player);
          draw_score_board(player, num_player);
          draw_text(player, PIXW/2 - BLOCK_WIDTH * 3 - 8 , PIXH+8,
                    get_level_name(level), FALSE, TRUE, 404, 32);
        }

      /* mark player positions */
      for (player = 0; player < num_player; player ++)
        {
          mark_maze( (int)((player_stat[player].x + SPRITE_X_OFF)
                           / BLOCK_WIDTH),
                    (int)((player_stat[player].y + SPRITE_Y_OFF)
                          / BLOCK_HEIGHT),
                    (int)((player_stat[player].x + SPRITE_X_OFF + SPRITE_WIDTH
                           -1) / BLOCK_WIDTH),
                    (int)((player_stat[player].y + SPRITE_HEIGHT + SPRITE_Y_OFF
                           -1) / BLOCK_HEIGHT) );
          print_player_lives(player, TRUE);
        }

      level_start();

      for (player = 0; player < num_disp; player ++)
        flush_pixmap(player, num_disp, FALSE);

      game_time = 0;
      game_next = TIME_STEP;
      time_led = MAZE_W*4;

      init_delta_time();
      last_delay = 0;
      time_flag = FALSE;
      do
        {
          if (pause_status < 0)
            game_time ++;

          if (game_time == (GAME_TIME - DEAD_TIME + 1))
            for (player = 0; player < num_player; player ++)
              if (player_stat[player].lives>0)
                {
                  player_stat[player].lives = 1;
                  player_stat[player].dying = DEAD_TIME;
                }

          if (game_time > game_next)
            {
              game_next += TIME_STEP;
              time_led --;
              time_flag = TRUE;
              mark_maze( (time_led>>2), MAZE_H+1, (time_led>>2), MAZE_H+1);
            }

          if (pause_status == -1 )
            {
              clear_sprite_list();

              /* execute keypress et al */
              for (player = 0; player <num_player; player ++)
                if (player_stat[player].lives != 0)
                  switch(player_stat[player].dying)
                    {
                    case 0:
                      do_walk(player);
                      break;
                    case 1:
                      revive_player(player);
                      break;
                    default:
                      do_die(player);
                      break;
                    }

	      (*special_game_function)();

              do_bombs();

              /* sort spites by y-position */
              sort_sprite_list();

              if(do_explosions())
		for (disp = 0; disp < num_disp; disp ++)
		  (*sound_function)(disp);

              ignite_bombs();

              infect_other_players();

              for (player = 0; player < num_player; player ++)
                if ((player_stat[player].lives !=0  )
                    && (player_stat[player].invincible==0)
                    && check_explosion
                    ( (player_stat[player].x + (BLOCK_WIDTH>>1))/BLOCK_WIDTH,
                     (player_stat[player].y + (BLOCK_HEIGHT>>1))/BLOCK_HEIGHT
                     +1 ))
                  {
                    player_stat[player].dying = DEAD_TIME;
                  }


              /* mark blocks to be redrawed */
            }
         
	  if(pause_text_flag)
	    {
	      mark_maze(4,MAZE_H,10,MAZE_H);
	      pause_text_flag = FALSE;
	    }
	  
	  set_redraw_rectangles();

          /* check all events for all players */
          for (player = 0; player < num_disp; player ++)
            {
              wait_event(player, game_expose, game_key);
            }
          time_flag = FALSE;


          /* delay */
          last_delay=wait_micsec(FRAME_TIME - get_delta_time() + last_delay);

          clear_redraw_map();
        }

      while( (game_time<= GAME_TIME)
#if 0	    
	    && (active_player != 0) 
#endif
            && ( active_player > 1 || number_of_explosions()!=0 ));

      level_end();
      unload_blocks(num_disp);

      clear_sprite_list();
      status_board();
      unload_blocks(num_disp);

    }
  if (num_victories == max_victories)
    {
      winning_the_game();
    }
  printf("Ready.\n");
#if DEVELOP
  print_frames();
#endif
  return(0);
}







