/**********************************************************************
 *
 *	tranout.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/**********************************************************************
 *	include files
 **********************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * standard string defs
 */
#include  <string.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * client exerciser defines
 */
#include  <xtm.h>
/*
 * input synthesis type defs
 */
#include  <X11/extensions/xtestext1.h>

/***********************************************************************
 *	external variables
 **********************************************************************/

extern int	G_translate_type;
extern int	G_newline_flag;
extern int	G_debug_flag;
extern FILE	*G_msg_file;

/***********************************************************************
 *	variables
 **********************************************************************/

/*
 * holds the last input action type
 */
int	G_l_i_a_type = NO_INPUT_ACTION_PENDING;
/*
 * buffer for the last key input action parsed (used in summary format)
 */
XTestKeyInfo	G_l_key;
/*
 * buffer for the accumulated key delay (used in summary format)
 */
XTestDelayInfo	G_delay_for_keys;
/*
 * if true then G_delay_for_keys contains some accumulated key delay
 */
int		G_key_delay_flag;
/*
 * buffer for the last jump input action parsed (used in summary format)
 */
XTestJumpInfo	G_l_jump;
/*
 * if true then use checksums for match data
 */
int             G_checksum_flag;

/**********************************************************************
 *	function declarations
 **********************************************************************/

void	readchk();
void	script_to_text();
void	flush_line_check();	
void	inttosum();	
void	inttoout();	
char	*getkeysym();
Display	*XOpenDisplay();

/**********************************************************************
 *	functions
 **********************************************************************/

/**********************************************************************
 *
 *      add_time_check
 *
 *      This routine adds two time intervals and checks for time 
 *	longer than XTestSHORT_DELAY_TIME.
 */
static CARD16
add_time_check(to_fptr, time1, time2)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * argument for one time parameter (were CARD16 but get promoted)
 */
unsigned int	time1;
/*
 * argument for the other time parameter (were CARD16 but get promoted)
 */
unsigned int	time2;
{
	/*
	 * Local for longer time intervals
	 */
	INT32	temp;

	/*
	 * add the two time intervals
	 */
	temp = time1 + time2;
	/*
	 * Check for value larger than XTestSHORT_DELAY_TIME
	 */
	if (temp > XTestSHORT_DELAY_TIME)
	{
		flush_line_check(to_fptr);
		/*
		 * write out a time delay line if the time
		 * won't fit in the short delay time
		 */
		(void) fprintf(to_fptr, "<TIME %6ld>\n", temp);
		return(0);
	}
	else
	{
		return(temp);
	}
}

/**********************************************************************
 *
 *      write_KeyActions
 *
 *      This routine translates the specified key input action
 *	and outputs text accordingly.
 */
static void
write_KeyActions(to_fptr, input_action_count_ptr, keyInfo, display)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
/*
 * pointer to a KeyInfo input action
 */
XTestKeyInfo	*keyInfo;
/*
 * pointer to X11 display
 */
Display 	*display;
{
	/*
	 * key code
	 */
        CARD8	code = 0;
	/*
	 * ???
	 */
        int	i;
	/*
	 * ???
	 */
        int	blanklen;

	code = keyInfo->keycode;
	
	if (G_translate_type == SCRIPT_TO_OUTPUT)
	{

		if ((keyInfo->header & XTestKEY_STATE_MASK) == XTestKEY_DOWN)
                {
                	(void) fprintf(to_fptr,
				       "KEY_DOWN   DEV [%d]   ",
				       XTestUnpackDeviceID(keyInfo->header));
                }
                else
                {
			(void) fprintf(to_fptr,
				       "KEY_UP     DEV [%d]   ",
				       XTestUnpackDeviceID(keyInfo->header));
                }
                blanklen = 14 - strlen(getkeysym(code, display));
                (void) fprintf(to_fptr, "KEY [%s]", getkeysym(code, display));

                for (i = 0; i < blanklen; i++)
                {
                	(void) fprintf(to_fptr, " ");
                }

                (void) fprintf(to_fptr,
			       "TIME (ms) [%6u]    ",
			       keyInfo->delay_time);
                (void) fprintf(to_fptr,
			       "COUNT [%lu]\n",
			       ++(*input_action_count_ptr));
	}
        else
        {
		/*
		 * Summary output on key down
		 */
                if ((keyInfo->header & XTestKEY_STATE_MASK) == XTestKEY_DOWN)
		{
                        (void) fprintf(to_fptr,
				       "<KEY_DOWN %s>\n",
				       getkeysym(code, display));
		}
                else
		{
                        (void) fprintf(to_fptr,
				       "<KEY_UP %s>\n",
				       getkeysym(code, display));
		}
		/*
		 * tell all that we are on a newline
		 */
                G_newline_flag = 1;
        }
}

/***********************************************************************
 *
 *      write_MotionActions
 *
 *      This routine translates the specified motion input action
 *	and outputs text accordingly.
 */
static void
write_MotionActions(to_fptr, input_action_count_ptr, motionInfo)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
/*
 * pointer to a motion input action
 */
XTestMotionInfo	*motionInfo;
{
	/*
	 * ???
	 */
        short           xdisp;
	/*
	 * ???
	 */
        short           ydisp;

        xdisp = XTestUnpackXMotionValue(motionInfo->motion_data);
        ydisp = XTestUnpackYMotionValue(motionInfo->motion_data);

        if (((motionInfo->header & XTestX_SIGN_BIT_MASK) == XTestX_NEGATIVE) &&
	    (xdisp > 0))
	{
        	xdisp = -xdisp;
	}
        if (((motionInfo->header & XTestY_SIGN_BIT_MASK) == XTestY_NEGATIVE) &&
	    (ydisp > 0))
	{
        	ydisp = -ydisp;
	}

	if (G_translate_type == SCRIPT_TO_OUTPUT)
	{
	        (void) fprintf(to_fptr,
			       "MOTION     DEV [%d]   ",
			       XTestUnpackDeviceID(motionInfo->header));
                (void) fprintf(to_fptr,
			       "X [%4d]  ",
			       xdisp);
                (void) fprintf(to_fptr,
			       "Y [%4d]  ",
			       ydisp);
                (void) fprintf(to_fptr,
			       "TIME (ms) [%6u]    ",
			       motionInfo->delay_time);
                (void) fprintf(to_fptr,
			       "COUNT [%lu]\n",
			       ++(*input_action_count_ptr));
	}
	else
	{
		/*
		 * must be SUMMARY translation (this should never happen)
		 */
		(void) fprintf(to_fptr,
			       "<MOTION %4d %4d %6u>\n",
			       xdisp,
			       ydisp,
			       motionInfo->delay_time);
	}
}

/***********************************************************************
 *
 *      write_JumpActions
 *
 *      This routine translates the specified jump input action
 *	and outputs text accordingly.
 */
static void
write_JumpActions(to_fptr, input_action_count_ptr, jumpInfo)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
/*
 * pointer to a JumpInfo input action
 */
XTestJumpInfo	*jumpInfo;
{
	if (G_translate_type == SCRIPT_TO_OUTPUT)
	{
        	(void) fprintf(to_fptr,
			       "JUMP       DEV [%d]   ",
                               XTestUnpackDeviceID(jumpInfo->header));
                (void) fprintf(to_fptr,
			       "X [%4d]  ",
			       jumpInfo->jumpx);
                (void) fprintf(to_fptr,
			       "Y [%4d]  ",
			       jumpInfo->jumpy);
                (void) fprintf(to_fptr,
			       "TIME (ms) [%6u]    ",
			       jumpInfo->delay_time);
                (void) fprintf(to_fptr,
			       "COUNT [%lu]\n",
			       ++(*input_action_count_ptr));
	}
	else
	{
		/*
		 * must be SUMMARY translation
		 */
		(void) fprintf(to_fptr,
			       "<JUMP   %4d %4d %6u>\n",
			       jumpInfo->jumpx,
			       jumpInfo->jumpy,
			       jumpInfo->delay_time);
		/*
		 * Zero out the time field
		 */
		jumpInfo->delay_time = 0;
	}
}

/***********************************************************************
 *
 *      write_DelayActions
 *
 *      This routine translates the specified delay input action
 *	and outputs text accordingly.
 */
static void
write_DelayActions(to_fptr, input_action_count_ptr, delayInfo)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
/*
 * pointer to a DelayInfo input action
 */
XTestDelayInfo	*delayInfo;
{
        if (G_translate_type == SCRIPT_TO_OUTPUT)
        {
        	(void) fprintf(to_fptr,
			       "TIME                                     ");
                (void) fprintf(to_fptr,
			       "TIME (ms) [%6lu]    ",
			       delayInfo->delay_time);
                (void) fprintf(to_fptr,
			       "COUNT [%lu]\n",
			       ++(*input_action_count_ptr));
        }
        else
	{
		(void) fprintf(to_fptr,
			       "<TIME %6lu>\n",
			       delayInfo->delay_time);
	}
}

/***********************************************************************
 *
 *      flush_key_check
 *
 *      This routine flushes any pending key input actions
 */
static void
flush_key_check(to_fptr, input_action_count_ptr, display)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
/*
 * pointer to X11 display
 */
Display *display;
{
	if (G_l_i_a_type == XTestKEY_ACTION)
        {
		write_KeyActions(to_fptr,
				 input_action_count_ptr,
				 &G_l_key,
				 display);
		G_l_i_a_type = NO_INPUT_ACTION_PENDING;
	}
	if (G_key_delay_flag)
	{
		write_DelayActions(to_fptr,
				   input_action_count_ptr,
				   &G_delay_for_keys);
		G_key_delay_flag = 0;
	}
}

/***********************************************************************
 *
 *      flush_jump_check
 *
 *      This routine flushes any pending jump input actions
 */
static void
flush_jump_check(to_fptr, input_action_count_ptr)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32		*input_action_count_ptr;
{
        if ((G_l_i_a_type == XTestJUMP_ACTION)  ||
            (G_l_i_a_type == XTestMOTION_ACTION))
        {
		write_JumpActions(to_fptr,
				  input_action_count_ptr,
				  &G_l_jump);
		G_l_i_a_type = NO_INPUT_ACTION_PENDING;
	}
}

/***********************************************************************
 *
 *      flush_line_check
 *
 *      This routine flushes any pending newlines
 */
static void
flush_line_check(to_fptr)
/*
 * points to output format file
 */
FILE	*to_fptr;
{
	if (!G_newline_flag) 
	{
		(void) fprintf(to_fptr, "\n");
		G_newline_flag = 1;
	}
}

/***********************************************************************
 *
 *      flush_last_input_action
 *
 *      This routine flushes any pending events during internal format to
 *	summary format translation.
 */
static void
flush_last_input_action(to_fptr, input_action_count_ptr, display)
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32	*input_action_count_ptr;
/*
 * pointer to X11 display
 */
Display *display;
{
	flush_line_check(to_fptr);
	flush_key_check(to_fptr, input_action_count_ptr, display);
	flush_jump_check(to_fptr, input_action_count_ptr);
}

/**********************************************************************
 *
 *      tran_input_actions_to_sum
 *
 *      This routine translates the recorded input actions and outputs
 *	text accordingly.
 */
static void
tran_input_actions_to_sum(from_fd, to_fptr, input_action_count_ptr, display)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * points to summary format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32	*input_action_count_ptr;
/*
 * display pointer
 */
Display	*display;
{
	/*
	 * local type: input action data type
	 */
        unsigned char	type;
	/*
	 * local buffer for input actions
	 */
        unsigned char	input_action_buffer[INPUT_ACTION_BUFFER_SIZE];
	/*
	 * local recordsize: size of key data
	 */
        int		recordsize;
	/*
	 * ???
	 */
        XTestKeyInfo	*fKeyInfo;
	/*
	 * ???
	 */
        XTestMotionInfo	*fMotionInfo;
	/*
	 * ???
	 */
        XTestJumpInfo	*fJumpInfo;
	/*
	 * ???
	 */
        XTestDelayInfo	*fDelayInfo;
	/*
	 * ???
	 */
        int		xdisp;
	/*
	 * ???
	 */
        int		ydisp;
	/*
	 * local pindex: present index of translation
	 */
        int		pindex = 0;

	/*
	 * ???
	 */
        recordsize = getint(from_fd);
	/*
	 * read in saved input actions
	 */
        readchk(from_fd,
		(char *) input_action_buffer,
		(unsigned int) recordsize,
		"error while reading input actions");
        while (pindex < recordsize)
        {
		/*
		 * get type of data
		 */
		type = input_action_buffer[pindex] & XTestACTION_TYPE_MASK;

		switch (type)
		{
		case XTestKEY_ACTION:
			flush_jump_check(to_fptr, input_action_count_ptr);
			fKeyInfo   = (XTestKeyInfo *) &input_action_buffer[pindex];
			if (G_l_i_a_type == XTestKEY_ACTION)
			{
				/*
				 * A key was pressed and released
				 */
				if ((fKeyInfo->keycode == G_l_key.keycode) &&
				    ((fKeyInfo->header & XTestKEY_STATE_MASK) ==
				     XTestKEY_UP) &&
				    ((G_l_key.header & XTestKEY_STATE_MASK) ==
				     XTestKEY_DOWN))
				{
					/*
					 * The key is a ASCII printable key
					 */
					if (strlen(getkeysym(fKeyInfo->keycode,
							     display)) == 1)
					{
						(void) fprintf(to_fptr,
							       "%s",
							       getkeysym(fKeyInfo->keycode, display));
						G_newline_flag = 0;
					}
					/*
					 * The key is a special key
					 */
					else
					{
						flush_line_check(to_fptr);
						(void) fprintf(to_fptr,
							       "<%s>\n",
							       getkeysym(fKeyInfo->keycode, display));
						G_newline_flag = 1;
					}
					G_l_i_a_type = NO_INPUT_ACTION_PENDING;
					/*
					 * already have some key delay,
					 * add this to it
					 */
					G_delay_for_keys.delay_time += fKeyInfo->delay_time;
				}
				/*
				 * Last Key pressed is being held down
				 */
				else
				{
					flush_line_check(to_fptr);
					/*
					 * Pending key input action was a KEY_DOWN
					 */
					if ((G_l_key.header &
					     XTestKEY_STATE_MASK) ==
					    XTestKEY_DOWN)
					{
						(void) fprintf(to_fptr,
							       "<KEY_DOWN %s>\n",
							       getkeysym(G_l_key.keycode, display));
					}
					/*
					 * Pending key input action was a KEY_UP
					 */
					if ((G_l_key.header &
					     XTestKEY_STATE_MASK) ==
					    XTestKEY_UP)
					{
						(void) fprintf(to_fptr,
							       "<KEY_UP %s>\n",
							       getkeysym(G_l_key.keycode, display));
					}
					/*
					 * move current input action to pending input action
					 */
					G_l_key.header = fKeyInfo->header;
					G_l_key.keycode = fKeyInfo->keycode;
					G_l_i_a_type = XTestKEY_ACTION;
					G_newline_flag = 1;
					/*
					 * already have some key delay,
					 * add this to it
					 */
					G_delay_for_keys.delay_time += fKeyInfo->delay_time;
				}
			}
			/*
			 * Last input action was NOT a key, so
			 * make this key a pending key 
			 */
			else
			{
				G_l_key.header = fKeyInfo->header;
				G_l_key.keycode = fKeyInfo->keycode;
				G_l_i_a_type = XTestKEY_ACTION;
				if (G_key_delay_flag)
				{
					/*
					 * already have some key delay,
					 * add this to it
					 */
					G_delay_for_keys.delay_time += fKeyInfo->delay_time;
				}
				else
				{
					/*
					 * start accumulating key delays
					 */
					G_delay_for_keys.delay_time = fKeyInfo->delay_time;
					G_key_delay_flag = 1;
				}
			}
			pindex += sizeof(XTestKeyInfo);
			break;

		case XTestMOTION_ACTION:
			flush_line_check(to_fptr);
			flush_key_check(to_fptr,
					input_action_count_ptr,
					display);
			fMotionInfo = (XTestMotionInfo *) &input_action_buffer[pindex];
			xdisp = XTestUnpackXMotionValue(fMotionInfo->motion_data);
			ydisp = XTestUnpackYMotionValue(fMotionInfo->motion_data);
			if (((fMotionInfo->header & XTestX_SIGN_BIT_MASK) ==
			     XTestX_NEGATIVE) &&
			    (xdisp > 0))
			{
				xdisp = -xdisp;
			}
			if (((fMotionInfo->header & XTestY_SIGN_BIT_MASK) ==
			     XTestY_NEGATIVE) &&
			    (ydisp > 0))
			{
				ydisp = -ydisp;
			}
			/*
			 * Move current into pending input action
			 */
			G_l_jump.delay_time =
			add_time_check(to_fptr,
				       G_l_jump.delay_time,
				       fMotionInfo->delay_time);
			G_l_jump.jumpx += xdisp;
			G_l_jump.jumpy += ydisp;
			G_l_i_a_type = XTestMOTION_ACTION;
			pindex += sizeof(XTestMotionInfo);
			break;

		case XTestJUMP_ACTION:
			/*
			 * flush any non-motion pending input actions
			 */
			flush_line_check(to_fptr);
			flush_key_check(to_fptr,
					input_action_count_ptr,
					display);
			/*
			 * Get pointers to current and pending input actions
			 */
			fJumpInfo = (XTestJumpInfo *) &input_action_buffer[pindex];
			/*
			 * if motion pending, add time of current input action
			 */
			if ((G_l_i_a_type == XTestJUMP_ACTION) ||
			    (G_l_i_a_type == XTestMOTION_ACTION))
			{
			fJumpInfo->delay_time =
			add_time_check(to_fptr,
				       G_l_jump.delay_time,
				       fJumpInfo->delay_time);
			}
			/*
			 * Move current into pending input action
			 */
			G_l_jump.delay_time = fJumpInfo->delay_time;
			G_l_jump.jumpx      = fJumpInfo->jumpx;
			G_l_jump.jumpy      = fJumpInfo->jumpy;
			G_l_i_a_type = XTestJUMP_ACTION;
			pindex += sizeof(XTestJumpInfo);
			break;

		default:
			flush_last_input_action(to_fptr,
						input_action_count_ptr,
						display);
			type = XTestUnpackDeviceID(input_action_buffer[pindex]);
			if (type == XTestDELAY_DEVICE_ID)
			{
				fDelayInfo = (XTestDelayInfo *) &input_action_buffer[pindex];
				write_DelayActions(to_fptr,
						   input_action_count_ptr,
						   fDelayInfo);
				pindex += sizeof(XTestDelayInfo);
			}
			else
			{
				pindex++;
			}
			G_l_i_a_type = NO_INPUT_ACTION_PENDING;
			break;
		}
	}
}

/***********************************************************************
 *
 *      tran_input_actions_to_out
 *
 *      This routine translate the recorded input actions and outputs
 *	text accordingly.
 *
 */
static void
tran_input_actions_to_out(from_fd, to_fptr, input_action_count_ptr, display)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * points to output format file
 */
FILE	*to_fptr;
/*
 * number of input actions that have occured
 */
CARD32	*input_action_count_ptr;
/*
 * pointer to X11 display
 */
Display *display;
{
	/*
	 * local type: input action data type
	 */
        unsigned char   type;
	/*
	 * local pindex: present index of translation
	 */
        short           pindex = 0;
	/*
	 * local buffer for input actions
	 */
        unsigned char   input_action_buffer[INPUT_ACTION_BUFFER_SIZE];
	/*
	 * local recordsize: size of input action data
	 */
        short           recordsize;
	/*
	 * ???
	 */
        XTestKeyInfo    *fKeyInfo;
	/*
	 * ???
	 */
        XTestMotionInfo *fMotionInfo;
	/*
	 * ???
	 */
        XTestJumpInfo   *fJumpInfo;
	/*
	 * ???
	 */
        XTestDelayInfo  *fDelayInfo;

	/*
	 * ???
	 */
        recordsize = getint(from_fd);
	/*
	 * read in saved input actions
	 */
        readchk(from_fd,
		(char *) input_action_buffer,
		(unsigned int) recordsize,
		"error while reading input actions");
        while (pindex < recordsize)
        {
		/*
		 * get type of data
		 */
		type = input_action_buffer[pindex] & XTestACTION_TYPE_MASK;
		switch (type)
		{
		case XTestKEY_ACTION:
			fKeyInfo = (XTestKeyInfo *) &input_action_buffer[pindex];
			write_KeyActions(to_fptr,
					 input_action_count_ptr,
					 fKeyInfo,
					 display);
			/*
			 * move pindex to point to next input action
			 */
			pindex += sizeof(XTestKeyInfo);
			break;

		case XTestMOTION_ACTION:
			fMotionInfo = (XTestMotionInfo *) &input_action_buffer[pindex];
			write_MotionActions(to_fptr,
					    input_action_count_ptr,
					    fMotionInfo);
			pindex += sizeof(XTestMotionInfo);
			break;

		case XTestJUMP_ACTION:
			fJumpInfo = (XTestJumpInfo *) &input_action_buffer[pindex];
			write_JumpActions(to_fptr,
					  input_action_count_ptr,
					  fJumpInfo);
			pindex += sizeof(XTestJumpInfo);
			break;

		default:
			type = XTestUnpackDeviceID(input_action_buffer[pindex]);
			if (type == XTestDELAY_DEVICE_ID)
			{
				fDelayInfo = (XTestDelayInfo *) &input_action_buffer[pindex];
				write_DelayActions(to_fptr,
						   input_action_count_ptr,
						   fDelayInfo);
				pindex += sizeof(XTestDelayInfo);
			}
			else
			{
				pindex++;
			}
			break;
		}
        }
}

/**********************************************************************
 *
 *      tran_version_number
 *
 *      This function translates the version number from a test script.
 *	It returns non-zero if the version number is missing or doesn't
 *	match the version number of xtmconvert.
 */
static int
tran_version_number(from_fd, to_fptr)
/*
 * file descriptor of internal format test script file
 */
int     from_fd;
/*
 * points to file to write to
 */
FILE	*to_fptr;
{
	/*
	 * holds a character read from the test script
	 */
	char	c;
	/*
	 * holds the version number read from the test script
	 */
	int	version_number;

	/*
	 * read a character from the test script
	 */
	if (read(from_fd, &c, 1) != 1)
	{
		return(1);
	}
	/*
	 * if the character is not a 'v' for the version number, then error
	 */
	if (c != 'v')
	{
		return(1);
	}
	/*
	 * read the version number from the test script
	 */
	version_number = getint(from_fd);
	/*
	 * if the version number of the test script and xtmexecute don't
	 * match, then error.
	 */
	if (version_number != VERSION_NUMBER)
	{
		return(1);
	}
	/*
	 * we have the correct version number, so translate it
	 */
	if (G_translate_type == SCRIPT_TO_OUTPUT)
	{
        	(void) fprintf(to_fptr, "VERSION [%d]\n", version_number);
	}
	else
	{
		(void) fprintf(to_fptr, "<VERSION %d>\n", version_number);
	}
	/*
	 * if we get to here, everything is fine
	 */
	return(0);
}

/***********************************************************************
 *
 *      tran_comment_out
 *
 *      This routine translates comments.
 */
static void
tran_comment_out(from_fd, to_fptr)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * points to file to write to
 */
FILE	*to_fptr;
{
	/*
	 * local record_size: number of bytes for this record
	 */
        int     record_size;
	/*
	 * local commentbuf: char buffer for comment record
	 */
        char    commentbuf[WRITE_BUFFER_SIZE];

        record_size = getint(from_fd);
	/*
	 * read in comment
	 */
        readchk(from_fd,
		commentbuf,
		(unsigned int) record_size,
		"error while reading comment");

	if (G_translate_type == SCRIPT_TO_OUTPUT)
	{
        	(void) fprintf(to_fptr, "COMMENT    %s\n", commentbuf);
	}
	else
	{
		(void) fprintf(to_fptr, "<COMMENT %s>\n", commentbuf);
	}
}

/***********************************************************************
 *
 *      tran_toggle_mode
 *
 *      This routine translates toggle mode commands.
 */
static void
tran_toggle_mode(from_fd, to_fptr)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * points to file to write to
 */
FILE	*to_fptr;
{
	/*
	 * points to either an "ON" or "OFF" string
	 */
	char	*on_off_string;
	/*
	 * holds a character
	 */
	char	c;

	/*
	 * read the type of mode to toggle
	 */
        readchk(from_fd,
		&c,
		1,
		"error while reading toggle mode type from test script\n");
        /*
         * process the mode type
         */
        switch(c)
        {
        case 'c':
                /*
                 * checksum mode
                 */
                readchk(from_fd,
                        &c,
                        1,
                        "error reading toggle mode type from test script\n");
                switch(c)
                {
                case 'd':
                        /*
                         * disable checksum mode
                         */
                        G_checksum_flag = 0;
			on_off_string = "OFF";
                        break;

                case 'e':
                        /*
                         * enable checksum mode
                         */
                        G_checksum_flag = 1;
			on_off_string = "ON";
                        break;

                default:
                        (void) fprintf(G_msg_file,
				       "Unrecognized character in test script (decimal): %d\n",
				       c);
			exit(1);
                        break;
                }
		if (G_translate_type == SCRIPT_TO_OUTPUT)
		{
			(void) fprintf(to_fptr,
				       "CHECKSUMS %s\n",
				       on_off_string);
		}
		else
		{
			(void) fprintf(to_fptr,
				       "<CHECKSUMS %s>\n",
				       on_off_string);
		}
		break;

        default:
		(void) fprintf(G_msg_file,
			       "Unrecognized character in test script (decimal): %d\n",
			       c);
		exit(1);
                break;
        }
}

/***********************************************************************
 *
 *      script_to_text
 *
 *      Main control logic for translating between internal and output 
 *	or summary formats.
 *
 *      Translate the test script.
 *
 *	This consists of records beginning with one of 4 characters:
 *	  1). v - indicates that the following record contains a version
 *		  number.
 *	  2). c - indicates that the following record contains a comment.
 *	  3). T - indicates that the following record contains a mode
 *		  toggle command.
 *	  4). k - indicates that the following record contains
 *	          a list of input actions.
 *	  5). m - indicates that the following record contains a request
 *	          for a comparison of image data (a "match").
 */
void
script_to_text(from_fd, to_fptr, to_basename, stripflag)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * points to summary format file
 */
FILE	*to_fptr;
/*
 * points to the the name of the <TO> file (without any extension)
 */
char	*to_basename;
/*
 * if non-zero, strip any image data during conversion
 */
int		stripflag;
{
	/*
	 * local type: the record type character described above
	 */
	char		recordtype;
	/*
	 * counter for number of screen matches
	 */
	int		matchcount;
	/*
	 * holds the name of the xwd file
	 */
	char		xwd_file_name[MAXPATH];
	/*
	 * counter for total number of input actions
	 */
	CARD32		input_action_count;
	/*
	 * pointer to X11 display
	 */
	Display 	*display;

	/*
	 * Open the default X display
	 */
        if ((display = XOpenDisplay((char *) 0)) == (Display *) 0)
        {
                (void) fprintf(G_msg_file, "error on XOpenDisplay\n");
                exit(1);
        }
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "In script to text, starting translation\n");
	}
	/*
	 * initialize counts to 0
	 */	
	matchcount = 0;
	input_action_count = 0;
	/*
	 * Clear x, y values for summary
	 */
	G_l_jump.jumpx      = 0;
	G_l_jump.jumpy      = 0;
	G_l_jump.delay_time = 0;
	G_l_i_a_type = NO_INPUT_ACTION_PENDING;
	/*
	 * start out indicating no accumulated key delays
	 */
	G_key_delay_flag = 0;
	/*
	 * initialize the checksum flag to false (not using checksums)
	 */
	G_checksum_flag = 0;
        /*
         * check for test script version
         */
        if (tran_version_number(from_fd, to_fptr) != 0)
        {
                (void) fprintf(G_msg_file,
                               "Missing or invalid version number in test script.\n");
                exit(1);
        }
	/*
	 * read first character in a record which indicates the record type 
	 */
        while (read(from_fd, &recordtype, 1) > 0)
        {
		/*
		 * flush out a pending input action if the next input action
		 * is not deferred
		 */
                if ((G_translate_type == SCRIPT_TO_SUMMARY) && 
                    (recordtype != 'k'))
		{
			flush_last_input_action(to_fptr,
						&input_action_count,
						display);
		}

                switch (recordtype)
                {
                case 'c':
			/*
			 * processing a comment
			 */
                        tran_comment_out(from_fd, to_fptr);
			G_l_i_a_type = NO_INPUT_ACTION_PENDING;
                        break;

                case 'T':
			/*
			 * processing a toggle mode
			 */
                        tran_toggle_mode(from_fd, to_fptr);
			G_l_i_a_type = NO_INPUT_ACTION_PENDING;
                        break;

                case 'k':
			/*
			 * processing input actions
			 */
			if (G_translate_type == SCRIPT_TO_OUTPUT)
			{
                        	tran_input_actions_to_out(from_fd,
							  to_fptr,
							  &input_action_count,
							  display);
			}
			else
			{
                        	tran_input_actions_to_sum(from_fd,
							  to_fptr,
							  &input_action_count,
							  display);
			}
                        break;

                case 'm':
			/*
			 * process match data
			 */
                        matchcount++;
			/*
			 * if there are too many matches in this test script,
			 * error out now because the error file names won't fit
			 */
			if (matchcount > MATCH_LIMIT)
			{
				(void) fprintf(G_msg_file,
					       "too many matches in test script\n");
				exit(1);
			}
			if (G_translate_type == SCRIPT_TO_OUTPUT)
			{
                        	inttoout(from_fd,
					 to_fptr,
					 matchcount,
					 stripflag);
			}
			else
			{
				if (G_checksum_flag)
				{
					/*
					 * create the xwd file name with a 'c'
					 * to indicate that it contains a
					 * checksum instead of image data
					 */
					(void) sprintf(xwd_file_name,
						       "%s.c%02d",
						       to_basename,
						       matchcount);
				}
				else
				{
					/*
					 * create the xwd file name
					 */
					(void) sprintf(xwd_file_name,
						       "%s.m%02d",
						       to_basename,
						       matchcount);
				}
				/*
				 * translate the internal format file to
				 * summary format
				 */
                        	inttosum(from_fd,
					 to_fptr,
					 xwd_file_name,
					 stripflag);
			}
			G_l_i_a_type = NO_INPUT_ACTION_PENDING;
                        break;

                default:
			G_l_i_a_type = NO_INPUT_ACTION_PENDING;
                        (void) fprintf(G_msg_file,
				       "Invalid data in test script. data is %x.\n",
			               recordtype);
			exit(1);
			break;
                }
        }
	/*
	 * flush pending input action, if any
	 */
	if (G_translate_type == SCRIPT_TO_SUMMARY)  
	{
		flush_last_input_action(to_fptr,
					&input_action_count,
					display);
	}
	/*
	 * Close the display, no longer needed
	 */
        XCloseDisplay(display);
}
