/*******************************************************************************
 *
 *	sumtoint.c
 *
 ******************************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/


/*******************************************************************************
 *	include files
 ******************************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * standard file defs
 */
#include  <fcntl.h>
/*
 * standard string defs
 */
#include  <string.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * client exerciser defines
 */
#include  <xtm.h>

/*******************************************************************************
 *	variables
 ******************************************************************************/

/*
 * size of the buffer for screen matches
 */
extern unsigned int     G_sb_size;
/*
 * the buffer holds the "uncompressed" match image data
 */
extern unsigned char    *G_sb;
extern FILE		*G_msg_file;
extern int		G_checksum_flag;
extern int		errno;

/*******************************************************************************
 *	function declarations
 ******************************************************************************/

void	readchk();
void	parse_summary_line();
void	make_dummy_match_data();
void	read_match_data_from_xwd();
void	write_match_data_to_internal();
void	getkey();
void	translate_xwd_to_internal();
void	gettmp();
void	copy_and_compress_match_data();
void	copy_match_data();
long	lseek();

/*******************************************************************************
 *	functions
 ******************************************************************************/

/*******************************************************************************
 *
 *	sumtoint
 *
 *	Translate a match from summary format to internal format.
 *
 ******************************************************************************/
void
sumtoint(from_fptr, to_fd, stripflag, playback_info_ptr)
/*
 * file pointer of summary format file to read from
 */
FILE	*from_fptr;
/*
 * file descriptor of internal format file to write to
 */
int	to_fd;
/*
 * if non-zero then strip out any image data
 */
int	stripflag;
/*
 * points to the structure containing the playback information
 */
struct playback_info    *playback_info_ptr;
{
	/*
	 * holds the match information temporarily
	 */
	struct match_data	match_data;
	/*
	 * holds the xwd format file name
	 */
	char			xwd_file_name[MAXPATH];

	/*
	 * Parse the rest of the line to isolate the match type and the
	 * xwd format file name, if any.  If there is an xwd format file
	 * name, also get the screen number and x and y coordinates of
	 * the window that the match data was recorded from.
	 */
	parse_summary_line(from_fptr, xwd_file_name, &match_data);
	if (stripflag || (strlen(xwd_file_name) == 0))
	{
		/*
		 * if the stripflag is set or there is no xwd format file name
		 * then make a dummy match data structure and write it to the
		 * internal format file
		 */
		make_dummy_match_data(&match_data, playback_info_ptr);
		write_match_data_to_internal(to_fd, &match_data);
	}
	else
	{
		/*
		 * read the xwd format file, translate it to internal format,
		 * and write it to the internal format file
		 */
		translate_xwd_to_internal(xwd_file_name,
					  to_fd,
					  &match_data,
					  playback_info_ptr);
	}
}

/*******************************************************************************
 *
 *	parse_summary_line
 *
 *	Parse the rest of the line from the summary format file
 *	to isolate the match type and the xwd format file name, if any.
 *
 ******************************************************************************/
static void
parse_summary_line(from_fptr, xwd_file_name, match_data_ptr)
/*
 * file pointer of summary format file to read from
 */
FILE	*from_fptr;
/*
 * points to where to put the name of the xwd format file
 */
char	*xwd_file_name;
/*
 * points to the structure to read the match data into
 */
struct match_data       *match_data_ptr;
{
	/*
	 * holds the rest of the line from the summary format file
	 */
	char	temp[WRITE_BUFFER_SIZE];
	/*
	 * holds the name of the type of match from the summary format file
	 */
	char	type_name[WRITE_BUFFER_SIZE];

	/*
	 * read the rest of the line from the summary format file
	 */
	getkey(from_fptr, temp, '>');
	/*
	 * look for a blank separating the match type from the xwd format
	 * file name (there may not be one)
	 */
	if (strrchr(temp, ' ') == (char *) 0)
	{
		/*
		 * if the strrchr result is NULL, then there was no blank,
		 * so there must not be an xwd format file name
		 */
		(void) strcpy(xwd_file_name, "");
		/*
		 * the only thing in the temp buffer is the match type name
		 */
		(void) strcpy(type_name, temp);
	}
	else
	{
		/*
		 * A blank was found.  Read in the match type name, the xwd
		 * format file name, the screen number, and the x and y
		 * coordinates of the window that the image data was
		 * recorded from.
		 */
		if (sscanf(temp,
			   "%s %s %d %d %d",
			   type_name,
			   xwd_file_name,
			   &(match_data_ptr->screen_number),
			   &(match_data_ptr->window_x),
			   &(match_data_ptr->window_y)) != 5)
		{
		(void) fprintf(G_msg_file,
			       "error while reading xwd file name from summary file\n");
		exit(1);
		}
	}
	/*
	 * figure out what type of match this is
	 */
	if (strcmp(type_name, "MOUSE") == 0)
	{
		match_data_ptr->type = (INT8) 'm';
	}
	else if (strcmp(type_name, "PARTIAL") == 0)
	{
		match_data_ptr->type = (INT8) 'p';
	}
	else if (strcmp(type_name, "SCREEN") == 0)
	{
		match_data_ptr->type = (INT8) 's';
	}
	else if (strcmp(type_name, "TOP") == 0)
	{
		match_data_ptr->type = (INT8) 't';
	}
	else
	{
		(void) fprintf(G_msg_file,
			       "unrecognizable match type '%s'\n",
			       type_name);
		exit(1);
	}
}

/*******************************************************************************
 *
 *	make_dummy_match_data
 *
 *	Fill out a match data structure when there is no match data.
 *
 ******************************************************************************/
static void
make_dummy_match_data(match_data_ptr, playback_info_ptr)
/*
 * points to the structure to read the match data into
 */
struct match_data       *match_data_ptr;
/*
 * points to the structure containing the playback information
 */
struct playback_info    *playback_info_ptr;
{
	/*
	 * assign values to dummy match data
	 */
	match_data_ptr->retry = playback_info_ptr->retries;
	match_data_ptr->intv = playback_info_ptr->interval;
	match_data_ptr->cnt = 0;
	match_data_ptr->checksum = 0;
	/*
	 * the type must be set by the calling routine
	 */
	match_data_ptr->screen_number = 0;
	match_data_ptr->window_x = 0;
	match_data_ptr->window_y = 0;
	match_data_ptr->window_width = 0;
	match_data_ptr->window_height = 0;
	match_data_ptr->window_border_width = 0;
	(void) strcpy(match_data_ptr->window_name, DEFAULT_WINDOW_NAME);
	match_data_ptr->x = 0;
	match_data_ptr->y = 0;
	match_data_ptr->width = 0;
	match_data_ptr->height = 0;
	match_data_ptr->depth = 0;
	match_data_ptr->format = 0;
	match_data_ptr->xoffset = 0;
	match_data_ptr->byte_order = 0;
	match_data_ptr->bitmap_unit = 0;
	match_data_ptr->bitmap_bit_order = 0;
	match_data_ptr->bitmap_pad = 0;
	match_data_ptr->bytes_per_line = 0;
	match_data_ptr->bits_per_pixel = 0;
	match_data_ptr->visual_class = 0;
	match_data_ptr->red_mask = 0;
	match_data_ptr->green_mask = 0;
	match_data_ptr->blue_mask = 0;
	match_data_ptr->bits_per_rgb = 0;
	match_data_ptr->colormap_entries = 0;
	match_data_ptr->ncolors = 0;
	match_data_ptr->color_ptr = (XColor *) 0;
}

/*******************************************************************************
 *
 *	translate_xwd_to_internal
 *
 *	Translate the match data from xwd format to internal format.
 *
 ******************************************************************************/
static void
translate_xwd_to_internal(xwd_file_name,
			  to_fd,
			  match_data_ptr,
			  playback_info_ptr)
/*
 * points to the name of the xwd format file
 */
char	*xwd_file_name;
/*
 * file descriptor of internal format file to write to
 */
int	to_fd;
/*
 * points to the structure to read the match data into
 */
struct match_data       *match_data_ptr;
/*
 * points to the structure containing the playback information
 */
struct playback_info    *playback_info_ptr;
{
	/*
	 * holds the file descriptor of the xwd format file
	 */
	int	xwd_fd;
        /*
         * holds the temporary file name
         */
        char	temp_file_name[MAXPATH];

	/*
	 * open the xwd format file
	 */
	xwd_fd = open(xwd_file_name, O_RDONLY);
	if (xwd_fd < 0)
	{
		(void) fprintf(G_msg_file,
			       "error %d in opening file: %s\n",
			       errno,
			       xwd_file_name);
		exit(1);
	}
	/*
	 * Read the match data from the xwd format file.  This
	 * allocates memory for the color map structures, if needed.
	 */
	read_match_data_from_xwd(xwd_fd, match_data_ptr, playback_info_ptr);
	if (!G_checksum_flag)
	{
		/*
		 * create a temporary file to hold the match data
		 */
		gettmp(&(match_data_ptr->tfd), temp_file_name);
		/*
		 * unlink the temporary file so that if the program stops
		 * unexpectedly the file will not be left laying around
		 */
		if (unlink(temp_file_name) != 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while unlinking the match data temp file\n",
				       errno);
		}
		/*
		 * read any image data from the xwd format file, compress it,
		 * and write it to the temporary file
		 */
		copy_and_compress_match_data(xwd_fd, match_data_ptr);
	}
	/*
	 * write the match data information to the internal format file
	 */
	write_match_data_to_internal(to_fd, match_data_ptr);
	if (!G_checksum_flag)
	{
		/*
		 * reset the file pointer to the beginning of the compressed
		 * match data temp file
		 */
		if (lseek(match_data_ptr->tfd, 0L, 0) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while seeking in match data temp file\n",
				       errno);
			exit(1);
		}
		/*
		 * copy the match data from the temporary file
		 * to the internal format file
		 */
		copy_match_data(match_data_ptr->tfd, to_fd, match_data_ptr->cnt);
		/*
		 * close the match data temp file
		 */
		if (close(match_data_ptr->tfd) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while closing match data temp file\n",
				       errno);
			exit(1);
		}
	}
	/*
	 * close the xwd format file
	 */
	if (close(xwd_fd) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while closing file: %s\n",
			       errno,
			       xwd_file_name);
		exit(1);
	}
	/*
	 * free color structure information space, if any
	 */
	if ((match_data_ptr->ncolors > 0) &&
	    (match_data_ptr->color_ptr != (XColor *) 0))
	 {
		 free ((char *) (match_data_ptr->color_ptr));
	 }
}

/*******************************************************************************
 *
 *	copy_and_compress_match_data
 *
 *	Read any image data from the xwd format file, compress it,
 *	and write it to a temporary file.
 *
 ******************************************************************************/
static void
copy_and_compress_match_data(xwd_fd, match_data_ptr)
/*
 * file descriptor of xwd format file to read from
 */
int	xwd_fd;
/*
 * points to the structure containing the match data information
 */
struct match_data       *match_data_ptr;
{
	/*
	 * holds the total number of bytes of uncompressed image data to read
	 * from the xwd format file
	 */
	INT32		total_count;
	/*
	 * holds the current number of bytes of uncompressed image data
	 * that have been read from the xwd format file
	 */
	INT32		current_count = 0;
	/*
	 * holds the maximum number of bytes of uncompressed image data
	 * to read at a time from the xwd format file
	 */
	unsigned int	max_count;
	/*
	 * holds the number of bytes of uncompressed image data to read
	 * at a time from the xwd format file
	 */
	int		uncomp_count;
	/*
	 * holds the number of bytes of compressed image data to write
	 * to the temp file
	 */
	INT32		comp_count;

	/*
	 * Compute the maximum number of bytes of uncompressed image data
	 * that will fit into the buffer at one time.  This is computed in
	 * terms of multiples of horizontal lines of pixels to be consistent
	 * with the wintotemp routine.
	 */
	max_count = (G_sb_size / match_data_ptr->bytes_per_line) *
		    match_data_ptr->bytes_per_line;
	/*
	 * figure out the total number of bytes of uncompressed image data
	 */
	total_count = match_data_ptr->bytes_per_line * match_data_ptr->height;
	/*
	 * set the total number of bytes of compressed image data to 0
	 */
	match_data_ptr->cnt = 0;
	/*
	 * loop until all of the image data has been read and compressed
	 */
	while (current_count < total_count)
	{
		/*
		 * figure out how much uncompressed data to read
		 */
		uncomp_count = ((total_count - current_count) < max_count) ?
			       (total_count - current_count) : max_count;
		/*
		 * read in some image data
		 */
		readchk(xwd_fd,
			(char *) G_sb,
			(unsigned int) uncomp_count,
			"error while reading xwd format image data\n");
		/*
		 * clear any pad bits in the image data to a known value
		 * so comparisons will work even if the pad bits change
		 */
		clear_pad_bits(G_sb,
			       (uncomp_count / match_data_ptr->bytes_per_line),
			       match_data_ptr);
		/*
		 * compress the image data and write it out to the temporary
		 * file
		 */
		comp_count = compress((char *) G_sb,
				      uncomp_count,
				      match_data_ptr->tfd);
		/*
		 * update the byte count variables
		 */
		match_data_ptr->cnt += comp_count;
		current_count += uncomp_count;
	}
}
