/*******************************************************************************
 *
 *	outtoint.c
 *
 ******************************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/


/*******************************************************************************
 *	include files
 ******************************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * client exerciser defines
 */
#include  <xtm.h>

/*******************************************************************************
 *	variables
 ******************************************************************************/

/*
 * Buffer for match image data
 */
extern unsigned char	*G_sb;
/*
 * Buffer for compressed match image data
 */
extern unsigned char	*G_c_data_buffer;
/*
 * size of match image data buffer
 */
extern unsigned int	G_sb_size;
extern FILE		*G_msg_file;
extern int		G_debug_flag;
extern int		G_checksum_flag;
extern int		errno;

/*******************************************************************************
 *	function declarations
 ******************************************************************************/

void	wrchk2();
void	read_match_data_from_output();
void	read_and_ignore_image_data();
void	read_and_checksum_image_data();
void	write_match_data_to_internal();
void	handle_image_data_when_checksum_modes_match();
void	handle_image_data_to_checksum_data();
void	handle_checksum_data_to_image_data();
void	copy_from_output_to_internal();
void	checksum_a_bufferful();
void	compute_checksum();
char	*malloc();

/*******************************************************************************
 *	functions
 ******************************************************************************/

/*******************************************************************************
 *
 *	outtoint
 *
 *	Translate a match from output format to internal format.
 *
 ******************************************************************************/
void
outtoint(from_fptr, to_fd, stripflag)
/*
 * file pointer of output format file to read from
 */
FILE	*from_fptr;
/*
 * file descriptor of internal format file to write to
 */
int	to_fd;
/*
 * if non-zero then strip out any image data
 */
int	stripflag;
{
	/*
	 * holds the match information temporarily
	 */
	struct match_data	match_data;
	/*
	 * holds the checksum mode the match was written out with
	 */
	int			match_checksum_mode;

	/*
	 * read the match data information into the match data structure
	 */
	read_match_data_from_output(from_fptr,
				    &match_data,
				    &match_checksum_mode);
	/*
	 * handle converting where someone might have modified the checksum
	 * mode in the output format file (by editing the file)
	 */
	if (match_checksum_mode == G_checksum_flag)
	{
		/*
		 * if the current checksum mode matches the checksum mode
		 * that the match data was written out with, then no
		 * special processing is needed
		 */
		handle_image_data_when_checksum_modes_match(from_fptr,
							    to_fd,
							    &match_data,
							    stripflag);
	}
	else
	{
		/*
		 * if the current checksum mode does not match the checksum
		 * mode that the match data was written out with, then
		 * special processing is needed
		 */
		if (G_checksum_flag && !match_checksum_mode)
		{
			handle_image_data_to_checksum_data(from_fptr,
							   to_fd,
							   &match_data,
							   stripflag);
		}
		else
		{
			handle_checksum_data_to_image_data(from_fptr,
							   to_fd,
							   &match_data,
							   stripflag);
		}
	}
	/*
	 * free the color structure information space, if any
	 */
	if ((match_data.ncolors > 0) &&
	    (match_data.color_ptr != (XColor *) 0))
	{
		free ((char *) (match_data.color_ptr));
	}
}

/*******************************************************************************
 *
 *	read_match_data_from_output
 *
 *	Reads the match data structure from an output format file.
 *
 ******************************************************************************/
static void
read_match_data_from_output(output_fmt_ptr,
			    match_data_ptr,
			    match_checksum_mode_ptr)
/*
 * file pointer of the file (assumed to be in output format)
 * from which the match data is to be read
 */
FILE			*output_fmt_ptr;
/*
 * points to the structure to read the match data into
 */
struct match_data	*match_data_ptr;
/*
 * points to the place to put the match checksum mode
 */
int			*match_checksum_mode_ptr;
{
	/*
	 * points to a color map structure
	 */
	XColor	*color_ptr;
	/*
	 * keeps track of the number of color map structures
	 */
	INT32	i;
	/*
	 * holds data before it is type-coerced to fit
	 */
	int	temp;
	/*
	 * holds character data before it is type-coerced to fit
	 */
	char	temp_char;

	/*
	 * read in the match count (ignored), match type, retry count,
	 * retry interval, and the count of the number of bytes
	 * of compressed image data
	 */
	if (fscanf(output_fmt_ptr,
		   " #%d TYPE[%c] REP[%ld] INTV[%ld] CNT[%ld] CM[%d] \n",
		   &temp,
		   &temp_char,
		   &(match_data_ptr->retry),
		   &(match_data_ptr->intv),
		   &(match_data_ptr->cnt),
		   match_checksum_mode_ptr) != 6)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading match data type information\n",
			       errno);
		exit(1);
	}
	/*
	 * fix up type-coerced data
	 */
	match_data_ptr->type = (INT8) temp_char;
	/*
	 * read in the match data window screen number, x and y coordinates,
	 * width, height, and border width
	 */
	if (fscanf(output_fmt_ptr,
		   "\tS#[%d] WX[%d] WY[%d] WW[%d] WH[%d] BW[%d] \n",
		   &(match_data_ptr->screen_number),
		   &(match_data_ptr->window_x),
		   &(match_data_ptr->window_y),
		   &(match_data_ptr->window_width),
		   &(match_data_ptr->window_height),
		   &(match_data_ptr->window_border_width)) != 6)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading match data window information\n",
			       errno);
		exit(1);
	}
	/*
	 * read in the window name
	 *
	 * This is somewhat tricky.  If you use a "%s" instead of the "%[^"]"
	 * in the format string, the fscanf will stop on a whitespace character
	 * instead of reading all of the characters up till the next '"'.
	 */
	if (fscanf(output_fmt_ptr,
		   "\tWN\"%[^\"]\"\n",
		   match_data_ptr->window_name) != 1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading match data window name\n",
			       errno);
		exit(1);
	}
	/*
	 * read in the match data x and y coordinates in the window,
	 * width, height, depth, format, and x offset
	 */
	if (fscanf(output_fmt_ptr,
		   "\tX[%d] Y[%d] W[%d] H[%d] D[%d] F[%d] XOFF[%d] \n",
		   &(match_data_ptr->x),
		   &(match_data_ptr->y),
		   &(match_data_ptr->width),
		   &(match_data_ptr->height),
		   &(match_data_ptr->depth),
		   &(match_data_ptr->format),
		   &(match_data_ptr->xoffset)) != 7)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading match data area information\n",
			       errno);
		exit(1);
	}
	/*
	 * read in the match data byte order, bitmap unit, bitmap bit order,
	 * bitmap pad, bytes per line, and bits per pixel
	 */
	if (fscanf(output_fmt_ptr,
	           "\tBO[%d] BU[%d] BBO[%d] BP[%d] BPL[%d] BPP[%d] \n",
		   &(match_data_ptr->byte_order),
		   &(match_data_ptr->bitmap_unit),
		   &(match_data_ptr->bitmap_bit_order),
		   &(match_data_ptr->bitmap_pad),
		   &(match_data_ptr->bytes_per_line),
		   &(match_data_ptr->bits_per_pixel)) != 6)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading match data image information\n",
			       errno);
		exit(1);
	}
	/*
	 * read in the match data visual class, red mask, green mask,
	 * blue mask, bits per rgb, number of entries in the colormap,
	 * and the number of color structures
	 */
	if (fscanf(output_fmt_ptr,
		   "\tVC[%d] RM[%lu] GM[%lu] BM[%lu] BPR[%d] CE[%d] C[%ld] \n",
		   &(match_data_ptr->visual_class),
		   &(match_data_ptr->red_mask),
		   &(match_data_ptr->green_mask),
		   &(match_data_ptr->blue_mask),
		   &(match_data_ptr->bits_per_rgb),
		   &(match_data_ptr->colormap_entries),
		   &(match_data_ptr->ncolors)) != 7)
	{
		(void) fprintf(G_msg_file,
			       "error %d while reading more match data image information\n",
			       errno);
		exit(1);
	}
	/*
	 * if there are any color map entries, read them in
	 */
	if (match_data_ptr->ncolors > 0)
	{
		/*
		 * allocate the space needed to hold the color map entries
		 */
		if ((color_ptr = (XColor *) malloc(sizeof(XColor) *
						   match_data_ptr->ncolors))
		    == (XColor *) 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while mallocing color map entries\n",
				       errno);
			exit(1);
		}
		/*
		 * remember where the color map entries are
		 */
		match_data_ptr->color_ptr = color_ptr;
		/*
		 * read the color map entries in
		 */
		for (i = 0; i < match_data_ptr->ncolors; i++, color_ptr++)
		{
			/*
			 * read in the data for one color map entry
			 */
			if (fscanf(output_fmt_ptr,
				   "\tPIXEL[%lu] RED[%hu] GREEN[%hu] BLUE[%hu] FLAGS[%d] \n",
				   &(color_ptr->pixel),
				   &(color_ptr->red),
				   &(color_ptr->green),
				   &(color_ptr->blue),
				   &temp) != 5)
			{
				(void) fprintf(G_msg_file,
					       "error %d while reading color map entry information\n",
					       errno);
				exit(1);
			}
			/*
			 * fix up the type-cast parameter
			 */
			color_ptr->flags = (char) temp;
		}
	}
	else
	{
		/*
		 * there are no color map entries
		 */
		match_data_ptr->color_ptr = (XColor *) 0;
	}
	/*
	 * if the match was output with a checksum, read it in
	 */
	if (*match_checksum_mode_ptr)
	{
		/*
		 * read in the match data checksum
		 */
		if (fscanf(output_fmt_ptr,
			   "\tCHECKSUM[%lu] \n",
			   &(match_data_ptr->checksum)) != 1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while reading match data checksum\n",
				       errno);
			exit(1);
		}
	}
}

/*******************************************************************************
 *
 *	handle_image_data_when_checksum_modes_match
 *
 *	Handles the image data from the output format file.
 *
 ******************************************************************************/
static void
handle_image_data_when_checksum_modes_match(from_fptr,
					    to_fd,
					    match_data_ptr,
					    stripflag)
/*
 * file pointer of output format file to read from
 */
FILE			*from_fptr;
/*
 * file descriptor of internal format file to write to
 */
int			to_fd;
/*
 * points to the structure to read the match data into
 */
struct match_data	*match_data_ptr;
/*
 * if non-zero then strip out any image data
 */
int			stripflag;
{
	/*
	 * strip the image data if needed
	 */
	if (stripflag)
	{
		/*
		 * free color structure information space, if any
		 */
		if ((match_data_ptr->ncolors > 0) &&
		    (match_data_ptr->color_ptr != (XColor *) 0))
		{
			free ((char *) (match_data_ptr->color_ptr));
		}
		/*
		 * set the color structure pointer to a null pointer
		 */
		match_data_ptr->color_ptr = (XColor *) 0;
		/*
		 * set the color structure count to 0
		 */
		match_data_ptr->ncolors = 0;
		/*
		 * if not using checksums read and ignore any image data
		 * in the output format file
		 */
		if (!G_checksum_flag)
		{
			read_and_ignore_image_data(from_fptr,
						   match_data_ptr->cnt);
		}
		/*
		 * set the image data size to 0
		 */
		match_data_ptr->cnt = 0;
		/*
		 * write the match data information to the internal format file
		 */
		write_match_data_to_internal(to_fd, match_data_ptr);
	}
	else
	{
		/*
		 * write the match data information to the internal format file
		 */
		write_match_data_to_internal(to_fd, match_data_ptr);
		/*
		 * copy any image data from the output format file to
		 * the internal format file
		 */
		copy_from_output_to_internal(from_fptr,
					     to_fd,
					     match_data_ptr->cnt);
	}
}

/*******************************************************************************
 *
 *	handle_image_data_to_checksum_data
 *
 *	Handles the case where the output format file has image data and
 *	the current checksum mode is checksums.
 *
 ******************************************************************************/
static void
handle_image_data_to_checksum_data(from_fptr, to_fd, match_data_ptr, stripflag)
/*
 * file pointer of output format file to read from
 */
FILE			*from_fptr;
/*
 * file descriptor of internal format file to write to
 */
int			to_fd;
/*
 * points to the structure to read the match data into
 */
struct match_data	*match_data_ptr;
/*
 * if non-zero then strip out any image data
 */
int			stripflag;
{
	/*
	 * strip the image data if needed
	 */
	if (stripflag)
	{
		/*
		 * free color structure information space, if any
		 */
		if ((match_data_ptr->ncolors > 0) &&
		    (match_data_ptr->color_ptr != (XColor *) 0))
		{
			free ((char *) (match_data_ptr->color_ptr));
		}
		/*
		 * set the color structure pointer to a null pointer
		 */
		match_data_ptr->color_ptr = (XColor *) 0;
		/*
		 * set the color structure count to 0
		 */
		match_data_ptr->ncolors = 0;
		/*
		 * read and ignore any image data
		 */
		read_and_ignore_image_data(from_fptr, match_data_ptr->cnt);
		/*
		 * set the image data size to 0
		 */
		match_data_ptr->cnt = 0;
		/*
		 * set the image checksum to 0
		 */
		match_data_ptr->checksum = (CARD32) 0;
		/*
		 * write the match data information to the internal format file
		 */
		write_match_data_to_internal(to_fd, match_data_ptr);
	}
	else
	{
		/*
		 * if now using checksums but the match data has image data
		 * in it, convert the image data to a checksum
		 */
		read_and_checksum_image_data(from_fptr, match_data_ptr);
		/*
		 * set the image data size to 0
		 */
		match_data_ptr->cnt = 0;
		/*
		 * write the match data information to the internal format file
		 */
		write_match_data_to_internal(to_fd, match_data_ptr);
	}
}

/*******************************************************************************
 *
 *	handle_checksum_data_to_image_data
 *
 *	Handles the case where the output format file has a checksum and
 *	the current checksum mode is images.
 *
 ******************************************************************************/
static void
handle_checksum_data_to_image_data(from_fptr, to_fd, match_data_ptr, stripflag)
/*
 * file pointer of output format file to read from
 */
FILE			*from_fptr;
/*
 * file descriptor of internal format file to write to
 */
int			to_fd;
/*
 * points to the structure to read the match data into
 */
struct match_data	*match_data_ptr;
/*
 * if non-zero then strip out any image data
 */
int			stripflag;
{
	/*
	 * strip the image data if needed
	 */
	if (!stripflag)
	{
		/*
		 * if now using image data but the match data has only
		 * a checksum, warn the user that an update is necessary and
		 * fix the match data structure
		 */
		(void) fprintf(G_msg_file,
			       "Warning: this test script must be updated before being used.\n");
	}
	/*
	 * set the checksum to 0
	 */
	match_data_ptr->checksum = 0;
	/*
	 * set the match image data count to 0
	 */
	match_data_ptr->cnt = 0;
	/*
	 * free the color structure information space, if any
	 */
	if ((match_data_ptr->ncolors > 0) &&
	    (match_data_ptr->color_ptr != (XColor *) 0))
	{
		free ((char *) (match_data_ptr->color_ptr));
	}
	/*
	 * set the color structure pointer to a null pointer
	 */
	match_data_ptr->color_ptr = (XColor *) 0;
	/*
	 * set the color structure count to 0
	 */
	match_data_ptr->ncolors = 0;
	/*
	 * write the match data information to the internal format file
	 */
	write_match_data_to_internal(to_fd, match_data_ptr);
}

/*******************************************************************************
 *
 *	copy_from_output_to_internal
 *
 *	Reads the image data from the output format file and 
 *	writes it internal format to the internal format file.
 *
 ******************************************************************************/
static void
copy_from_output_to_internal(from_fptr, to_fd, count)
/*
 * file pointer to read from
 */
FILE	*from_fptr;
/*
 * file descriptor to write to
 */
int	to_fd;
/*
 * number of bytes to copy
 */
INT32	count;
{
	/*
	 * holds the number of bytes to put into the buffer at a time
	 */
	unsigned int    size;
	/*
	 * used to count through the bytes in the internal format file
	 */
	INT32		i;
	/*
	 * used to count through the bytes in the buffer
	 */
	unsigned int	j;
	/*
	 * holds one byte value
	 */
	int		temp;

	/*
	 * loop through all of the image bytes in the output format file
	 */
	for (i = 0; i < count;)
	{
		/*
		 * figure out how many bytes will fit into the buffer
		 * at a time
		 */
		size = ((count - i) < G_sb_size) ? (count - i) : G_sb_size;
		/*
		 * read and convert that many bytes from the output format file
		 */
		for (j = 0; j < size; j++)
		{
			/*
			 * read in one byte from two characters representing
			 * the byte value in hexadecimal
			 */
			if (fscanf(from_fptr, "%2x", &temp) != 1)
			{
				(void) fprintf(G_msg_file,
					       "error %d while reading image data\n",
					       errno);
				exit(1);
			}
			/*
			 * put the byte value in the buffer
			 */
			G_sb[j] = temp;
		}
		/*
		 * write the converted bytes to the internal format file
		 */
		wrchk2(to_fd,
		       (char *) G_sb,
		       size,
		       "Error writing match data to test script file.\n");
		/*
		 * update the count of bytes written to the internal format file
		 */
		i += size;
	}
}

/*******************************************************************************
 *
 *	read_and_ignore_image_data
 *
 *	Reads the image data from the output format file and ignores it.
 *
 ******************************************************************************/
static void
read_and_ignore_image_data(from_fptr, count)
/*
 * file pointer to read from
 */
FILE	*from_fptr;
/*
 * number of bytes to copy
 */
INT32	count;
{
	/*
	 * used to count through the bytes in the internal format file
	 */
	INT32		i;
	/*
	 * holds one byte value
	 */
	int		temp;

	/*
	 * loop through all of the image bytes in the output format file
	 */
	for (i = 0; i < count; i++)
	{
		/*
		 * read in one byte from two characters representing
		 * the byte value in hexadecimal
		 */
		if (fscanf(from_fptr, "%2x", &temp) != 1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while reading image data\n",
				       errno);
			exit(1);
		}
	}
}

/*******************************************************************************
 *
 *	read_and_checksum_image_data
 *
 *	Reads the image data from the output format file and checksums it.
 *
 ******************************************************************************/
static void
read_and_checksum_image_data(from_fptr, match_data_ptr)
/*
 * file pointer to read from
 */
FILE			*from_fptr;
/*
 * points to the structure to read the match data into
 */
struct match_data	*match_data_ptr;
{
	/*
	 * holds the number of bytes currently in the compressed data buffer
	 */
	unsigned int	comp_count;
	/*
	 * used to count the number of bytes in the compressed data buffer
	 */
	unsigned int	j;
	/*
	 * holds the total number of bytes of compressed data to read
	 */
	INT32		count;
	/*
	 * holds the total number of bytes of compressed data read so far
	 */
	INT32		current_count = 0;
	/*
	 * holds one byte value
	 */
	int		temp;

	/*
	 * start with a checksum of 0
	 */
	match_data_ptr->checksum = (CARD32) 0;
	/*
	 * copy the count into a local variable for understandability
	 */
	count = match_data_ptr->cnt;
	/*
	 * while there is still compressed data to read
	 */
	while (current_count < count)
	{
		/*
		 * figure out how many bytes of compressed data to read 
		 * into the compressed data buffer
		 */
		comp_count = ((count - current_count) < (2 * G_sb_size)) ?
			     (count - current_count) :
			     (2 * G_sb_size);
		/*
		 * read and convert that many bytes from the output format file
		 */
		for (j = 0; j < comp_count; j++)
		{
			/*
			 * read in one byte from two characters representing
			 * the byte value in hexadecimal
			 */
			if (fscanf(from_fptr, "%2x", &temp) != 1)
			{
				(void) fprintf(G_msg_file,
					       "error %d while reading image data\n",
					       errno);
				exit(1);
			}
			/*
			 * put the byte value in the buffer
			 */
			G_c_data_buffer[j] = temp;
		}
		/*
		 * uncompress and checksum the compressed data buffer
		 */
		checksum_a_bufferful(&(match_data_ptr->checksum), comp_count);
		/*
		 * update the current count of compressed data
		 */
		current_count += comp_count;
	}
}

/***********************************************************************
 *
 *	checksum_a_bufferful
 *
 *	This routine uncompresses and checksums a bufferful of compressed
 *	match image data.
 */
static void
checksum_a_bufferful(checksum_ptr, count)
/*
 * points to the checksum
 */
CARD32		*checksum_ptr;
/*
 * holds the number of bytes of compressed image data in the buffer
 */
unsigned int	count;
{
	/*
	 * the index of the current byte in the uncompressed data buffer
	 */
	unsigned int	uncomp_index;
	/*
	 * the index of the current byte in the compressed data buffer
	 */
	unsigned int	comp_index;
	/*
	 * used to count the number of uncompressed bytes in one count:value 
	 * pair
	 */
	int		i;

	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "uncompressing %u bytes of compressed data\n",
			       count);
	}
	/*
	 * start at the beginning of the uncompressed data buffer
	 */
	uncomp_index = 0;
	/*
	 * loop through the count:value pairs in the compressed data buffer
	 */
	for (comp_index = 0; comp_index < count; comp_index += 2)
	{
		/*
		 * if the next count:value pair would not fit into the 
		 * uncompressed image data buffer then checksum the 
		 * uncompressed image data
		 */
		if ((uncomp_index + G_c_data_buffer[comp_index]) > G_sb_size)
		{
			/*
			 * checksum the uncompressed image data
			 */
			compute_checksum(checksum_ptr,
					 G_sb,
					 uncomp_index);
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "compressed index is %u\n",
					       comp_index);
				(void) fprintf(G_msg_file,
					       "checksummed %u bytes of uncompressed data\n",
					       uncomp_index);
			}
			/*
			 * restart at the beginning
			 * of the uncompressed data buffer
			 */
			uncomp_index = 0;
		}
		/*
		 * uncompress the next count:value pair
		 */
		for (i = 0; i < G_c_data_buffer[comp_index]; i++)
		{
			G_sb[uncomp_index] = G_c_data_buffer[comp_index + 1];
			uncomp_index++;
		}
	}
	/*
	 * checksum any leftover uncompressed image data
	 */
	if (uncomp_index != 0)
	{
		compute_checksum(checksum_ptr,
				 G_sb,
				 uncomp_index);
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "compressed index is %u\n",
			       comp_index);
		(void) fprintf(G_msg_file,
			       "checksummed %u bytes of uncompressed data\n",
			       uncomp_index);
	}
}
