/***********************************************************************
 *
 *	matdtoxwd.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/***********************************************************************
 *	include files
 **********************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * more client exerciser type defs
 */
#include  <xtm.h>
/*
 * definitions for the xwd format file
 */
#include  <X11/XWDFile.h>

/***********************************************************************
 *	external variables
 **********************************************************************/

/*
 * size of the buffer for screen matches
 */
extern unsigned int     G_sb_size;
/*
 * this buffer holds the "compressed" match image data
 */
extern unsigned char    *G_c_data_buffer;
/*
 * the buffer holds the "uncompressed" match image data
 */
extern unsigned char	*G_sb;
extern int		G_debug_flag;
extern int		G_checksum_flag;
extern FILE		*G_msg_file;
extern int		errno;

/***********************************************************************
 *	function declarations
 **********************************************************************/

void	uncompress_and_write_image_data();
void	uncompress_a_bufferful();
void	swaplong();
void	swapshort();
void	readchk();
char	*malloc();

/***********************************************************************
 *	functions
 **********************************************************************/

/***********************************************************************
 *
 *	write_match_data_to_xwd
 *
 *	Writes out match data in xwd format.  Much of this code is
 *	adapted from the xwd client source.
 */
void
write_match_data_to_xwd(xwd_fd, image_fd, match_data_ptr)
/*
 * file descriptor of the xwd format file
 */
int			xwd_fd;
/*
 * file descriptor of the file containing the image data
 */
int			image_fd;
/*
 * points to the structure containing the match information
 */
struct match_data	*match_data_ptr;
{
	/*
	 * holds the xwd header information
	 */
	XWDFileHeader	header;
	/*
	 * holds the size of the header
	 */
	int		header_size;
	/*
	 * holds the size of the window name
	 */
	int		window_name_size;
	/*
	 * used as a flag for determining if the header information
	 * needs to be byte-swapped
	 */
	unsigned long	swaptest = 1;
	/*
	 * used to point to the local copy of the color map information
	 */
	XColor		*colors;
	/*
	 * used for debugging
	 */
	char		*temp_ptr;
	/*
	 * used to hold the size of the color map data
	 */
	unsigned int	color_size;
	/*
	 * used to count through the color maps
	 */
	INT32		i;
	/*
	 * holds the root-relative ("absolute") x coordinate of the
	 * image data
	 */
	int		absx;
	/*
	 * holds the root-relative ("absolute") y coordinate of the
	 * image data
	 */
	int		absy;

	/*
	 * Is this ok if the window is partially off the screen???
	 * I think it is because the clipping is done in wintotemp
	 * before the image is read.
	 */

	/*
	 * compute the absolute x and y coordinates of the image data
	 */
	absx = match_data_ptr->window_x +
	       match_data_ptr->window_border_width +
	       match_data_ptr->x;
	absy = match_data_ptr->window_y +
	       match_data_ptr->window_border_width +
	       match_data_ptr->y;
	/*
	 * figure out the size of the window name (including a null terminator)
	 */
	window_name_size = (strlen(match_data_ptr->window_name) + 1) *
			   sizeof(char);
	/*
	 * figure out the size of the header
	 */
	header_size = sizeof(header) + window_name_size;
	/*
	 * fill out the xwd structure
	 */
	header.header_size      = (xwdval) header_size;
	header.file_version     = (xwdval) XWD_FILE_VERSION;
	header.pixmap_format    = (xwdval) match_data_ptr->format;
	header.pixmap_depth     = (xwdval) match_data_ptr->depth;
	header.pixmap_width     = (xwdval) match_data_ptr->width;
	header.pixmap_height    = (xwdval) match_data_ptr->height;
	header.xoffset          = (xwdval) match_data_ptr->xoffset;
	header.byte_order       = (xwdval) match_data_ptr->byte_order;
	header.bitmap_unit      = (xwdval) match_data_ptr->bitmap_unit;
	header.bitmap_bit_order = (xwdval) match_data_ptr->bitmap_bit_order;
	header.bitmap_pad       = (xwdval) match_data_ptr->bitmap_pad;
	header.bits_per_pixel   = (xwdval) match_data_ptr->bits_per_pixel;
	header.bytes_per_line   = (xwdval) match_data_ptr->bytes_per_line;
	header.visual_class     = (xwdval) match_data_ptr->visual_class;
	header.red_mask         = (xwdval) match_data_ptr->red_mask;
	header.green_mask       = (xwdval) match_data_ptr->green_mask;
	header.blue_mask        = (xwdval) match_data_ptr->blue_mask;
	header.bits_per_rgb     = (xwdval) match_data_ptr->bits_per_rgb;
	header.colormap_entries = (xwdval) match_data_ptr->colormap_entries;
	header.ncolors          = (xwdval) match_data_ptr->ncolors;
	header.window_width     = (xwdval) match_data_ptr->window_width;
	header.window_height    = (xwdval) match_data_ptr->window_height;
	header.window_bdrwidth  = (xwdval) match_data_ptr->window_border_width;
	/*
	 * The window_x and window_y values in the header structure are
	 * misnamed.  They actually contain the absolute coordinates of
	 * the image data.  They are normally used by xwud to create a
	 * window in the correct place on the screen that just fits
	 * the image data.
	 */
	header.window_x         = (xwdval) absx;
	header.window_y         = (xwdval) absy;
	/*
	 * byte-swap the header if needed
	 */
	if (*(char *) &swaptest)
	{
		swaplong((char *) &header, sizeof(header));
	}
	/*
	 * write the xwd header to the xwd format file
	 */
	wrchk2(xwd_fd,
	       (char *) &header,
	       sizeof(header),
	       "error while writing xwd header to xwd file\n");
	/*
	 * write the window name to the xwd format file
	 */
	wrchk2(xwd_fd,
	       match_data_ptr->window_name,
	       (unsigned int) window_name_size,
	       "error while writing window name to xwd file\n");
	/*
	 * write out color map information, if any
	 */
	if ((match_data_ptr->ncolors > 0) &&
	    (match_data_ptr->color_ptr != (XColor *) 0))
	{
		/*
		 * Allocate space for the local copy of the color map
		 * information.  We need this copy because the color map
		 * information might get byte-swapped, and we don't want
		 * to change the original match data.
		 */
		if ((sizeof(XColor) * match_data_ptr->ncolors) >
		    (unsigned int) UNSIGNED_INT_MAX)
		{
			(void) fprintf(G_msg_file,
				       "color map data is too large in write_match_data_to_xwd\n");
			exit(1);
		}
		color_size = sizeof(XColor) * match_data_ptr->ncolors;
		temp_ptr = malloc(color_size);
		if (temp_ptr == (char *) 0)
		{
			(void) fprintf(G_msg_file,
				       "out of memory in write_match_data_to_xwd\n");
			exit(1);
		}
		colors = (XColor *) temp_ptr;
		/*
		 * copy the color map information to the local copy
		 */
		for (i = 0; i < match_data_ptr->ncolors; i++)
		{
			colors[i].pixel = match_data_ptr->color_ptr[i].pixel;
			colors[i].red = match_data_ptr->color_ptr[i].red;
			colors[i].green = match_data_ptr->color_ptr[i].green;
			colors[i].blue = match_data_ptr->color_ptr[i].blue;
			colors[i].flags = match_data_ptr->color_ptr[i].flags;
			colors[i].pad = match_data_ptr->color_ptr[i].pad;
		}
		/*
		 * byte-swap the local copy of the color map information
		 * if needed
		 */
		if (*(char *) &swaptest)
		{
			for (i = 0; i < match_data_ptr->ncolors; i++)
			{
				swaplong((char *) &(colors[i].pixel),
					 sizeof(long));
				swapshort((char *) &(colors[i].red),
					  (3 * sizeof(short)));
			}
		}
		/*
		 * write out the color map information
		 */
		wrchk2(xwd_fd,
		       (char *) colors,
		       (match_data_ptr->ncolors * sizeof(XColor)),
		       "error while writing color map information to xwd file\n");
		/*
		 * free the local copy of the color map information
		 */
		free((char *) (colors));
	}
	if (G_checksum_flag)
	{
		/*
		 * If using checksums, there is no image data to write.
		 * Just write out the checksum instead.
		 */
		wrchk2(xwd_fd,
		       (char *) &(match_data_ptr->checksum),
		       sizeof(match_data_ptr->checksum),
		       "error while writing match checksum to xwd file\n");
	}
	else
	{
		/*
		 * uncompress and write out the image data
		 */
		uncompress_and_write_image_data(xwd_fd,
						image_fd,
						match_data_ptr->cnt);
	}
}

/***********************************************************************
 *
 *	uncompress and write image data
 *
 *	This routine uncompresses match image data.
 */
static void
uncompress_and_write_image_data(xwd_fd, image_fd, count)
/*
 * file descriptor of the xwd format file
 */
int	xwd_fd;
/*
 * file descriptor of the file containing the image data
 */
int	image_fd;
/*
 * holds the number of bytes of compressed image data
 */
INT32	count;
{
	/*
	 * holds the number of bytes currently in the compressed data buffer
	 */
	unsigned int	comp_count;
	/*
	 * holds the total number of bytes of compressed data read so far
	 */
	INT32		current_count = 0;

	/*
	 * while there is still compressed data to read
	 */
	while (current_count < count)
	{
		/*
		 * figure out how many bytes of compressed data to read 
		 * into the compressed data buffer
		 */
		comp_count = ((count - current_count) < (2 * G_sb_size)) ?
			     (count - current_count) :
			     (2 * G_sb_size);
		/*
		 * read the bytes of compressed data
		 */
		readchk(image_fd,
			(char *) G_c_data_buffer,
			comp_count,
			"error while reading match image data from file\n");
		/*
		 * uncompress and write out the compressed data buffer
		 */
		uncompress_a_bufferful(xwd_fd, comp_count);
		/*
		 * update the current count of compressed data
		 */
		current_count += comp_count;
	}
}

/***********************************************************************
 *
 *	uncompress_a_bufferful
 *
 *	This routine uncompresses a buffferful of compressed match image data.
 *	The image data is in pairs of one byte of count followed by one byte
 *	of data.
 */
static void
uncompress_a_bufferful(xwd_fd, count)
/*
 * file descriptor of the xwd format file
 */
int		xwd_fd;
/*
 * holds the number of bytes of compressed image data in the buffer
 */
unsigned int	count;
{
	/*
	 * the index of the current byte in the uncompressed data buffer
	 */
	unsigned int	uncomp_index;
	/*
	 * the index of the current byte in the compressed data buffer
	 */
	unsigned int	comp_index;
	/*
	 * used to count the number of uncompressed bytes in one count:value 
	 * pair
	 */
	int		i;

	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "uncompressing %u bytes of compressed data\n",
			       count);
	}
	/*
	 * start at the beginning of the uncompressed data buffer
	 */
	uncomp_index = 0;
	/*
	 * loop through the count:value pairs in the compressed data buffer
	 */
	for (comp_index = 0; comp_index < count; comp_index += 2)
	{
		/*
		 * if the next count:value pair would not fit into the 
		 * uncompressed image data buffer then write the
		 * uncompressed image data out to the xwd file
		 */
		if ((uncomp_index + G_c_data_buffer[comp_index]) > G_sb_size)
		{
			/*
			 * write out the uncompressed image data
			 * out to the xwd file
			 */
			wrchk2(xwd_fd,
			       (char *) G_sb,
			       uncomp_index,
			       "error while writing match image data to xwd file\n");
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "compressed index is %u\n",
					       comp_index);
				(void) fprintf(G_msg_file,
					       "wrote out %u bytes of uncompressed data\n",
					       uncomp_index);
			}
			/*
			 * restart at the beginning
			 * of the uncompressed data buffer
			 */
			uncomp_index = 0;
		}
		/*
		 * uncompress the next count:value pair
		 */
		for (i = 0; i < G_c_data_buffer[comp_index]; i++)
		{
			G_sb[uncomp_index] = G_c_data_buffer[comp_index + 1];
			uncomp_index++;
		}
	}
	/*
	 * write out any leftover uncompressed image data to the xwd file
	 */
	if (uncomp_index != 0)
	{
		wrchk2(xwd_fd,
		       (char *) G_sb,
		       uncomp_index,
		       "error while writing match image data to xwd file\n");
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "compressed index is %u\n",
			       comp_index);
		(void) fprintf(G_msg_file,
			       "wrote out %u bytes of uncompressed data\n",
			       uncomp_index);
	}
}
