/***********************************************************************
 *
 *	inttoout.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/***********************************************************************
 *	include files
 **********************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * more client exerciser type defs
 */
#include  <xtm.h>

/***********************************************************************
 *	external variables
 **********************************************************************/

/*
 * Buffer for match image data
 */
extern unsigned char	*G_sb;
/*
 * size of matbuf
 */
extern unsigned int	G_sb_size;
extern int		G_debug_flag;
extern int		G_checksum_flag;
extern FILE		*G_msg_file;
extern int		errno;

/***********************************************************************
 *	function declarations
 **********************************************************************/

void	readchk();
void	getmatchdata();
void	read_match_data_from_internal();
void	write_match_data_to_output();
void	handle_image_data_from_internal();
void	copy_from_internal_to_output();
long	lseek();

/***********************************************************************
 *	functions
 **********************************************************************/

/***********************************************************************
 *
 *	inttoout
 *
 *	This routine translates match data from internal format to
 *	output format.
 */
void
inttoout(from_fd, to_fptr, matchcount, stripflag)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * file pointer of output format file
 */
FILE	*to_fptr;
/*
 * count of the number of matches so far
 */
int	matchcount;
/*
 * strip image data if non-zero
 */
int	stripflag;
{
	/*
	 * holds the match information temporarily
	 */
	struct match_data	match_data;

	/*
	 * read the match data information into the match data structure
	 */
	read_match_data_from_internal(from_fd, &match_data);
	/*
	 * write out the match data structure information
	 */
	write_match_data_to_output(to_fptr, &match_data, matchcount, stripflag);
	/*
	 * if not in checksum mode handle the match data image information
	 */
	if (!G_checksum_flag)
	{
		handle_image_data_from_internal(from_fd,
						to_fptr,
						match_data.cnt,
						stripflag);
	}
	/*
	 * free the color map structure memory, if any
	 */
	if ((match_data.ncolors > 0) &&
	    (match_data.color_ptr != (XColor *) 0))
	{
		free((char *) (match_data.color_ptr));
	}
}

/***********************************************************************
 *
 *	write_match_data_to_output
 *
 *	Writes the match data structure information out to an output format
 *	file.
 */
static void
write_match_data_to_output(output_fptr, match_data_ptr, matchcount, stripflag)
/*
 * holds the file pointer of the file to write to
 */
FILE			*output_fptr;
/*
 * points to the match data to write out
 */
struct match_data	*match_data_ptr;
/*
 * holds the count of the number of matches so far
 */
int			matchcount;
/*
 * strip image data if non-zero
 */
int	stripflag;
{
	/*
	 * holds the number of bytes of image data to be written to
	 * the output format file
	 */
	INT32	image_size;
	/*
	 * holds the number of color map structures to write out
	 * to the output format file
	 */
	INT32	ncolors;
	/*
	 * used to count through the color map entries
	 */
	INT32	i;
	/*
	 * used to point through the color map entries
	 */
	XColor	*color_ptr;

	/*
	 * if stripping then don't put out any color map structures
	 * or image data
	 */
	if (stripflag)
	{
		ncolors = (INT32) 0;
		image_size = (INT32) 0;
	}
	else
	{
		ncolors = match_data_ptr->ncolors;
		image_size = match_data_ptr->cnt;
	}
	/*
	 * output the match count, type, retry count, retry interval,
	 * compressed image size and the checksum mode
	 */
	(void) fprintf(output_fptr,
		       "MATCH #%d TYPE[%c] REP[%ld] INTV[%ld] CNT[%ld] CM[%d] \n",
		       matchcount,
		       (unsigned char) match_data_ptr->type,
		       match_data_ptr->retry,
		       match_data_ptr->intv,
		       image_size,
		       G_checksum_flag);
	/*
	 * output the match data window screen number, x and y coordinates,
	 * width, height, and border width
	 */
	(void) fprintf(output_fptr,
		       "\tS#[%d] WX[%d] WY[%d] WW[%d] WH[%d] BW[%d] \n",
		       match_data_ptr->screen_number,
		       match_data_ptr->window_x,
		       match_data_ptr->window_y,
		       match_data_ptr->window_width,
		       match_data_ptr->window_height,
		       match_data_ptr->window_border_width);
	/*
	 * output the match data window name
	 */
	(void) fprintf(output_fptr,
		       "\tWN\"%s\"\n",
		       match_data_ptr->window_name);
	/*
	 * output the match data x and y coordinates in the window,
	 * width, height, depth, format, and x offset
	 */
	(void) fprintf(output_fptr,
		       "\tX[%d] Y[%d] W[%d] H[%d] D[%d] F[%d] XOFF[%d] \n",
		       match_data_ptr->x,
		       match_data_ptr->y,
		       match_data_ptr->width,
		       match_data_ptr->height,
		       match_data_ptr->depth,
		       match_data_ptr->format,
		       match_data_ptr->xoffset);
	/*
	 * output the match data byte order, bitmap unit, bitmap bit order,
	 * bitmap pad, bytes per line, and bits per pixel
	 */
	(void) fprintf(output_fptr,
		       "\tBO[%d] BU[%d] BBO[%d] BP[%d] BPL[%d] BPP[%d] \n",
		       match_data_ptr->byte_order,
		       match_data_ptr->bitmap_unit,
		       match_data_ptr->bitmap_bit_order,
		       match_data_ptr->bitmap_pad,
		       match_data_ptr->bytes_per_line,
		       match_data_ptr->bits_per_pixel);
	/*
	 * output the match data visual class, red mask, green mask,
	 * blue mask, bits per rgb, number of entries in the colormap,
	 * and the number of color structures
	 */
	(void) fprintf(output_fptr,
		       "\tVC[%d] RM[%lu] GM[%lu] BM[%lu] BPR[%d] CE[%d] C[%ld] \n",
		       match_data_ptr->visual_class,
		       match_data_ptr->red_mask,
		       match_data_ptr->green_mask,
		       match_data_ptr->blue_mask,
		       match_data_ptr->bits_per_rgb,
		       match_data_ptr->colormap_entries,
		       ncolors);
	/*
	 * output the color map structures, if any
	 */
	for (i = 0, color_ptr = match_data_ptr->color_ptr;
	     i < ncolors;
	     i++, color_ptr++)
	{
		(void) fprintf(output_fptr,
			       "\tPIXEL[%lu] RED[%hu] GREEN[%hu] BLUE[%hu] FLAGS[%d] \n",
			       color_ptr->pixel,
			       color_ptr->red,
			       color_ptr->green,
			       color_ptr->blue,
			       (int) color_ptr->flags);
	}
	/*
	 * if we are using checksums, output the checksum
	 */
	if (G_checksum_flag)
	{
		/*
		 * output the match data checksum
		 */
		(void) fprintf(output_fptr,
			       "\tCHECKSUM[%lu] \n",
			       match_data_ptr->checksum);
	}
}

/***********************************************************************
 *
 *	handle_image_data_from_internal
 *
 *	Handle the image data from the internal format file.
 */
static void
handle_image_data_from_internal(from_fd, to_fptr, count, stripflag)
/*
 * file descriptor of internal format file
 */
int	from_fd;
/*
 * file pointer of output format file
 */
FILE	*to_fptr;
/*
 * number of bytes to copy
 */
INT32	count;
/*
 * strip image data if non-zero
 */
int	stripflag;
{
	if (stripflag)
	{
		/*
		 * skip over the image data in the internal format file
		 */
		if (lseek(from_fd, (long) count, 1) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while seeking in internal format file\n",
				       errno);
			exit(1);
		}
	}
	else
	{
		/*
		 * read the image data from the internal format file
		 * and write it to the output format file
		 */
		copy_from_internal_to_output(from_fd, to_fptr, count);
	}	
}

/***********************************************************************
 *
 *	copy_from_internal_to_output
 *
 *	Read the image data from the internal format file and
 *	write it in output format to the output format file.
 */
static void
copy_from_internal_to_output(from_fd, to_fptr, count)
/*
 * file descriptor to read from
 */
int	from_fd;
/*
 * file pointer to write to
 */
FILE	*to_fptr;
/*
 * number of bytes to copy
 */
INT32	count;
{
	/*
	 * holds the number of bytes that have been read from 
	 * the internal format file
	 */
	INT32		tread = 0;  
	/*
	 * holds the number of bytes to read into the buffer
	 */
	unsigned int	size;  
	/*
	 * used to count through the bytes in the buffer
	 */
	int		j;  

	/*
	 * read all of the bytes from the internal format file
	 */
	for (tread = 0; tread < count;)
	{
		if (G_debug_flag)
		{
			(void) fprintf(G_msg_file,
				       "count: %d, read: %d\n",
				       count,
				       tread);
		}
		/*
		 * figure out how many bytes will fit into the buffer
		 * at a time
		 */
		size = ((count - tread) < G_sb_size) ? (count - tread) :
						       G_sb_size;
		/*
		 * read that many bytes from the internal format file
		 */
		readchk(from_fd,
			(char *) G_sb,
			size,
			"match data read error\n");
		/*
		 * update the count of bytes read from the internal format file
		 */
		tread += size;
		/*
		 * loop through the buffer, writing the bytes out in ascii
		 * hex characters
		 */
		for (j = 1; j <= size; j++)
		{
			/*
			 * write out a byte as two characters representing
			 * the byte value in hexadecimal
			 */
			(void) fprintf(to_fptr, "%02x", G_sb[j - 1]);
			/*
			 * put a newline after every 80 characters
			 */
			if (j % 40 == 0)
			{
				(void) fprintf(to_fptr, "\n");
			}
		}
	}
	/*
	 * put a newline after all of the image data
	 */
	(void) fprintf(to_fptr, "\n");
}
