/**********************************************************************
 *
 *	clrpadbits.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/**********************************************************************
 * include files
 **********************************************************************/

/*
 * contains NULL constant
 */
#include  <stdio.h>
/*
 * X library files
 */
#include  <X11/Xlib.h>
/*
 * ???
 */
#include  <X11/Xmd.h>
/*
 * client exerciser constants
 */
#include  <xtm.h>

/**********************************************************************
 * external variables
 **********************************************************************/

/*
 * undocumented debug invocation option flag
 */
extern int           	G_debug_flag;
/*
 * pointer to diagnostic/error message file
 */
extern FILE		*G_msg_file;

/**********************************************************************
 * variables
 **********************************************************************/

/*
 * these flags are non-zero if a warning message has already been output
 */
static int		pad_bits_warning_flag;
static int		pad_bytes_warning_flag;
static int		per_pixel_pad_bits_warning_flag;
/*
 * used to determine if the current machine is LSBFirst or MSBFirst
 */
static unsigned long	swaptest = 1;
static int		local_byte_order;

/**********************************************************************
 * function declarationss
 **********************************************************************/

void	clear_end_of_line_pad_bits();
void	clear_per_pixel_pad_bits();
void	compute_pad_bits_mask();
void	clear_end_pad_bytes();
void	clear_end_pad_bits();
void	clear_one_line_of_pad_bits();

/**********************************************************************
 * functions
 **********************************************************************/

/**********************************************************************
 *
 *	clrpadbits
 *
 *	The pad bits (if any) in the image data are not guaranteed to
 *	be a repeatable value when the same image is read more than
 *	once.  This causes problems for matches because the code
 *	works on all of the image data including the pad bits.
 *
 *	The "fix", such as it is, is to set the pad bits to a fixed
 *	value (0), so that the image will be repeatable.
 *
 *	Perhaps a better choice would have been to change the image
 *	compression, comparison, and checksum algorithms to work on
 *	individual pixels instead of treating the image data as one
 *	big block of bytes.  Unfortunately, the full nature of this
 *	problem was not discovered until after it was too late to
 *	make such major modifications to the way things worked.
 *	
 *	I don't know how to work with image data in the general case,
 *	and the existing Xlib routines are not very useful for working
 *	with all of the pixels in the image data.  Therefore, the
 *	following code handles the special cases that I know how to do,
 *	and puts out a warning message for the other cases.
 *
 *	There are two types of pad bits to deal with:
 *
 *	1)  Pad bits at the end of each scan line.  For example,
 *	    if an image stored in 8 bits per pixel has a width
 *	    of 126 pixels and the X server pads each scan line
 *	    to a multiple of 32 bits, there would be 16 pad bits
 *	    at the end of each scan line.
 *
 *	2)  Pad bits associated with each pixel.  For example,
 *	    an image of depth 6 stored in 8 bits per pixel has
 *	    two pad bits for each pixel.
 */
void
clear_pad_bits(image_data_ptr, image_height, p)
/*
 * this points to the image data
 */
unsigned char		*image_data_ptr;
/*
 * holds the number of scan lines of image data
 */
int			image_height;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	/*
	 * holds the number of pad bits
	 */
	CARD32		pad_bits;

	/*
	 * figure out the byte order of the local machine
	 * (the one that this software is running on)
	 */
	if (*(char *) &swaptest)
	{
		local_byte_order = LSBFirst;
	}
	else
	{
		local_byte_order = MSBFirst;
	}
	/*
	 * figure out the number of bits per scan line
	 */
	pad_bits = p->bytes_per_line * 8;
	/*
	 * subtract the number of bits the pixels take up to leave
	 * the number of pad bits per scan line
	 */
	pad_bits -= p->bits_per_pixel * p->width;
	/*
	 * if there are end of line pad bits, clear them
	 */
	if (pad_bits != 0)
	{
		clear_end_of_line_pad_bits(image_data_ptr,
					   image_height,
					   p,
					   pad_bits);
	}
	/*
	 * if needed, clear any per-pixel pad bits in the image
	 */
	if (p->depth != p->bits_per_pixel)
	{
		clear_per_pixel_pad_bits(image_data_ptr, image_height, p);
	}
}

/**********************************************************************
 *
 *	clear_end_of_line_pad_bits
 *
 *	Clear any pad bits at the end of each scan line of pixels.
 */
static void
clear_end_of_line_pad_bits(image_data_ptr, image_height, p, pad_bits)
/*
 * this points to the image data
 */
unsigned char		*image_data_ptr;
/*
 * holds the number of scan lines of image data
 */
int			image_height;
/*
 * this points to the match data structure
 */
struct match_data	*p;
/*
 * holds the number of pad bits to clear
 */
CARD32			pad_bits;
{
	/*
	 * holds the number of pad bytes to clear
	 */
	CARD32		pad_bytes;
	/*
	 * holds the mask for pad bits
	 */
	unsigned char	pad_bits_mask;
	/*
	 * used as a loop counter
	 */
	CARD32		i;

	/*
	 * clear the pad bits and pad bytes warning flags
	 */
	pad_bits_warning_flag = 0;
	pad_bytes_warning_flag = 0;
	/*
	 * figure out how many pad bytes there are
	 */
	pad_bytes = pad_bits / 8;
	/*
	 * figure out the remaining number of pad bits
	 */
	pad_bits = pad_bits % 8;
	/*
	 * compute the pad bit mask if it is needed
	 */
	if (pad_bits != 0)
	{
		compute_pad_bits_mask(pad_bits, &pad_bits_mask, p);
	}
	/*
	 * loop for each scan line of pixels in the image
	 */
	for (i = 0; i < image_height; i++)
	{
		/*
		 * if needed, set the pad bytes at the end of each line
		 * of pixels to 0
		 */
		if (pad_bytes != 0)
		{
			clear_end_pad_bytes(image_data_ptr, pad_bytes, p);
		}
		/*
		 * if needed, set the pad bits at the end of each line
		 * of pixels to 0
		 */
		if (pad_bits != 0)
		{
			clear_end_pad_bits(image_data_ptr,
					   pad_bytes,
					   pad_bits,
					   pad_bits_mask,
					   p);
		}
		/*
		 * advance to the next scan line of pixels in the image
		 */
		image_data_ptr += p->bytes_per_line;
	}
}

/**********************************************************************
 *
 *	compute_pad_bits_mask
 *
 *	Compute the pad bits mask to be "anded" later with the byte that
 *	has one or more end of scan line pad bits in it.
 */
static void
compute_pad_bits_mask(pad_bits, pad_bits_mask_ptr, p)
/*
 * holds the number of pad bits
 */
CARD32			pad_bits;
/*
 * points to where to put the pad bits mask
 */
unsigned char		*pad_bits_mask_ptr;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	/*
	 * this is a special case that I know how to handle
	 */
	if ((p->byte_order == MSBFirst) &&
	    (p->bitmap_bit_order == MSBFirst) &&
	    (local_byte_order == MSBFirst))
	{
		/*
		 * figure out the pad bit mask for this case
		 */
		*pad_bits_mask_ptr = ~((1 << pad_bits) - 1);
	}
	else
	{
		/*
		 * if there has been no warning message, warn the user
		 */
		if (!pad_bits_warning_flag)
		{
			(void) fprintf(G_msg_file, "Warning:  This image has pad bits that this program does not know how to clear.\n");
			(void) fprintf(G_msg_file, "          Matches may fail if the pad bits are not a repeatable value.\n");
			/*
			 * set the flag so the warning message
			 * will not be repeated
			 */
			pad_bits_warning_flag = 1;
		}
	}
}

/**********************************************************************
 *
 *	clear_end_pad_bytes
 *
 *	Set all of the end of scan line pad bytes to 0.
 */
static void
clear_end_pad_bytes(scan_line_ptr, pad_bytes, p)
/*
 * this points to the first byte of the scan line
 */
unsigned char		*scan_line_ptr;
/*
 * holds the number of pad bytes to clear
 */
CARD32			pad_bytes;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	/*
	 * used to count the number of pad bytes to clear
	 */
	CARD32		j;

	/*
	 * this is a special case that I know how to handle
	 */
	if ((p->byte_order == MSBFirst) &&
	    (p->bitmap_bit_order == MSBFirst) &&
	    (local_byte_order == MSBFirst))
	{
		/*
		 * point the scan line pointer at the end of the scan line 
		 * plus one byte
		 */
		scan_line_ptr += p->bytes_per_line;
		/*
		 * work backwards clearing pad bytes
		 */
		for (j = 0; j < pad_bytes; j++)
		{
			*(scan_line_ptr - (j + 1)) = 0;
		}
	}
	else
	{
		/*
		 * if there has been no warning message, warn the user
		 */
		if (!pad_bytes_warning_flag)
		{
			(void) fprintf(G_msg_file, "Warning:  This image has pad bytes that this program does not know how to clear.\n");
			(void) fprintf(G_msg_file, "          Matches may fail if the pad bytes are not a repeatable value.\n");
			/*
			 * set the flag so the warning message
			 * will not be repeated
			 */
			pad_bytes_warning_flag = 1;
		}
	}
}

/**********************************************************************
 *
 *	clear_end_pad_bits
 *
 *	Clear the pad bits at the end of each scan line of pixels.
 */
static void
clear_end_pad_bits(scan_line_ptr, pad_bytes, pad_bits, pad_bits_mask, p)
/*
 * this points to the first byte of the scan line
 */
unsigned char		*scan_line_ptr;
/*
 * holds the number of pad bytes to clear
 */
CARD32			pad_bytes;
/*
 * holds the number of pad bits
 */
CARD32			pad_bits;
/*
 * holds the pad bits mask
 */
unsigned char		pad_bits_mask;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	if ((p->byte_order == MSBFirst) &&
	    (p->bitmap_bit_order == MSBFirst) &&
	    (local_byte_order == MSBFirst))
	{
		/*
		 * point at the correct byte in the scan line
		 */
		scan_line_ptr += p->bytes_per_line - (pad_bytes + 1);
		/*
		 * clear any pad bits
		 */
		*(scan_line_ptr) &= pad_bits_mask;
	}
	else
	{
		/*
		 * there was already a warning message generated when the
		 * pad bits mask was computed, so no message is needed here
		 */
	}
}

/**********************************************************************
 *
 *	clear_per_pixel_pad_bits
 *
 *	Clear the pad bits associated with each pixel.
 */
static void
clear_per_pixel_pad_bits(image_data_ptr, image_height, p)
/*
 * this points to the image data
 */
unsigned char		*image_data_ptr;
/*
 * holds the number of scan lines of image data
 */
int			image_height;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	/*
	 * used as a loop counter
	 */
	CARD32		i;

	/*
	 * clear the per pixel pad bits warning flag
	 */
	per_pixel_pad_bits_warning_flag = 0;
	/*
	 * loop for each scan line of pixels in the image
	 */
	for (i = 0; i < image_height; i++)
	{
		/*
		 * clear the per-pixel pad bits in one scan line
		 */
		clear_one_line_of_pad_bits(image_data_ptr, p);
		/*
		 * advance to the next scan line of pixels in the image
		 */
		image_data_ptr += p->bytes_per_line;
	}
}

/**********************************************************************
 *
 *	clear_one_line_of_pad_bits
 *
 *	Clear the pad bits associated with each pixel in one scan line.
 */
static void
clear_one_line_of_pad_bits(scan_line_ptr, p)
/*
 * this points to the image data
 */
unsigned char		*scan_line_ptr;
/*
 * this points to the match data structure
 */
struct match_data	*p;
{
	/*
	 * used as a loop counter
	 */
	int		i;
	/*
	 * used as a mask for 8 bits per pixel
	 */
	unsigned char	pixel_mask_8;
	/*
	 * used as a mask for 16 bits per pixel
	 */
	CARD16		pixel_mask_16;
	/*
	 * used as a mask for 32 bits per pixel
	 */
	CARD32		pixel_mask_32;

	/*
	 * this is a special case that I know how to handle
	 */
	if ((p->byte_order == MSBFirst) &&
	    (p->bitmap_bit_order == MSBFirst) &&
	    (local_byte_order == MSBFirst))
	{
		/*
		 * compute the appropriate pad bits mask
		 */
		switch(p->bits_per_pixel)
		{
		case 8:
			pixel_mask_8 = (1 << p->depth) - 1;
			break;
		case 16:
			pixel_mask_16 = (1 << p->depth) - 1;
			break;
		case 32:
			pixel_mask_32 = (1 << p->depth) - 1;
			break;
		default:
			(void) fprintf(G_msg_file, "This image has a bits-per-pixel that this program doesn't know how to handle.\n");
			exit(1);
			break;
		}
		/*
		 * loop for each pixel in the scan line
		 */
		for (i = 0; i < p->width; i++)
		{
			switch(p->bits_per_pixel)
			{
			case 8:
				*scan_line_ptr &= pixel_mask_8;
				scan_line_ptr += 1;
				break;
			case 16:
				*((CARD16 *) scan_line_ptr) &= pixel_mask_16;
				scan_line_ptr += 2;
				break;
			case 32:
				*((CARD32 *) scan_line_ptr) &= pixel_mask_32;
				scan_line_ptr += 4;
				break;
			default:
				/*
				 * there was already an error exit caused when
				 * the pad bits mask was computed, so no code is
				 * needed here
				 */
				break;
			}
		}
	}
	else
	{
		(void) fprintf(G_msg_file, "Warning: this image has per-pixel pad bits\n");
		(void) fprintf(G_msg_file, "         that this program does not know how to clear.\n");
		(void) fprintf(G_msg_file, "         Matches may fail if the per-pixel pad bits are not a repeatable value.\n");
		/*
		 * set the flag so the warning message
		 * will not be repeated
		 */
		per_pixel_pad_bits_warning_flag = 1;
	}
}
