#!/usr/X/bin/wksh -motif

#
#	lpx:	Replacement for lp command.  Pops up an X window to allow you
#		to specify destination and other lp options.
#
#	Rick Richardson, Digi Intl., rick@digibd.com
#

#
#	WKSH Programming notes:
#
#	- You cannot XtGetValues for menuHistory.  Instead, I set up
#	  a callback on each option menu button, which just sets
#	  a shell variable whenever the button is selected.
#	- You cannot use panedWindow - wksh doesn't know about it.
#

#########################################################################
#									#
#	Useful functions						#
#									#
#########################################################################
function postscript	# Return true or false if $* is a postscript file
{
	# A file is PS if it starts with "%!" or "^D%!"
	ctrl_d=`echo "\004\c"`
	cat $* | read a
        case "$a" in
        %!*)            return 0;;
        "$ctrl_d"%!*)   return 0;;
        *)              return 1;;
        esac
}

function pcl		# Return true or false if $* is a HP PLC file
{
	# A file is PCL if there are escape characters in the file
	esc=`cat $* | tr -d '[\001-\032][\034-\377]' | wc -c`
	if [ $esc -gt 0 ]; then return 0; else return 1; fi
}

#########################################################################
#									#
#	Callbacks 							#
#									#
#########################################################################
function ok_cb
{
	cmd="lp"
	cmd="echo lp"		# For debugging

	case "$Dest" in
	"")		;;
	DigiFAX)	do_fax;; 
	*)		cmd="$cmd -d '$Dest'";;
	esac

	case "$Filetype" in
	ps)		cmd="$cmd -o ps";;
	pcl)		cmd="$cmd -o pcl";;
	esac

	XtGetValues $copies value:Copies
	if [ "$Copies" -lt 1 ]; then Copies=1; fi
	if [ "$Copies" != "" ]; then cmd="$cmd -n '$Copies'"; fi

	XtGetValues $title value:Title
	if [ "$Title" != "" ]; then cmd="$cmd -t '$Title'"; fi

	XtGetValues $other value:Other
	if [ "$Other" != "" ]; then cmd="$cmd $Other"; fi

	if [ "$TmpFile" != "" ]; then
		cmd="$cmd -c $Files"
		$cmd
		rm -f $TmpFile
	else
		cmd="$cmd $Files"
		$cmd
	fi
	exit 0
}

function cancel_cb
{
	if [ "$TmpFile" != "" ]; then
		rm -f $TmpFile
	fi
	exit
}

function new_copies
{
	XtGetValues $copies value:Copies
	((Copies=$Copies+$1))
	if [ "$Copies" -lt 1 ]; then Copies=1; fi
	XtSetValues $copies value:$Copies
	XtSetValues $copies cursorPosition:"${#Copies}"
}

function do_fax
{
	if [ "$TmpFile" != "" ]; then
		(xfaxsend $FFiles; rm -f $TmpFile) &
	else
		xfaxsend $FFiles &
	fi
	exit 0
}

#########################################################################
#									#
#	Program Starts here						#
#									#
#########################################################################

#
#	Parse the options accepted by "lp"
#
#	We ignore most, but you may want to add support for them yourself
#
Command="$0 $@"
Copies=1
Other=
while getopts 'cd:f:H:mn:o:P:q:rsS:t:T:wy:' let
do
	case "$let" in
	c)	Copyit=1;;
	d)	Dest=$OPTARG;;
	f)	Form=$OPTARG;		Other="${Other}-f '$OPTARG' ";;
	H)	Handling=$OPTARG;	Other="${Other}-H '$OPTARG' ";;
	m)	Mail=1;			Other="${Other}-m ";;
	n)	Copies=$OPTARG;;
	o)	Options=$OPTARG;	Other="${Other}-o '$OPTARG' ";;
	P)	Pages=$OPTARG;		Other="${Other}-P '$OPTARG' ";;
	q)	Priority=$OPTARG;	Other="${Other}-q '$OPTARG' ";;
	r)	Reject=1;		Other="${Other}-r ";;
	s)	Supress=1;		Other="${Other}-s ";;
	S)	Priority=$OPTARG;	Other="${Other}-S '$OPTARG' ";;
	t)	Title=$OPTARG;;
	T)	Content=$OPTARG;	Other="${Other}-T '$OPTARG' ";;
	w)	Write=1;		Other="${Other}-w ";;
	y)	Modes=$OPTARG;		Other="${Other}-y '$OPTARG' ";;
	'?')	usage() ;;
	esac
done
shift `expr $OPTIND - 1`

if [ $# = 0 ]; then
	TmpFile=/tmp/lpx$$
	Files=$TmpFile
	FFiles="-f $TmpFile"
	cat > $Files
else
	TmpFile=
	Files=
	for i in $*
	do
		Files="$Files $i"
		FFiles="$FFiles -f $i"
	done
fi

#
#	Guess what type the file is
#
if postscript $Files; then
	Filetype=ps
elif pcl $Files; then
	Filetype=pcl
else
	Filetype=text
fi

#
#	Create the widgets
#
XtAppInitialize topLevel lpx lpx "$@" \

XtSetValues $topLevel title:"$Command"

#
#	Create a control area and an action area, with a separator
#		NOTE: panedWindow is missing in the current wksh, so we use
#		a form, instead.
#
XmCreateForm pane $topLevel pane

	# This order of creation allows the form to stretch the way we want
	# it to (action area never grows in height)

XtCreateWidget aa aa form $pane  \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_form \
				fractionBase:59 \
				# Fraction base is 20*nbuttons-1

XtCreateManagedWidget panesep panesep separator $pane \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_widget \
				bottomWidget:$aa \

XtCreateWidget ca ca rowColumn $pane \
				topAttachment:attach_form \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_widget \
				bottomWidget:$panesep \

#
#	Create the destination option menu
#
XmCreatePulldownMenu destMenu $ca destMenu \

XtCreateManagedWidget destDefault destDefault pushButton $destMenu \
				labelString:"System Default Printer" \
				activateCallback:"Dest=" \

XtCreateManagedWidget destFAX destFAX pushButton $destMenu \
				labelString:"FAX it using DigiFAX" \
				activateCallback:"Dest=DigiFAX" \

XtCreateManagedWidget destSEP destSEP separator $destMenu \

destHistory=$destDefault
lpstat -p | awk '{print $2}' |
while read name
do
	XtCreateManagedWidget printer printer pushButton $destMenu \
				labelString:"$name" \
				activateCallback:"Dest=$name" \

	if [ "$name" = "$Dest" ]; then
		destHistory=$printer
	fi
done

XmCreateOptionMenu destOptions $ca destOptions \
				subMenuId:$destMenu \
				menuHistory:$destHistory \
				labelString:Destination \

XtManageChildren $destOptions

#
#	Create the file type option menu
#
XmCreatePulldownMenu typeMenu $ca typeMenu

XtCreateManagedWidget textContents textContents pushButton $typeMenu \
				labelString:"Plain Text" \
				activateCallback:"Filetype=text" \

XtCreateManagedWidget psContents psContents pushButton $typeMenu \
				labelString:"PostScript" \
				activateCallback:"Filetype=ps" \

XtCreateManagedWidget pclContents pclContents pushButton $typeMenu \
				labelString:"HP LaserJet PCL" \
				activateCallback:"Filetype=pcl" \

eval typeHistory=\$${Filetype}Contents

XmCreateOptionMenu typeOptions $ca typeOptions \
				subMenuId:$typeMenu \
				menuHistory:$typeHistory \
				labelString:"File Contents" \

XtManageChildren $typeOptions

#
#	Create the copies field
#
XtCreateManagedWidget copiesForm copiesForm rowColumn $ca \
				orientation:horizontal \

XtCreateManagedWidget copiesLabel copiesLabel label $copiesForm \
				labelString:"Copies" \

XtCreateManagedWidget copies copies textField $copiesForm \
				value:"$Copies" \
				cursorPosition:"${#Copies}" \
				columns:2 \

XtCreateManagedWidget copiesDown copiesDown arrowButton $copiesForm \
				arrowDirection:arrow_down \
				activateCallback:"new_copies -1" \

XtCreateManagedWidget copiesUp copiesUp arrowButton $copiesForm \
				arrowDirection:arrow_up \
				activateCallback:"new_copies +1" \

#
#	Create the title field
#
XtCreateManagedWidget titleForm titleForm rowColumn $ca \
				orientation:horizontal \

XtCreateManagedWidget titleLabel titleLabel label $titleForm \
				labelString:"Title" \

XtCreateManagedWidget title title textField $titleForm \
				value:"$Title" \
				cursorPosition:"${#Title}" \
				columns:40 \

#
#	Create the other options field
#
XtCreateManagedWidget otherForm otherForm rowColumn $ca \
				orientation:horizontal \

XtCreateManagedWidget otherLabel otherLabel label $otherForm \
				labelString:"Other Options" \

XtCreateManagedWidget other other textField $otherForm \
				value:"$Other" \
				cursorPosition:"${#Other}" \
				columns:40 \

#
#	Create the action area buttons
#
XtCreateManagedWidget okButton okButton pushButton $aa \
				labelString:"OK" \
				activateCallback:ok_cb \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_form \
				rightAttachment:attach_position \
				rightPosition:19 \

XtCreateManagedWidget cancelButton cancelButton pushButton $aa \
				labelString:"Cancel" \
				activateCallback:cancel_cb \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_position \
				leftPosition:20 \
				rightAttachment:attach_position \
				rightPosition:39 \

XtCreateManagedWidget helpButton helpButton pushButton $aa \
				labelString:"Help" \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_position \
				leftPosition:40 \
				rightAttachment:attach_form \

#
#	The main loop
#
XtManageChildren $ca
XtManageChildren $aa
XtManageChildren $pane
XtRealizeWidget $topLevel
XtMainLoop
