/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/time.h>
#include <sys/errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "rplayd.h"
#include "connection.h"
#include "command.h"
#include "version.h"
#include "sound.h"
#include "server.h"
#include "host.h"
#include "cache.h"
#include "rplay.h"
#include "spool.h"
#include "buffer.h"

CONNECTION	*connections = NULL;
static int	nconnections = 0;

/*
 * create a connection object
 * types are CONNECTION_CLIENT and CONNECTION_SERVER
 */
#ifdef __STDC__
CONNECTION	*connection_create(int type)
#else
CONNECTION	*connection_create(type)
int		type;
#endif
{
	CONNECTION	*c;

	c = (CONNECTION *)malloc(sizeof(CONNECTION));
	if (c == NULL)
	{
		report(REPORT_ERROR, "connection_create: out of memory\n");
		done(1);
	}
	c->type = type;
	c->fd = -1;
	c->event = NULL;
	c->ep = &c->event;
	c->server = NULL;
	c->time = time(0);
			
	/*
	 * insert new connection into the connection list
	 */
	c->prev = NULL;
	c->next = connections;
	if (connections)
	{
		connections->prev = c;
	}
	connections = c;

	if (type == CONNECTION_CLIENT)
	{
		nconnections++;
	}
	
	return c;
}

/*
 * destroy the connection and remove it from the connection list
 */
#ifdef __STDC__
void	connection_destroy(CONNECTION *c)
#else
void	connection_destroy(c)
CONNECTION	*c;
#endif
{
	if (c->type == CONNECTION_CLIENT)
	{
		nconnections--;
	}
	if (c->prev)
	{
		c->prev->next = c->next;
	}
	else
	{
		connections = c->next;
	}
	if (c->next)
	{
		c->next->prev = c->prev;
	}
	free((char *)c);
}

/*
 * start a server connection to try and find the given sound
 *
 * The sound can be NULL which means that events have already
 * been created and will be forwarded to this server.
 */
#ifdef __STDC__
CONNECTION	*connection_server_open(SERVER *server, SOUND *sound)
#else
CONNECTION	*connection_server_open(server, sound)
SERVER	*server;
SOUND	*sound;
#endif
{
	int		n;
	CONNECTION	*c;
	EVENT		*e;
	
	/*
	 * see if there is already a connection to that server
	 */
	for (c = connections; c; c = c->next)
	{
		if (c->type == CONNECTION_SERVER && c->server == server)
		{
			break;
		}
	}
	if (c == NULL)
	{
		c = connection_create(CONNECTION_SERVER);
		c->server = server;
		c->sin = server->sin;
	}
	if (c->fd == -1)
	{
		c->fd = socket(AF_INET, SOCK_STREAM, 0);
		if (c->fd < 0)
		{
			report(REPORT_ERROR, "connection_open_server: socket: %s\n", sys_errlist[errno]);
			done(1);
		}
		fd_nonblock(c->fd);

		/*
		 * ping the server
		 */
		connection_server_ping(c);
		
		/* 
		 * try to connect to the server immediately
		 */
		report(REPORT_DEBUG, "%s server connection attempt\n", inet_ntoa(c->server->sin.sin_addr));
		n = connect(c->fd, (struct sockaddr *)&c->server->sin, sizeof(c->server->sin));
		e = event_create(EVENT_CONNECT);
		e->time = time(0);
		event_insert(c, e);
	}

	/*
	 * create events to try and find the sound
	 */
	if (sound)
	{
		BUFFER	*b;
		
		b = buffer_create();
		sprintf(b->buf, "find %s\r\n", sound->name);
		b->nbytes = strlen(b->buf);
		e = event_create(EVENT_WRITE_FIND, b, sound);
		event_insert(c, e);
	}

	return c;
}

#ifdef __STDC__
void	connection_server_reopen(CONNECTION *c)
#else
void	connection_server_reopen(c)
CONNECTION	*c;
#endif
{
	EVENT		*e;
	CONNECTION	*server;
	
	close(c->fd);
	c->fd = -1;
	e = c->event;
	c->event = NULL;
	c->ep = &c->event;
	
	server = connection_server_open(c->server, NULL);
	if (server != c)
	{
		report(REPORT_ERROR, "connection_server_reopen: server != c\n");
		done(1);
	}
	event_insert(server, e);
}

/*
 * send a ping packet to the server
 */
#ifdef __STDC__
void	connection_server_ping(CONNECTION *c)
#else
void	connection_server_ping(c)
CONNECTION	*c;
#endif
{
	struct sockaddr_in	ping_addr;
	
	memcpy((char *)&ping_addr, (char *)&c->server->sin, sizeof(ping_addr));
	ping_addr.sin_port = htons(RPLAY_PORT);
	rplay_ping_sockaddr_in(&ping_addr);
}

/*
 * open a client connection
 */
void	connection_client_open()
{
	struct sockaddr_in	f;
	int			fd, flen = sizeof(f);
	CONNECTION		*c;
	static char		message[RPTP_MAX_LINE];

	fd = accept(rptp_fd, (struct sockaddr *)&f, &flen);
	if (fd < 0)
	{
		report(REPORT_ERROR, "accept: %s\n", sys_errlist[errno]);
		return;
	}

	report(REPORT_DEBUG, "%s client connection request\n", inet_ntoa(f.sin_addr));

#ifdef AUTH
	/*
	 * see if this host has any access
	 */
	if (!host_access(f, HOST_READ) && !host_access(f, HOST_WRITE)
	    && !host_access(f, HOST_WRITE))
	{
		report(REPORT_NOTICE, "%s RPTP access denied\n", inet_ntoa(f.sin_addr));
		write(fd, "-permission denied\r\n", 20);
		close(fd);
		return;
	}
#endif /* AUTH */

	/*
	 * see if there are too many connections
	 */
	if (nconnections+1 == RPTP_MAX_CONNECTIONS)
	{
		write(fd, "-too many connections\r\n", 23);
		close(fd);
		return;
	}
	
	c = connection_create(CONNECTION_CLIENT);
	c->fd = fd;
	fd_nonblock(c->fd);
	c->sin = f;

	report(REPORT_NOTICE, "%s client connection established\n", inet_ntoa(c->sin.sin_addr));

	connection_reply(c, "%c%s RPTP server %s ready.", RPTP_OK, hostname, rplay_version);
}

/*
 * update connections by monitoring the read and write fd_sets
 *
 * If the given fd_sets are NULL then select must be called to update
 * the fd_sets.  (This should not be necessary anymore)
 */
#ifdef __STDC__
void	connection_update(fd_set *read_fds, fd_set *write_fds)
#else
void	connection_update(read_fds, write_fds)
fd_set	*read_fds;
fd_set	*write_fds;
#endif
{
	int		i, n, nfds;
	CONNECTION	*c, *next;
	struct timeval	tv;
	fd_set		rfds, wfds;
	long		t;
	char		buf[RPTP_MAX_LINE];

	if (read_fds == NULL && write_fds == NULL)
	{
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		rfds = read_mask;
		wfds = write_mask;
		FD_CLR(rplay_fd, &rfds);
#ifdef __hpux
		nfds = select(FD_SETSIZE, (int *)&rfds, (int *)&wfds, 0, &tv);
#else
		nfds = select(FD_SETSIZE, &rfds, &wfds, 0, &tv);
#endif
	}
	else
	{
		rfds = *read_fds;
		wfds = *write_fds;
		nfds = 1;
	}

	if (nfds == 0)
	{
		/*
		 * nothing to do...
		 */
		return;
	}
	else if (nfds < 0)
	{
		report(REPORT_ERROR, "connection_update: select: %s\n", sys_errlist[errno]);
		done(1);
	}

	/*
	 * check for new connections
	 */
	if (FD_ISSET(rptp_fd, &rfds))
	{
		connection_client_open();
	}

	t = time(0);
	
	for (c = connections; c; c = next)
	{
		next = c->next;	/* just in case c is destroyed */

		/*
		 * connection has no events
		 */
		if (c->event == NULL)
		{
			continue;
		}

		/*
		 * read events
		 */
		if (FD_ISSET(c->fd, &rfds))
		{
			c->time = t;

			switch (c->event->type)
			{
			case EVENT_READ_SOUND:
restart:
#ifdef MMAP
				n = read(c->fd, c->event->ptr, c->event->nleft);
#else /* MMAP */
				n = read(c->fd, c->event->buffer->buf,
					MIN(c->event->nleft, sizeof(c->event->buffer->buf)));
#endif /* MMAP */
				if (n < 0 && errno == EINTR)
				{
					goto restart;
				}
				report(REPORT_DEBUG, "%s read %d (%d), errno=%d\n",
				       inet_ntoa(c->sin.sin_addr),
				       n, c->event->nleft, errno);
				if (n <= 0)
				{
					event_update(c);
				}
				else
				{
					int	r;
#ifdef MMAP
					c->event->ptr += n;
					c->event->nleft -= n;
#else /* MMAP */
restart1:
					r = write(c->event->fd, c->event->buffer->buf, n);
					if (r < 0 && errno == EINTR)
					{
						goto restart1;
					}
					c->event->nleft -= n;
#endif /* MMAP */
					if (c->event->nleft == 0)
					{
						c->event->success++;
						event_update(c);
					}
				}
				break;
					
			default:
				/*
				 * read a line
				 */
restart2:
				n = recv(c->fd, c->event->ptr, c->event->nleft, MSG_PEEK);
				if (n < 0 && errno == EINTR)
				{
					goto restart2;
				}
				report(REPORT_DEBUG, "%s recv %d (%d), errno=%d\n", 
				       inet_ntoa(c->sin.sin_addr),
				       n, c->event->nleft, errno);
				if (n <= 0)
				{
					event_update(c);
				}
				else
				{
					/*
				 	 * search for a newline and replace \r and \n with \0
				 	 */
					for (i = 0; i < n; i++)
					{
						if (c->event->ptr[i] == '\r')
						{
							c->event->ptr[i] = '\0';
						}
						else if (c->event->ptr[i] == '\n')
						{
							c->event->ptr[i] = '\0';
							c->event->success++;
							break;
						}
					}
restart3:
					n = read(c->fd, buf, i == n ? n : i+1);
					if (n < 0 && errno == EINTR)
					{
						goto restart3;
					}
					if (n <= 0)
					{
						c->event->success = 0;
						event_update(c);
					}
					else
					{
						c->event->ptr += n;
						c->event->nleft -= n;
					}
					if (c->event->success || c->event->nleft == 0)
					{
						event_update(c);
					}
				}
				break;
			}
		}
		/*
		 * write events
		 */
		else if (FD_ISSET(c->fd, &wfds))
		{
			c->time = t;

			switch (c->event->type)
			{
			case EVENT_CONNECT:
				event_update(c);
				break;
				
			default:
again:
				n = write(c->fd, c->event->ptr, c->event->nleft);
				report(REPORT_DEBUG, "%s write %d (%d), errno=%d\n",
				       inet_ntoa(c->sin.sin_addr), n,
				       c->event->nleft, errno);
				if (n <= 0)
				{
					if (errno == EINTR)
					{
						goto again;
					}
					event_update(c);
				}
				else
				{
					c->event->ptr += n;
					c->event->nleft -= n;
					if (c->event->nleft == 0)
					{
						c->event->success++;
						event_update(c);
					}
				}
				break;
			}
		}
	}
}

/*
 * close and destroy the connection
 */
#ifdef __STDC__
void	connection_close(CONNECTION *c)
#else
void	connection_close(c)
CONNECTION	*c;
#endif
{
	EVENT	*e, *next;
	
	switch (c->type)
	{
	case CONNECTION_CLIENT:
		report(REPORT_NOTICE, "%s client connection closed\n", inet_ntoa(c->sin.sin_addr));
		break;
		
	case CONNECTION_SERVER:
		report(REPORT_NOTICE, "%s server connection closed\n", inet_ntoa(c->sin.sin_addr));
		break;
	}

	FD_CLR(c->fd, &read_mask);
	FD_CLR(c->fd, &write_mask);

	close(c->fd);

	/*
	 * destroy any events that are left
	 */
	for (e = c->event; e; e = next)
	{
		next = e->next;
		switch (e->type)
		{
		case EVENT_WRITE_FIND:
		case EVENT_READ_FIND_REPLY:
		case EVENT_WRITE_GET:
		case EVENT_READ_GET_REPLY:
		case EVENT_READ_SOUND:
			spool_remove(e->sound);
			sound_delete(e->sound);
			break;
		}
		event_destroy(e);
	}

	c->event = NULL;
	c->ep = &c->event;

	connection_destroy(c);
}

/*
 * update the fd_sets for the connect
 */
#ifdef __STDC__
void	connection_update_fdset(CONNECTION *c)
#else
void	connection_update_fdset(c)
CONNECTION	*c;
#endif
{
	FD_CLR(c->fd, &read_mask);
	FD_CLR(c->fd, &write_mask);

	if (c->event == NULL)
	{
		return;
	}
	
	switch (c->event->type)
	{
	case EVENT_READ_COMMAND:
	case EVENT_READ_SOUND:
	case EVENT_READ_CONNECT_REPLY:
	case EVENT_READ_FIND_REPLY:
	case EVENT_READ_GET_REPLY:
		FD_SET(c->fd, &read_mask);
		break;

	case EVENT_WRITE:
	case EVENT_WRITE_SOUND:
	case EVENT_WRITE_FIND:
	case EVENT_WRITE_GET:
	case EVENT_CONNECT:
	case EVENT_WRITE_TIMEOUT:
		FD_SET(c->fd, &write_mask);
		break;

	default:
		report(REPORT_ERROR, "connection_update_fdset: unknown event type '%d'\n", c->event->type);
		done(1);
	}

#ifdef DEBUG
	event_print(c);
#endif
}

/*
 * send the connection a reply message
 */
#ifdef __STDC__
void	connection_reply(CONNECTION *c, char *fmt, ...)
#else
void	connection_reply(va_alist)
va_dcl
#endif
{
	va_list	args;
	EVENT	*e;
	BUFFER	*b;

#ifdef __STDC__
	va_start(args, fmt);
#else
	CONNECTION	*c;
	char		*fmt;
	va_start(args);
	c = va_arg(args, CONNECTION *);
	fmt = va_arg(args, char *);
#endif

	b = buffer_create();

	vsprintf(b->buf, fmt, args);
	va_end(args);
	strcat(b->buf, "\r\n");

	b->nbytes += strlen(b->buf);
	
	e = event_create(EVENT_WRITE, b);
	event_insert(c, e);
}

/*
 * close all connections
 */
void	connection_cleanup()
{
	CONNECTION	*c;

	for (c = connections; c; c = c->next)
	{
		connection_close(c);
	}
}

/*
 * timeout all idle connections
 */
void	connection_check_timeout()
{
	long		t;
	CONNECTION	*c, *next;
	
	if (!rptp_timeout)
	{
		return;
	}

	t = time(0);
	for (c = connections; c; c = next)
	{
		next = c->next;
		if (t - c->time >= rptp_timeout)
		{
			connection_timeout(c);
		}
	}
}

/*
 * timeout an idle connection
 */
#ifdef __STDC__
void	connection_timeout(CONNECTION *c)
#else
void	connection_timeout(c)
CONNECTION	*c;
#endif
{
	BUFFER	*b;
	EVENT	*e;

	if (c->type == CONNECTION_SERVER)
	{		
		report(REPORT_DEBUG, "%s server connection timeout\n", inet_ntoa(c->sin.sin_addr));
		connection_close(c);
	}
	else
	{
		b = buffer_create();
		sprintf(b->buf, "%cConnection timed out after %d idle seconds.\r\n",
			RPTP_TIMEOUT, rptp_timeout);
		b->nbytes += strlen(b->buf);
		e = event_create(EVENT_WRITE_TIMEOUT, b);
		event_replace(c, e);
	}
}

/*
 * forward the sound request to the next server
 */
#ifdef __STDC__
void	connection_server_forward_sound(CONNECTION *c, SOUND *s)
#else
void	connection_server_forward_sound(c, s)
CONNECTION	*c;
SOUND		*s;
#endif
{
	CONNECTION	*server;
	
	if (c->server->next)
	{
		server = connection_server_open(c->server->next, s);
	}
	else
	{
		spool_remove(s);
		sound_delete(s);
	}
}

/*
 * forward all events to the next server
 */
#ifdef __STDC__
void	connection_server_forward(CONNECTION *c)
#else
void	connection_server_forward(c)
CONNECTION	*c;
#endif
{
	EVENT		*e, *next;
	CONNECTION	*server;
	
	if (c->server->next)
	{
		server = connection_server_open(c->server->next, NULL);
		event_insert(server, c->event);
	}
	else
	{
		for (e = c->event; e; e = next)
		{
			next = e->next;
			switch (e->type)
			{
			case EVENT_WRITE_FIND:
			case EVENT_READ_FIND_REPLY:
			case EVENT_WRITE_GET:
			case EVENT_READ_GET_REPLY:
			case EVENT_READ_SOUND:
				spool_remove(e->sound);
				sound_delete(e->sound);
				break;
			}
			event_destroy(e);
		}
	}
	c->event = NULL;
	c->ep = &c->event;
}

BUFFER	*connection_list_create()
{
	CONNECTION	*c;
	BUFFER		*b, *connection_list;
	char		buf[RPTP_MAX_LINE];
	char		idle_buf[16];
	char		what_buf[512];
	int		n;
	long		t, idle, days, hours, mins;
	
	b = buffer_create();
	connection_list = b;
	sprintf(b->buf, "+connections\r\nHOST            TYPE    IDLE WHAT\r\n");
	b->nbytes += strlen(b->buf);
	
	t = time(0);

	for (c = connections; c; c = c->next)
	{
		sprintf(buf, "%-15s ", inet_ntoa(c->sin.sin_addr));
		switch (c->type)
		{
		case CONNECTION_SERVER:
			strcat(buf, "server ");
			break;
			
		case CONNECTION_CLIENT:
			strcat(buf, "client ");
			break;
		}
		
		idle = t - c->time;
		days = idle / (60*60*24);
		idle %= (60*60*24); 
		hours = idle / (60*60);
		idle %= (60*60);
		mins = idle / 60;

		if (days > 0)
		{
			sprintf(idle_buf, "%dday%s", days, days > 1 ? "s" : "");
		}
		else if (hours > 0 && mins > 0)
		{
			sprintf(idle_buf, "%2d:%2d", hours, mins);
		}
		else 
		{
			if (hours > 0)
			{
				sprintf(idle_buf, "%dhr%s", hours,
					hours > 1 ? "s" : "");
			}
			else if (mins > 0)
			{
				sprintf(idle_buf, "%d", mins);
			}
			else
			{
				sprintf(idle_buf, "");
			}
		}

		if (c->event == NULL)
		{
			sprintf(what_buf, "(idle)");
		}
		else
		{
			switch (c->event->type)
			{
			case EVENT_READ_COMMAND:
				sprintf(what_buf, "(idle)");
				break;
				
			case EVENT_CONNECT:
			case EVENT_READ_CONNECT_REPLY:
				sprintf(what_buf, "(connect)");
				break;
				
			case EVENT_WRITE_FIND:
			case EVENT_READ_FIND_REPLY:
				sprintf(what_buf, "find %s", c->event->sound->name);
				break;
				
			case EVENT_WRITE_GET:
			case EVENT_READ_GET_REPLY:
				sprintf(what_buf, "get %s", c->event->sound->name);
				break;

			case EVENT_READ_SOUND:
				if (c->type == CONNECTION_SERVER)
					sprintf(what_buf, "get %s", c->event->sound->name);
				else
					sprintf(what_buf, "put %s", c->event->sound->name);
				break;
				
			case EVENT_WRITE:
				sprintf(what_buf, "(write)");
				break;
				
			case EVENT_WRITE_SOUND:
				sprintf(what_buf, "get %s", c->event->sound->name);
				break;
			}
		}

		sprintf(buf+23, "%5s %s\r\n", idle_buf, what_buf);

		n = strlen(buf);
		if (b->nbytes + n > BUFFER_SIZE)
		{
			b->next = buffer_create();
			b = b->next;
		}
		strcat(b->buf, buf);
		b->nbytes += n;
	}
	
	if (b->nbytes + 3 > BUFFER_SIZE)
	{
		b->next = buffer_create();
		b = b->next;
	}
	strcat(b->buf, ".\r\n");
	b->nbytes += 3;
	
	return connection_list;
}

#ifdef __STDC__
void	event_update(CONNECTION *c)
#else
void	event_update(c)
CONNECTION	*c;
#endif
{
	int		n, size;
#ifndef MMAP
	int		fd;
#endif
	EVENT		*e;
	CONNECTION	*server;
	long		t;
	char		*p;
	BUFFER		*b;
	
	switch (c->event->type)
	{
	case EVENT_READ_COMMAND:
		if (c->event->success)
		{
			if (c->event->start[0] == '\0')
			{
				event_delete(c);
			}
			else if (command(c, c->event->start) == 0)
			{
				event_delete(c);
			}
		}
		else
		{
			connection_close(c);
		}
		break;

	case EVENT_CONNECT:
		/*
		 * check the time to delay between connect attempts
		 */
		t = time(0);
		if (t - c->event->time < RPTP_PING_DELAY)
		{
			break;
		}
		c->event->time = t;
		
		/*
		 * check the socket to see if it is connected
		 */
		n = connect(c->fd, (struct sockaddr *)&c->server->sin, sizeof(c->server->sin));
		if (n == 0 || errno == EISCONN)
		{	
			/*
			 * got a connection
			 */
			report(REPORT_NOTICE, "%s server connection established\n", inet_ntoa(c->server->sin.sin_addr));
			/*
			 * prepare to read the connect reply, this must replace the
			 * connect event
			 */
			e = event_create(EVENT_READ_CONNECT_REPLY, buffer_create());
			event_replace(c, e);
		}
		else
		{
			/*
			 * socket is not connected
			 */
			c->event->nconnects++;
			report(REPORT_DEBUG, "%s server connection failed attempt #%d\n", 
			       inet_ntoa(c->server->sin.sin_addr), 
			       c->event->nconnects);
			if (c->event->nconnects < RPTP_CONNECT_ATTEMPTS)
			{
				switch (errno)
				{
				case ECONNREFUSED:
				case EINTR:
				case EINVAL:
				case EPIPE:
					/*
					 * ping the server again
					 */
					connection_server_ping(c);
					
					FD_CLR(c->fd, &read_mask);
					FD_CLR(c->fd, &write_mask);
					close(c->fd);
					c->fd = socket(AF_INET, SOCK_STREAM, 0);
					fd_nonblock(c->fd);
					
					/*
					 * try and connect again
					 */
					connect(c->fd, (struct sockaddr *)&c->server->sin, sizeof(c->server->sin));
					connection_update_fdset(c);
					return;

				default:
					break;
				}
			}
			
			/*
			 * connection failed, remove connect event and forward all other
			 * events to the next server
			 */
			report(REPORT_NOTICE, "%s server connection failed\n", inet_ntoa(c->server->sin.sin_addr));
			event_delete(c);
			connection_server_forward(c);
			connection_close(c);
		}
		break;

	case EVENT_READ_CONNECT_REPLY:
		if (c->event->success)
		{
			if (c->event->start[0] == RPTP_OK)
			{
				event_delete(c);
			}
			else
			{
				event_delete(c);
				connection_server_forward(c);
				connection_close(c);
			}
		}
		else
		{
			report(REPORT_NOTICE, "cannot read connect reply from %s\n",
				inet_ntoa(c->server->sin.sin_addr));
			event_delete(c);
			connection_server_forward(c);
			connection_close(c);
		}
		break;

	case EVENT_WRITE_FIND:
		if (c->event->success)
		{
			e = event_create(EVENT_READ_FIND_REPLY, buffer_create(), c->event->sound);
			event_replace(c, e);
		}
		else
		{
			connection_server_forward(c);
			connection_close(c);
		}
		break;

	case EVENT_READ_FIND_REPLY:
		if (c->event->success)
		{
			switch (c->event->start[0])
			{
			case RPTP_OK:
				p = strtok(c->event->start, " ");
				size = atoi(strtok(NULL, "\r\n"));
				/*
				 * see if the sound will fit in the cache
				 */
				if (cache_free(size) < 0)
				{
					/*
					 * the sound is too big for the cache
					 */
					spool_remove(c->event->sound);
					sound_delete(c->event->sound);
					event_delete(c);
					break;
				}
				b = buffer_create();
				sprintf(b->buf, "get %s\r\n", c->event->sound->name);
				b->nbytes = strlen(b->buf);
				e = event_create(EVENT_WRITE_GET, b, c->event->sound);
				event_replace(c, e);
				break;
				
			case RPTP_ERROR:
				report(REPORT_NOTICE, "%s server does not have %s\n",
					inet_ntoa(c->sin.sin_addr), c->event->sound->name);
				connection_server_forward_sound(c, c->event->sound);
				event_delete(c);
				break;
				
			case RPTP_TIMEOUT:
				report(REPORT_NOTICE, "%s server connection timed out\n",
				       inet_ntoa(c->sin.sin_addr));
				b = buffer_create();
				sprintf(b->buf, "find %s\r\n", c->event->sound->name);
				b->nbytes = strlen(b->buf);
				e = event_create(EVENT_WRITE_FIND, b, c->event->sound);
				event_replace(c, e);
				connection_server_reopen(c);
				break;

			default:
				report(REPORT_ERROR, "event_update: unknown find reply '%c'\n", c->event->start[0]);
				done(1);
			}
		}
		else
		{
			report(REPORT_NOTICE, "cannot read find reply from %s\n", inet_ntoa(c->sin.sin_addr));
			b = buffer_create();
			sprintf(b->buf, "find %s\r\n", c->event->sound->name);
			b->nbytes = strlen(b->buf);
			e = event_create(EVENT_WRITE_FIND, b, c->event->sound);
			event_replace(c, e);
			connection_server_forward(c);
			connection_close(c);
		}
		break;
			
	case EVENT_WRITE_GET:
		if (c->event->success)
		{
			e = event_create(EVENT_READ_GET_REPLY, buffer_create(), c->event->sound);
			event_replace(c, e);
		}
		else
		{
			b = buffer_create();
			sprintf(b->buf, "find %s\r\n", c->event->sound->name);
			b->nbytes = strlen(b->buf);
			e = event_create(EVENT_WRITE_FIND, b, c->event->sound);
			event_replace(c, e);
			connection_server_forward(c);
			connection_close(c);
		}
		break;
		
	case EVENT_READ_GET_REPLY:
		if (c->event->success)
		{
			switch (c->event->start[0])
			{
			case RPTP_OK:
				p = strtok(c->event->start, " ");
				p++;
				free((char *)c->event->sound->path);
				c->event->sound->path = strdup(cache_name(p));
				c->event->sound->name = c->event->sound->path[0] == '/' ? 
					strrchr(c->event->sound->path, '/') + 1 :
					c->event->sound->path;
				size = atoi(strtok(NULL, "\r\n"));
#ifdef MMAP
				p = cache_create(c->event->sound->path, size);
				if (p == NULL)
#else /* MMAP */
				fd = cache_create(c->event->sound->path, size);
				if (fd < 0)
#endif /* MMAP */
				{
					spool_remove(c->event->sound);
					sound_delete(c->event->sound);
					event_delete(c);
				}
				else
				{
#ifdef MMAP
					e = event_create(EVENT_READ_SOUND, p, size, c->event->sound);
#else /* MMAP */
					e = event_create(EVENT_READ_SOUND, fd, buffer_create(), size, c->event->sound);
#endif /* MMAP */
					c->event->sound->status = SOUND_NOT_READY;
					event_replace(c, e);
				}
				break;
				
			case RPTP_ERROR:
				report(REPORT_NOTICE, "%s server does not have %s\n",
					inet_ntoa(c->sin.sin_addr), c->event->sound->name);
				connection_server_forward_sound(c, c->event->sound);
				event_delete(c);
				break;
				
			case RPTP_TIMEOUT:
				report(REPORT_NOTICE, "%s server connection timed out\n",
				       inet_ntoa(c->sin.sin_addr));
				b = buffer_create();
				sprintf(b->buf, "get %s\r\n", c->event->sound->name);
				b->nbytes = strlen(b->buf);
				e = event_create(EVENT_WRITE_GET, b, c->event->sound);
				event_replace(c, e);
				connection_server_reopen(c);
				break;
				
			default:
				report(REPORT_ERROR, "event_update: unknown get reply '%c'\n", c->event->start[0]);
				done(1);
			}
		}
		else
		{
			report(REPORT_NOTICE, "cannot read get reply from %s\n",
			       inet_ntoa(c->server->sin.sin_addr));
			b = buffer_create();
			sprintf(b->buf, "find %s\r\n", c->event->sound->name);
			b->nbytes = strlen(b->buf);
			e = event_create(EVENT_WRITE_FIND, b, c->event->sound);
			event_replace(c, e);
			connection_server_forward(c);
			connection_close(c);
		}
		break;
		
	case EVENT_READ_SOUND:
		if (c->event->success)
		{
#ifdef MMAP
			sound_map(c->event->sound, c->event->start, c->event->nbytes);
#else /* MMAP */
			close(c->event->fd);
			sound_load(c->event->sound);
#endif /* MMAP */
			spool_ready(c->event->sound);
			c->event->sound->status = SOUND_READY;
			report(REPORT_DEBUG, "sound %s is ready\n", c->event->sound->name);
			event_delete(c);
		}
		else
		{
			report(REPORT_NOTICE, "cannot read sound %s from %s\n", c->event->sound->name,
			       inet_ntoa(c->sin.sin_addr));
			if (c->type == CONNECTION_SERVER)
			{
				b = buffer_create();
				sprintf(b->buf, "find %s\r\n", c->event->sound->name);
				b->nbytes = strlen(b->buf);
				e = event_create(EVENT_WRITE_FIND, b, c->event->sound);
				event_replace(c, e);
				connection_server_forward(c);
			}
			connection_close(c);
		}
		break;
		
	case EVENT_WRITE_SOUND:
		if (c->event->success)
		{
			event_delete(c);
		}
		else
		{
			connection_close(c);
		}
		break;

	case EVENT_WRITE:
		if (c->event->success)
		{
			b = c->event->buffer;
			c->event->buffer = c->event->buffer->next;
			if (b->status == BUFFER_FREE)
			{
				buffer_destroy(b);
			}
			if (c->event->buffer)
			{
				c->event->start = c->event->buffer->buf;
				c->event->ptr = c->event->start;
				c->event->nleft = c->event->buffer->nbytes;
				c->event->nbytes = c->event->nleft;
				c->event->success = 0;
			}
			else
			{
				event_delete(c);
			}
		}
		else
		{
			connection_close(c);
		}
		break;

	case EVENT_WRITE_TIMEOUT:
		report(REPORT_DEBUG, "%s client connection timeout\n", inet_ntoa(c->sin.sin_addr));
		connection_close(c);
		break;

	default:
		report(REPORT_ERROR, "event_update: unknown event type %d\n", c->event->type);
		done(1);
	}
}

#ifdef __STDC__
void	event_insert(CONNECTION *c, EVENT *e)
#else
void	event_insert(c, e)
CONNECTION	*c;
EVENT		*e;
#endif
{
	EVENT	*tail;

	for (tail = e; tail->next; tail = tail->next);
	
	*c->ep = e;
	c->ep = &tail->next;

	if (c->event == e)
	{
		connection_update_fdset(c);
	}
}

#ifdef __STDC__
void	event_replace(CONNECTION *c, EVENT *e)
#else
void	event_replace(c, e)
CONNECTION	*c;
EVENT		*e;
#endif
{
	if (c->event)
	{
		e->next = c->event->next;
		event_destroy(c->event);
	}
	c->event = e;
	if (c->event->next == NULL)
	{
		c->ep = &c->event->next;
	}
	connection_update_fdset(c);
}

#ifdef __STDC__
void	event_delete(CONNECTION *c)
#else
void	event_delete(c)
CONNECTION	*c;
#endif
{
	EVENT	*e;
	
	if (c->event == NULL)
	{
		connection_update_fdset(c);
		return;
	}

	e = c->event;
	c->event = c->event->next;

	event_destroy(e);
	
	if (c->event == NULL)
	{
		c->ep = &c->event;
		switch (c->type)
		{
		case CONNECTION_CLIENT:
			e = event_create(EVENT_READ_COMMAND, buffer_create());
			event_insert(c, e);
			break;

		case CONNECTION_SERVER:
			break;
		}
	}

	connection_update_fdset(c);
}

/*
 *	EVENT_READ_COMMAND, BUFFER *b
 *	EVENT_CONNECT
 *	EVENT_READ_CONNECT_REPLY, BUFFER *b
 *	EVENT_WRITE_FIND, BUFFER *b, SOUND *sound
 *	EVENT_READ_FIND_REPLY, BUFFER *b, SOUND *sound
 *	EVENT_WRITE_GET, BUFFER *b, SOUND *sound
 *	EVENT_READ_GET_REPLY, BUFFER *b, SOUND *sound
 *	EVENT_READ_SOUND, char *start, int nbytes, SOUND *sound (MMAP)
 *		or
 *	EVENT_READ_SOUND, int fd, BUFFER *b, int nbytes, SOUND *sound
 *	EVENT_WRITE, BUFFER *buffer
 *	EVENT_WRITE_SOUND, SOUND *sound
 */
#ifdef __STDC__
EVENT	*event_create(int type, ...)
#else
EVENT	*event_create(va_alist)
va_dcl
#endif
{
	EVENT	*e;
	va_list	args;

#ifdef __STDC__
	va_start(args, type);
#else
	int	type;
	va_start(args);
	type = va_arg(args, int);
#endif

	e = (EVENT *)malloc(sizeof(EVENT));
	if (e == NULL)
	{
		report(REPORT_ERROR, "event_create: out of memory\n");
		done(1);
	}
       
	/*
	 * defaults
	 */
	e->next = NULL;
	e->type = type;
	e->nleft = 0;
	e->nbytes = 0;
	e->start = NULL;
	e->ptr = NULL;
	e->sound = NULL;
	e->fd = -1;
	e->nconnects = 0;
	e->time = 0;
	e->success = 0;
	e->buffer = NULL;
	
	switch (e->type)
	{
	case EVENT_READ_COMMAND:
	case EVENT_READ_CONNECT_REPLY:
		e->buffer = va_arg(args, BUFFER *);
		e->nleft = RPTP_MAX_LINE;
		e->nbytes = e->nleft;
		e->ptr = e->buffer->buf;
		e->start = e->ptr;
		break;

	case EVENT_CONNECT:
		break;
		
	case EVENT_WRITE_FIND:
	case EVENT_WRITE_GET:
		e->buffer = va_arg(args, BUFFER *);
		e->nleft = e->buffer->nbytes;
		e->nbytes = e->nleft;
		e->ptr = e->buffer->buf;
		e->start = e->ptr;
		e->sound = va_arg(args, SOUND *);
		break;
		
	case EVENT_READ_FIND_REPLY:
	case EVENT_READ_GET_REPLY:
		e->buffer = va_arg(args, BUFFER *);
		e->nleft = RPTP_MAX_LINE;
		e->nbytes = e->nleft;
		e->ptr = e->buffer->buf;
		e->start = e->ptr;
		e->sound = va_arg(args, SOUND *);
		break;

	case EVENT_READ_SOUND:
#ifdef MMAP
		e->ptr = va_arg(args, char *);
		e->start = e->ptr;
		e->nleft = va_arg(args, int);
		e->nbytes = e->nleft;
		e->sound = va_arg(args, SOUND *);
#else /* MMAP */
		e->fd = va_arg(args, int);
		e->buffer = va_arg(args, BUFFER *);
		e->nbytes = va_arg(args, int);
		e->nleft = e->nbytes;
		e->sound = va_arg(args, SOUND *);
#endif /* MMAP */
		break;
	   
	case EVENT_WRITE_SOUND:
		e->sound = va_arg(args, SOUND *);
		e->nleft = e->sound->size;
		e->nbytes = e->nleft;
		e->ptr = e->sound->buf;
		e->start = e->ptr;
		break;

	case EVENT_WRITE:
	case EVENT_WRITE_TIMEOUT:
		e->buffer = va_arg(args, BUFFER *);
		e->ptr = e->buffer->buf;
		e->start = e->ptr;
		e->nleft = e->buffer->nbytes;
		e->nbytes = e->nleft;
		break;
	}

	va_end(args);
	
	return e;
}

#ifdef __STDC__
void	event_destroy(EVENT *e)
#else
void	event_destroy(e)
EVENT	*e;
#endif
{
	if (e->buffer && e->buffer->status == BUFFER_FREE)
	{
		buffer_destroy(e->buffer);
	}
	free((char *)e);
}

#ifdef DEBUG
event_print(CONNECTION *c)
{
	EVENT	*e;
	
	printf("%s events: ", inet_ntoa(c->sin.sin_addr));
	e = c->event;
	for (; e; e = e->next)
	{
		switch (e->type)
		{
		case EVENT_READ_COMMAND:
			printf("read_command ");
			break;
			
		case EVENT_CONNECT:
			printf("connect ");
			break;
			
		case EVENT_READ_CONNECT_REPLY:
			printf("read_connect_reply ");
			break;
			
		case EVENT_WRITE_FIND:
			printf("write_find ");
			break;

		case EVENT_READ_FIND_REPLY:
			printf("read_find_reply ");
			break;
			
		case EVENT_WRITE_GET:
			printf("write_get ");
			break;
			
		case EVENT_READ_GET_REPLY:
			printf("read_get_reply ");
			break;
			
		case EVENT_READ_SOUND:
			printf("read_sound ");
			break;
			
		case EVENT_WRITE:
			printf("write ");
			break;
			
		case EVENT_WRITE_SOUND:
			printf("write_sound ");
			break;
			
		case EVENT_WRITE_TIMEOUT:
			printf("write_timeout ");
			break;
			
		default:
			printf("%d ", e->type);
			break;
		}
	}
	printf("\n");
}
#endif
