/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include "rplayd.h"
#include "command.h"
#include "connection.h"
#include "server.h"
#include "host.h"
#include "spool.h"
#include "cache.h"
#include "sound.h"
#include "audio.h"

static COMMAND	commands[] =
{
/*	
 *	command		min	max	usage				function
 */
	{ "access",	-1,	-1,	"" ,				command_access, },
	{ "continue",	1,	-1,	"[-PNnv] sound ...",		command_play, },
	{ "find",		1,	1,	"sound",		command_find, },
	{ "get",		1,	1,	"sound",		command_get, },
	{ "help",		-1,	-1,	"",			command_help, },
#ifdef AUTH
	{ "list",		0,	1,	"[connections|hosts|servers|spool|sounds]",	command_list, },
#else
	{ "list",		0,	1,	"[connections|servers|sounds}",		command_list, },
#endif /* AUTH */
	{ "pause",	1,	-1,	"[-PNnv] sound ...",		command_play, },
	{ "play",		1,	-1,	"[-PNnv] sound ...",	command_play, },
	{ "put",		2,	2,	"sound size",		command_put, },
	{ "stop",		1,	-1,	"[-PNnv] sound ...",	command_play, },
	{ "quit",		0,	0,	"",			command_quit, },
	{ "volume",	0,	1,	"[volume]",			command_volume, },
};
#define NCOMMANDS	(sizeof(commands)/sizeof(COMMAND))

#ifdef __STDC__
int	command(CONNECTION *c, char *buf)
#else
int	command(c, buf)
CONNECTION	*c;
char		*buf;
#endif
{
	char	*argv[RPTP_MAX_ARGS], *p;
	int	argc = 0, first = 1, i;

	report(REPORT_INFO, "%s %s\n", inet_ntoa(c->sin.sin_addr), buf);

	while (p = strtok(first ? buf : NULL, " \t"))
	{
		argv[argc++] = p;
		first = 0;
	}
	argv[argc] = NULL;

	for (i = 0; i < NCOMMANDS; i++)
	{
		if (strcmp(commands[i].name, argv[0]) == 0)
		{
			if (commands[i].min_args >= 0 && argc-1 < commands[i].min_args
				|| commands[i].max_args >= 0 && argc-1 > commands[i].max_args)
			{
				connection_reply(c, "%cUsage: %s %s",
					RPTP_ERROR, commands[i].name, commands[i].usage);
				return 0;
			}
			else
			{
				return (*commands[i].func)(c, argc, argv);
			}
		}
	}

	return command_unknown(c, argc, argv);
}

#ifdef __STDC__
int	command_quit(CONNECTION *c, int argc, char **argv)
#else
int	command_quit(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	connection_close(c);
	return -1;
}

#ifdef __STDC__
int	command_unknown(CONNECTION *c, int argc, char **argv)
#else
int	command_unknown(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
        connection_reply(c, "%cunknown command '%s'", RPTP_ERROR, argv[0]);

	return 0;
}

#ifdef __STDC__
int	command_help(CONNECTION *c, int argc, char **argv)
#else
int	command_help(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	int		i;
	char		fmt[64];
	EVENT		*e;
	static BUFFER  	*b;
	
	
	if (b == NULL)
	{	
		b = buffer_create();
		b->status = BUFFER_KEEP;
		sprintf(b->buf, "%ccommand summary\r\n", RPTP_OK);
		for (i = 0; i < NCOMMANDS; i++)
		{
			sprintf(fmt, "%-8s %s\r\n", commands[i].name, commands[i].usage);
			strcat(b->buf, fmt);
		}
		strcat(b->buf, ".\r\n");
		b->nbytes = strlen(b->buf);
	}

	e = event_create(EVENT_WRITE, b);
	event_insert(c, e);
	
	return 0;
}

#ifdef __STDC__
int	command_get(CONNECTION *c, int argc, char **argv)
#else
int	command_get(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	SOUND	*s;
	EVENT	*e;
	
#ifdef AUTH
	if (!host_access(c->sin, HOST_READ))
	{
		report(REPORT_NOTICE, "%s get %s - read access denied\n", inet_ntoa(c->sin.sin_addr),
			argv[1]);
		connection_reply(c, "%cread access denied", RPTP_ERROR);
		return 0;
	}
#endif /* AUTH */
	
	s = sound_lookup(argv[1], SOUND_DONT_FIND, NULL);
	if (s == NULL || s->status != SOUND_READY)
	{
		report(REPORT_NOTICE, "%s get %s - not found\n", inet_ntoa(c->sin.sin_addr),
			argv[1]);
		connection_reply(c, "%c%s not found", RPTP_ERROR, argv[1]);
	}
	else
	{
		report(REPORT_NOTICE, "%s get %s %d\n", inet_ntoa(c->sin.sin_addr), s->name, s->size);
		connection_reply(c, "%c%s %d", RPTP_OK, s->name, s->size);
		e = event_create(EVENT_WRITE_SOUND, s);
		event_insert(c, e);
	}
	
	return 0;
}

#ifdef __STDC__
int	command_put(CONNECTION *c, int argc, char **argv)
#else
int	command_put(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	EVENT	*e;
	int	size;
	char	*name, *p, *sound_name;
	SOUND	*s;
#ifndef MMAP
	int	fd;
#endif
	
#ifdef AUTH
	if (!host_access(c->sin, HOST_WRITE))
	{
		report(REPORT_NOTICE, "%s put %s - write access denied\n", inet_ntoa(c->sin.sin_addr),
			argv[1]);
		connection_reply(c, "%cwrite access denied", RPTP_ERROR);
		return 0;
	}
#endif /* AUTH */
	
	/*
	 * strip pathnames, files can only be put in the cache directory
	 */
	sound_name = strrchr(argv[1], '/');
	if (sound_name == NULL)
	{
	        sound_name = argv[1];
	}
	else
	{
		sound_name++;
	}
	name = cache_name(sound_name);

	s = sound_lookup(name, SOUND_DONT_FIND, NULL);
	if (s != NULL)
	{
		connection_reply(c, "%c%s already is in the cache", RPTP_ERROR, sound_name);
		return 0;
	}

	size = atoi(argv[2]);

	if (cache_free(size) < 0)
	{
		connection_reply(c, "%cthe cache is full", RPTP_ERROR);
	}
	else
	{
#ifdef MMAP
		p = cache_create(name, size);
		if (p == NULL)
#else
		fd = cache_create(name, size);
		if (fd < 0)
#endif /* MMAP */
		{
			connection_reply(c, "%ccache error", RPTP_ERROR);
		}
		else
		{
			report(REPORT_NOTICE, "%s put %s %d\n", inet_ntoa(c->sin.sin_addr), sound_name, size);
			s = sound_insert(name, SOUND_NOT_READY);
			connection_reply(c, "%c%s %d", RPTP_OK, sound_name, size);
#ifdef MMAP
			e = event_create(EVENT_READ_SOUND, p, size, s);
#else
			e = event_create(EVENT_READ_SOUND, fd, buffer_create(), size, s);
#endif /* MMAP */
			event_insert(c, e);
		}
	}

	return 0;
}

#ifdef __STDC__
int	command_list(CONNECTION *c, int argc, char **argv)
#else
int	command_list(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	EVENT	*e;

	if (argv[1] == NULL || strcmp(argv[1], "sounds") == 0)
	{
		e = event_create(EVENT_WRITE, sound_list_create());
		event_insert(c, e);
	}
	else if (strcmp(argv[1], "connections") == 0)
	{
		e = event_create(EVENT_WRITE, connection_list_create());
		event_insert(c, e);
	}		
	else if (strcmp(argv[1], "servers") == 0)
	{
		e = event_create(EVENT_WRITE, server_list);
		event_insert(c, e);
	}
	else if (strcmp(argv[1], "spool") == 0)
	{
		e = event_create(EVENT_WRITE, spool_list_create());
		event_insert(c, e);
	}
#ifdef AUTH
	else if (strcmp(argv[1], "hosts") == 0)
	{
		e = event_create(EVENT_WRITE, host_list);
		event_insert(c, e);
	}
#endif /* AUTH */
	else
	{
		connection_reply(c, "%ccannot list %s", RPTP_ERROR, argv[1]);
	}

	return 0;
}

#ifdef __STDC__
int	command_find(CONNECTION *c, int argc, char **argv)
#else
int	command_find(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	SOUND	*s;

	s = sound_lookup(argv[1], SOUND_DONT_FIND, NULL);
	if (s == NULL || s->status != SOUND_READY)
	{
		report(REPORT_NOTICE, "%s find %s - not found\n", inet_ntoa(c->sin.sin_addr), argv[1]);
		connection_reply(c, "%c%s not found", RPTP_ERROR, argv[1]);
	}
	else
	{
		report(REPORT_NOTICE, "%s find %s %d\n", inet_ntoa(c->sin.sin_addr), argv[1], s->size);
		connection_reply(c, "%c%s %d", RPTP_OK, s->name, s->size);
	}

	return 0;
}

#ifdef __STDC__
int	command_access(CONNECTION *c, int argc, char **argv)
#else
int	command_access(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	char	buf[4];
	
	buf[0] = '\0';
#ifdef AUTH
	if (host_access(c->sin, HOST_READ))
	{
		strcat(buf, "r");
	}
	if (host_access(c->sin, HOST_WRITE))
	{
		strcat(buf, "w");
	}
	if (host_access(c->sin, HOST_EXECUTE))
	{
		strcat(buf, "x");
	}
#else /* AUTH */
	strcat(buf, "rwx");
#endif /* AUTH */
	
	connection_reply(c, "%c%s", RPTP_OK, buf);
	
	return 0;
}

#ifdef __STDC__
int	command_play(CONNECTION *c, int argc, char **argv)
#else
int	command_play(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	int		i, n, command, val;
	RPLAY		*rp;  
	int		volume, list_count, count, priority;
	extern char	*optarg;
	extern int	optind;

#ifdef AUTH
	if (!host_access(c->sin, HOST_EXECUTE))
	{
		report(REPORT_NOTICE, "%s %s permission denied\n", argv[0], inet_ntoa(c->sin.sin_addr));
		connection_reply(c, "%cexecute access denied", RPTP_ERROR);
		return 0;
	}
#endif /* AUTH */
	if (strcmp(argv[0], "play") == 0)
	{
		command = RPLAY_PLAY;
	}
	else if (strcmp(argv[0], "stop") == 0)
	{
		command = RPLAY_STOP;
	}
	else if (strcmp(argv[0], "pause") == 0)
	{
		command = RPLAY_PAUSE;
	}
	else if (strcmp(argv[0], "continue") == 0)
	{
		command = RPLAY_CONTINUE;
	}

	volume = RPLAY_DEFAULT_VOLUME;
	count = RPLAY_DEFAULT_COUNT;
	list_count = RPLAY_DEFAULT_LIST_COUNT;
	priority = RPLAY_DEFAULT_PRIORITY;

	optind = 1;
	while ((n = getopt(argc, argv, "P:N:n:v:")) != -1)
	{
		switch (n)
		{
		case 'v':
			volume = atoi(optarg);
			break;

		case 'n':
			count = atoi(optarg);
			break;

		case 'N':
			list_count = atoi(optarg);
			break;

		case 'P':
			priority = atoi(optarg);
			break;

		default:
			argc = optind;
			break;
		}
	}

	if (argc == optind)
	{
		for (i = 0; i < NCOMMANDS; i++)
		{
			if (strcmp(commands[i].name, argv[0]) == 0)
			{
				connection_reply(c, "%cUsage: %s %s",
					RPTP_ERROR, commands[i].name, commands[i].usage);
				return 0;
			}
		}
	}

	rp = rplay_create(command);
	if (rp == NULL)
	{
		connection_reply(c, "%c%s failed", RPTP_ERROR, argv[0]);
		return 0;
	}
	  
	if (rplay_set(rp, RPLAY_LIST_COUNT, list_count, NULL) < 0)
	{
		connection_reply(c, "%c%s failed", RPTP_ERROR, argv[0]);
		return 0;
	}

	if (rplay_set(rp, RPLAY_PRIORITY, priority, NULL) < 0)
	{
		connection_reply(c, "%c%s failed", RPTP_ERROR, argv[0]);
		return 0;
	}

	while (argv[optind] != NULL)
	{
		val = rplay_set(rp, RPLAY_APPEND,
			RPLAY_SOUND,	argv[optind++],
			RPLAY_VOLUME,	volume,
			RPLAY_COUNT,	count,
			NULL);
		if (val < 0)
		{
			connection_reply(c, "%c%s failed", RPTP_ERROR, argv[0]);
			return 0;
		}
	}

	switch (command)
	{
	case RPLAY_PLAY:
		val = rplayd_play(rp, c->sin);
		break;

	case RPLAY_STOP:
		val = rplayd_stop(rp, c->sin);
		break;

	case RPLAY_PAUSE:
		val = rplayd_pause(rp, c->sin);
		break;

	case RPLAY_CONTINUE:
		val = rplayd_continue(rp, c->sin);
		break;
	}

	if (val < 0)
	{
		connection_reply(c, "%c%s failed", RPTP_ERROR, argv[0]);
	}
	else
	{
		if (command == RPLAY_PLAY)
		{
			connection_reply(c, "%c%d", RPTP_OK, val);
		}
		else
		{
			connection_reply(c, "%c%s successful", RPTP_OK, argv[0]);
		}
	}

	return 0;
}

#ifdef __STDC__
int	command_volume(CONNECTION *c, int argc, char **argv)
#else
int	command_volume(c, argc, argv)
CONNECTION	*c;
int		argc;
char		**argv;
#endif
{
	int	volume;
	int	n;
	
#ifdef AUTH
	if (!host_access(c->sin, HOST_EXECUTE))
	{
		report(REPORT_NOTICE, "%s volume permission denied\n", inet_ntoa(c->sin.sin_addr));
		connection_reply(c, "%cexecute access denied", RPTP_ERROR);
		return 0;
	}
#endif /* AUTH */

	if (argv[1] == NULL)
	{
		volume = audio_get_volume();
		if (volume >= 0)
		{
			connection_reply(c, "%c%d", RPTP_OK, volume);
		}
		else
		{
			connection_reply(c, "%cvolume failed", RPTP_ERROR);
		}
	}
	else
	{
		volume = atoi(argv[1]);
		n = audio_set_volume(volume);
		if (n >= 0)
		{
			connection_reply(c, "%c%d", RPTP_OK, n);
		}
		else
		{
			connection_reply(c, "%cvolume failed", RPTP_ERROR);
		}
	}

	return 0;
}
