/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/errno.h>
#ifdef ultrix
#include <fcntl.h>
#else
#include <sys/fcntl.h>
#endif
#include <stdio.h>
#include <unistd.h>
#include "audio.h"
#include "rplayd.h"
#include "spool.h"
#include "sound.h"
#include "ulaw.h"
#include "timer.h"
#ifdef __386BSD__
#include <machine/sblast.h>
#endif
#ifdef sgi
#include <audio.h>
#endif
#ifdef sun
#include <sys/ioctl.h>
#include <sun/audioio.h>
#endif

#ifdef sgi
static ALport	audio_fd = (ALport)-1;
static ALconfig	sgitheconfig;
static short	*audio_buf;
#else
static int	audio_fd = -1;
static char	*audio_buf;
#endif
static int	audio_size = 0;
int		audio_timeout = AUDIO_TIMEOUT;
int		audio_bufsize = AUDIO_BUFSIZE;
char		*audio_device = AUDIO_DEVICE;
int		audio_open_count = 0;

/*
 * initialize the audio device
 */
void	audio_init()
{
#ifdef sgi
	long	sgisampfmt	= AL_SAMPLE_16;
	long	sginchannels	= AL_MONO;
	long	sgipvbuf[4];
	long	sgipvbuflen;

	sgipvbuf[0] = AL_INPUT_RATE;
	sgipvbuf[1] = AL_RATE_8000;
	sgipvbuf[2] = AL_OUTPUT_RATE;
	sgipvbuf[3] = AL_RATE_8000;
	sgipvbuflen = 4;

	ALsetparams(AL_DEFAULT_DEVICE, sgipvbuf, sgipvbuflen);

	sgitheconfig = ALnewconfig();
	ALsetwidth(sgitheconfig, sgisampfmt);
	ALsetchannels(sgitheconfig, sginchannels);

	audio_buf = (short *)malloc(AUDIO_SAMPLE_RATE * sizeof(short));
#else
	audio_buf = (char *)malloc(AUDIO_SAMPLE_RATE * sizeof(char));
#endif
}

/*
 * open the audio device
 */
void	audio_open()
{
	report(REPORT_DEBUG, "opening %s\n", audio_device);
#ifdef sgi
	audio_fd = ALopenport("rplay", "w", sgitheconfig);
	if ((int)audio_fd < 0)
#else
	audio_fd = open(audio_device, O_WRONLY|O_NDELAY, 0);
	if (audio_fd < 0)
#endif
	{
		report(REPORT_ERROR, "audio_open: open %s: %s\n", audio_device, sys_errlist[errno]);
	}
#ifndef sgi
	if (audio_fd > 0)
	{
#if 0
		/*
		 * Make the audio device writes non-blocking.
		 */
		int	flags;

		flags = fcntl(audio_fd, F_GETFL, 0);
		flags |= O_NONBLOCK;
		fcntl(audio_fd, F_SETFL, flags);
#else
		/*
		 * Make sure the audio device writes are blocking.
		 */
		fd_nonblock(audio_fd);
#endif
	}
#endif /* !sgi */
	audio_open_count = 0;
}

/*
 * Is the audio device open?
 */
int	audio_isopen()
{
#ifdef sgi
	return audio_fd != (ALport)-1;
#else
	return audio_fd != -1;
#endif
}

/*
 * close the audio device
 */
void	audio_close()
{
#ifdef sgi
	if ((int)audio_fd != -1)
	{
		ALcloseport(audio_fd);
	}
	audio_fd = (ALport)-1;
#else
	if (audio_fd > 0)
	{
		report(REPORT_DEBUG, "closing %s\n", audio_device);
		close(audio_fd);
	}
	audio_fd = -1;
#endif
}

/*
 * write a buffer from the spool to the audio device
 */
#ifdef __STDC__
void	audio_write(int amount)
#else
void	audio_write(amount)
int	amount;
#endif
{
	int	i, n;
	int	total, val, vol;
	SPOOL	*sp;
	char	*p;

	while (audio_size != amount && spool_size)
	{
		total = 0;
		for (i = 0; i < SPOOL_SIZE && audio_size != amount; i++)
		{
			sp = &spool[i];
			if (sp->state == SPOOL_PLAY)
			{
				vol = sp->curr_attrs->volume;
				if (spool_prio && spool_size > 1)
				{
					vol -= (spool_prio - sp->rp->priority) >> 1;
					vol = MAX(vol, RPLAY_MIN_VOLUME);
				}
				val = ulaw_to_linear((unsigned char)*sp->ptr)*vol;
				total += val >> 7;
				if (sp->ptr == sp->end)
				{
					if (sp->curr_count > 1)
					{
						sp->curr_count--;
					}
					else if (sp->curr_count)
					{
						sp->curr_sound++;
						if (sp->curr_sound == sp->rp->nsounds)
						{
							if (sp->list_count > 1)
							{
								sp->list_count--;
								sp->curr_attrs = sp->rp->attrs;
								sp->curr_sound = 0;
							}
							else if (sp->list_count == 0)
							{
								sp->curr_attrs = sp->rp->attrs;
								sp->curr_sound = 0;
							}
							else 
							{
								spool_clear(sp);
								spool_size--;
								continue;
							}
						}
						else
						{
							sp->curr_attrs = sp->curr_attrs->next;
						}
						sp->curr_count = sp->curr_attrs->count;
					}
					sp->ptr = sp->sound[sp->curr_sound]->start;
					sp->end = sp->sound[sp->curr_sound]->stop;
				}
				else
				{
					sp->ptr++;
				}
			}
		}
#ifdef __386BSD__
		/*
		 * convert signed short to unsigned 8 bit value.
		 */
		audio_buf[audio_size++] = (total ^ 0x8000) >> 8;
#else
#ifdef sgi
		audio_buf[audio_size++] = total;
#else
		audio_buf[audio_size++] = linear_to_ulaw(total);
#endif
#endif
	}
#ifdef sgi
	if ((int)audio_fd < 0 && audio_size)
#else
	if (audio_fd < 0 && audio_size)
#endif
	{
		audio_open();
	}
#ifdef sgi
	if ((int)audio_fd > 0)
#else
	if (audio_fd > 0)
#endif
	{
#ifdef sgi
		ALwritesamps(audio_fd, audio_buf, audio_size);
#else /* !sgi */
		for (p = audio_buf; audio_size; audio_size -= n, p += n)
		{
			n = write(audio_fd, p, audio_size);
			if (n < 0 && errno != EINTR)
			{
				report(REPORT_ERROR, "audio_write: write: n=%d errno=%d: %s\n", n, errno, sys_errlist[errno]);
				break;
			}
		}
#ifdef __386BSD__
		if (spool_size == 0)
		{
			ioctl(audio_fd, DSP_IOCTL_FLUSH, 0);
		}
#endif /* !__386BSD__ */
#endif /* !sgi */
	}
	audio_size = 0;
}

/*
 * Return the volume of the audio device.
 *
 * Currently this only works on Suns.
 */
int	audio_get_volume()
{
#ifdef sun
	audio_info_t    a;
		
	if (audio_fd < 0)
	{
		audio_open();
	}
	if (audio_fd < 0)
	{
		return -1;
	}
	
	if (ioctl(audio_fd, AUDIO_GETINFO, &a) < 0)
	{
		report(REPORT_DEBUG, "audio_get_volume: ioctl: %s\n", sys_errlist[errno]);
		return -1;
	}
	else
	{
		return a.play.gain;
	}
#else /* !sun */
	return -1;
#endif /* !sun */
}

/*
 * Set the volume of the audio device.
 *
 * Currently this only works on Suns.
 */
#ifdef __STDC__
int	audio_set_volume(int volume)
#else
int	audio_set_volume(volume)
int	volume;
#endif
{
#ifdef sun
	audio_info_t	a;
	
	if (audio_fd < 0)
	{
		audio_open();
	}
	if (audio_fd < 0)
	{
		return -1;
	}
	
	if (ioctl(audio_fd, AUDIO_GETINFO, &a) < 0)
	{
		report(REPORT_DEBUG, "audio_set_volume: ioctl: %s\n", sys_errlist[errno]);
		return -1;
	}
	a.play.gain = volume;
	if (ioctl(audio_fd, AUDIO_SETINFO, &a) < 0)
	{
		report(REPORT_DEBUG, "audio_set_volume: ioctl: %s\n", sys_errlist[errno]);
		return -1;
	}
	
	return audio_get_volume();
#else /* !sun */
	return -1;
#endif /* !sun */
}

