/**********************************************************************/
/*                                                                    */
/*  File:          xvolume.c                                          */
/*  Author:        Andrew W. Scherpbier                               */
/*                 with RPTP code added by Mark Boyns                 */
/*  Version:       1.00                                               */
/*  Created:       24 Oct 1993                                        */
/*                                                                    */
/*  Copyright (c) 1991, 1992 Andrew Scherpbier                        */
/*                All Rights Reserved.                                */
/*                                                                    */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*  Description:  XVolume with slider using XLib only                 */
/*                                                                    */
/**********************************************************************/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <stdio.h>
#include <rplay.h>
#include "Button.h"
#include "Slider.h"
#include "Message.h"
#include "icon.h"
#include "icon_mask.h"
#include "al.h"

#define	PUBLIC
#define	PRIVATE		static

#ifndef TRUE
#define	TRUE		(1)
#endif
#ifndef FALSE
#define	FALSE		(0)
#endif

#define	OK			(0)
#define	NOTOK		(-1)

#define	when		break;case
#define	orwhen		case
#define	otherwise	break;default


/* Private routines
 * ================
 */
PRIVATE void slider_proc(Slider *slider, int value);
PRIVATE void quit_proc(Button *button);
PRIVATE void button_proc(Button *button);
PRIVATE void resize();
PRIVATE void redraw();
PRIVATE void main_loop();
PRIVATE int do_volume(int volume);
PRIVATE void min_proc(Button *button);
PRIVATE void mid_proc(Button *button);
PRIVATE void max_proc(Button *button);


/* Private variables
 * =================
 */


/* Public routines
 * ===============
 */


/* Public variables
 * ================
 */
Display		*display;
int			screen_num;
Screen		*screen_ptr;
Window		win;
char		*icon_name = "XVolume";
Button		*button;
Button		*quit_button;
Button		*min, *mid, *max;
Slider		*slider;
Message		*message;
Cursor		cursor_busy;

char		*rptp_host = NULL;
int			rptp_port = RPTP_PORT;
char		window_title[128], *window_name = window_title;
char		*xvolume_version = "XVolume 1.0";

/**************************************************************************
 * PUBLIC main(int ac, char **av)
 *
 */
PUBLIC main(int ac, char **av)
{
	int	i;
	char	*display_name = NULL;
	Pixmap  icon_pixmap;
	Pixmap	iconmask_pixmap;

	for (i = 1; i < ac; i++)
	{
		if (strcmp(av[i], "-display") == 0)
		{
			i++;
			display_name = av[i];
		}
		else if (strcmp(av[i], "-h") == 0)
		{
			i++;
			rptp_host = av[i];
		}
		else if (strcmp(av[i], "-p") == 0)
		{
			i++;
			rptp_port = atoi(av[i]);
		}
		else if (strcmp(av[i], "-v") == 0)
		{
			printf("%s\n", xvolume_version);
			exit(0);
		}
	}

	if (rptp_host == NULL)
	{
		rptp_host = rplay_default_host();
	}

	sprintf(window_title, "XVolume: %s", rptp_host);

	/*
	 * Build connection with X server
	 */
	if ((display = XOpenDisplay(display_name)) == NULL)
	{
		fprintf(stderr, "%s: cannot connect to X server %s\n",
			av[0], XDisplayName(display_name));
		exit(1);
	}
	screen_num = DefaultScreen(display);
	screen_ptr = DefaultScreenOfDisplay(display);

	al_init(display);

	/*
	 * Create the window
	 */
	win = CreateWindow(360, 100);

	cursor_busy = XCreateFontCursor(display, XC_watch);

	icon_pixmap = XCreateBitmapFromData(display, win,
		speaker_bits, speaker_width, speaker_height);

	iconmask_pixmap = XCreateBitmapFromData(display, win,
		speaker_mask_bits, speaker_mask_width, speaker_mask_height);

	/*
	 * Set WN Hints
	 */
	{
		XWMHints		wm_hints;
		XClassHint		class_hints;
		XTextProperty	windowName;
		XTextProperty	iconName;
		XSizeHints		sizehints;

		if (XStringListToTextProperty(&window_name, 1, &windowName) == 0)
		{
			fprintf(stderr, "%s: structure allocation for windowName failed.\n", av[0]);
			exit(1);
		}
		if (XStringListToTextProperty(&icon_name, 1, &iconName) == 0)
		{
			fprintf(stderr, "%s: structure allocation for iconName failed.\n", av[0]);
			exit(1);
		}

		wm_hints.initial_state = NormalState;
		wm_hints.input = True;
		wm_hints.icon_pixmap = icon_pixmap;
		wm_hints.icon_mask = iconmask_pixmap;
		wm_hints.flags = StateHint | IconPixmapHint | IconMaskHint | InputHint;
		class_hints.res_name = av[0];
		class_hints.res_class = "Slider";
		sizehints.flags = PSize | PMinSize | PMaxSize | PResizeInc;
		sizehints.width = sizehints.min_width = sizehints.max_width = 360;
		sizehints.height = sizehints.min_height = sizehints.max_height = 100;
		sizehints.width_inc = sizehints.height_inc = 0;

		XSetWMProperties(display, win, &windowName, &iconName,
			av, ac, &sizehints, &wm_hints, &class_hints);
	}

	/*
	 * Select the input events we will be wanting to deal with
	 */
	XSelectInput(display, win, 
		ExposureMask |
		KeyPressMask |
		ButtonPressMask |
		ButtonReleaseMask |
		ButtonMotionMask |
		StructureNotifyMask);

	/*
	 * Display the window
	 */
	XMapWindow(display, win);
	XFlush(display);

	slider = CreateSlider(win);
	SetSliderLocation(slider, 18, 45);
	SetSliderLabel(slider, "Volume:");
	SetSliderCallback(slider, slider_proc);
	SetSliderRange(slider, 0, 255);
	SetSliderWidth(slider, 250);
	SetSliderHeight(slider, 30);

	quit_button = CreateButton(win);
	SetButtonLocation(quit_button, 10, 10);
	SetButtonLabel(quit_button, "Quit");
	SetButtonCallback(quit_button, quit_proc);

	min = CreateButton(win);
	SetButtonLocation(min, 89, 10);
	SetButtonLabel(min, "0%");
	SetButtonCallback(min, min_proc);

	mid = CreateButton(win);
	SetButtonLocation(mid, 189, 10);
	SetButtonLabel(mid, "50%");
	SetButtonCallback(mid, mid_proc);

	max = CreateButton(win);
	SetButtonLocation(max, 289, 10);
	SetButtonLabel(max, "100%");
	SetButtonCallback(max, max_proc);

	message = CreateMessage(win);
	SetMessageLocation(message, 10, 80);
	SetMessageLabel(message, "foobar");

	do_volume(-1);

	/*
	 * Now do the main loop
	 */
	main_loop();
}


/**************************************************************************
 * PRIVATE void slider_proc(Slider *slider, int value)
 *
 */
PRIVATE void slider_proc(Slider *slider, int value)
{
	int	n = do_volume(value);

	if (n < 0)
	{
		SetSliderValue(slider, 0);
	}
	SliderRedraw(slider);
}


/**************************************************************************
 * PRIVATE void quit_proc(Button *button)
 *
 */
PRIVATE void quit_proc(Button *button)
{
	XCloseDisplay(state.display);
	exit(0);
}


PRIVATE void	min_proc(Button *button)
{
	do_volume(RPLAY_MIN_VOLUME);
	SliderRedraw(slider);
}


PRIVATE void	mid_proc(Button *button)
{
	do_volume(RPLAY_DEFAULT_VOLUME);
	SliderRedraw(slider);
}

PRIVATE void	max_proc(Button *button)
{
	do_volume(RPLAY_MAX_VOLUME);
	SliderRedraw(slider);
}

/**************************************************************************
 * PRIVATE void main_loop()
 *
 */
PRIVATE void main_loop()
{
	XEvent	report;

	while (TRUE)
	{
		XNextEvent(display, &report);
		if (ButtonEvent(min, &report))
			continue;
		if (ButtonEvent(mid, &report))
			continue;
		if (ButtonEvent(max, &report))
			continue;
		if (ButtonEvent(quit_button, &report))
			continue;
		if (SliderEvent(slider, &report))
			continue;
		switch (report.type)
		{
			case Expose:
				if (report.xexpose.count != 0)
					break;
				redraw();
			when ConfigureNotify:
				resize();
			when ButtonPress:
			when MotionNotify:
			when KeyPress:
			otherwise:
				break;
		}
	}
}


/**************************************************************************
 * PRIVATE void redraw()
 *
 */
PRIVATE void redraw()
{
	ButtonRedraw(min);
	ButtonRedraw(mid);
	ButtonRedraw(max);
	ButtonRedraw(quit_button);
	SliderRedraw(slider);
	MessageRedraw(message);
}


/**************************************************************************
 * PRIVATE void resize()
 *
 */
PRIVATE void resize()
{
}

int	do_volume(int volume)
{
	static int	rptp_fd;
	static char	response[RPTP_MAX_LINE];
	static char	command[RPTP_MAX_LINE];
	int		n;

	busy_cursor(display, win);
	if (rptp_fd <= 0)
	{
		do_message("Connecting...");
		rptp_fd = rptp_open(rptp_host, rptp_port, response, sizeof(response));
		if (rptp_fd < 0)
		{
			SetSliderValue(slider, 0);
			do_message(rptp_errlist[rptp_errno]);
			normal_cursor(display, win);
			return -1;
		}
		do_message("");
	}

	if (volume >= 0)
	{
		sprintf(command, "volume %d", volume);
	}
	else
	{
		sprintf(command, "volume");
	}

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		do_message(rptp_errlist[rptp_errno]);
		if (rptp_errno == RPTP_ERROR_TIMEOUT)
		{
			rptp_fd = -1;
			rptp_close(rptp_fd);
			return do_volume(volume);
		}
		rptp_close(rptp_fd);
		rptp_fd = -1;
		SetSliderValue(slider, 0);
		normal_cursor(display, win);
		return -1;

	case 1:
		if (rptp_fd > 0)
		{
			rptp_close(rptp_fd);
		}
		rptp_fd = -1;
		SetSliderValue(slider, 0);
		do_message(response);
		normal_cursor(display, win);
		return -1;

	default:
		n = atoi(response+1);
		SetSliderValue(slider, n);
		do_message("");
		normal_cursor(display, win);
		return n;
	}
}



busy_cursor(Display *display, Window win)
{
	XDefineCursor(display, win, cursor_busy);
	XFlush(display);
}

normal_cursor(Display *display, Window win)
{
	XUndefineCursor(display, win);
	XFlush(display);
}

do_message(char *string)
{
	int		direction, ascent, descent;
	XCharStruct	overall;
	int		width;
	
	XTextExtents(state.normal_font, message->label, strlen(message->label),
		     &direction, &ascent, &descent, &overall);
	
	width = overall.width + 16;
						   
	XClearArea(display, win, message->x, message->y, width, message->string_y, False);

	SetMessageLabel(message, string);
	MessageRedraw(message);
}
