#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <stdio.h>
#include <string.h>
#include "Slider.h"
#include "al.h"


Slider *CreateSlider(Window win)
{
	Slider	*slider = (Slider *) malloc(sizeof(Slider));
	slider->win = win;
	slider->x = slider->y = 0;
	slider->width = slider->height = 1;
	slider->label = NULL;
	slider->min_value = 0;
	slider->max_value = 100;
	slider->slider_width = 100;
	slider->slider_height = 10;
	slider->value = 0;

	return slider;
}


void SetSliderLocation(Slider *slider, int x, int y)
{
	slider->x = x;
	slider->y = y;
}


void SetSliderLabel(Slider *slider, char *label)
{
	if (slider->label)
		free(slider->label);
	slider->label = strdup(label);
}


void SetSliderCallback(Slider *slider, void (*callback)(Slider *, int))
{
	slider->callback = callback;
}


void SetSliderRange(Slider *slider, int r_min, int r_max)
{
	slider->min_value = r_min;
	slider->max_value = r_max;
}


void SetSliderValue(Slider *slider, int value)
{
	if (value < slider->min_value || value > slider->max_value)
		return;
	slider->value = value;
}


void SetSliderWidth(Slider *slider, int width)
{
	slider->slider_width = width;
}


void SetSliderHeight(Slider *slider, int height)
{
	slider->slider_height = height;
}


void SliderRedraw(Slider *slider)
{
	int			direction, ascent, descent;
	XCharStruct	overall;

	/*
	 * Compute the slider dimensions from the label string
	 */
	XTextExtents(state.bold_font, slider->label, strlen(slider->label),
		&direction, &ascent, &descent, &overall);

	slider->width = overall.width + 20 + slider->slider_width;
	slider->height = state.normal_font->ascent + state.normal_font->descent + 5;
	slider->string_x = slider->x + 8;
	slider->string_y = slider->y + state.normal_font->ascent +  3;

	XSetFont(state.display, state.gc[GC_NORMAL], state.bold_font->fid);
	XDrawString(state.display, slider->win, state.gc[GC_NORMAL],
		slider->string_x, slider->string_y, slider->label, strlen(slider->label));
	XSetFont(state.display, state.gc[GC_NORMAL], state.normal_font->fid);

	/*
	 * Draw the outline
	 */
	slider->slider_x = slider->x + overall.width + 16;
	slider->slider_y = slider->y + slider->height / 2 - slider->slider_height / 2;
	al_drawbox(slider->win,
		slider->slider_x, slider->slider_y,
		slider->slider_width, slider->slider_height, AL_3D_IN, THREE_D_WIDTH);

	/*
	 * Draw the value
	 */
	SliderUpdate(slider, slider->value, 0);
}


SliderUpdate(Slider *slider, int value, int busy)
{
	int	value_x;
	char	string[10];
	int	x;

	value_x = value * (slider->slider_width - 6 )/ (slider->max_value - slider->min_value);
	XFillRectangle(state.display, slider->win, state.gc[GC_SELECTED],
		slider->slider_x, slider->slider_y,
		slider->slider_width, slider->slider_height);

	/*
	 * Compute the value to be displayed
	 */
	x = slider->slider_width / 2 + slider->slider_x - 10;
	sprintf(string, "%d", value);

	/*
	 * Draw the actual bar
	 */
	al_drawbox(slider->win,
		slider->slider_x + 4, slider->slider_y + 4,
		value_x - THREE_D_WIDTH, slider->slider_height - 8, AL_3D_OUT, THREE_D_WIDTH);
	if (value_x - THREE_D_WIDTH > 0)
		XClearArea(state.display, slider->win,
			slider->slider_x + 4, slider->slider_y + 4,
			value_x - THREE_D_WIDTH, slider->slider_height - 8, False);
	XDrawString(state.display, slider->win, state.gc[GC_REVERSE],
		x - 1, slider->string_y - 1,
		string, strlen(string));
	XDrawString(state.display, slider->win, state.gc[GC_REVERSE],
		x - 1, slider->string_y,
		string, strlen(string));
	XDrawString(state.display, slider->win, state.gc[GC_REVERSE],
		x - 2, slider->string_y - 2,
		string, strlen(string));
	XDrawString(state.display, slider->win, state.gc[GC_NORMAL],
		x, slider->string_y,
		string, strlen(string));
	XSetFunction(state.display, state.gc[GC_NORMAL], GXcopy);

	if (busy)
	{
		XFillRectangle(state.display, slider->win, state.gc[GC_BUSY],
			slider->slider_x + 4, slider->slider_y + 4,
			value_x - THREE_D_WIDTH, slider->slider_height - 8);
	}
}


void SliderEventLoop(Slider *slider, XEvent *orig)
{
	XEvent	event;
	int	temp_value;
	int	range = slider->max_value - slider->min_value;

	temp_value = (orig->xbutton.x - slider->slider_x - 2) * range / (slider->slider_width - THREE_D_WIDTH);
	SliderUpdate(slider, temp_value, 0);
	while (1)
	{
		XNextEvent(state.display, &event);
		switch (event.type)
		{
			case ButtonRelease:
				slider->value = temp_value;
				if (slider->callback)
				{
					SliderUpdate(slider, temp_value, 1);
					(*slider->callback)(slider, temp_value);
					SliderUpdate(slider, temp_value, 0);
				}
				return;
			case MotionNotify:
				temp_value = (event.xmotion.x - slider->slider_x - 2) * range / (slider->slider_width - THREE_D_WIDTH);
				if (temp_value < slider->min_value)
					temp_value = slider->min_value;
				if (temp_value > slider->max_value)
					temp_value = slider->max_value;
				SliderUpdate(slider, temp_value, 0);
				break;
			default:
				break;
		}
	}
}

int SliderEvent(Slider *slider, XEvent *event)
{
	switch (event->type)
	{
		case ButtonPress:
			if (in_rectangle(event->xbutton.x, event->xbutton.y,
				slider->slider_x, slider->slider_y,
				slider->slider_width, slider->slider_height))
			{
				SliderEventLoop(slider, event);
				return 1;
			}
			else
				return 0;
		default:
			return 0;
	}
}


