/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include "version.h"
#include "rplay.h"
#include <sys/param.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <netdb.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "rplayd.h"
#include "connection.h"
#include "audio.h"
#include "spool.h"
#include "sound.h"
#ifdef AUTH
#include "host.h"
#endif /* AUTH */
#include "server.h"
#include "hash.h"
#include "misc.h"
#include "cache.h"

fd_set		read_mask;
fd_set		write_mask;

int		debug = 0;
int		inetd = 1;
int		rplayd_timeout = RPLAYD_TIMEOUT;
int		rptp_timeout = RPTP_CONNECTION_TIMEOUT;
int	       	sound_cleanup_timeout = SOUND_CLEANUP_TIMEOUT;

static char	*progname;
static char	*sounds_file = RPLAY_CONF;
#ifdef AUTH
static char	*hosts_file = RPLAY_HOSTS;
#endif /* AUTH */
static char	*servers_file = RPLAY_SERVERS;
static char	*cache_dir = RPLAY_CACHE;

extern char	*sys_errlist[];
extern char	*optarg;
extern int	optind;

char		hostname[MAXHOSTNAMELEN];

int		rplay_fd, rptp_fd;

#ifdef sparc
#ifdef __STDC__
extern int	gethostname(char *name, int namelen);
#else
extern int	gethostname(/* char *name, int namelen */);
#endif
#endif

#ifdef __STDC__
main(int argc, char **argv)
#else
main(argc, argv)
int	argc;
char	**argv;
#endif
{
	int		nfds, tcount = 0, c;
	struct timeval	select_timeout;
	fd_set		rfds, wfds;

	progname = argv[0];

#ifdef AUTH
	while ((c = getopt(argc, argv, "C:D:H:S:c:dns:t:T:v")) != -1)
#else /* AUTH */
	while ((c = getopt(argc, argv, "C:D:S:c:dns:t:T:v")) != -1)
#endif /* AUTH */
	{
		switch (c)
		{
		case 'C':
			sounds_file = optarg;
			break;
		
		case 'D':
			cache_dir = optarg;
			break;

#ifdef AUTH
		case 'H':
			hosts_file = optarg;
			break;
#endif /* AUTH */
		
		case 'S':
			servers_file = optarg;
			break;

		case 'c':
			audio_timeout = atoi(optarg);
			break;

		case 'd':
			debug = 1;
			break;

		case 'n':
			inetd = 0;
			break;

		case 's':
			cache_max_size = atoi(optarg);
			break;

		case 't':
			rplayd_timeout = atoi(optarg);
			break;

		case 'T':
			rptp_timeout = atoi(optarg);
			break;

		case 'v':
			printf("%s\n", rplay_version);
			done(0);
			break;

		case '?':
			usage();
			done(1);
		}
	}

	if (gethostname(hostname, sizeof(hostname)) < 0)
	{
		report("gethostname: %s\n", sys_errlist[errno]);
		done(1);
	}

	/*
	 * ingore SIGPIPE since read will handled closed TCP connections
	 */
	signal(SIGPIPE, SIG_IGN);

	rplayd_init();
	hash_init(MAX_SOUNDS);
	sound_read(sounds_file);
	cache_init(cache_dir);
	cache_read();
#ifdef AUTH
	host_read(hosts_file);
#endif /* AUTH */
	server_read(servers_file);
	spool_init();
	audio_init();

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	FD_ZERO(&read_mask);
	FD_ZERO(&write_mask);
	FD_SET(rplay_fd, &read_mask);
	FD_SET(rptp_fd, &read_mask);

	if (debug)
	{
		report("%s rplayd ready.\n", hostname);
	}
	
	for (;;)
	{
		if (spool_size)
		{
			rplayd_read();
			audio_write();
			connection_update((fd_set *)NULL, (fd_set *)NULL);
		}
		else
		{
			select_timeout.tv_sec = 1;
			select_timeout.tv_usec = 0;
			rfds = read_mask;
			wfds = write_mask;
#ifdef __hpux
			nfds = select(FD_SETSIZE, (int *)&rfds, (int *)&wfds, 0, &select_timeout);
#else
			nfds = select(FD_SETSIZE, &rfds, &wfds, 0, &select_timeout);
#endif
			switch (nfds)
			{
			case -1:
				report("select: %s\n", sys_errlist[errno]);
				done(1);

			case 0:
				tcount++;
				if (audio_timeout && tcount == audio_timeout)
				{
/*
 * SGI does not need to close the audio device
 */
#ifndef sgi
					audio_close();
#endif
				}
				
				if (rplayd_timeout && tcount == rplayd_timeout)
				{
					done(0);
				}
				if (rptp_timeout && connections)
				{
					connection_check_timeout();
				}
				if (sound_cleanup_timeout && tcount == sound_cleanup_timeout)
				{
					sound_cleanup();
				}
				break;

			default:
				tcount = 0;
				if (FD_ISSET(rplay_fd, &rfds))
				{
					rplayd_read();
				}
				if (spool_size)
				{
					audio_write();
				}
				connection_update(&rfds, &wfds);
				break;
			}
		}
	}
}

/*
 * set up the sockets to receive rplay packets and connection connections
 */
void	rplayd_init()
{
	struct sockaddr_in	s;
	int			on = 1;
	
	if (inetd)
	{
		rplay_fd = 0;
	}
	else
	{
		rplay_fd = socket(AF_INET, SOCK_DGRAM, 0);
		if (rplay_fd < 0)
		{
			report("socket: %s\n", sys_errlist[errno]);
			done(1);
		}

		s.sin_family = AF_INET;
		s.sin_port = htons(RPLAY_PORT);
		s.sin_addr.s_addr = INADDR_ANY;

		if (bind(rplay_fd, (struct sockaddr *)&s, sizeof(s)) < 0)
		{
			report("bind: %s\n", sys_errlist[errno]);
			done(1);
		}
	}

	socket_nonblock(rplay_fd);

	rptp_fd = socket(AF_INET, SOCK_STREAM, 0);
	if (rptp_fd < 0)
	{
		report("socket: %s\n", sys_errlist[errno]);
		done(1);
	}

	if (setsockopt(rptp_fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) < 0)
	{
		report("setsockopt SO_REUSEADDR: %s\n", sys_errlist[errno]);
		done(1);
	}

	s.sin_family = AF_INET;
	s.sin_port = htons(RPTP_PORT);
	s.sin_addr.s_addr = INADDR_ANY;

	if (bind(rptp_fd, (struct sockaddr *)&s, sizeof(s)) < 0)
	{
		report("bind: %s\n", sys_errlist[errno]);
		done(1);
	}

	if (listen(rptp_fd, 5) < 0)
	{
		report("listen: %s\n", sys_errlist[errno]);
		done(1);
	}
}

/*
 * read rplay packets from the rplay_fd UDP socket
 */
#ifdef __STDC__
void	rplayd_read(void)
#else
void	rplayd_read()
#endif
{
	static char		recv_buf[MAX_PACKET];
	char			*packet;
	struct sockaddr_in	f;
	int			j, n, flen;
	RPLAY			*rp;

	for (j = 0; j < SPOOL_SIZE; j++)
	{
		flen = sizeof(f);
		
		n = recvfrom(rplay_fd, recv_buf, sizeof(recv_buf), 0, (struct sockaddr *)&f, &flen);
		if (n <= 0)
		{
			usleep(1); /* believe me this helps xtank a lot */
			continue;
		}
#ifdef AUTH
		if (!host_access(f, HOST_EXECUTE))
		{
			if (debug && !host_access(f, HOST_READ) && !host_access(f, HOST_WRITE))
			{
				report("%s permission denied\n", inet_ntoa(f.sin_addr));
			}
			continue;
		}
#endif /* AUTH */

		packet = recv_buf;

		switch (*packet)
		{
#ifdef OLD_RPLAY
		case OLD_RPLAY_PLAY:
		case OLD_RPLAY_STOP:
		case OLD_RPLAY_PAUSE:
		case OLD_RPLAY_CONTINUE:
			packet = rplay_convert(recv_buf);
#endif /* OLD_RPLAY */
				
		case RPLAY_VERSION:
			rp = rplay_unpack(packet);
			if (rp == NULL)
			{
				report("%s\n", rplay_errlist[rplay_errno]);
				break;
			}

			switch (rp->command)
			{
			case RPLAY_PING:
				if (debug)
				{
					report("received a ping packet\n");
				}
				rplay_destroy(rp);
				break;

			case RPLAY_PLAY:
				if (debug)
				{
					report("play sound=%s volume=%d from %s\n",
						(char *)rplay_get(rp, RPLAY_SOUND, 0),
						rplay_get(rp, RPLAY_VOLUME, 0),
						inet_ntoa(f.sin_addr));
				}
				rplayd_play(rp, f);
				break;

			case RPLAY_STOP:
				if (debug)
				{
					report("stop sound=%s volume=%d from %s\n",
						(char *)rplay_get(rp, RPLAY_SOUND, 0),
						rplay_get(rp, RPLAY_VOLUME, 0),
						inet_ntoa(f.sin_addr));
				}
				rplayd_stop(rp, f);
				break;

			case RPLAY_PAUSE:
				if (debug)
				{
					report("pause sound=%s volume=%d from %s\n",
						(char *)rplay_get(rp, RPLAY_SOUND, 0),
						rplay_get(rp, RPLAY_VOLUME, 0),
						inet_ntoa(f.sin_addr));
				}
				rplayd_pause(rp, f);
				break;

			case RPLAY_CONTINUE:
				if (debug)
				{
					report("continue sound=%s volume=%d from %s\n",
						(char *)rplay_get(rp, RPLAY_SOUND, 0),
						rplay_get(rp, RPLAY_VOLUME, 0),
						inet_ntoa(f.sin_addr));
				}
				rplayd_continue(rp, f);
				break;
			}
			break;
				
		default:
			if (debug)
			{
				report("unknown packet received from %s\n", inet_ntoa(f.sin_addr));
			}
			break;
		}
	}
}

#ifdef __STDC__
int	rplayd_play(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_play(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	SPOOL	*sp;
	int	i, n;

	sp = spool_next(rp->priority);
	if (!sp)
	{
		if (debug)
		{
			report("spool full\n");
		}
		return -1;
	}
	n = 0;
	for (i = 0; i < rp->nsounds; i++)
	{
		sp->sound[i] = sound_lookup((char *)rplay_get(rp, RPLAY_SOUND, i), SOUND_FIND);
		if (sp->sound[i] == NULL)
		{
			rplay_destroy(rp);
			return -1;
		}
		if (sp->sound[i]->status != SOUND_READY)
		{
			n++;
		}
	}

	sp->rp = rp;
	sp->curr_attrs = rp->attrs;
	sp->curr_sound = 0;
	sp->curr_count = sp->curr_attrs->count;
	sp->list_count = rp->count;
	sp->sin = sin;

	if (n)
	{
		sp->state = SPOOL_WAIT;
	}
	else
	{
		sp->ptr = sp->sound[0]->start;
		sp->end = sp->sound[0]->stop;
		sp->state = SPOOL_PLAY;
		spool_size++;
		spool_setprio();
	}

	return 0;
}

#ifdef __STDC__
int	rplayd_stop(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_stop(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	SPOOL	*sp;

	sp = spool_match(rp);
	if (sp == NULL || sp->state == SPOOL_WAIT)
	{
		return -1;
	}

	sp->sin = sin;
	rplay_destroy(rp);
	if (sp)
	{
		if (sp->state == SPOOL_PLAY)
		{
			spool_size--;
		}
		spool_clear(sp);
		return 0;
	}
	else if (debug)
	{
		report("stop sound not found\n");
		return -1;
	}
}

#ifdef __STDC__
int	rplayd_pause(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_pause(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	SPOOL	*sp;

	sp = spool_match(rp);
	if (sp == NULL || sp->state == SPOOL_WAIT)
	{
		return -1;
	}

	sp->sin = sin;
	rplay_destroy(rp);
	if (sp)
	{
		if (sp->state == SPOOL_PLAY)
		{
			spool_size--;		/* sort of remove it from the spool! */
		}
		sp->state = SPOOL_PAUSE;
		return 0;
	}
	else if (debug)
	{
		report("pause sound not found\n");
		return -1;
	}
}

#ifdef __STDC__
int	rplayd_continue(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_continue(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	SPOOL	*sp;

	sp = spool_match(rp);
	if (sp == NULL || sp->state == SPOOL_WAIT)
	{
		return -1;
	}

	sp->sin = sin;
	rplay_destroy(rp);
	if (sp)
	{
		if (sp->state == SPOOL_PAUSE)
		{
			spool_size++;		/* put it back in the spool! */
		}
		sp->state = SPOOL_PLAY;
		return 0;
	}
	else if (debug)
	{
		report("continue sound not found\n");
		return -1;
	}
}

void	usage()
{
	printf("\n");
	printf("%s\n", rplay_version);
	printf("\n");
	printf("usage: %s [options]\n", progname);
	printf("\t-c n    Close %s after n idle seconds, disabled with 0 (%d).\n", AUDIO_DEVICE, audio_timeout);
	printf("\t-C file Use file for alternate rplay.conf (%s).\n", RPLAY_CONF);
	printf("\t-d      Debug mode (%s).\n", debug ? "on" : "off");
	printf("\t-D dir  Use dir for alternate rplay.cache (%s).\n", RPLAY_CACHE);
#ifdef AUTH
	printf("\t-H file Use file for alternate rplay.hosts (%s).\n", RPLAY_HOSTS);
#endif /* AUTH */
	printf("\t-n      Disable inetd mode (%s).\n", inetd ? "enabled" : "disabled");
	printf("\t-s n    Maximum size in bytes of the rplay cache, disabled with 0 (%d).\n", RPLAY_CACHE_SIZE);
	printf("\t-S file Use file for alternate rplay.servers (%s).\n", RPLAY_SERVERS);
	printf("\t-t n    Exit after n idle seconds, disabled with 0 (%d).\n", rplayd_timeout);
	printf("\t-T n    Close idle RPTP connections after n seconds, disabled with 0 (%d).\n", rptp_timeout);
	printf("\t-v      Print rplay version and exit.\n");
	printf("\n");
}

#ifdef __STDC__
void	done(int exit_value)
#else
void	done(exit_value)
int	exit_value;
#endif
{
	connection_cleanup();
	audio_close();
	close(rplay_fd);
	close(rptp_fd);
	if (debug)
	{
		report("done(%d)\n", exit_value);
	}
	exit(exit_value);
}

#ifdef __STDC__
void	report(char *fmt, ...)
#else
void	report(va_alist)
va_dcl
#endif
{
	va_list	args;

#ifdef __STDC__
	va_start(args, fmt);
#else
	char	*fmt;
	va_start(args);
	fmt = va_arg(args, char *);
#endif

	fprintf(stderr, "%s: ", progname);
	vfprintf(stderr, fmt, args);
	va_end(args);
}
