/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include "rplayd.h"
#include <netdb.h>
#include <string.h>
#include "host.h"
#include "regex.h"
#include "buffer.h"

#ifdef AUTH

static struct re_pattern_buffer	access_read, access_write, access_execute;

BUFFER	*host_list = NULL;

/*
 * read the rplay hosts file
 *
 * This routine will build a regular expression for each type of access.
 * The expressions are of the form:
 * 	\(130\.191\.224\.2\|130\.191\..*\)
 * See regex.[ch] for more details.
 */
#ifdef __STDC__
void	host_read(char *filename)
#else
void	host_read(filename)
char	*filename;
#endif
{
	FILE		*fp;
	char		buf[BUFSIZ], *perms, *name, *p;
	unsigned long	addr;
	struct in_addr	addr_in;
	struct hostent	*hp;
	int		n;
	char		expr_read[HOST_EXPR_SIZE];
	char		expr_write[HOST_EXPR_SIZE];
	char		expr_execute[HOST_EXPR_SIZE];
	BUFFER		*b;

	fp = fopen(filename, "r");
	if (fp == NULL)
	{
		report("cannot open %s\n", filename);
		done(1);
	}
	
	b = buffer_create();
	b->status = BUFFER_KEEP;
	strcpy(b->buf, "+hosts\r\n");
	b->nbytes += strlen(b->buf);
	host_list = b;

	memset((char *)&access_read, 0, sizeof(access_read));
	memset((char *)&access_write, 0, sizeof(access_write));
	memset((char *)&access_execute, 0, sizeof(access_execute));

	strcpy(expr_read, "\\(");
	strcpy(expr_write, "\\(");
	strcpy(expr_execute, "\\(");

	while (fgets(buf, sizeof(buf), fp))
	{
		switch (buf[0])
		{
		case '#':
		case ' ':
		case '\t':
		case '\n':
			continue;
		}

		p = strchr(buf, '\n');
		*p = '\0';

		name = buf;
		perms = strchr(buf, ':');
		if (perms)
		{
			*perms = '\0';
			perms++;
		}
		else
		{
			perms = HOST_DEFAULT_ACCESS;
		}

		n = strlen(buf) + strlen(perms) + 3; /* room for : \r\n */

		if (b->nbytes + n > BUFFER_SIZE)
		{
			b->next = buffer_create();
			b = b->next;
			b->status = BUFFER_KEEP;
		}

		strcat(b->buf, name);
		strcat(b->buf, ":");
		strcat(b->buf, perms);
		strcat(b->buf, "\r\n");
		b->nbytes += n;

		if (strchr(name, '*'))
		{
			name = host_ip_to_regex(name);
		}
		else
		{
			addr = inet_addr(name);
			if (addr == 0xffffffff)
			{
				hp = gethostbyname(name);
				if (hp == NULL)
				{
					report("%s unknown host\n", name);
					done(1);
				}
				memcpy((char *)&addr, (char *)hp->h_addr, hp->h_length);
			}
			memcpy((char *)&addr_in, (char *)&addr, sizeof(addr_in));
			name = host_ip_to_regex(inet_ntoa(addr_in));
		}
		
		for (p = perms; *p; p++)
		{
			switch (*p)
			{
			case HOST_READ:
				strcat(expr_read, name);
				strcat(expr_read, "\\|");
				break;
				
			case HOST_WRITE:
				strcat(expr_write, name);
				strcat(expr_write, "\\|");
				break;
				
			case HOST_EXECUTE:
				strcat(expr_execute, name);
				strcat(expr_execute, "\\|");
				break;

			default:
				report("'%c' unknown host access permission\n", *p);
				done(1);
			}
		}
	}

	fclose(fp);
	
	if (b->nbytes + 3 > BUFFER_SIZE) /* room for .\r\n */
	{
		b->next = buffer_create();
		b = b->next;
		b->status = BUFFER_KEEP;
	}
	strcat(b->buf, ".\r\n");
	b->nbytes += 3;

	if (strlen(expr_read) == 2)
	{
		strcat(expr_read, "\\)");
	}
	else
	{
		expr_read[strlen(expr_read)-1] = ')';
	}
	if (strlen(expr_write) == 2)
	{
		strcat(expr_write, "\\)");
	}
	else
	{
		expr_write[strlen(expr_write)-1] = ')';
	}
	if (strlen(expr_execute) == 2)
	{
		strcat(expr_execute, "\\)");
	}
	else
	{
		expr_execute[strlen(expr_execute)-1] = ')';
	}

#ifdef DEBUG
	printf("expr_read: %s\n", expr_read);
	printf("expr_write: %s\n", expr_write);
	printf("expr_execute: %s\n", expr_execute);
#endif

	p = re_compile_pattern(expr_read, strlen(expr_read), &access_read);
	if (p != NULL)
	{
		report("re_compile_pattern: %s\n", p);
		done(1);
	}

	p = re_compile_pattern(expr_write, strlen(expr_write), &access_write);
	if (p != NULL)
	{
		report("re_compile_pattern: %s\n", p);
		done(1);
	}
	
	p = re_compile_pattern(expr_execute, strlen(expr_execute), &access_execute);
	if (p != NULL)
	{
		report("re_compile_pattern: %s\n", p);
		done(1);
	}
}

#ifdef __STDC__
int	host_access(struct sockaddr_in sin, char access_mode)
#else
int	host_access(sin, access_mode)
struct sockaddr_in	sin;
char			access_mode;
#endif
{
	char				*p;
	struct re_pattern_buffer	*re;

	p = inet_ntoa(sin.sin_addr);
	
	switch (access_mode)
	{
	case HOST_READ:
		re = &access_read;
		break;
		
	case HOST_WRITE:
		re = &access_write;
		break;
		
	case HOST_EXECUTE:
		re = &access_execute;
		break;

	default:
		report("host_access: unknown access mode '%s'\n", access_mode);
		done(1);
	}

	return re_match(re, p, strlen(p), 0, 0) > 0 ? 1 : 0;
}

#ifdef __STDC__
char	*host_ip_to_regex(char *p)
#else
char	*host_ip_to_regex(p)
char	*p;
#endif
{
	static char	buf[64];
	char		*q;
	
	for (q = buf; *p; p++, q++)
	{
		switch (*p)
		{
		case '.':
			*q++ = '\\';
			*q = '.';
			break;
			
		case '*':
			*q++ = '.';
			*q = '*';
			break;
			
		default:
			*q = *p;
		}
	}
	
	*q = '\0';
	
	return buf;
}

#endif /* AUTH */
