-- THIS SOFTWARE IS PROVIDED ''AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
-- WARRANTIES, INCLUDING, WITHOUT LIMITATION, IMPLIED WARRANTIES OF
-- NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- Digital assumes no responsibility AT ALL for the use or reliability
-- of this software.

-- XM_TEXT.ADA
--
-- Abstract: Demo of X Window text editor widget.
--
with UNCHECKED_CONVERSION, C_TYPES,
     XT, XM, XM_CONVENIENCE, XM_WIDGET, XM_STRING;
 
package TEXT_PKG is

    --
    -- This program shows how the use the text widget to make a simple
    -- text editor.  It is based on the example in section 10.5 on
    -- page 158 of the first edition of "Motif Programming" by Marshall
    -- Brain, published by Digital Press.
    --
    -- Note the use of the exception, EXIT_TEXT, in place of a call
    -- to exit in the C run-time library.
    --
 
    type MENU_TYPE  is (OPEN, CLOSE, QUIT, CUT, CLEAR, COPY, PASTE);
	for MENU_TYPE'size use 8;

    type REPLY_TYPE is (OK, CANCEL);
	for REPLY_TYPE'size use 8;
 
    function TO_POINTER_TYPE is new XT.GENERIC_TO_POINTER_TYPE(REPLY_TYPE);
    function TO_POINTER_TYPE is new XT.GENERIC_TO_POINTER_TYPE(MENU_TYPE);
 
    CONTEXT  : XT.APP_CONTEXT_TYPE;
    CHAR_SET : XM.STRING_CHAR_SET_TYPE :=
	XM.STRING_CHAR_SET_TYPE(XM_STRING.STRING_DEFAULT_CHARSET_PTR);
 
    TOP_LEVEL    : XT.WIDGET_TYPE;
    TEXT_WIDGET  : XT.WIDGET_TYPE;
    FORM_WIDGET  : XT.WIDGET_TYPE;
    LABEL        : XT.WIDGET_TYPE;
    OPEN_OPTION  : XT.WIDGET_TYPE;
    CLOSE_OPTION : XT.WIDGET_TYPE;
    QUIT_OPTION  : XT.WIDGET_TYPE;
    CUT_OPTION   : XT.WIDGET_TYPE;
    CLEAR_OPTION : XT.WIDGET_TYPE;
    COPY_OPTION  : XT.WIDGET_TYPE;
    PASTE_OPTION : XT.WIDGET_TYPE;
    OPEN_DIALOG  : XT.WIDGET_TYPE;
    SAVE_DIALOG  : XT.WIDGET_TYPE;
 
    NULL_STRING  : constant C_TYPES.CHAR_POINTER :=
	C_TYPES.TO_CHAR_POINTER(C_TYPES.NULL_TERMINATED.NULL_STRING'address);
 
    FILE_NAME    : C_TYPES.CHAR_POINTER := NULL_STRING;
    TEXT_CHANGED : BOOLEAN              := false;

    EXIT_TEXT    : exception;           -- Done exiting when raised.
 
    package MANIPULATE_STRING is new XT.MANIPULATE_AVT(XM.STRING_TYPE);
 
    procedure CHANGE_SENSITIVITY(
	OPEN_STATE : in BOOLEAN );
 
    procedure CHANGED_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in XT.POINTER_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE );

    pragma EXPORT_PROCEDURE(CHANGED_CALLBACK,
	MECHANISM => (VALUE, VALUE, VALUE) );
 
    procedure OPEN_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in REPLY_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_FILE_SELECTION_CALLBACK_TYPE );

        pragma EXPORT_PROCEDURE(OPEN_CALLBACK,
	    MECHANISM => (VALUE, VALUE, VALUE) );
 
    procedure HANDLE_SAVE;
 
    procedure SAVE_DIALOG_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in REPLY_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE );

	pragma EXPORT_PROCEDURE(SAVE_DIALOG_CALLBACK,
		MECHANISM => (VALUE, VALUE, VALUE) );
 
    function MAKE_MENU_OPTION(
	OPTION_NAME : in C_TYPES.NULL_TERMINATED.STRING;
	CLIENT_DATA : in MENU_TYPE;
	MENU        : in XT.WIDGET_TYPE )
	return XT.WIDGET_TYPE;
 
    function MAKE_MENU(
	MENU_NAME : in C_TYPES.NULL_TERMINATED.STRING;
	MENU_BAR  : in XT.WIDGET_TYPE )
	return XT.WIDGET_TYPE;
 
    procedure CREATE_MENUS(
	MENU_BAR : in XT.WIDGET_TYPE );
 
    procedure MENU_CALLBACK(
	WIDGET	  : in XT.WIDGET_TYPE; CLIENT_DATA : in MENU_TYPE;
	CALL_DATA : in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE );

        pragma EXPORT_PROCEDURE(MENU_CALLBACK,
		MECHANISM => (VALUE, VALUE, VALUE) );
 
end TEXT_PKG;
 
with TEXT_IO, INTEGER_TEXT_IO, X_LIB;
package body TEXT_PKG is
 
    use C_TYPES.NULL_TERMINATED;
 
    procedure CHANGE_SENSITIVITY(
	OPEN_STATE : in BOOLEAN) is

    -- Changes the menu sensitivities as needed for opened and closed states.
 
    begin
	XT.SET_SENSITIVE(OPEN_OPTION, OPEN_STATE);
	XT.SET_SENSITIVE(QUIT_OPTION, OPEN_STATE);
	XT.SET_SENSITIVE(CLOSE_OPTION, not OPEN_STATE);
	XT.SET_SENSITIVE(CUT_OPTION, not OPEN_STATE);
	XT.SET_SENSITIVE(COPY_OPTION, not OPEN_STATE);
	XT.SET_SENSITIVE(PASTE_OPTION, not OPEN_STATE);
	XT.SET_SENSITIVE(CLEAR_OPTION, not OPEN_STATE);
    end CHANGE_SENSITIVITY;
 
    procedure CHANGED_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in XT.POINTER_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE ) is
 
    -- Triggered everytime a character is inserted or deleted in the text widget.
    -- Text_Changed is used to decide if the file needs saving or not.
 
    begin
	TEXT_CHANGED := TRUE;
    end CHANGED_CALLBACK;

    procedure OPEN_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in REPLY_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_FILE_SELECTION_CALLBACK_TYPE ) is
 
    -- Handles the file selection box callbacks.
 
	FILE   : TEXT_IO.FILE_TYPE;
	CH     : CHARACTER;
	RESULT : BOOLEAN;                      --dummy argument

	type ACCESS_STRING is access STRING;
	FILE_CONTENTS : ACCESS_STRING;
	FILE_LENGTH   : INTEGER;
  
    begin
        -- Do nothing if cancel is selected.
	--
	if CLIENT_DATA = CANCEL then
	    XT.UNMANAGE_CHILD(OPEN_DIALOG);
	    return;
	end if;

	-- Free up the file name if it exists.
	--
	if C_TYPES.NULL_TERMINATED.LENGTH(FILE_NAME) /= 0 then
	    XT.FREE(FILE_NAME);
	    FILE_NAME := NULL_STRING;
	end if;

        --  Get the file name from the file selection box.
        --
	XM_CONVENIENCE.STRING.GET_L_TO_R(
	    RESULT, XM.STRING_TYPE(CALL_DATA.all.VALUE), CHAR_SET, FILE_NAME);

        --  Open and read the file.
	--
	TEXT_IO.OPEN(FILE, TEXT_IO.IN_FILE,
	    C_TYPES.NULL_TERMINATED.TO_STRING(FILE_NAME) );

	FILE_LENGTH := 0;
 
	while not TEXT_IO.END_OF_FILE(FILE) loop
	    if TEXT_IO.END_OF_LINE(FILE) then
		TEXT_IO.SKIP_LINE(FILE);

	    else
		TEXT_IO.GET(FILE, CH);
	    end if;

	    FILE_LENGTH := FILE_LENGTH + 1;
	end loop;

	TEXT_IO.CLOSE(FILE);
--	pc := direct_io.size(file);

--	text_io.put("length = ");
--	integer_text_io.put(integer(pc));
--	text_io.new_line;

	if FILE_LENGTH /= 0 then
	    -- Malloc a place for the string to be read to.
	    --
	    FILE_CONTENTS := new STRING(1 .. FILE_LENGTH + 10);
 
            -- Read the file string.
	    --
	    TEXT_IO.OPEN(FILE, TEXT_IO.IN_FILE,
		C_TYPES.NULL_TERMINATED.TO_STRING(FILE_NAME) );

	    FILE_CONTENTS.all(1) := ASCII.NUL;
 
	    for I in 1 .. FILE_LENGTH loop
		if TEXT_IO.END_OF_LINE(FILE) then
	    	    TEXT_IO.SKIP_LINE(FILE);
		    FILE_CONTENTS.all(I) := ASCII.LF;

		else 
		    TEXT_IO.GET(FILE, CH);
		    FILE_CONTENTS.all(I) := CH;
		end if;
	    end loop;
 
	    TEXT_IO.CLOSE(FILE);
 
	    FILE_CONTENTS.all(FILE_LENGTH + 1) := ASCII.NUL;
 
            -- Give the string to the text widget.
	    -- 
	    XM_CONVENIENCE.TEXT.SET_STRING(TEXT_WIDGET,
	        C_TYPES.NULL_TERMINATED.TO_STRING(
		    C_TYPES.TO_CHAR_POINTER(FILE_CONTENTS.all'address) ));
 
	    XT.FREE(C_TYPES.TO_CHAR_POINTER(FILE_CONTENTS.all'address));
 
            -- Set up all resources as needed to make menus and the
	    -- text widget sensitive.
	    --
	    CHANGE_SENSITIVITY(false);
 
	    XT.SET_SENSITIVE(TEXT_WIDGET, true);
 
	    XM_CONVENIENCE.TEXT.SET_EDITABLE(TEXT_WIDGET, true);
 
	    XM_CONVENIENCE.TEXT.SET_CURSOR_POSITION(TEXT_WIDGET, 0);
 
	    TEXT_CHANGED := false;
	end if;
 
	XT.UNMANAGE_CHILD(OPEN_DIALOG);

    end OPEN_CALLBACK;
 
    procedure HANDLE_SAVE is
	-- Saves the text widget's string to a file.
	--
	FILE : TEXT_IO.FILE_TYPE;
	TEXT : C_TYPES.CHAR_POINTER;
 
    begin
	TEXT_IO.OPEN(FILE, TEXT_IO.OUT_FILE,
	    C_TYPES.NULL_TERMINATED.TO_STRING(FILE_NAME) );
 
        -- Get the string from the text widget.
	--
	TEXT := XM_CONVENIENCE.TEXT.GET_STRING(TEXT_WIDGET);
 
	if C_TYPES.NULL_TERMINATED.LENGTH(TEXT) /= 0 then
 
	    -- Write the file.
	    --
	    for I in 1 .. C_TYPES.NULL_TERMINATED.LENGTH(TEXT) loop
		TEXT_IO.PUT( FILE, C_TYPES.NULL_TERMINATED.TO_STRING(TEXT)(I) );
	    end loop;
 
            -- Make sure the last line is terminated by a new line so that
            -- vi and compilers like it.
	    --
	    if ( C_TYPES.NULL_TERMINATED.TO_STRING(TEXT) 
		(C_TYPES.NULL_TERMINATED.LENGTH(TEXT) - 1) ) /= ASCII.LF then
 
		TEXT_IO.NEW_LINE(FILE);
	    end if;
 
	    XT.FREE(TEXT);
	end if;
 
	TEXT_IO.CLOSE(FILE);
    end HANDLE_SAVE;

    procedure SAVE_DIALOG_CALLBACK(
	WIDGET      : in XT.WIDGET_TYPE;
	CLIENT_DATA : in REPLY_TYPE;
	CALL_DATA   : in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE ) is
 
    begin
	-- Handles save_dialog buttons.
	--
	case CLIENT_DATA is
	    when OK	=> HANDLE_SAVE;
	    when CANCEL => null;
	end case;
 
        -- Get rid of the text in the text widget and set it so it can't be used.
 
	XT.SET_SENSITIVE(TEXT_WIDGET, false);
	XM_CONVENIENCE.TEXT.SET_EDITABLE(TEXT_WIDGET, false);
	XM_CONVENIENCE.TEXT.SET_STRING(
		TEXT_WIDGET, C_TYPES.NULL_TERMINATED.TO_STRING("") );
 
        -- Change menu sensativities and make the dialog invisible.
 
	CHANGE_SENSITIVITY(true);
 
	XT.UNMANAGE_CHILD(SAVE_DIALOG);
 
    end SAVE_DIALOG_CALLBACK;
 
    procedure MENU_CALLBACK(
	WIDGET		: in XT.WIDGET_TYPE;
	CLIENT_DATA	: in MENU_TYPE;
	CALL_DATA	: in XM_WIDGET.ACCESS_ANY_CALLBACK_TYPE ) is
 
	-- Handles menu option.
 
	function TO_BUTTON_EVENT is new UNCHECKED_CONVERSION(
	    X_LIB.EVENT_TYPE, X_LIB.EVENTLESS_BUTTON_EVENT_TYPE );
 
	TIME   : X_LIB.TIME_TYPE;
	RESULT : BOOLEAN;               --dummy argument
 
    begin
 
	case CLIENT_DATA is
 
	    when OPEN =>
                -- Make the file selection box appear.
		--
		XT.MANAGE_CHILD(OPEN_DIALOG);
 
	    when CLOSE =>
		if TEXT_CHANGED then
		    -- If the text was changed, ask the user about saving it.
		    --
		    XT.MANAGE_CHILD(SAVE_DIALOG);
 
		else
                    -- Loose the text and set the widget insensitive.
		    -- 
		    XT.SET_SENSITIVE(TEXT_WIDGET, false);
 
		    XM_CONVENIENCE.TEXT.SET_EDITABLE(TEXT_WIDGET, false);
 
		    XM_CONVENIENCE.TEXT.SET_STRING(
			TEXT_WIDGET,
			C_TYPES.NULL_TERMINATED.TO_STRING(NULL_STRING) );

		    CHANGE_SENSITIVITY(true);
		end if;
 
	    when QUIT => raise EXIT_TEXT;
 
	    when CUT =>
		TIME   := TO_BUTTON_EVENT(CALL_DATA.all.EVENT.all).TIME;
		RESULT := XM_CONVENIENCE.TEXT.CUT(TEXT_WIDGET, TIME);
 
	    when CLEAR =>
		RESULT := XM_CONVENIENCE.TEXT.REMOVE(TEXT_WIDGET);
 
	    when PASTE =>
		RESULT := XM_CONVENIENCE.TEXT.PASTE(TEXT_WIDGET);
 
	    when COPY =>
		TIME   := TO_BUTTON_EVENT(CALL_DATA.all.EVENT.all).TIME;
		RESULT := XM_CONVENIENCE.TEXT.COPY(TEXT_WIDGET, TIME);
	end case;
    end MENU_CALLBACK;
 
    function MAKE_MENU_OPTION(
	OPTION_NAME : in C_TYPES.NULL_TERMINATED.STRING;
	CLIENT_DATA : in MENU_TYPE;
	MENU        : in XT.WIDGET_TYPE )
	return XT.WIDGET_TYPE is

	AL     : XT.ARG_LIST_TYPE(0 .. 0);
	BUTTON : XT.WIDGET_TYPE;
 
    begin
	AL(0) := (
	    XM_STRING.NLABEL_STRING_PTR,
	    XT.TO_ARG_VALUE_TYPE(C_TYPES.CHAR_POINTER(
		XM_CONVENIENCE.STRING.CREATE_L_TO_R(OPTION_NAME, CHAR_SET) )));
 
	BUTTON := XM_WIDGET.CREATE_PUSH_BUTTON(MENU, OPTION_NAME, AL);
	XT.MANAGE_CHILD(BUTTON);

	XT.ADD_CALLBACK(
	    BUTTON,
	    XM_STRING.NACTIVATE_CALLBACK,
	    XT.BASE_PROC_TYPE(MENU_CALLBACK'address),
	    TO_POINTER_TYPE(CLIENT_DATA) );

	return BUTTON;
 
    end MAKE_MENU_OPTION;
 
    function MAKE_MENU(
	MENU_NAME : in C_TYPES.NULL_TERMINATED.STRING;
	MENU_BAR  : in XT.WIDGET_TYPE)
	return XT.WIDGET_TYPE is

	AL      : XT.ARG_LIST_TYPE(0 .. 1);
	MENU    : XT.WIDGET_TYPE;
	CASCADE : XT.WIDGET_TYPE;
 
    begin
	MENU := XM_WIDGET.CREATE_PULLDOWN_MENU(MENU_BAR, MENU_NAME, AL(1..0));
 
	AL(0) := (XM_STRING.NSUB_MENU_ID_PTR, XT.TO_ARG_VALUE_TYPE(MENU));
 
	AL(1) := (
	    XM_STRING.NLABEL_STRING_PTR,
	    XT.TO_ARG_VALUE_TYPE( C_TYPES.CHAR_POINTER(
		XM_CONVENIENCE.STRING.CREATE_L_TO_R(MENU_NAME, CHAR_SET) )));
 
	CASCADE := XM_WIDGET.CREATE_CASCADE_BUTTON(MENU_BAR, MENU_NAME, AL);
 
	XT.MANAGE_CHILD(CASCADE);
 
	return MENU;
    end MAKE_MENU;
 

    procedure CREATE_MENUS(
	MENU_BAR : in XT.WIDGET_TYPE) is
 
	MENU : XT.WIDGET_TYPE;
 
    begin
	MENU := MAKE_MENU(C_TYPES.NULL_TERMINATED.TO_STRING("File"), MENU_BAR);
 
	OPEN_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Open"), OPEN, MENU);
 
	CLOSE_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Close"), CLOSE, MENU);
 
	XT.SET_SENSITIVE(CLOSE_OPTION, false);
 
	QUIT_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Quit"), QUIT, MENU);
 
	MENU := MAKE_MENU(
		C_TYPES.NULL_TERMINATED.TO_STRING("Edit"), MENU_BAR);
 
	CUT_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Cut"), CUT, MENU);
 
	COPY_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Copy"), COPY, MENU);
 
	PASTE_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Paste"), PASTE, MENU);
 
	CLEAR_OPTION := MAKE_MENU_OPTION(
		C_TYPES.NULL_TERMINATED.TO_STRING("Clear"), CLEAR, MENU);

    end CREATE_MENUS;
end TEXT_PKG;
 
with TEXT_PKG; use  TEXT_PKG;
with SYSTEM, C_TYPES, X_RESOURCE, XT, XM, 
     XM_CONVENIENCE, XM_WIDGET, XM_STRING;

use SYSTEM;
procedure XM_TEXT is
 
    package MANIPULATE_ATTACH is new XT.MANIPULATE_AVT(XM_WIDGET.ATTACH_TYPE);

    package MANIPULATE_EDIT is new XT.MANIPULATE_AVT(XM_WIDGET.EDIT_MODE_TYPE);

    MENU_BAR      : XT.WIDGET_TYPE;
    AL            : XT.ARG_LIST_TYPE(0 .. 5);
    ARGC	  : NATURAL := 0;
    ARGV	  : ADDRESS := NULL_ADDRESS;

begin

    -- Create the top level shell.
    --
    XT.APP_INITIALIZE(TOP_LEVEL, CONTEXT,
	C_TYPES.NULL_TERMINATED.NULL_STRING,
	X_RESOURCE.NULL_VECTOR_OF_OPTION_DESC,
	ARGC,
	ARGV,
	C_TYPES.NULL_TERMINATED.NULL_STRING,
	AL(1..0) );
 
    -- Default window size.
    --
    AL(0) := (XM_STRING.NHEIGHT_PTR, XT.TO_ARG_VALUE_TYPE(200));
    AL(1) := (XM_STRING.NWIDTH_PTR,  XT.TO_ARG_VALUE_TYPE(200));
 
    XT.SET_VALUES(TOP_LEVEL, AL(0 .. 1));
 

    -- Create a Form widget.
    -- 
    FORM_WIDGET := XM_WIDGET.CREATE_FORM(
	TOP_LEVEL, C_TYPES.NULL_TERMINATED.TO_STRING("form"), AL(1..0) );
 
    XT.MANAGE_CHILD(FORM_WIDGET);
 

    -- Create a menu bar and attach it to the form.
    -- 
    AL(0) := (
	XM_STRING.NTOP_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM));

    AL(1) := (
	XM_STRING.NRIGHT_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM) );

    AL(2) := (
	XM_STRING.NLEFT_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM));

    MENU_BAR := XM_WIDGET.CREATE_MENU_BAR(
	FORM_WIDGET, C_TYPES.NULL_TERMINATED.TO_STRING("menu_bar"), AL(0..2) );
 
    XT.MANAGE_CHILD(MENU_BAR);
  
 
    -- Create a text widget and attach it to the form.
    --
    AL(0) := (
	XM_STRING.NTOP_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE( XM_WIDGET.ATTACH_TYPE(
	    XM_WIDGET.WIDGET )));

    AL(1) := (XM_STRING.NTOP_WIDGET_PTR, XT.TO_ARG_VALUE_TYPE(MENU_BAR));

    AL(2) := (
	XM_STRING.NRIGHT_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM) );

    AL(3) := (
	XM_STRING.NLEFT_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM));

    AL(4) := (
	XM_STRING.NBOTTOM_ATTACHMENT_PTR,
	MANIPULATE_ATTACH.TO_ARG_VALUE_TYPE(XM_WIDGET.FORM) );

    AL(5) := (
	XM_STRING.NEDIT_MODE_PTR,
	MANIPULATE_EDIT.TO_ARG_VALUE_TYPE(XM_WIDGET.MULTI_LINE_EDIT) );

    TEXT_WIDGET := XM_CONVENIENCE.TEXT.CREATE_SCROLLED(
	FORM_WIDGET, C_TYPES.NULL_TERMINATED.TO_STRING("text"), AL );
 
    XT.ADD_CALLBACK(
	TEXT_WIDGET,
	XM_STRING.NVALUE_CHANGED_CALLBACK,
	XT.BASE_PROC_TYPE(CHANGED_CALLBACK'address),
	XT.NULL_POINTER);
 
    XT.MANAGE_CHILD(TEXT_WIDGET);
 
    XT.SET_SENSITIVE(TEXT_WIDGET, false);
 
    XM_CONVENIENCE.TEXT.SET_EDITABLE(TEXT_WIDGET, false);
 
    CREATE_MENUS(MENU_BAR);
  
 
    -- Create the file selection box used by open option.
    --
    AL(0) := (XM_STRING.NMUST_MATCH_PTR, XT.TO_ARG_VALUE_TYPE(true));
 
    AL(1) := (XM_STRING.NAUTO_UNMANAGE_PTR, XT.TO_ARG_VALUE_TYPE(false));
 
    OPEN_DIALOG := XM_WIDGET.CREATE_FILE_SELECTION_DIALOG(
	TOP_LEVEL, C_TYPES.NULL_TERMINATED.TO_STRING("open_dialog"), AL(0..2) );
 
    XT.ADD_CALLBACK(
	OPEN_DIALOG,
	XM_STRING.NOK_CALLBACK,
	XT.BASE_PROC_TYPE(OPEN_CALLBACK'address),
	TO_POINTER_TYPE(OK));
 
    XT.ADD_CALLBACK(
	OPEN_DIALOG,
	XM_STRING.NCANCEL_CALLBACK,
	XT.BASE_PROC_TYPE(OPEN_CALLBACK'address),
	TO_POINTER_TYPE(CANCEL));
 
    XT.UNMANAGE_CHILD(
	XM_WIDGET.SELECTION_BOX_GET_CHILD(OPEN_DIALOG, XM_WIDGET.HELP_BUTTON) );
  
 
    -- Create the file saving dialog.
    --
    AL(0) := (
	XM_STRING.NMESSAGE_STRING_PTR,
	XT.TO_ARG_VALUE_TYPE( C_TYPES.CHAR_POINTER(
	    XM_CONVENIENCE.STRING.CREATE_L_TO_R(
		C_TYPES.NULL_TERMINATED.TO_STRING(
		    "The text was changed.  Save it?" ),
		CHAR_SET ))));
 
    SAVE_DIALOG := XM_WIDGET.CREATE_MESSAGE_DIALOG(
	TOP_LEVEL, C_TYPES.NULL_TERMINATED.TO_STRING("ok_dialog"), AL(0..0) );
 
    XT.ADD_CALLBACK(
	SAVE_DIALOG,
	XM_STRING.NOK_CALLBACK,
	XT.BASE_PROC_TYPE(SAVE_DIALOG_CALLBACK'address),
	TO_POINTER_TYPE(OK));
 
    XT.ADD_CALLBACK(
	SAVE_DIALOG,
	XM_STRING.NCANCEL_CALLBACK,
	XT.BASE_PROC_TYPE(SAVE_DIALOG_CALLBACK'address),
	TO_POINTER_TYPE(CANCEL));
 
    XT.UNMANAGE_CHILD(
	XM_WIDGET.MESSAGE_BOX_GET_CHILD(SAVE_DIALOG, XM_WIDGET.HELP_BUTTON));
 
    XT.REALIZE_WIDGET(TOP_LEVEL);
    XT.APP_MAIN_LOOP(CONTEXT);

exception
    when EXIT_TEXT => null;      -- Leave the main program.
end XM_TEXT;
