-- THIS SOFTWARE IS PROVIDED ''AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
-- WARRANTIES, INCLUDING, WITHOUT LIMITATION, IMPLIED WARRANTIES OF
-- NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- Digital assumes no responsibility AT ALL for the use or reliability
-- of this software.

-- XLIB_HELLO.ADA
--
-- Abstract: Displays Hello World within an X window (click in the window).
--
-- This is a sample program that shows a simple usage of the Ada binding to
-- X Windows. This program shows the basic calls needed to construct
-- a client application based on the binding.
--
-- In specific, this program shows how to:
--
--     Open a display.
--     Get the default Screen.
--     Get the root window of that screen.
--     Create A Window.
--     Create a cursor for that window.
--     Set up input event masks.
--     Set up Icon names.
--     Load a font style.
--     Create a graphic context.
--     Map a window to the display.
--     Set the foreground and backround colors for the window.
--     Read in events from the event queue.
--     Handle those events.
--     Draw strings of text to the window.
--
--==========================================================================
 
with TEXT_IO;
with C_TYPES, X, X_ATOM, X_LIB_SUPPORT, X_LIB;
use  C_TYPES, X, X_ATOM, X_LIB_SUPPORT, X_LIB;
procedure XLIB_HELLO is
 
    DISPLAY_NOT_OPEN         : exception;

    type ACCESS_STRING is access STRING;

    BUFFER                   : ACCESS_STRING := new STRING'(" ");
    DISPLAY_ID               : DISPLAY_TYPE;

    DRAWABLE_WINDOW          : DRAWABLE_PKG.DRAWABLE_TYPE;
    GC_VALUE_MASK            : GC_VALUE_MASK_TYPE := 
	(FONT => true, others => false);
 
    GC_VALUES                : GC_VALUES_TYPE;
 
    HELLO_WORLD_BORDER_WIDTH : WIDTH_PIXEL_COUNT_UNSIGNED_SHORT_SUBTYPE := 2;
 
    WINDOW_X                 : WIDTH_PIXEL_COUNT_SHORT_SUBTYPE  := 5;
    WINDOW_Y                 : HEIGHT_PIXEL_COUNT_SHORT_SUBTYPE := 5;
 
    WINDOW_WIDTH             : WIDTH_PIXEL_COUNT_SHORT_SUBTYPE  := 810;
    WINDOW_HEIGHT            : HEIGHT_PIXEL_COUNT_SHORT_SUBTYPE := 672;
 
    MAIN_ATTRIBUTES          : SET_WINDOW_ATTRIBUTES_TYPE;
    MAIN_SCREEN              : SCREEN_NUMBER_TYPE;
    MAIN_WINDOW              : WINDOW_TYPE;
 
    POINTER_STRING           : STRING(1 .. 8) := "Hi There";
    ROOT_WIND                : WINDOW_TYPE;
    STATUS                   : COMPOSE_STATUS_TYPE;
    THE_SYMBOL               : KEYSYM_TYPE;

    TITLE_X                  : WIDTH_PIXEL_COUNT_SHORT_SUBTYPE;
    TITLE_Y                  : HEIGHT_PIXEL_COUNT_SHORT_SUBTYPE;
    TITLE_STRING             : STRING(1 .. 11) := "Hello World";
 
    LENGTH                   : NATURAL;       --Dummy result.
 
    WINDOW_ATTRIBUTES        : SET_WINDOW_ATTRIBUTES_MASK_TYPE := (
	BACKGROUND_PIXEL => true,
	BORDER_PIXEL	 => true,
	CURSOR		 => true,
	others		 => false );
 
    WINDOW_CONTEXT           : GC_TYPE;
    X_EVENT                  : EVENT_TYPE;
 
    --
    -- Beginning of main procedure
    --
begin
 
-- Opening the display, get the new Display_Id, get the default
-- screen from the environment variable DISPLAY, set up the bounds
-- of the panel, get the root window id and then create a window
-- with our new defaults for the screen.
--
    DISPLAY_ID := OPEN_DISPLAY(NULL_TERMINATED.NULL_POINTER);
 
    if (DISPLAY_ID = NULL_DISPLAY) then
	raise DISPLAY_NOT_OPEN;
    end if;
 
    MAIN_SCREEN := DEFAULT_SCREEN(DISPLAY_ID);
    ROOT_WIND := ROOT_WINDOW(DISPLAY_ID, MAIN_SCREEN);
 
    -- Here we are setting up the bounds for our window
    -- after we set up the bounds, we calculate
    -- the starting position point
    -- from those bounds.
    --
    TITLE_X := WINDOW_WIDTH / 2;
    TITLE_Y := WINDOW_HEIGHT / 2;
 
    -- Create the font cursor with the shape of a dot
    --
    MAIN_ATTRIBUTES.CURSOR := CREATE_FONT_CURSOR(DISPLAY_ID, CURSORS.DOT);
 
    MAIN_ATTRIBUTES.BORDER_PIXEL := BLACK_PIXEL(DISPLAY_ID, MAIN_SCREEN);
    MAIN_ATTRIBUTES.BACKGROUND_PIXEL := WHITE_PIXEL(DISPLAY_ID, MAIN_SCREEN);
 
    MAIN_ATTRIBUTES.BIT_GRAVITY := CENTER;
 
    -- With many of the parameters that we have just populated
    -- we now create the window. The parameter Main_Window will
    -- be the new window_id associated with this new window.
    --
    MAIN_WINDOW := CREATE_WINDOW(
	DISPLAY_ID, ROOT_WIND,
	WINDOW_X,   WINDOW_Y,
	WINDOW_WIDTH, WINDOW_HEIGHT,
	HELLO_WORLD_BORDER_WIDTH,
	DEFAULT_DEPTH(DISPLAY_ID, MAIN_SCREEN),
	INPUT_OUTPUT,
	DEFAULT_VISUAL(DISPLAY_ID, MAIN_SCREEN).all,
	WINDOW_ATTRIBUTES, MAIN_ATTRIBUTES );
 
    -- Set up the event mask for use with the event handler. Then call
    -- Select_Input to make the mask visable to the server.
    --
    MAIN_ATTRIBUTES.EVENT_MASK := (
	EXPOSURE	 => true,
	BUTTON_PRESS	 => true,
	BUTTONS_MOTION   => (BUTTON1 => true, others => false),
	KEY_PRESS	 => true,
	STRUCTURE_NOTIFY => true,
	others		 => false );

    SELECT_INPUT(DISPLAY_ID, MAIN_WINDOW, MAIN_ATTRIBUTES.EVENT_MASK);
 
    --
    -- Set the icon name and the name that would be used as the title
    -- bar if we had one.
    --
    SET_ICON_NAME(
	DISPLAY_ID, MAIN_WINDOW, NULL_TERMINATED.TO_STRING("Hello_World") );
 
    STORE_NAME(
	DISPLAY_ID, MAIN_WINDOW, NULL_TERMINATED.TO_STRING("Hello World") );
 
    DEFINE_CURSOR(DISPLAY_ID, MAIN_WINDOW, MAIN_ATTRIBUTES.CURSOR);
 
    DRAWABLE_WINDOW := DRAWABLE_TYPE(MAIN_WINDOW);
 
    --
    -- Here we set the text font to a 9X15 size and
    -- create a Graphic_Context for the specified drawable
    -- window.
    --
    GC_VALUES.FONT := LOAD_FONT(
	DISPLAY_ID, NULL_TERMINATED.TO_STRING("fixed"));
 
    WINDOW_CONTEXT := CREATE_GC(
	DISPLAY_ID, DRAWABLE_WINDOW, GC_VALUE_MASK, GC_VALUES);
 
    --
    -- Now we are going to Map the Main_Window to the specified display
    --
    MAP_WINDOW(DISPLAY_ID, MAIN_WINDOW);
    FLUSH(DISPLAY_ID);
 
    --
    -- This sets the text foreground color to be black
    -- and the background color to be white
    --
    SET_BACKGROUND(DISPLAY_ID, WINDOW_CONTEXT, 0);
    SET_FOREGROUND(DISPLAY_ID, WINDOW_CONTEXT, 1);
 
    loop
    --
    -- Inside this loop is where we process events. The next call,
    -- Next_Event, returns the next event that is in our event queue.
    -- We may then apply the event kind to a case statement
    -- and perform the processing.
    --
	NEXT_EVENT(DISPLAY_ID, X_EVENT);
 
	case X_EVENT.EVENT_NAME is
 
	    when EXPOSE =>
		if (X_EVENT.EXPOSE_EVENT.COUNT = 0) then
                --
                -- here we will redraw the title string to the window
                -- each time we receive an expose event, we want to draw the
                -- Title_String at the title point
                --
		    DRAW_IMAGE_STRING(
			DISPLAY_ID,
			DRAWABLE_WINDOW,
			WINDOW_CONTEXT,
			TITLE_X, TITLE_Y,
			TITLE_STRING );
		end if;
 
	    when BUTTON_PRESS =>
            --
            -- if a button press event is received by this client then
            -- we want to display the pointer string at those coordinates.
            --
		DRAW_IMAGE_STRING(
		    DISPLAY_ID,
		    DRAWABLE_WINDOW,
		    WINDOW_CONTEXT,
		    X_EVENT.BUTTON_EVENT.X,
		    X_EVENT.BUTTON_EVENT.Y,
		    POINTER_STRING );
 
	    when KEY_PRESS =>
            --
            -- when a key press event is received by this client, we are
            -- finished so we set the flag
            --
		LOOKUP_STRING(LENGTH, X_EVENT, BUFFER.all, THE_SYMBOL, STATUS);
 
		if (BUFFER.all (1) = 'q') then
		    TEXT_IO.PUT_LINE(
			"Program has been terminated with 'q' key" );

		    exit;
		end if;
 
	    when others =>
		null;
	end case;
    end loop;
 
exception
 
    when DISPLAY_NOT_OPEN =>
	TEXT_IO.PUT_LINE("Could not open Display");
 
    when others =>
	TEXT_IO.PUT_LINE("Unexpected exception.");
end XLIB_HELLO;
