--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with CONDITION_HANDLING, SYSTEM;
package ETHERNET_IO_ERROR_MONITOR is
    procedure ERROR(
	STATUS	    : CONDITION_HANDLING.COND_VALUE_TYPE;
	DEV_INFO    : SYSTEM.UNSIGNED_LONGWORD;
	REASON	    : STRING);
end;


with Text_IO, Calendar;
package body ETHERNET_IO_ERROR_MONITOR is
    use CONDITION_HANDLING, SYSTEM;

    type Seen_Info is
	record
	    Status	: CONDITION_HANDLING.COND_VALUE_TYPE;
	    Last_Shown	: Calendar.Time;
	end record;
    Already_Seen : array(1..10) of Seen_Info;
    Already_Seen_Last : Natural := 0;

    procedure ERROR(
	STATUS	    : CONDITION_HANDLING.COND_VALUE_TYPE;
	DEV_INFO    : SYSTEM.UNSIGNED_LONGWORD;
	REASON	    : STRING)
    is
	use Calendar;
	I   : Integer;
	Now : constant Time := Clock;
    begin
	-- See if we know about this code
	I := Already_Seen'first;
	while I <= Already_Seen_Last loop
	    exit when Already_Seen(I).Status = STATUS;
	    I := I+1;
	end loop;

	-- Update our knowlege of it
	if I <= Already_Seen_Last then
	    if Now - Already_Seen(I).Last_Shown < 10.0 then return; end if;
	elsif Already_Seen_Last < Already_Seen'Last then
	    Already_Seen_Last := Already_Seen_Last + 1;
	    I := Already_Seen_Last;
	    Already_Seen(I) := (STATUS, Now);
	else
	    return;
	end if;

	-- Emit msg about it
	Text_IO.Put_Line(
	    "Ethernet error :"&REASON&
	    "STATUS => "&COND_VALUE_TYPE'image(STATUS)&
	    "DEV_INFO => "&SYSTEM.UNSIGNED_LONGWORD'image(DEV_INFO));

	-- Record when this message was shown
	Already_Seen(I).Last_Shown := Now;
    end;
end;



with ETHERNET_IO_ERROR_MONITOR;
with BUG, Calendar, CONDITION_HANDLING,
    Logical_To_Float,
    STARLET, SYSTEM, ETHERNET, LIB,
    TASKING_SERVICES, UNCHECKED_DEALLOCATION,
    CONTROLS;

pragma ELABORATE(
    BUG, Calendar, CONDITION_HANDLING,
    TASKING_SERVICES, UNCHECKED_DEALLOCATION);

package body ETHERNET_IO is

    type REAL_IO_STATUS_TYPE is new STARLET.IO_STATUS_BLOCK_TYPE;
    type REAL_CHANNEL_TYPE   is new STARLET.CHANNEL_TYPE;

    type DESCRIPTOR_TYPE is
	record
	    SIZE    : INTEGER;
	    ADDR    : SYSTEM.ADDRESS;
	end record;
    for DESCRIPTOR_TYPE'size use 64;
    for DESCRIPTOR_TYPE use
	record
	    SIZE    at 0 range 0..31;
	    ADDR    at 4 range 0..31;
	end record;

    -- Avoid Canceling to often
    --
    Time_Of_Last_Cancel : Calendar.Time := Calendar.Clock;
    Min_Time_Between_Cancels : constant Duration
	:= Duration(Logical_To_Float("FCTM_MIN_TIME_BETWEEN_CANCELS", 1.0));

    -- values obtained from $NMADEF in SYS$SHARE:LIB.MLB
    --
    NMA_PCLI_ACC    : constant := 2846;
    NMA_ACC_SHR     : constant := 1;
    NMA_ACC_LIM     : constant := 2;
    NMA_ACC_EXC     : constant := 3;

    NMA_PCLI_BFN    : constant := 1105;

    NMA_PCLI_BUS    : constant := 2801;

    NMA_PCLI_CON    : constant := 1110;
    NMA_LINCN_NOR   : constant := 0;
    NMA_LINCN_LOO   : constant := 1;

    NMA_PCLI_PHA    : constant := 16#1B04#;

    NMA_PCLI_PTY    : constant := 2830; 

    NMA_PCLI_PAD    : constant := 2842;
    NMA_STATE_ON    : constant := 0;
    NMA_STATE_OFF   : constant := 1;

    type SETMODE_DATA_ITEM_TYPE is
	record
	    CODE  : SYSTEM.UNSIGNED_WORD;
	    VALUE : SYSTEM.UNSIGNED_LONGWORD;
	end record;
	pragma PACK(SETMODE_DATA_ITEM_TYPE);
	for SETMODE_DATA_ITEM_TYPE'size use 6*8;

    type SETMODE_DATA_TYPE is
	array(POSITIVE range <>) of SETMODE_DATA_ITEM_TYPE;
	pragma PACK(SETMODE_DATA_TYPE);


    procedure INIT_AS_OKAY(IO_STATUS : in out IO_STATUS_TYPE) is
    begin
	if IO_STATUS = null then
	    IO_STATUS := new REAL_IO_STATUS_TYPE;
	end if;
	IO_STATUS.STATUS := 1;
    end;

    procedure ASSIGN(CHANNEL : in out CHANNEL_TYPE) is
	type VECTOR_OF_STRING4 is array(POSITIVE range <>) of STRING(1..4);

	ETHERNET_DEVICES    : constant VECTOR_OF_STRING4
			    := ("XQA0", "ESA0", "ETA0", "EZA0", "ISA0");

	STATUS : CONDITION_HANDLING.COND_VALUE_TYPE;
    begin
	if CHANNEL = null then
	    CHANNEL := new REAL_CHANNEL_TYPE;
	end if;

	for I in ETHERNET_DEVICES'range loop
	    STARLET.ASSIGN(STATUS, ETHERNET_DEVICES(I),
		STARLET.CHANNEL_TYPE(CHANNEL.all));
	    exit when CONDITION_HANDLING.SUCCESS(STATUS);
	end loop;

	if not CONDITION_HANDLING.SUCCESS(STATUS) then
	    BUG("Unable to $ASSIGN to Ethernet device " &
		ETHERNET_DEVICES(1) & ", or " &
		ETHERNET_DEVICES(2) & ", or " &
		ETHERNET_DEVICES(3) & ", or " &
		ETHERNET_DEVICES(4) & ", or " &
		ETHERNET_DEVICES(5)
		);
	end if;
    end;

    procedure STARTUP(CHANNEL : in out CHANNEL_TYPE) is

	SETMODE_DATA : constant SETMODE_DATA_TYPE
	    := ((NMA_PCLI_PTY, PROTOCOL),	-- see 6.2.1.1 ETHERNET/802
						-- device driver manual
		(NMA_PCLI_BUS, SYSTEM.UNSIGNED_LONGWORD(
				    ETHERNET.MAX_MESSAGE_TYPE_SIZE/8)),
						-- 6.4.1,   same manual
		(NMA_PCLI_PAD, NMA_STATE_OFF),	-- 6.2.1.2, same manual
		(NMA_PCLI_BFN, 10));		-- 6.4.3.1, same manual

	SETMODE_DESCRIPTOR : DESCRIPTOR_TYPE
	    := (SIZE => SETMODE_DATA'size/8,
		ADDR => SETMODE_DATA'address);

	STATUS	: CONDITION_HANDLING.COND_VALUE_TYPE;
	IOSB	: STARLET.IO_STATUS_BLOCK_TYPE;
    begin

	STARLET.QIOW (
	    STATUS	=> STATUS,
	    CHAN	=> STARLET.CHANNEL_TYPE(CHANNEL.all),
	    IOSB	=> IOSB,
	    FUNC	=> SYSTEM.UNSIGNED_WORD(
			    STARLET.IO_SETMODE
			    + STARLET.IO_M_CTRL
			    + STARLET.IO_M_STARTUP),
	    P2		=> SYSTEM.TO_UNSIGNED_LONGWORD(
			    SETMODE_DESCRIPTOR'address));

	if not CONDITION_HANDLING.SUCCESS(STATUS)
	or not CONDITION_HANDLING.SUCCESS(IOSB.STATUS)
	then
	    BUG("Unable to set ethernet device to correct protocol, size, etc");
	end if;
    end;

    function GET_ETHERNET_PHYSICAL_ADDRESS(CHANNEL : CHANNEL_TYPE)
	return ETHERNET_PHYSICAL_ADDRESS_TYPE is

	BUFFER	: SYSTEM.UNSIGNED_BYTE_ARRAY(0..1023); pragma VOLATILE(BUFFER);
	BUFFER_DESCRIPTOR : constant DESCRIPTOR_TYPE
	    := (SIZE => BUFFER'length,
		ADDR => BUFFER'address);

	STATUS	: CONDITION_HANDLING.COND_VALUE_TYPE;
	I	: INTEGER;
	IOSB    : STARLET.IO_STATUS_BLOCK_TYPE;

    begin
	STARLET.QIOW (
	    STATUS	=> STATUS,
	    CHAN	=> STARLET.CHANNEL_TYPE(CHANNEL.all),
	    IOSB	=> IOSB,
	    FUNC	=> SYSTEM.UNSIGNED_WORD(
			    STARLET.IO_SENSEMODE + STARLET.IO_M_CTRL),
	    P2		=> SYSTEM.TO_UNSIGNED_LONGWORD(
			    BUFFER_DESCRIPTOR'address));

	I := BUFFER'first;
	while I < INTEGER(IOSB.COUNT)
	and then INTEGER(BUFFER(I+1))*256+INTEGER(BUFFER(I)) /= NMA_PCLI_PHA
	loop
	    if SYSTEM.TO_BIT_ARRAY_8(BUFFER(I+1))(4) then
		I := I + 4 + INTEGER(BUFFER(I+3))*256+INTEGER(BUFFER(I+2));
	    else
		I := I + 6;
	    end if;
	end loop;

	if I >= INTEGER(IOSB.COUNT) then
	    BUG("Bad format for ethernet SENSEMODE result");
	end if;

	declare
	    P : ETHERNET_PHYSICAL_ADDRESS_TYPE;
	    for P use at BUFFER(I+4)'address;
	begin
	    return P;
	end;
    end;

    function BAD_COND(STATUS : CONDITION_HANDLING.COND_VALUE_TYPE)
	return BOOLEAN
    is
	use SYSTEM;
	RESULT : UNSIGNED_LONGWORD;
    begin
	if CONDITION_HANDLING.SUCCESS(STATUS) then return false; end if;
	LIB.MATCH_COND(RESULT, STATUS, STARLET.SS_CANCEL);
	return RESULT /= 0;
    end;


    package ASYNC_WRITE is

	type REQUEST(NUM_BYTES : NATURAL);
	type ACCESS_REQUEST is access REQUEST;

	type REQUEST(NUM_BYTES : NATURAL) is
	    record
		NEXT	    : ACCESS_REQUEST;
		CHANNEL     : CHANNEL_TYPE;
		DESTINATION : ETHERNET_PHYSICAL_ADDRESS_TYPE;
		BYTES	    : SYSTEM.UNSIGNED_BYTE_ARRAY(1..NUM_BYTES);
	    end record;

	procedure DEALLOCATE is new Unchecked_Deallocation(
	    Object => REQUEST,
	    Name   => ACCESS_REQUEST);

	procedure INSERT(R : in out ACCESS_REQUEST);
    end;

    package body ASYNC_WRITE is

	task BUFFER is
	    entry INSERT(R : in out ACCESS_REQUEST);
	    entry REMOVE(R : out ACCESS_REQUEST);
	end;

	procedure INSERT(R : in out ACCESS_REQUEST) is
	begin
	    BUFFER.INSERT(R);
	    R := null;
	exception
	    when TASKING_ERROR =>
		DEALLOCATE(R);
		if not CONTROLS.EXIT_PRESSED then
		    raise;
		end if;
	end;

	task WRITER;

	task body WRITER is
	    AR : ACCESS_REQUEST;
	begin
	    loop
	<<RESTART>>
		exit when CONTROLS.EXIT_PRESSED;
		begin
		    select
			BUFFER.REMOVE(AR);
		    or
		    	delay 2.0;
		    	goto RESTART;
		    end select;
		exception
		    when TASKING_ERROR => exit;
		end;

		declare
		    R : REQUEST renames AR.all;
		    CHANNEL	: CHANNEL_TYPE renames R.CHANNEL;
		    NUM_BYTES   : NATURAL renames R.NUM_BYTES;
		    BYTES	: SYSTEM.ADDRESS := R.BYTES'address;
		    DESTINATION : ETHERNET_PHYSICAL_ADDRESS_TYPE
				    renames R.DESTINATION;
		    STATUS	: CONDITION_HANDLING.COND_VALUE_TYPE;
		    IOSB	: REAL_IO_STATUS_TYPE;
		begin

		    TASKING_SERVICES.TASK_QIOW (
			STATUS  => STATUS,
			CHAN    => STARLET.CHANNEL_TYPE(CHANNEL.all),
			IOSB    => STARLET.IO_STATUS_BLOCK_TYPE(IOSB),
			FUNC    => SYSTEM.UNSIGNED_WORD(STARLET.IO_WRITEVBLK),
			P1	=> SYSTEM.TO_UNSIGNED_LONGWORD(BYTES),
			P2	=> SYSTEM.UNSIGNED_LONGWORD(NUM_BYTES),
			P5	=> SYSTEM.TO_UNSIGNED_LONGWORD(DESTINATION'address));

		    if BAD_COND(STATUS) then
			ETHERNET_IO_ERROR_MONITOR.ERROR(
			    STATUS,
			    IOSB.DEV_INFO,
			    "Status from QIO bad");

		    elsif BAD_COND(CONDITION_HANDLING.COND_VALUE_TYPE(IOSB.STATUS)) then
			ETHERNET_IO_ERROR_MONITOR.ERROR(
			    CONDITION_HANDLING.COND_VALUE_TYPE(IOSB.STATUS),
			    IOSB.DEV_INFO,
			    "Status from IOSB bad");

			declare
			    use Calendar;
			    Now : Time := Clock;
			begin
			    if Now-Time_Of_Last_Cancel > Min_Time_Between_Cancels then
				STARLET.CANCEL(
				    STATUS  => STATUS,
				    CHAN    => STARLET.CHANNEL_TYPE(CHANNEL.all));
				Time_Of_Last_Cancel := Now;
			    end if;
			end;
		    end if;
		end;

		DEALLOCATE(AR);
	    end loop;
	end;


	task body BUFFER is
	    THROWN_AWAY : NATURAL := 0;
	    MAX_PENDING_COUNT : constant POSITIVE := 20;
	    PENDING_COUNT : NATURAL := 0;
	    PENDING_HEAD,
	    PENDING_TAIL  : ACCESS_REQUEST := null;
	begin
	    while not CONTROLS.EXIT_PRESSED loop
		select
		    when not CONTROLS.EXIT_PRESSED =>
			accept INSERT(R : in out ACCESS_REQUEST) do
				-- Remember the request
				if PENDING_TAIL /= null then
				    PENDING_TAIL.NEXT := R;
				else
				    PENDING_HEAD := R;
				end if;
				PENDING_TAIL := R;
				R := null;
			    end INSERT;
			    PENDING_COUNT := PENDING_COUNT+1;

			    -- Throw away old messages when we get too many
			    while PENDING_COUNT>MAX_PENDING_COUNT loop
				ETHERNET_IO_ERROR_MONITOR.ERROR(0,0,
				    "Throwing away old messages");
				declare
				    NEXT : ACCESS_REQUEST := PENDING_HEAD.NEXT;
				begin
				    THROWN_AWAY := THROWN_AWAY+1;
				    DEALLOCATE(PENDING_HEAD);
				    PENDING_HEAD := NEXT;
				    PENDING_COUNT := PENDING_COUNT-1;
				end;
			    end loop;

		    or when PENDING_COUNT>0 =>
			accept REMOVE(R : out ACCESS_REQUEST) do
				R := PENDING_HEAD;
				PENDING_HEAD := PENDING_HEAD.NEXT;
			    end REMOVE;
			    PENDING_COUNT := PENDING_COUNT-1;
			    if PENDING_COUNT = 0 then
				PENDING_TAIL := null;
			    end if;
		    or
			delay 2.0;
		end select;
	    end loop;

	    raise TASKING_ERROR;
	exception
	    when TASKING_ERROR =>
		if THROWN_AWAY > 0 then
		    ETHERNET_IO_ERROR_MONITOR.ERROR(0,1,
			"Total thrown away ="&Integer'image(THROWN_AWAY));
		end if;
		if not CONTROLS.EXIT_PRESSED then
		    raise;
		end if;
	end;

    end;

    procedure SYNC_AND_WRITE(
	IO_STATUS   : in IO_STATUS_TYPE;
	CHANNEL     : in CHANNEL_TYPE;
	NUM_BYTES   : in NATURAL;
	BYTES	    : in SYSTEM.ADDRESS;
	DESTINATION : in ETHERNET_PHYSICAL_ADDRESS_TYPE)
    is
	ACTUAL_BYTES : SYSTEM.UNSIGNED_BYTE_ARRAY(1..NUM_BYTES);
	    for ACTUAL_BYTES use at BYTES;

	R : ASYNC_WRITE.ACCESS_REQUEST := new ASYNC_WRITE.REQUEST(NUM_BYTES);
    begin
	R.NEXT		:= null;
	R.CHANNEL	:= CHANNEL;
	R.DESTINATION	:= DESTINATION;
	R.BYTES 	:= ACTUAL_BYTES;
	ASYNC_WRITE.INSERT(R);
    end;

    function DID_READ_NOW(
	IO_STATUS   : in IO_STATUS_TYPE;
	CHANNEL     : in CHANNEL_TYPE;
	NUM_BYTES   : in NATURAL;
	BYTES       : in SYSTEM.ADDRESS;
	PKT_HEAD    : in SYSTEM.ADDRESS) return BOOLEAN is

	STATUS	    : CONDITION_HANDLING.COND_VALUE_TYPE;
    begin
	STARLET.QIOW (
	    STATUS	=> STATUS,
	    CHAN	=> STARLET.CHANNEL_TYPE(CHANNEL.all),
	    IOSB	=> STARLET.IO_STATUS_BLOCK_TYPE(IO_STATUS.all),
	    FUNC	=> SYSTEM.UNSIGNED_WORD(
			    STARLET.IO_READVBLK+STARLET.IO_M_NOW),
	    P1		=> SYSTEM.TO_UNSIGNED_LONGWORD(BYTES),
	    P2		=> SYSTEM.UNSIGNED_LONGWORD(NUM_BYTES),
	    P5		=> SYSTEM.TO_UNSIGNED_LONGWORD(PKT_HEAD));

	return  CONDITION_HANDLING.SUCCESS(STATUS)
	    and CONDITION_HANDLING.SUCCESS(IO_STATUS.STATUS);
    end;

end;
