#include <sys/socket.h>
#include <sys/time.h>	/* for timeval struct in pfilt.h */
#include <sys/file.h>
#include <sys/errno.h>
#include <sys/types.h>
#include <net/if.h>
#include <net/pfilt.h>
#include <stdio.h>
#include <string.h>

#include <netinet/in.h>
#include <netinet/if_ether.h>

#define	PFPREFIX	"/dev/pf/pfilt"		/* prefix for device names */
#define	PFMAXMINORS	256			/* 8-bit minor device field */
extern int errno;

/*
 * pfopen(ifname, flags): to support access to the Ethernet Packet Filter.
 * (using kernel options PACKETFILTER, pseudo-device packetfilter)
 *
 * ifname is a ptr to the Ethernet device name ("ln0", "xna1", "pf0", etc.)
 *	or NULL for default
 * flags are passed to the open() system call.
 *
 * return value:
 *	special device file descriptor on success
 *	-1 on failure with errno set to indicate the error
 *
 */
static pfopen(ifname, flags)
char *ifname;			/* "ln0", "pf0", etc. or NULL */
int flags;
{
	int i;			/* loop counter */
	int fd;			/* file descriptor */
	char tryname[128];	/* device name: "/dev/pf/pfiltnn" */
	static int setif();

	if (ifname && (ifname[0] == 0))
	    ifname = NULL;	/* change empty string to NULL string */

	/* find next available device under the /dev/pf directory */
	for (i = 0; i < PFMAXMINORS; i++) {
		sprintf(tryname, "%s%d", PFPREFIX, i);
		fd = open(tryname, flags, 0);
		if (fd < 0) {
			switch (errno) {
			case EBUSY:	/* device in use */
				continue;	/* try the next entry */
			case ENOENT:	/* ran out of filenames */
			case ENXIO:	/* no more configured in kernel */
			default:	/* something else went wrong */
				return(-1);
			}
		}
		/* open succeeded, set the interface name */
		return(setif(fd, ifname));
	}
	return(-1);	/* didn't find an openable device */
}

static int setif(fd, ifname)
    int fd;
    char *ifname;
{
	if (ifname == NULL)	/* use default */
	    return(fd);

	if (ioctl(fd, EIOCSETIF, ifname) < 0) {
		close(fd);
		return(-1);
	}
	/* return the file descriptor */
	return(fd);
}

/*----------------------------------------------------------------------------*/

static char ifname[256],buf[4096];
static int fd;

static void err(s)
    char *s;
{
    perror(s);
    exit(1);
}

static struct endevp param;

void fctm_get_ethernet_physical_address(char *pa)
{   
    if(ioctl(fd, EIOCDEVP, &param))
	err("ioctl: EIOCDEVP");
    memcpy(pa, (char *)(&param.end_addr[0]), 6);
    return;
}

void fctm_ethernet_write(int cnt, char *bytes, char *pa)
{   short protocol = htons(0x6006);

    memcpy(&buf[0],pa,6);
    memcpy(&buf[6],(char *)(&param.end_addr[0]),6);
    memcpy(&buf[12],(char *)(&protocol),2);
    memcpy(&buf[14],bytes,cnt);

    if (write(fd,buf,cnt+14) < 0)
	err("write : ");

    return;
}

#define PKT_HEAD_LEN (sizeof(struct ether_header))

int fctm_ethernet_read(int num_bytes, char *bytes, struct ether_header *pkt_head)
{
	int i;

	if ((i=read(fd,buf,4096)) < 0)
		err("read");

	if (i<PKT_HEAD_LEN)
		return 0;       /* not enough to read */

	memcpy(pkt_head, &buf[0], PKT_HEAD_LEN);
	memcpy(bytes, &buf[PKT_HEAD_LEN], i-PKT_HEAD_LEN);

	return 1;
}
    

static struct enfilter filter;

void fctm_ethernet_io_assign()
{   u_int timeout = -1;
    u_short bic = (ENTSTAMP|ENBATCH|ENPROMISC|ENCOPYALL|ENNONEXCL|ENBPFHDR);

    ifname[0] = NULL;

    if ((fd = pfopen(ifname, O_RDWR) ) < 0)
	    err("pfopen O_RDONLY:");

    if (ioctl(fd, EIOCMBIC, &bic))
	    err("ioctl: EIOCMBIC");

    filter.enf_Priority = 37;       /* anything > 2 should work */
    filter.enf_FilterLen = 0;

    /* packet type is last short in header */
    filter.enf_Filter[filter.enf_FilterLen++] =
	    ENF_PUSHWORD + 6;
    filter.enf_Filter[filter.enf_FilterLen++] =
	    ENF_PUSHLIT | ENF_CAND;
    filter.enf_Filter[filter.enf_FilterLen++] =
	    htons(0x6006);	/* match all 0x6006 packets */

    if (ioctl(fd, EIOCSETF, &filter))
	err("ioctl: EIOCSETF");

    if (ioctl(fd, EIOCSRTIMEOUT, &timeout))
	err("ioctl: EIOCSRTIMEOUT");

}
