--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    SCALE_TYPE_TRIG;

pragma ELABORATE(
    SCALE_TYPE_TRIG);

package body VIEW_DIRECTION_MANAGER is
    use SCALAR_PHYSICS;

    FIRST_DIRECTION : constant DIRECTION_INDEX_TYPE := 0;
    FORWARD : constant DIRECTION_INDEX_TYPE := FIRST_DIRECTION;
    BACK    : constant DIRECTION_INDEX_TYPE := 1;
    LEFT    : constant DIRECTION_INDEX_TYPE := 2;
    RIGHT   : constant DIRECTION_INDEX_TYPE := 3;
    UP	    : constant DIRECTION_INDEX_TYPE := 4;
    DOWN    : constant DIRECTION_INDEX_TYPE := 5;
    LAST_DIRECTION : constant DIRECTION_INDEX_TYPE := DOWN;

    type JUMP_TYPE is (JUMP_LEFT, JUMP_UP, JUMP_RIGHT);

    SWIVEL_TABLE : constant array(FIRST_DIRECTION..LAST_DIRECTION,
				  JUMP_TYPE) of DIRECTION_INDEX_TYPE
	    := (FORWARD => (LEFT, UP     , RIGHT),
	        BACK    => (LEFT, FORWARD, RIGHT),
	        LEFT	=> (BACK, FORWARD, RIGHT),
	        RIGHT	=> (LEFT, FORWARD, BACK ),
	        UP	=> (LEFT, FORWARD, RIGHT),
	        DOWN	=> (LEFT, FORWARD, RIGHT));

    VIEW_BASIS_TABLE : constant array(FIRST_DIRECTION..LAST_DIRECTION)
	of OBJECT_PHYSICS.POSITION_BASIS
	:= (FORWARD =>
		(( 1.0,  0.0,  0.0), ( 0.0,  1.0,  0.0), ( 0.0,  0.0,  1.0)),
	    BACK    =>
		((-1.0,  0.0,  0.0), ( 0.0, -1.0,  0.0), ( 0.0,  0.0,  1.0)),
	    LEFT    =>
		(( 0.0,  1.0,  0.0), (-1.0,  0.0,  0.0), ( 0.0,  0.0,  1.0)),
	    RIGHT   =>
		(( 0.0, -1.0,  0.0), ( 1.0,  0.0,  0.0), ( 0.0,  0.0,  1.0)),
	    UP	    =>
		(( 1.0,  0.0,  0.0), ( 0.0,  0.0,  1.0), ( 0.0, -1.0,  0.0)),
	    DOWN    => 
		(( 1.0,  0.0,  0.0), ( 0.0,  0.0, -1.0), ( 0.0,  1.0,  0.0))
		);


    procedure SWIVEL(
	VIEW_DIRECTION	    : in out VIEW_DIRECTION_TYPE;
	CONTROL 	    : CONTROL_TYPE;
	ELAPSED_TIME	    : SCALAR_PHYSICS.SECONDS := 0.0) is
    begin
	if VIEW_SMOOTH then
	    declare
		use TRIG;

		SECONDS_TO_MAX_SWIVEL : constant SCALAR_PHYSICS.SECONDS := 0.3;

		ANGLE_UP_RATE    : constant SCALE_TYPE -- RADIANS_PER_SECOND
				 := RADIANS_PER_CIRCLE/3.0;
		ANGLE_UP_LIMIT	 : constant RADIANS
				 := RADIANS_PER_CIRCLE/4.0;
		ANGLE_DOWN_LIMIT : constant RADIANS
				 := RADIANS_PER_CIRCLE/12.0;

		ANGLE_RIGHT_RATE : constant SCALE_TYPE -- RADIANS_PER_SECOND
				 := RADIANS_PER_CIRCLE/3.0;
		ANGLE_RIGHT_LIMIT: constant RADIANS
				 := RADIANS_PER_CIRCLE/2.2;

		ANGLE_UP_OFF_J	  : RADIANS := VIEW_DIRECTION.ANGLE_UP_OFF_J;
		ANGLE_RIGHT_OFF_J : RADIANS := VIEW_DIRECTION.ANGLE_RIGHT_OFF_J;

		SWIVEL_RIGHT_RATE : SCALE_TYPE;
		SWIVEL_UP_RATE	  : SCALE_TYPE;

		procedure ADJUST(
		    OFF     : in out RADIANS;
		    RATE    : SCALE_TYPE;
		    POSITIVE_LIMIT, NEGATIVE_LIMIT : RADIANS) is

		    LIMIT   : RADIANS;

		begin
		    OFF := OFF + RADIANS(RATE*SCALE_TYPE(ELAPSED_TIME));

		    if VIEW_DIRECTION.CONTROL.CONTROL_STATE = TO_FRONT then
			if (OFF > 0.0) = (RATE > 0.0) then
			    OFF := 0.0;
			end if;

		    elsif RATE > 0.0 then
			if OFF > POSITIVE_LIMIT then
			    OFF := POSITIVE_LIMIT;
			end if;

		    else
			if OFF < NEGATIVE_LIMIT then
			    OFF := NEGATIVE_LIMIT;
			end if;
		    end if;
		end;

	    begin
		-- Record the new control setting, if any
		--
		if CONTROL.CONTROL_STATE /= UNCHANGED then
		    VIEW_DIRECTION.CONTROL := CONTROL;
		end if;

		-- Compute the swivel rates
		--
		case VIEW_DIRECTION.CONTROL.CONTROL_STATE is

		    when STOP|UNCHANGED =>
			VIEW_DIRECTION.ELAPSED_TIME := 0.0;
			VIEW_DIRECTION.CONTROL := CONTROL;
			return;

		    when TO_FRONT =>
			if ANGLE_RIGHT_OFF_J = 0.0
			and ANGLE_UP_OFF_J   = 0.0
			then
			    VIEW_DIRECTION.ELAPSED_TIME := 0.0;
			    VIEW_DIRECTION.CONTROL := (CONTROL_STATE => STOP);
			    return;
			end if;

			if ANGLE_RIGHT_OFF_J > 0.0 then
			    SWIVEL_RIGHT_RATE := -1.0;
			elsif ANGLE_RIGHT_OFF_J < 0.0 then
			    SWIVEL_RIGHT_RATE := 1.0;
			else
			    SWIVEL_RIGHT_RATE := 0.0;
			end if;

			if ANGLE_UP_OFF_J > 0.0 then
			    SWIVEL_UP_RATE := -1.0;
			elsif ANGLE_UP_OFF_J < 0.0 then
			    SWIVEL_UP_RATE :=  1.0;
			else
			    SWIVEL_UP_RATE := 0.0;
			end if;

		    when TO_ARBITRARY =>
			SWIVEL_RIGHT_RATE   :=
			    SCALE_TYPE(VIEW_DIRECTION.CONTROL.SWIVEL_RIGHT_LEFT);
			SWIVEL_UP_RATE	    :=
			    SCALE_TYPE(VIEW_DIRECTION.CONTROL.SWIVEL_UP_DOWN);

		end case;

		VIEW_DIRECTION.ELAPSED_TIME :=
		    VIEW_DIRECTION.ELAPSED_TIME + ELAPSED_TIME;

		if SCALAR_PHYSICS."<"(VIEW_DIRECTION.ELAPSED_TIME,
		    SECONDS_TO_MAX_SWIVEL)
		then
		    declare
			use SCALAR_PHYSICS;
			PARTIAL_RATE : constant SCALE_TYPE :=
			    SCALE_TYPE(
				VIEW_DIRECTION.ELAPSED_TIME/
				SECONDS_TO_MAX_SWIVEL)**2;
		    begin
			SWIVEL_RIGHT_RATE := SWIVEL_RIGHT_RATE * PARTIAL_RATE;
			SWIVEL_UP_RATE    := SWIVEL_UP_RATE * PARTIAL_RATE;
		    end;
		end if;

		-- Obey the swivel rates
		--
		ADJUST(
		    ANGLE_RIGHT_OFF_J,
		    SWIVEL_RIGHT_RATE*ANGLE_RIGHT_RATE,
		    ANGLE_RIGHT_LIMIT,
		    -ANGLE_RIGHT_LIMIT);

		ADJUST(
		    ANGLE_UP_OFF_J,
		    SWIVEL_UP_RATE*ANGLE_UP_RATE,
		    ANGLE_UP_LIMIT,
		    -ANGLE_DOWN_LIMIT);

		VIEW_DIRECTION.ANGLE_UP_OFF_J :=
		    ANGLE_UP_OFF_J;

		VIEW_DIRECTION.ANGLE_RIGHT_OFF_J :=
		    ANGLE_RIGHT_OFF_J;

		declare
		    use SCALE_TYPE_TRIG;
		    SU,
		    CU,
		    SR,
		    CR : METRES;

		    O : OBJECT_PHYSICS.POSITION_BASIS
			renames VIEW_DIRECTION.ORIENTATION;
		begin
		    SU := METRES(SIN(ANGLE_UP_OFF_J));
		    CU := METRES(COS(ANGLE_UP_OFF_J));
		    SR := METRES(SIN(ANGLE_RIGHT_OFF_J));
		    CR := METRES(COS(ANGLE_RIGHT_OFF_J));
		    O.I := ( CR,      -SR, 0.0);
		    O.J := ( CU*SR, CU*CR,  SU);
		    O.K := (-SU*SR,-SU*CR,  CU);
		end;

	    end;
	else
	    case CONTROL.CONTROL_STATE is
		when UNCHANGED|STOP =>
		    null;
		when TO_FRONT =>
		    VIEW_DIRECTION := DEFAULT_VIEW_DIRECTION;
		when TO_ARBITRARY =>
		    declare
			I : JUMP_TYPE;
		    begin

			if	CONTROL.SWIVEL_RIGHT_LEFT < 0.0 then
			    I := JUMP_LEFT; 
			elsif	CONTROL.SWIVEL_RIGHT_LEFT > 0.0 then
			    I := JUMP_RIGHT;
			elsif	CONTROL.SWIVEL_UP_DOWN	/= 0.0  then
			    I := JUMP_UP;
			else
			    return;
			end if;

			VIEW_DIRECTION :=
			    VIEW_DIRECTION_TYPE'(FALSE,
				SWIVEL_TABLE(
				    VIEW_DIRECTION.DIRECTION_INDEX,
				    I));
		    end;
	    end case;
	end if;
    end;

    function VIEW_ORIENTATION(
	VIEW_DIRECTION	    : VIEW_DIRECTION_TYPE;
	RELATIVE_TO	    : WORLD_PHYSICS.POSITION_BASIS)
	return WORLD_PHYSICS.POSITION_BASIS is

	ORIENTATION	    : WORLD_PHYSICS.POSITION_BASIS;

	procedure PROJECT_OBJECT_TO_WORLD_SAME_ORIGIN(
	    O	: in OBJECT_PHYSICS.POSITION;
	    W	: out WORLD_PHYSICS.POSITION) is

	    R	: WORLD_PHYSICS.POSITION_BASIS renames RELATIVE_TO;

	    O_I : constant METRES := O.I;
	    O_J : constant METRES := O.J;
	    O_K : constant METRES := O.K;
	begin
	    W.I := O_I*R.I.I+O_J*R.J.I+O_K*R.K.I;
	    W.J := O_I*R.I.J+O_J*R.J.J+O_K*R.K.J;
	    W.K := O_I*R.I.K+O_J*R.J.K+O_K*R.K.K;
	end;

	procedure COMPUTE_ORIENTATION(FROM : OBJECT_PHYSICS.POSITION_BASIS) is
	begin
	    PROJECT_OBJECT_TO_WORLD_SAME_ORIGIN(
		FROM.I, ORIENTATION.I);
	    PROJECT_OBJECT_TO_WORLD_SAME_ORIGIN(
		FROM.J, ORIENTATION.J);
	    PROJECT_OBJECT_TO_WORLD_SAME_ORIGIN(
		FROM.K, ORIENTATION.K);
	end;

    begin
	if VIEW_SMOOTH then
	    COMPUTE_ORIENTATION(
		VIEW_DIRECTION.ORIENTATION);
	else
	    COMPUTE_ORIENTATION(
		VIEW_BASIS_TABLE(VIEW_DIRECTION.DIRECTION_INDEX));
	end if;

	return ORIENTATION;
    end;

end;
