--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with LOGICAL_TO_FLOAT, APPEARANCE_MANAGER;
package body VARIOUS_AEROPLANES_DATA is
    use TRIG;


    LOD_DISTANCE : constant METRES :=
	METRES(LOGICAL_TO_FLOAT("FCTM_LOD", 6_000.0));


    AF_C	: constant AEROFOIL_NAME_TYPE := (14, "Clark-y (1922)");
    AF_F4	: constant AEROFOIL_NAME_TYPE := ( 3, "F-4");
    AF_MiG21	: constant AEROFOIL_NAME_TYPE := ( 6, "MiG-21");

    AF_Spit	: constant AEROFOIL_NAME_TYPE := ( 4, "Spit");
    AF_P47	: constant AEROFOIL_NAME_TYPE := ( 3, "P47");
    AF_P51	: constant AEROFOIL_NAME_TYPE := ( 3, "P51");
    AF_BF109	: constant AEROFOIL_NAME_TYPE := ( 5, "BF109");
    AF_FW190	: constant AEROFOIL_NAME_TYPE := ( 5, "FW190");

    AF_DFLT	: constant AEROFOIL_NAME_TYPE := AF_F4;


    type KNOWN_PLANES is
	(F15, F16, F18, F5, F4, MiG21, Jaguar, MiG25,
	 Spit, P47, P51, BF109, FW190,
	 FkDVII);


    DEFAULT_APPEARANCE_INFO
	: constant INFO_MANAGER.APPEARANCE_INFO_TYPE
        := APPEARANCE_MANAGER.CREATE(
		APPEARANCE_MANAGER.PREALLOCATED_APPEARANCES_PKG.AEROPLANES);

    type KNOWN_PLANES_TO_APPEARANCE_INFO_TYPE is
	array(KNOWN_PLANES) of INFO_MANAGER.APPEARANCE_INFO_TYPE;

    APPEARANCE_INFO : constant KNOWN_PLANES_TO_APPEARANCE_INFO_TYPE
	:= KNOWN_PLANES_TO_APPEARANCE_INFO_TYPE'(
	    MiG21|FkDVII|Spit|F4|F15|F16|F18
			=> APPEARANCE_MANAGER.CREATE(
			    APPEARANCE_MANAGER.PREALLOCATED_APPEARANCES_PKG.
				AEROPLANES),
	    others	=> DEFAULT_APPEARANCE_INFO);


    -- Various engines
    --
    J79 	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("J79");
    F404	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("F404");
    F100	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("F100");
    TF30	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("TF30");
    RB199	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("RB199");
    ADOUR	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("ADOUR");
    M53_5	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("M53_5");
    TUMANSKY_R11F   : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("TUMANSKY_R11F");
    TUMANSKY_R25    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("TUMANSKY_R25");
    TUMANSKY_R33    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("TUMANSKY_R33");
    TUMANSKY_RD_F   : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("TUMANSKY_RD_F");
    WW1_A1	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("WW1_A1");
    WW1_E1	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("WW1_E1");
    WW2_A1	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("WW2_A1");
    WW2_E1	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("WW2_E1");
    GRIFFON	    : constant ENGINE_HANDLE_TYPE := ENGINE_TYPE_VALUE("GRIFFON");


    METRES_PER_FOOT : constant := 0.3048;
    NEWTONS_PER_LB  : constant := 0.4536*9.8;
    NPM2_PER_LBPFT2 : constant := (NEWTONS_PER_LB)/(METRES_PER_FOOT**2);


    VIEW_LOCATION_UNSPECIFIED : constant OBJECT_PHYSICS.POSITION
	:= (0.0, 0.0, 0.0);

    DEGREES_TO_RADIANS : constant RADIANS := RADIANS(2.0*PI/360.0);

    -- The following table is from	MODERN COMBAT AIRCRAFT DESIGN,
    --					ISBN 0-087021-426-8
    --					Naval Institute Press
    --
    -- Aerofoil_name added by me
    -- Extra_zero_drag added by me
    -- Dihedral added by me
    -- Wing Placement added by me
    -- G limits, collision radii, etc. added by me
    --
    -- The first two Extra_Radii must be the wings, because the collision code
    -- decreases the area of these for shots off the K axis.
    --
    ACCESS_PLANE_DATA : constant ACCESS_VECTOR_OF_PLANE_DATA_TYPE := new VECTOR_OF_PLANE_DATA_TYPE'(
  ("F-15  ", APPEARANCE_INFO(F15), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00, 13.04, 59.3, 2.88, 0.23, 0.0, 00.0, 45.0,
    06.6, 3.0, 3556.0, FS, 1.17, 0.46, 0.18, 0.54, 0.21, 0.83, 0.19, 0.16, 2,
    F100,
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.8, ( 0.0,  0.0,  0.0)),
			(0.8, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 200.0,
    FLAP_G_LIMIT    => 13.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

  ("F-16  ", APPEARANCE_INFO(F16), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00,  9.14, 27.9, 3.00, 0.21, 0.0, 00.0, 40.0,
    04.0, 4.0, 3449.0, FS, 1.39, 0.34, 0.21, 0.47, 0.29, 1.01, 0.20, 0.20, 1,
    F100, 
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.5, ( 0.0,  0.0,  0.0)),
			(0.5, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 200.0,
    FLAP_G_LIMIT    => 13.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

  ("F-18  ", APPEARANCE_INFO(F18), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00, 11.43, 37.2, 3.52, 0.37, 0.0, 00.0, 26.0,
    05.0, 3.5, 3250.0, FS, 1.65, 0.36, 0.24, 0.60, 0.39, 0.61, 0.28, 0.17, 2,
    F404, 
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.6, ( 0.0,  0.0,  0.0)),
			(0.6, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 200.0,
    FLAP_G_LIMIT    => 13.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

--("F-14 f", APPEARANCE_INFO(F14A), VIEW_LOCATION_UNSPECIFIED,
--  AF_DFLT, 0.00, 19.54, 53.3, 7.16, 0.31, 0.0, 00.0, 20.0,
--  10.0, 5.0, 4787.0, FS, 1.88, 0.67, 0.24, 1.26, 0.45, 0.58, 0.31, 0.18, 2,
--  TF30, ...),

--("F-14 b", APPEARANCE_INFO(F14B), VIEW_LOCATION_UNSPECIFIED,
--  AF_DFLT, 0.00, 11.72, 66.3, 2.07, 0.25, 0.0, 00.0, 68.0,
--  10.0, 5.0, 3847.0, FS, 0.91, 0.54, 0.19, 0.49, 0.17, 1.19, 0.31, 0.37, 2,
--  TF30, ...),

  ("F-5   ", APPEARANCE_INFO(F5), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00,  7.70, 15.8, 3.75, 0.20, 0.0, 00.0, 31.0,
    04.8, 4.8, 3989.0, FS, 1.65, 0.32, 0.18, 0.52, 0.29, 0.87, 0.20, 0.17, 2,
    ADOUR,  --engine guess
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.6, ( 0.0,  0.0,  0.0)),
			(0.6, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 200.0,
    FLAP_G_LIMIT    => 13.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

  ("F-4   ", APPEARANCE_INFO(F4), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00, 11.80, 49.3, 2.82, 0.17, 0.0, 00.0, 51.0,
    06.4, 3.0, 3545.0, FS, 1.52, 0.17, 0.15, 0.25, 0.22, 1.05, 0.13, 0.13, 2,
    J79,
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 9.6,
    EXTRA_RADII     => ((2.5, ( 2.5,  0.0, -0.3)),  -- wing and engine
			(2.5, (-2.5,  0.0, -0.3)),  -- other wing
			(0.8, ( 0.0, -7.0,  0.0)),  -- tail
			(0.9, ( 0.0,  5.0,  0.2)),  -- cockpit etc
			(0.4, ( 0.0,  9.0, -0.4))), -- nose
    CAS_FLAP_LIMIT  => 140.0,
    FLAP_G_LIMIT    => 3.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0 -- use default calc
    ),

--("F-104 ", APPEARANCE_INFO(F104), VIEW_LOCATION_UNSPECIFIED,
--  AF_DFLT, 0.00,  6.68, 18.2, 2.45, 0.38, 0.0, 00.0, 27.0,
--  03.4, 3.4, 4856.0, FS, 1.98, 0.25, 0.23, 0.49, 0.46, 1.39, 0.21, 0.30, 1,
--  J79, ...),

--("Mirage", APPEARANCE_INFO(MIRAGE), VIEW_LOCATION_UNSPECIFIED,
--  AF_DFLT, 0.00,  8.22, 34.8, 1.94, 0.06, 0.0, 00.0, 60.0,
--  04.5, 3.5, 2220.0, FS, 0.00, 0.00, 0.00, 0.00, 0.00, 0.97, 0.14, 0.14, 1,
--  M53_5, ...),

  ("MiG-21", APPEARANCE_INFO(MiG21), VIEW_LOCATION_UNSPECIFIED,
    AF_MiG21, 0.00,  7.15, 23.2, 2.20, 0.08, 0.0, 00.0, 57.0,
    04.3, 4.3, 3463.0, FS, 1.35, 0.31, 0.19, 0.41, 0.26, 1.22, 0.21, 0.26, 1,
    TUMANSKY_R11F, --engine for MiG-21F Fishbed-C
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 6.5,
    EXTRA_RADII     => ((1.0, ( 1.5, -1.0,  0.0)),  -- wing
			(1.0, (-1.5, -1.0,  0.0)),  -- other wing
			(0.6, ( 0.0, -4.0,  0.0)),  -- tail
			(0.8, ( 0.0,  2.5,  0.0)),  -- cockpit area
			(0.6, ( 0.0,  5.5,  0.0))), -- nose
    CAS_FLAP_LIMIT  => 200.0,
    FLAP_G_LIMIT    => 13.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

  ("Jaguar", APPEARANCE_INFO(JAGUAR), VIEW_LOCATION_UNSPECIFIED,
    AF_DFLT, 0.00,  8.49, 24.0, 3.00, 0.23, 0.0, 00.0, 48.0,
    06.0, 5.0, 4263.0, FS, 1.68, 0.33, 0.26, 0.55, 0.44, 1.35, 0.17, 0.23, 2,
    ADOUR,
    POSITIVE_G_LIMIT => 13.0, NEGATIVE_G_LIMIT => -3.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.8, ( 0.0,  0.0,  0.0)),
			(0.6, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 0.0,
    FLAP_G_LIMIT    => 3.0,

    CAS_AEROLON_LIMIT => 340.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(18.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 50.0,

    MAX_FUEL_MASS   => 0.0  -- use default calc
    ),

--("MiG-25", APPEARANCE_INFO(MiG25), VIEW_LOCATION_UNSPECIFIED,
--  AF_DFLT, 0.00, 14.00, 56.0, 3.07, 0.32, 0.0, 00.0, 40.0,
--  03.5, 3.5, 4764.0, FS, 1.29, 0.34, 0.18, 0.44, 0.23, 0.88, 0.21, 0.19, 2,
--  F100, ...),--engine guess

    -- added by me, mostly trash guesses
    --
    -- although "Quest for Performance" NASA SP-468 gives the following
    -- Cd.min values
    --
    -- FkDVII	    0.0404	    Albatross DIII  0.046
    -- EIII	    0.077	    FkDr1	    0.03
    -- DH2	    0.043	    P-38	    0.027
    -- Nieuport 17  0.049	    P-51	    0.016
    --
-- Fokker D.VII max speed 189 km/hr at 3000 ft, weight 900 kg, wing area 20 m2
  ("FkDVII", APPEARANCE_INFO(FkDVII), VIEW_LOCATION_UNSPECIFIED,
    AF_C, 0.03, 9.00, 20.0, 2.25, 1.00, 2.0, 15.0, 00.0,
    10.0,10.0,  450.0,  N, 1.00, 0.10, 0.10, 1.00, 0.50, 0.60, 0.30, 0.20, 1,
    WW1_A1,
    POSITIVE_G_LIMIT => 6.0, NEGATIVE_G_LIMIT => -2.0,
    RADIUS	    => 4.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),
			(1.0, (-3.0,  0.0,  0.0)),
			(0.5, ( 0.0,  0.0,  0.0)),
			(0.5, ( 0.0, -3.0,  0.0)),
			(0.0, ( 0.0,  0.0,  0.0))),
    CAS_FLAP_LIMIT  => 0.0,
    FLAP_G_LIMIT    => 6.0,

    CAS_AEROLON_LIMIT => 50.0,
    AEROLON_PER_WING_AREA => 1.0,
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(30.0),
    CAS_PITCH_LIMIT => 340.0,
    ROTATE_SPEED    => 25.0,

    MAX_FUEL_MASS   => 40.0	-- bad guess
    ),


    (
    NAME		=> "Spit  ",
    APPEARANCE_INFO	=> APPEARANCE_INFO(Spit),
    VIEW_LOCATION	=> VIEW_LOCATION_UNSPECIFIED,

    AEROFOIL_NAME	=> AF_Spit,
    EXTRA_ZERO_DRAG	=> 0.000,
    SPAN		=> METRES(37.0*METRES_PER_FOOT),
    WING_AREA		=> METRES_SQUARED(244.0*(METRES_PER_FOOT**2)),
    ASPECT_RATIO	=> 3.0,
    TAPER_RATIO 	=> 0.5,
    DIHEDRAL		=> 2.0,     -- guess
    WING_PLACEMENT	=> 0.0,     -- guess
    SWEEPBACK_LEAD	=> 0.0,
    THICKNESS_INNER|
    THICKNESS_OUTER	=> 0.0,
    WING_LOADING	=> NEWTONS_PER_METRE_SQUARED((9400.0/244.0)*NPM2_PER_LBPFT2),
    LIFT_DEVICES	=> N,
    IH_PER_C		=> 2.0,
    SH_PER_S|
    SHN_PER_S		=> 0.0,
    VH|
    VHN 		=> 0.0,
    IV_PER_S		=> 0.0,
    SV_PER_S		=> 0.0,
    Vv			=> 0.0,

    ENGINE_COUNT	=> 1,
    ENGINE		=> GRIFFON,

    POSITIVE_G_LIMIT	=> 9.0,
    NEGATIVE_G_LIMIT	=> -3.0,

    RADIUS		=> 7.0,
    EXTRA_RADII     => ((1.0, ( 3.0,  0.0,  0.0)),  -- right wing
			(1.0, (-3.0,  0.0,  0.0)),  -- left wing
			(0.8, ( 0.0,  1.5,  0.0)),  -- engine
			(1.0, ( 0.0, -6.0,  0.5)),  -- tail
			(1.0, ( 0.0, -1.5,  0.5))), -- pilot

    CAS_FLAP_LIMIT	=> 0.0,
    FLAP_G_LIMIT	=> 3.0,

    CAS_AEROLON_LIMIT	=> 90.0,
    CAS_PITCH_LIMIT	=> 110.0, -- aiming at a 6g max pull
    MAX_ELEVATOR_DEFLECTION => DEGREES_TO_RADIANS*RADIANS(24.0),
    AEROLON_PER_WING_AREA => 1.0,
    ROTATE_SPEED	=> 30.0,

    MAX_FUEL_MASS	=> 200.0    -- Bad guess
    )

);


    function PLANE_DATA return ACCESS_VECTOR_OF_PLANE_DATA_TYPE is
    begin
	return ACCESS_PLANE_DATA;
    end;


begin
    declare
	use APPEARANCE_MANAGER;

	AEROPLANE_APPEARANCE : APPEARANCE_MANAGER.ACCESS_APPEARANCE_TYPE;

	procedure SET_VIEW_LOCATION(
	    PLANE	    : in KNOWN_PLANES;
	    VIEW_LOCATION   : in OBJECT_PHYSICS.POSITION) is

	    PLANE_DATA : VECTOR_OF_PLANE_DATA_TYPE
		renames ACCESS_PLANE_DATA.all;
	begin
	    for I in PLANE_DATA'range loop
		-- HACK, there is no other way of recognising the right place!!!
		if INFO_MANAGER."="(
		    PLANE_DATA(I).APPEARANCE_INFO,
		    APPEARANCE_INFO(PLANE))
		then
		    PLANE_DATA(I).VIEW_LOCATION := VIEW_LOCATION;
		    return;
		end if;
	    end loop;
	end;

	procedure ADD_TRIVIAL_PLANE(
	    CLOSEST_VISIBLE,
	    WW,
	    FD,
	    BK,
	    TB,
	    TF,
	    TT : METRES) is
	begin
	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		CLOSEST_VISIBLE => CLOSEST_VISIBLE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,
		VECTOR_OF_POSITION'(		-- triangle for wing
		( -WW,  BK, 0.0),
		(  WW,  BK, 0.0),
		( 0.0,  FD, 0.0),
		( -WW,  BK, 0.0))
		);
	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		CLOSEST_VISIBLE => CLOSEST_VISIBLE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,	-- triangle for tail
		VECTOR_OF_POSITION'(
		( 0.0, TF, 0.0),
		( 0.0, TB, TT),
		( 0.0, TB, 0.0),
		( 0.0, TF, 0.0))
		);
	end;

	procedure DRAW_AEROPLANE(
	    PLANE		: KNOWN_PLANES;
	    SCALE_FACTOR	: SCALE_TYPE;
	    CI_UNSCALED,
	    CJ_UNSCALED,
	    CK_UNSCALED		: METRES;
	    RIGHT_SIDE_UNSCALED : VECTOR_OF_POSITION;
	    PROFILE_UNSCALED	: VECTOR_OF_POSITION;
	    VIEW_LOCATION_UNSCALED : OBJECT_PHYSICS.POSITION)
	is
	    SF : SCALE_TYPE renames SCALE_FACTOR;

	    CI : constant METRES := CI_UNSCALED*SF;
	    CJ : constant METRES := CJ_UNSCALED*SF;
	    CK : constant METRES := CK_UNSCALED*SF;

	    RIGHT_SIDE		: VECTOR_OF_POSITION(RIGHT_SIDE_UNSCALED'range);
	    LEFT_SIDE		: VECTOR_OF_POSITION(RIGHT_SIDE_UNSCALED'range);
	    PROFILE		: VECTOR_OF_POSITION(PROFILE_UNSCALED'range);
	    VIEW_LOCATION	: OBJECT_PHYSICS.POSITION;

	    MAX_I, MAX_J, MAX_K : METRES := 0.0;
	    MIN_I, MIN_J, MIN_K : METRES := 0.0;

	    procedure SCALE_AND_TRANSLATE(
		V1 : in OBJECT_PHYSICS.POSITION;
		V2 : out OBJECT_PHYSICS.POSITION) is

		procedure P(
		    UNSCALED	: in METRES;
		    BIAS	: in METRES;
		    SCALED	: out METRES;
		    MAX 	: in out METRES;
		    MIN 	: in out METRES) is

		    S : constant METRES := UNSCALED * SF - BIAS;
		begin
		    SCALED := S;
		    if S > MAX then MAX := S; end if;
		    if S < MIN then MIN := S; end if;
		end;
	    begin
		P(V1.I, CI, V2.I, MAX_I, MIN_I);
		P(V1.J, CJ, V2.J, MAX_J, MIN_J);
		P(V1.K, CK, V2.K, MAX_K, MIN_K);
	    end;

	    procedure SCALE_AND_TRANSLATE(
		V1 : in VECTOR_OF_POSITION;
		V2 : out VECTOR_OF_POSITION) is
	    begin
		for N in V1'range loop
		    SCALE_AND_TRANSLATE(V1(N), V2(N));
		end loop;
	    end;

	begin
	    AEROPLANE_APPEARANCE :=
		APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
		    APPEARANCE_INFO(PLANE));

	    SCALE_AND_TRANSLATE(RIGHT_SIDE_UNSCALED,RIGHT_SIDE);
	    SCALE_AND_TRANSLATE(PROFILE_UNSCALED,PROFILE);

	    ADD_TRIVIAL_PLANE(LOD_DISTANCE,
		MAX_I,    -- wing width
		MAX_J,    -- forward
		MIN_J,    -- backward
		MIN_J,    -- back of tail
		MIN_J,    -- front of tail
		MAX_K);   -- up

	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);

	    ADD_POINTS(AEROPLANE_APPEARANCE, PROFILE);

	    -- Make the left side the mirror of the right side, but in reverse
	    --
	    declare
		Li : INTEGER := LEFT_SIDE'last;
	    begin
		for Ri in RIGHT_SIDE'range loop
		    LEFT_SIDE(Li) := RIGHT_SIDE(Ri);
		    LEFT_SIDE(Li).I := -LEFT_SIDE(Li).I;
		    Li := Li - 1;
		end loop;
	    end;

	    -- do the two sides
	    --
	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);

	    ADD_POINTS(AEROPLANE_APPEARANCE,
		RIGHT_SIDE &
		LEFT_SIDE(LEFT_SIDE'first+1..LEFT_SIDE'last)
		);

	    FINISH_CREATING_SUBPICTURES(AEROPLANE_APPEARANCE);

	    SCALE_AND_TRANSLATE(VIEW_LOCATION_UNSCALED, VIEW_LOCATION);
	    SET_VIEW_LOCATION(PLANE, VIEW_LOCATION);
	end;
    
    begin

	-- Default
	--
	AEROPLANE_APPEARANCE :=
	    APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
		DEFAULT_APPEARANCE_INFO);

	ADD_TRIVIAL_PLANE(0.0,
	    +3.0,    -- wing width
	    +3.0,    -- forward
	    -3.0,    -- backward
	    -4.0,    -- back of tail
	    -2.0,    -- front of tail
	    +2.0);   -- up

        FINISH_CREATING_SUBPICTURES(AEROPLANE_APPEARANCE);

	-- FkDVII biplane (bad rendition)
	--
	AEROPLANE_APPEARANCE :=
	    APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
		APPEARANCE_INFO(FkDVII));

        ADD_POINTS(AEROPLANE_APPEARANCE,
	    VECTOR_OF_POSITION'(		-- top wing
	    (  3.0,  0.5, 1.5),
	    ( -3.0,  0.5, 1.5),
	    ( -3.0, -0.5, 1.5),
	    (  3.0, -0.5, 1.5),
	    (  3.0,  0.5, 1.5))
	    );
	SET_VIEW_LOCATION(F4, (0.0,0.0,1.3));

        ADD_SUBPICTURE(AEROPLANE_APPEARANCE);
        ADD_POINTS(AEROPLANE_APPEARANCE,
	    VECTOR_OF_POSITION'(		-- bottom wing
	    (  2.5,  0.3, 0.0),
	    ( -2.5,  0.7, 0.0),
	    ( -2.5, -0.7, 0.0),
	    (  2.5, -0.7, 0.0),
	    (  2.5,  0.3, 0.0))
	    );
        ADD_SUBPICTURE(AEROPLANE_APPEARANCE);
        ADD_POINTS(AEROPLANE_APPEARANCE,
	    VECTOR_OF_POSITION'(		-- sides of body and elevator
	    ( -0.5,  2.0, 0.5),
	    ( -0.25,-3.0, 0.5),
	    ( -1.25,-4.0, 0.5),
	    (  1.25,-4.0, 0.5),
	    (  0.25,-3.0, 0.5),
	    (  0.5,  2.0, 0.5),
	    ( -0.5,  2.0, 0.5))
	    );
        ADD_SUBPICTURE(AEROPLANE_APPEARANCE);
        ADD_POINTS(AEROPLANE_APPEARANCE,
	    VECTOR_OF_POSITION'(		-- top/bottom of body and tail
	    ( 0.0,  2.0, 1.0),
	    ( 0.0,  2.0, 0.0),
	    ( 0.0, -4.0, 0.0),
	    ( 0.0, -4.0, 1.0),
	    ( 0.0, -3.0, 0.2),
	    ( 0.0,  2.0, 1.0))
	    );
        FINISH_CREATING_SUBPICTURES(AEROPLANE_APPEARANCE);

	-- MiG21
	--
	AEROPLANE_APPEARANCE :=
	    APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
		APPEARANCE_INFO(MiG21));

	declare
	    CI : constant METRES := 0.0;
	    CJ : constant METRES := 6.5;
	    CK : constant METRES := 1.0;

	    RIGHT_SIDE : constant VECTOR_OF_POSITION
		:=
			VECTOR_OF_POSITION'(		-- left and right wings
			(0.53-CI,13.1-CJ,0.6-CK),	-- nose
			(0.53-CI,9.1-CJ,1.0-CK), 	-- lead edge of wing
			(3.6-CI, 4.0-CJ,0.6-CK),	-- tip
			(0.53-CI, 4.0-CJ,1.0-CK),	-- trail edge of wing
			(0.53-CI, 2.8-CJ,1.0-CK),	-- lead of tail
			(1.9-CI, 0.8-CJ,1.0-CK),	-- lead tip
			(1.9-CI, 0.0-CJ,1.0-CK),	-- trail tip
			(0.53-CI,0.8-CJ,1.0-CK));	-- trail edge of tail

	    LEFT_SIDE : VECTOR_OF_POSITION(RIGHT_SIDE'range);

	begin
	    ADD_TRIVIAL_PLANE(LOD_DISTANCE,
		3.6-CI,     -- wing width
		13.1-CJ,    -- forward
		0.8-CJ,     -- backward
		-0.2-CJ,    -- back of tail
		4.4-CJ,     -- front of tail
		3.0-CK);    -- up

	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);

	    ADD_POINTS(AEROPLANE_APPEARANCE,
		VECTOR_OF_POSITION'(		-- body, tail
		( 0.0-CI,13.1-CJ,1.2-CK),		--  top nose
		( 0.0-CI,10.3-CJ,1.5-CK),		--  top of canopy
		( 0.0-CI, 4.4-CJ,1.5-CK),		--  leading base of tail
		( 0.0-CI, 1.2-CJ,3.0-CK),		--  leading top of tail
		( 0.0-CI,-0.2-CJ,3.0-CK),		--  trailing top of tail
		( 0.0-CI, 0.8-CJ,1.5-CK),		--  trailing base of tail
		( 0.0-CI, 0.8-CJ,0.5-CK),		--  bottom of jet
		( 0.0-CI, 1.1-CJ,0.4-CK),		--  underfin rear base
		( 0.0-CI, 1.1-CJ,0.25-CK),		--  underfin rear low
		( 0.0-CI, 4.0-CJ,-0.25-CK),		--  underfin point
		( 0.0-CI, 4.3-CJ,0.0-CK),		--  underfin front base
		( 0.0-CI,13.1-CJ,0.0-CK),		--  bottom of nose
		( 0.0-CI,13.1-CJ,1.2-CK))		--  top nose
		);
	    SET_VIEW_LOCATION(MiG21, (0.0-CI,9.8-CJ,1.3-CK));

	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,
		VECTOR_OF_POSITION'(		-- spike
		 (0.0-CI,13.1-CJ,1.0-CK),		-- top base
		 (0.0-CI,14.3-CJ,0.6-CK),		-- tip
		 (0.0-CI,13.1-CJ,0.2-CK)) 		-- bottom base
		);

	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,
		VECTOR_OF_POSITION'(		-- spike
		 (0.37-CI,13.1-CJ,0.6-CK),		-- right base
		 (0.0-CI,14.3-CJ,0.6-CK),		-- tip
		 (-0.37-CI,13.1-CJ,0.6-CK)) 		-- left base
		);

	    -- Make the left side the mirror of the right side, but in reverse
	    --
	    declare
		Li : INTEGER := LEFT_SIDE'last;
	    begin
		for Ri in RIGHT_SIDE'range loop
		    LEFT_SIDE(Li) := RIGHT_SIDE(Ri);
		    LEFT_SIDE(Li).I := -LEFT_SIDE(Li).I;
		    Li := Li - 1;
		end loop;
	    end;

	    -- do the two sides
	    --
	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,
		RIGHT_SIDE & LEFT_SIDE & RIGHT_SIDE(RIGHT_SIDE'first)
		);

	    FINISH_CREATING_SUBPICTURES(AEROPLANE_APPEARANCE);
	end;


	-- F-4
	--
	AEROPLANE_APPEARANCE :=
	    APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
		APPEARANCE_INFO(F4));

	declare
	    CI : constant METRES := 0.0;
	    CJ : constant METRES := 8.0;
	    CK : constant METRES := 1.0;
	begin
	    ADD_TRIVIAL_PLANE(LOD_DISTANCE,
		5.9-CI,     -- wing width
		18.3-CJ,    -- forward
		0.0-CJ,     -- backward
		0.0-CJ,     -- back of tail
		5.3-CJ,     -- front of tail
		4.5-CK);    -- up

	    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
		FURTHEREST_VISIBLE => LOD_DISTANCE);
	    ADD_POINTS(AEROPLANE_APPEARANCE,
		VECTOR_OF_POSITION'(		-- body, tail
		( 0.0-CI,13.6-CJ,0.0-CK),		--  under chin
		( 0.0-CI,18.3-CJ,0.6-CK),		--  nose
		( 0.0-CI,13.6-CJ,2.4-CK),		--  top of canopy
		( 0.0-CI, 5.3-CJ,2.4-CK),		--  leading base of tail
		( 0.0-CI, 2.0-CJ,4.5-CK),		--  leading top of tail
		( 0.0-CI, 0.0-CJ,4.5-CK),		--  trailing top of tail
		( 0.0-CI, 0.0-CJ,1.8-CK),		--  trailing base of tail
		( 0.0-CI, 0.6-CJ,1.8-CK),		--  trailing middle of hori tail
		( 1.7-CI, 0.3-CJ,0.9-CK),		--  right trail ht tip
		( 1.7-CI, 0.6-CJ,0.9-CK),		--  right lead ht tip
		( 0.0-CI, 3.0-CJ,1.8-CK),		--  leading center of ht
		(-1.7-CI, 0.6-CJ,0.9-CK),		--  left lead ht tip
		(-1.7-CI, 0.3-CJ,0.9-CK),		--  left trail ht tip
		( 0.0-CI, 0.6-CJ,1.8-CK))		--  trailing middle of hori tail
		);

	    SET_VIEW_LOCATION(F4, (0.0-CI,13.1-CJ,2.2-CK));

	    -- engine and wing, symmetric hence the loop
	    --
	    for I in 0..1 loop
		declare
		    LR : constant SCALE_TYPE := SCALE_TYPE(I*2-1);
		begin
		    ADD_SUBPICTURE(AEROPLANE_APPEARANCE,
			FURTHEREST_VISIBLE => LOD_DISTANCE);
		    ADD_POINTS(AEROPLANE_APPEARANCE,
			VECTOR_OF_POSITION'(		-- left and right wings
			(1.5*LR-CI,13.6-CJ,0.0-CK),
			(1.5*LR-CI,13.6-CJ,1.5-CK),
			(1.0*LR-CI, 4.1-CJ,1.3-CK),
			(1.0*LR-CI, 4.1-CJ,0.2-CK),
			(1.0*LR-CI, 5.9-CJ,0.0-CK),
			(5.9*LR-CI, 4.7-CJ,1.1-CK),
			(5.9*LR-CI, 6.2-CJ,1.1-CK),
			(1.5*LR-CI,11.8-CJ,0.0-CK),
			(1.5*LR-CI,13.6-CJ,0.0-CK))
			);
		end;
	    end loop;

	    FINISH_CREATING_SUBPICTURES(AEROPLANE_APPEARANCE);
	end;

	-- Spitfire
	--
	DRAW_AEROPLANE(
	    PLANE	    => Spit,
	    SCALE_FACTOR    => SCALE_TYPE(36.0*METRES_PER_FOOT/3.5),
	    CI_UNSCALED	    => 0.0,
	    CJ_UNSCALED	    => 2.5,
	    CK_UNSCALED	    => 0.125,
	    RIGHT_SIDE_UNSCALED =>
			    -- partial picture
			    --
		VECTOR_OF_POSITION'(
			(0.15, 3.25, 0.37),	-- side of nose
			(0.15, 2.87, 0.12),	-- f wing root
			(1.75, 2.80, 0.25),	-- f wing tip
			(2.12, 2.50, 0.25),	-- wing tip
			(1.50, 2.00, 0.23),	-- b wing tip
			(0.15, 1.87, 0.12),	-- b wing root
			(0.08, 0.62, 0.56),	-- f hori stab
			(0.32, 0.62, 0.56),	-- f hori s tip
			(0.50, 0.44, 0.56),	-- f hori s tip
			(0.50, 0.25, 0.56),	-- b hori s tip
			(0.08, 0.25, 0.56),	-- b hori s "^"
			(0.00, 0.30, 0.56)),	-- b hori stab

	    PROFILE_UNSCALED =>
		VECTOR_OF_POSITION'(
			(0.00, 3.50, 0.40),	-- tip of prop <
			(0.00, 3.40, 0.50),	-- f of cowling
			(0.00, 2.50, 0.50),	-- f of canopy
			(0.00, 2.32, 0.68),	-- f t of canopy
			(0.00, 2.18, 0.68),	-- b t of canopy
			(0.00, 2.00, 0.50),	-- b b of canopy
			(0.00, 0.56, 0.50),	-- b of tail
			(0.00, 0.25, 1.00),	-- t of tail
			(0.00, 0.00, 0.37),	-- b of tail
			(0.00, 1.88, 0.00),	-- b of wing
			(0.00, 3.20, 0.00),	-- f of wing
			(0.00, 3.40, 0.30),	-- bottom of prop <
			(0.00, 3.50, 0.40)),	-- tip of prop <
	    VIEW_LOCATION_UNSCALED =>
			(0.00, 2.25, 0.60)
	);

	-- F-15
	--
	DRAW_AEROPLANE(
	    PLANE	    => F15,
	    SCALE_FACTOR    => SCALE_TYPE(13.04/10.5),
	    CI_UNSCALED	    => 0.0,
	    CJ_UNSCALED	    => 5.0,
	    CK_UNSCALED	    => 1.0,
	    RIGHT_SIDE_UNSCALED =>
			    -- partial picture
			    --
		VECTOR_OF_POSITION'(
			(00.0, 05.0, 01.7),	-- on back
			(00.8, 11.2, 01.7),	-- top inner front of engine
			(01.3, 11.2, 01.7),	-- top outer front of engine
			(01.8, 07.5, 01.5),	-- front wing root
			(05.2, 04.0, 01.5),	-- front wing tip
			(04.5, 03.0, 01.5),	-- rear wing tip
			(03.0, 03.5, 01.5),	-- bend in trailing wing edge
			(01.8, 03.5, 01.5),	-- rear wing root
			(01.8, 02.8, 01.0),	-- front hori stab root
			(03.2, 00.8, 01.0),	-- front hori stab tip
			(03.0, 00.0, 01.0),	-- rear hori stab tip
			(01.8, 00.3, 01.0),	-- rear hori stab root
			(01.8, 01.0, 01.0),	-- rear root vert stab
			(01.8, 01.0, 03.8),	-- rear tip vert stab
			(01.8, 01.9, 03.8),     -- front tip vert stab
			(01.8, 02.5, 01.5),	-- front root vert stab
			(01.3, 11.2, 01.7),	-- top outer front of engine
			(01.3, 08.5, 00.6),     -- bottom of inlet
			(01.3, 01.0, 00.6),	-- bottom of outlet
			(00.0, 01.0, 00.6)),    -- outlets middle

	    PROFILE_UNSCALED =>
		VECTOR_OF_POSITION'(
			(0.00, 01.0, 00.6),	-- bottom of outlets
			(0.00, 12.0, 00.6),	-- base of nose start
			(0.00, 14.5, 01.0),	-- nose
			(0.00, 11.5, 02.0),	-- base of canopy
			(0.00, 10.5, 02.5),	-- f t of canopy
			(0.00, 09.0, 02.4),	-- b t of canopy
			(0.00, 08.0, 02.0),	-- b b of canopy
			(0.00, 03.0, 01.5)),	-- between tails
	    VIEW_LOCATION_UNSCALED =>
			(0.00, 09.8, 01.2)
	);

	-- F-16
	--
	DRAW_AEROPLANE(
	    PLANE	    => F16,
	    SCALE_FACTOR    => SCALE_TYPE(9.14/09.0),
	    CI_UNSCALED	    => 0.0,
	    CJ_UNSCALED	    => 5.5,
	    CK_UNSCALED	    => 1.0,
	    RIGHT_SIDE_UNSCALED =>
			    -- partial picture
			    --
		VECTOR_OF_POSITION'(
			(00.0, 14.0, 01.0),	-- nose
			(00.3, 13.5, 01.0),	-- side of nose
			(01.2, 07.8, 01.0),	-- front wing root
			(04.5, 05.0, 01.0),	-- front wing tip
			(04.5, 04.0, 01.0),	-- rear wing tip
			(00.9, 04.0, 01.0),	-- rear wing root
			(00.9, 03.0, 01.0),	-- front hori stab root
			(02.5, 01.5, 01.0),	-- front hori stab tip
			(02.5, 01.0, 01.0),	-- rear hori stab tip
			(01.0, 01.0, 01.0),	-- rear hori stab trailing edge
			(00.7, 02.0, 01.0),     -- rear hori stab root
			(00.4, 00.5, 01.0),	-- engine end
			(00.0, 00.5, 01.0)),	-- engine end middle

	    PROFILE_UNSCALED =>
		VECTOR_OF_POSITION'(
			(0.00, 14.0, 01.0),	-- nose
			(0.00, 13.5, 01.3),	-- f of cowling, upper
			(0.00, 12.0, 01.6),	-- f of canopy
			(0.00, 11.5, 02.0),	-- f t of canopy
			(0.00, 10.0, 02.0),	-- b t of canopy
			(0.00, 08.0, 01.7),	-- b b of canopy
			(0.00, 05.0, 01.7),	-- tail "chine"
			(0.00, 04.0, 02.0),	-- root of tail
			(0.00, 02.0, 03.5),	-- forward tip of tail
			(0.00, 01.2, 03.5),	-- rear tip of tail
			(0.00, 02.0, 01.5),	-- rear root of tail
			(0.00, 02.7, 01.2),	-- rear root of tail
			(0.00, 01.0, 01.2),	-- top of engine end
			(0.00, 01.0, 00.6),	-- bottom of engine end
			(0.00, 09.0, 00.0),	-- front of engine, lower
			(0.00, 09.0, 01.0),     -- front of engine, upper
			(0.00, 13.5, 00.7),	-- f of cowling, lower
			(0.00, 14.0, 01.0)),	-- nose
	    VIEW_LOCATION_UNSCALED =>
			(0.00, 10.7, 01.9)
	);

	-- F-18
	--
	DRAW_AEROPLANE(
	    PLANE	    => F18,
	    SCALE_FACTOR    => SCALE_TYPE(11.43/09.0),
	    CI_UNSCALED	    => 0.0,
	    CJ_UNSCALED	    => 4.5,
	    CK_UNSCALED	    => 1.0,
	    RIGHT_SIDE_UNSCALED =>
			    -- partial picture
			    --
		VECTOR_OF_POSITION'(
			(00.0, 03.0, 01.5),	-- between tails
			(00.5, 10.0, 01.0),	-- front of chine
			(01.1, 06.1, 01.0),	-- front wing root
			(04.5, 04.5, 01.0),	-- front wing tip
			(04.5, 03.5, 01.0),	-- rear wing tip
			(00.7, 03.0, 01.0),	-- rear wing root
			(00.7, 02.2, 00.8),	-- front hori stab root
			(02.5, 00.9, 00.8),	-- front hori stab tip
			(02.5, 00.0, 00.8),	-- rear hori stab tip
			(00.7, 00.9, 00.8),	-- rear hori stab root
			(00.5, 00.4, 00.8),	-- outlet behind hori stab
			(00.4, 00.4, 01.2),	-- outlet behind vert stab
			(00.4, 01.8, 01.5),	-- rear root vert stab
			(01.4, 01.2, 03.0),	-- rear tip vert stab
			(01.4, 02.2, 03.0),     -- front tip vert stab
			(00.4, 03.7, 01.5),	-- front root vert stab
			(01.0, 06.0, 01.0),	-- top of inlet
			(01.0, 06.0, 00.5),     -- bottom of inlet
			(00.5, 00.4, 00.6),	-- bottom of outlet
			(00.0, 00.4, 00.6)),    -- outlets middle

	    PROFILE_UNSCALED =>
		VECTOR_OF_POSITION'(
			(0.00, 00.7, 00.6),	-- bottom of outlets
			(0.00, 06.0, 00.5),	-- between inlets
			(0.00, 09.0, 00.5),	-- under chine
			(0.00, 13.0, 01.0),	-- nose
			(0.00, 10.0, 01.5),	-- base of canopy
			(0.00, 09.0, 02.0),	-- f t of canopy
			(0.00, 08.0, 01.7),	-- b b of canopy
			(0.00, 03.0, 01.5)),	-- between tails
	    VIEW_LOCATION_UNSCALED =>
			(0.00, 09.0, 01.8)
	);
    end;
end;
