--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with SYSTEM_INTERFACE;
package body TRIG is


    NUMBER_OF_ENTRIES_IN_ASIN_TABLE 
		    : constant := 1024;
    ASIN_TABLE_LAST : constant := (NUMBER_OF_ENTRIES_IN_ASIN_TABLE*1.0)/
				  1.0;

    type ASIN_SCALE_TYPE is digits 4 range 0.0 .. ASIN_TABLE_LAST;

    type ASIN_INDEX is range 0..NUMBER_OF_ENTRIES_IN_ASIN_TABLE;
    ASIN_TABLE : array(ASIN_INDEX) of RADIANS;


    RADIANS_PER_RIGHT_ANGLE_TIMES_TWO : constant := RADIANS_PER_RIGHT_ANGLE*2.0;

    NUMBER_OF_ENTRIES_IN_SIN_TABLE 
		    : constant := 1024;
    SIN_TABLE_LAST  : constant := (NUMBER_OF_ENTRIES_IN_SIN_TABLE*1.0)/
				  RADIANS_PER_RIGHT_ANGLE_TIMES_TWO;

    type SIN_SCALE_TYPE is digits 4 range 0.0 .. SIN_TABLE_LAST;

    type SIN_INDEX is range 0..NUMBER_OF_ENTRIES_IN_SIN_TABLE;
    SIN_TABLE : array(SIN_INDEX) of SYSTEM_INTERFACE.FAST_FLOAT;


    package body TRIG_FUNCTIONS is

	function "*"(LEFT : SOME_FLOAT; RIGHT : RADIANS) return RADIANS is
	begin
	    return RADIANS(LEFT)*RIGHT;
	end;

	function "*"(LEFT : RADIANS; RIGHT : SOME_FLOAT) return RADIANS is
	begin
	    return LEFT*RADIANS(RIGHT);
	end;

	function "/"(LEFT : RADIANS; RIGHT : SOME_FLOAT) return RADIANS is
	begin
	    return LEFT/RADIANS(RIGHT);
	end;

	function HARD_SIN(R : RADIANS) return SOME_FLOAT is
	    M   : SOME_FLOAT := 1.0;
	    SIN : SYSTEM_INTERFACE.FAST_FLOAT; 
	    X   : RADIANS    := R;
	begin

	    if X < 0.0 then M := -M; X := -X; end if;

	    while X >= RADIANS_PER_RIGHT_ANGLE_TIMES_TWO loop
		X := X - RADIANS_PER_RIGHT_ANGLE_TIMES_TWO;
		M := -M;
	    end loop;

	    SIN := SIN_TABLE(SIN_INDEX(SIN_SCALE_TYPE(X)*
					SIN_SCALE_TYPE(SIN_TABLE_LAST)));

	    return M*SOME_FLOAT(SIN);
	end;

	function SIN(R : RADIANS) return SOME_FLOAT is
	    S : constant SOME_FLOAT := SOME_FLOAT(R);
	begin
	    if abs(S) < 10.0**(-RADIANS'digits/3)*3.0 then
		return S;
	    end if;
	    return HARD_SIN(R);
	end;

	function COS(R : RADIANS) return SOME_FLOAT is
	begin
	    return SIN(R + RADIANS(RADIANS_PER_RIGHT_ANGLE));
	end;

	function TAN(R : RADIANS) return SOME_FLOAT is
	begin
	    return SIN(R)/COS(R);
	end;

	function HARD_ASIN_ACOS(SIN,COS : SOME_FLOAT) return RADIANS is
	    R : RADIANS;
	begin
	    if abs(SIN) >= abs(COS) then
		R := ACOS(COS);
		if SIN < 0.0 then
		    R := -R;
		end if;
	    else
		R := ASIN(SIN);
		if COS < 0.0 then
		    if SIN < 0.0 then
			R := RADIANS(-2.0*RADIANS_PER_RIGHT_ANGLE) - R;
		    else
			R := RADIANS(+2.0*RADIANS_PER_RIGHT_ANGLE) - R;
		    end if;
		end if;
	    end if;
	    return R;
	end;

	function ASIN_ACOS(SIN,COS : SOME_FLOAT) return RADIANS is
	begin
	    if abs(SIN) < 10.0**(-RADIANS'digits/3)*3.0 then
		if COS > 0.0 then
		    return RADIANS(SIN);
		end if;
	    end if;
	    return HARD_ASIN_ACOS(SIN, COS);
	end;

	function ASIN(SIN : SOME_FLOAT) return RADIANS is
	    X : SOME_FLOAT := abs(SIN);
	    R : RADIANS;
	begin
	    R := ASIN_TABLE(ASIN_INDEX( ASIN_SCALE_TYPE(X)*
					ASIN_SCALE_TYPE(ASIN_TABLE_LAST)));
	    if SIN >= 0.0 then
		return R;
	    else
		return -R;
	    end if;
	end;

	function ACOS(COS : SOME_FLOAT) return RADIANS is
	    X : SOME_FLOAT := abs(COS);
	    R : RADIANS;
	begin
	    R := ASIN_TABLE(ASIN_INDEX( ASIN_SCALE_TYPE(X)*
					ASIN_SCALE_TYPE(ASIN_TABLE_LAST)));
	    if COS > 0.0 then
		return RADIANS_PER_RIGHT_ANGLE-R;
	    else
		return RADIANS_PER_RIGHT_ANGLE+R;
	    end if;
	end;

--	function ATAN(TAN : SOME_FLOAT) return ANGLES is
--	begin
--	    This is trickier to do efficiently and may not be needed
--	end;

    end TRIG_FUNCTIONS;


begin

    for I in ASIN_INDEX loop
	ASIN_TABLE(I) :=
	    RADIANS(SYSTEM_INTERFACE.ASIN(
			SYSTEM_INTERFACE."/"(
			    SYSTEM_INTERFACE.FAST_FLOAT(I),
			    ASIN_TABLE_LAST)));
    end loop;

    for I in SIN_INDEX loop
	SIN_TABLE(I) :=
	    (SYSTEM_INTERFACE.SIN(
		SYSTEM_INTERFACE."/"(
		    SYSTEM_INTERFACE.FAST_FLOAT(I),
		    SIN_TABLE_LAST)));
    end loop;

end;
